/*
Copyright (c) 2000-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	dkslsupp.c	Syslog support module.
*/



#include "dk.h"
#include "dkstr.h"


/**	Inside the dkslsupp module.
*/
#define DK_SLSUPP_C 1

#include "dkslsupp.h"



#if DK_HAVE_SYSLOG_H
#include <syslog.h>
#endif
#if DK_HAVE_STRING_H
#include <string.h>
#endif
#if DK_HAVE_STRINGS_H
#include <strings.h>
#endif



#ifndef LOG_EMERG
/**	Log priority: emerg (emergency).
*/
#define LOG_EMERG 0
#endif
#ifndef LOG_ALERT
/**	Log priority: alert.
*/
#define LOG_ALERT 1
#endif
#ifndef LOG_CRIT
/**	Log priority: crit (critical error).
*/
#define LOG_CRIT  2
#endif
#ifndef LOG_ERR
/**	Log priority: err (error).
*/
#define LOG_ERR   3
#endif
#ifndef LOG_WARNING
/**	Log priority: warning.
*/
#define LOG_WARNING 4
#endif
#ifndef LOG_NOTICE
/**	Log priority: notice.
*/
#define LOG_NOTICE  5
#endif
#ifndef LOG_INFO
/**	Log priority: info.
*/
#define LOG_INFO 6
#endif
#ifndef LOG_DEBUG
/**	Log priority: debug.
*/
#define LOG_DEBUG 7
#endif

#ifndef LOG_KERN
/**	Log feature: kern (kernel).
*/
#define LOG_KERN (0<<3)
#endif
#ifndef LOG_USER
/**	Log feature: user.
*/
#define LOG_USER (1<<3)
#endif
#ifndef LOG_MAIL
/**	Log feature: mail.
*/
#define LOG_MAIL (2<<3)
#endif
#ifndef LOG_DAEMON
/**	Log feature: daemon.
*/
#define LOG_DAEMON (3<<3)
#endif
#ifndef LOG_AUTH
/**	Log feature: auth.
*/
#define LOG_AUTH (4<<3)
#endif
#ifndef LOG_SYSLOG
/**	Log feature: syslog.
*/
#define LOG_SYSLOG (5<<3)
#endif
#ifndef LOG_LPR
/**	Log feature: lpr.
*/
#define LOG_LPR (6<<3)
#endif
#ifndef LOG_NEWS
/**	Log feature: news.
*/
#define LOG_NEWS (7<<3)
#endif
#ifndef LOG_UUCP
/**	Log feature: uucp.
*/
#define LOG_UUCP (8<<3)
#endif
#ifndef LOG_CRON
/**	Log feature: cron.
*/
#define LOG_CRON (9<<3)
#endif
#ifndef LOG_AUTHPRIV
/**	Log feature: authpriv.
*/
#define LOG_AUTHPRIV (10<<3)
#endif
#ifndef LOG_FTP
/**	Log feature: ftp.
*/
#define LOG_FTP (11<<3)
#endif
#ifndef LOG_LOCAL0
/**	Log feature: local0.
*/
#define LOG_LOCAL0 (16<<3)
#endif
#ifndef LOG_LOCAL1
/**	Log feature: local1.
*/
#define LOG_LOCAL1 (17<<3)
#endif
#ifndef LOG_LOCAL2
/**	Log feature: local2.
*/
#define LOG_LOCAL2 (18<<3)
#endif
#ifndef LOG_LOCAL3
/**	Log feature: local3.
*/
#define LOG_LOCAL3 (19<<3)
#endif
#ifndef LOG_LOCAL4
/**	Log feature: local4.
*/
#define LOG_LOCAL4 (20<<3)
#endif
#ifndef LOG_LOCAL5
/**	Log feature: local5.
*/
#define LOG_LOCAL5 (21<<3)
#endif
#ifndef LOG_LOCAL6
/**	Log feature: local6.
*/
#define LOG_LOCAL6 (22<<3)
#endif
#ifndef LOG_LOCAL7
/**	Log feature: local7.
*/
#define LOG_LOCAL7 (23<<3)
#endif



/**	Assign codes to priority names.
*/
typedef struct _name_and_code_ {
  char *n;	/**< Priority name. */
  int   i;	/**< Priority code. */
} name_and_code;



/**	Get numeric code for a keyword.
	@param	nac	Array listing name to code translations.
	@param	what	Name for code.
	@return	Numeric code for \a what.
*/
static
int
get_code_for DK_P2(name_and_code *, nac, char *, what)
{
  int back = 0;
  int found;
  name_and_code *ptr;
  ptr = nac; found = 0;
  while((ptr->n) && (!found)) {
    if(!dkstr_casecmp(ptr->n, what)) {
      back = ptr->i; found = 1;
    }
    ptr++;
  }
  return back;
}



/**	Syslog priority codes.
*/
static name_and_code priorities[] = {
  { (char *)"alert", LOG_ALERT },
  { (char *)"crit", LOG_CRIT },
  { (char *)"debug", LOG_DEBUG },
  { (char *)"emerg", LOG_EMERG },
  { (char *)"err", LOG_ERR },
  { (char *)"error", LOG_ERR },
  { (char *)"info", LOG_INFO },
  { (char *)"none", 0x10 },
  { (char *)"notice", LOG_NOTICE },
  { (char *)"panic", LOG_EMERG },
  { (char *)"warning", LOG_WARNING },
  { (char *)"warn", LOG_WARNING },
  { NULL, -1 }
};



/**	Syslog feature codes.
*/
static name_and_code features[] = {
  { (char *)"auth", LOG_AUTH },
  { (char *)"authpriv", LOG_AUTHPRIV },
  { (char *)"cron", LOG_CRON },
  { (char *)"daemon", LOG_DAEMON },
  { (char *)"ftp", LOG_FTP },
  { (char *)"kern", LOG_KERN },
  { (char *)"lpr", LOG_LPR },
  { (char *)"mail", LOG_MAIL },
  { (char *)"mark", (24<<3) },
  { (char *)"news", LOG_NEWS },
  { (char *)"security", LOG_AUTH },
  { (char *)"syslog", LOG_SYSLOG },
  { (char *)"user", LOG_USER },
  { (char *)"uucp", LOG_UUCP },
  { (char *)"local0", LOG_LOCAL0 },
  { (char *)"local1", LOG_LOCAL1 },
  { (char *)"local2", LOG_LOCAL2 },
  { (char *)"local3", LOG_LOCAL3 },
  { (char *)"local4", LOG_LOCAL4 },
  { (char *)"local5", LOG_LOCAL5 },
  { (char *)"local6", LOG_LOCAL6 },
  { (char *)"local7", LOG_LOCAL7 },
  { NULL, -1}
};



int
dkslsupp_get_facility DK_P1(char *, str)
{
  int fac;
  fac = get_code_for(features, str);
  return fac;
}



int
dkslsupp_get_code DK_P1(char *, str)
{
  int back = 0;
  int fac, prio;
  char *ptr;
  char mybuffer[64];
  if(str) {
    if(strlen(str) < sizeof(mybuffer)) {
      strcpy(mybuffer,str);
      ptr = dkstr_chr(mybuffer, '.');
      if(ptr) {
	*ptr++ = '\0';
	prio = get_code_for(priorities, ptr);
	fac  = get_code_for(features, mybuffer);
	back = fac | prio;
      }
    }
  }
  return back;
}


#if DKSLSUPP_MAIN

char *test[] = {
    "auth.err", "auth.warn", "auth.notice", "lpr.panic",
    NULL
};
int main(int argc, char *argv[])
{
  char **ptr;
  int  i;
  ptr = test;
  while(*ptr) {
    i = dkslsupp_get_code(*ptr);
    printf("%d %s\n", i, *ptr);
    ptr++;
  }
}

#endif

