/*
Copyright (c) 1998-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	tracecc.h Include file for the tracecc modules.
*/



#ifndef TRACECC_H_INC
#define TRACECC_H_INC 1

#include <stdio.h>
#include <dk.h>
#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#else
#error "no stdlib.h"
#endif
#if DK_HAVE_UNISTD_H
#include <unistd.h>
#endif
#if DK_HAVE_PROCESS_H
#include <process.h>
#endif
#include <dkapp.h>
#include <dklogc.h>
#include <dkmem.h>
#include <dkstr.h>
#include <dkerror.h>
#include <dklic.h>
#include <dksf.h>
#include <dkcp.h>



/**	Option: Produce debug output. */
#define TRACECC_OPT_DEBUG		    1

/**	Option: Write debug output to stdout. */
#define TRACECC_OPT_DEBST		    2

/**	Option: Include trace keyword in debug output. */
#define TRACECC_OPT_KEYWORD		    4

/**	Option: Leave C++ style comments unchanged. */
#define TRACECC_OPT_CPPCOMMENT		    8

/**	Option: Run in make mode. */
#define TRACECC_OPT_MAKE		   16

/**	Option: Include line numbers in output. */
#define TRACECC_OPT_LINENO		   32

/**	Option: Use Java 2. */
#define TRACECC_OPT_JAVA2		   64

/**	Option: Include timestamps in debug output. */
#define TRACECC_OPT_DATETIME		  128

/**	Option: Show help. */
#define TRACECC_OPT_HELP		16384

/**	Option: Show versin number. */
#define TRACECC_OPT_VERSION		 8192

/**	Option: Configure application. */
#define TRACECC_OPT_CONFIG		 4096

/**	Option: Unconfigure application. */
#define TRACECC_OPT_UNCONF		 2048

/**	Option: Show configuration. */
#define TRACECC_OPT_SHOWCONF		 1024



/**	Option set.
 */
typedef struct {
  int o;	/**< Options or-connected. */
  int b;	/**< Comment box width. */
  int i;	/**< IDE style. */
} tracecc_opt_t ;

/**	Tracecc job.
*/
typedef struct {
  /*
    initialized during tracecc_open
  */
  dk_app_t *app;	/**< Application. */
  tracecc_opt_t	o;	/**< Options as specified by setup. */
  tracecc_opt_t f;	/**< Option set changed while processing file. */
  char **messages;	/**< Message texts. */
  int error_code;	/**< Code for last error. */
  long mpl;		/**< Max path length. */
  char *out_name_buf;	/**< Output name buffer, mpl bytes long. */
  char *tmp_name_buf;	/**< Buffer for temporary file name. */
  size_t ibl;		/**< Input buffer length in bytes. */
  char *inbuffer;	/**< Input buffer. */
  char *icmdbuf;	/**< Buffer for interpreter commands. */
  size_t icmdsz;	/**< Size of buffer. */
  /*
    set up when a run is started
  */
  int what;		/**< What to do: 0=no setup, 1=directories, 2=files. */
  char *dir_ori;	/**< Directory as specified, fnsep corrected. */
  dk_fne_t *fne_ori;	/**< Directory name expander. */
  char *dir_cur;	/**< Current directory as obtained from expander. */
  dk_dir_t *dir_tra;	/**< Directory traverser. */
  char *in_ori;		/**< Input file name as specified, fnsep corrected. */
  char *out_ori;	/**< Output file name as specified, fnsep corrected. */
  dk_fne_t *fne_in;	/**< File name expander for input file. */
  dk_fne_t *fne_out;	/**< File name expander for output file. */
  char *in_real;	/**< Real input name, not dynamically allocated. */
  char *out_real;	/**< Real output name, not dynamically allocated. */
  /*
    set by run_for_real_in_and_real_out
  */
  int   newft;		/**< Type of new file 1=C, 2=Java. */
  char *in_short;	/**< Short name of input file. */
  FILE *infile;		/**< Full name of input file. */
  FILE *outfile;	/**< Full name of output file. */
  /*
    initialized by run_for_opened_files
  */
  FILE *tmpfile;	/**< Temporary file. */
  unsigned long lineno;	/**< Line number currently processed. */
  size_t ibu;		/**< Used bytes in input buffer. */
  size_t ibc;		/**< Characters in input buffer. */
  char   currentchar;	/**< Current character to process. */
  int    textflags;	/**< Flags for text handling. */
  int    indent;	/**< Indent for new lines (number of spaces). */
  int    state;		/**< Current state. */
  int    reaction;	/**< Reaction for current transition. */
  size_t icmdu;		/**< Bytes used in interpreter command buffer. */
  int    cmbll;		/**< Comment box line length. */
  int    cmbmax;	/**< Maximum comment box length. */
  /*
    messages we already printed
  */
  int    line_too_long_printed;	/**< Flag: error message alreay printed. */
} tracecc_t;


#if defined(EXTERN)
#undef EXTERN
#endif
#ifndef TRACECC_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif
#if defined(__cplusplus)
extern "C" {
#endif
#if defined(__cplusplus)
}
#endif

#if defined(EXTERN)
#undef EXTERN
#endif
#ifndef TRACECO_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Create new Tracecc job.
 * 	@param	app	Application.
 * 	@return	Pointer to new Tracecc job on success, NULL on error.
 */
EXTERN tracecc_t *tracecc_open DK_PR((dk_app_t *app));

/**	Close Tracecc job,
 * 	@param	ptr	Tracecc job.
 */
EXTERN void tracecc_close DK_PR((tracecc_t *ptr));

/**	Run for a directory.
 * 	@param	ptr	Tracecc job.
 * 	@param	d	Directory name.
 */
EXTERN int tracecc_dir DK_PR((tracecc_t *ptr, char *d));

/**	Run for files.
 * 	@param	ptr	Tracecc job.
 * 	@param	i	Input file name.
 * 	@param	o	output file name.
 */
EXTERN int tracecc_files DK_PR((tracecc_t *ptr, char *i, char *o));

/**	Get mode.
 * 	@param	tc	Tracecc job.
 * 	@return	Current mode.
 */
EXTERN int tracecc_get_mode DK_PR((tracecc_t *tc));

/**	Get box width.
 * 	@param	tc	Tracecc job.
 * 	@return	Comment box width.
 */
EXTERN int tracecc_get_boxwidth DK_PR((tracecc_t *tc));

/**	Set mode.
 * 	@param	tc	Tracecc job.
 * 	@param	mode	New mode.
 */
EXTERN void tracecc_set_mode DK_PR((tracecc_t *tc, int mode));

/**	Set comment box width.
 * 	@param	tc	Tracecc job.
 * 	@param	bw	new box width.
 */
EXTERN void tracecc_set_boxwidth DK_PR((tracecc_t *tc, int bw));

/**	Save preferences.
 * 	@param	app	Application.
 * 	@param	mode	Mode.
 * 	@param	bw	Comment box width.
 * 	@param	is	IDE style.
 */
EXTERN void tracecc_save_preferences DK_PR((dk_app_t *app,int mode, int bw, int is));

/**	Get IDE style.
 * 	@param	tc	Tracecc job.
 * 	@return	Current IDE style.
 */
EXTERN int tracecc_get_idestyle DK_PR((tracecc_t *tc));

/**	Set IDE style.
 * 	@param	tc	Tracecc job.
 * 	@param	bw	New IDE style.
 */
EXTERN void tracecc_set_idestyle DK_PR((tracecc_t *tc, int bw));
#if defined(__cplusplus)
}
#endif


#endif

