/*
jlayout - A Java code generator for GUI layout
Copyright (c) 2007-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	jsrc.c	The jsrc module in the jlayout program.
	This module reads and modifies a Java source file.
*/



/**	Inside the jsrc module.
*/
#define JSRC_C	1

#include	"jl.h"




#line 55 "jsrc.ctr"




/** Package import variants. */
static char *dklibs_kw[] = {
  "dirk_krause.tools.DkTool",
  "dirk_krause.tools.*",
  "dirk_krause.wstools.DkTool",
  "dirk_krause.wstools.*",
  NULL
};

/** Fill modes. */
static char *fill_kw[] = {
  "NONE", "HORIZONTAL", "VERTICAL", "BOTH", NULL
};

/** Anchor positions. */
static char *anchor_kw[] = {
  "NORTH",
  "NORTHEAST",
  "EAST",
  "SOUTHEAST",
  "SOUTH",
  "SOUTHWEST",
  "WEST",
  "NORTHWEST",
  "PAGE_START",
  "PAGE_END",
  "LINE_START",
  "LINE_END",
  "FIRST_LINE_START",
  "FIRST_LINE_END",
  "LAST_LINE_START",
  "LAST_LINE_END",
  "BASELINE",
  "BASELINE_LEADING",
  "BASELINE_TRAILING",
  "ABOVE_BASELINE",
  "ABOVE_BASELINE_LEADING",
  "ABOVE_BASELINE_TRAILING",
  "BELOW_BASELINE",
  "BELOW_BASELINE_LEADING",
  "BELOW_BASELINE_TRAILING",
  "CENTER",
};

/** Events we can create listeners for. */
static char *listener_kw[] = {
  "Action",
  "Window",
  "Item",
  "Caret",
  "Change",
  "Component",
  "Container",
  "Document",
  "Focus",
  "InternalFrame",
  "Key",
  "ListData",
  "ListSelection",
  "Mouse",
  "MouseMotion",
  "MouseWheel",
  "PropertyChange",
  "TableModel",
  "TreeExpansion",
  "TreeModel",
  "TreeSelection",
  "TreeWillExpand",
  "UndoableEdit",
};

/** Horizontal align types. */
static char *h_align_types[] = {
	"LEADING",
	"LEFT",
	"CENTER",
	"RIGHT",
	"TRAILING",
	NULL
};

/** Vertical align types. */
static char *v_align_types[] = {
	"TOP",
	"CENTER",
	"BOTTOM",
	NULL
};

/** Horizontal text positions.
*/
static char *h_text_positions[] = {
	"LEFT",
	"CENTER",
	"RIGHT",
	"LEADING",
	"TRAILING",
	NULL
};

/** Operations when closing a frame. */
static char *str_close_operation[] = {
  "DO_NOTHING", "HIDE", "DISPOSE", "EXIT"
};

/** String: true, false . */
static char str_font_on[] = { "true, false" };

/** String: false, true . */
static char str_font_off[] = { "false, true" };

/** String: false, false . */
static char str_font_unchanged[] = { "false, false" };

/** String: setEnabled. */
static char str_set_enabled[] = { "setEnabled" };

/** String: setDefaultcloseOperation. */
static char str_set_default_close_operation[] = { "setDefaultCloseOperation" };

/** String: WindowConstants. */
static char str_window_constants[] = { "WindowConstants" };

/** String: JFrame. */
static char str_jframe[] = { "JFrame" };

/** String: _ON_CLOSE. */
static char str_on_close[] = { "_ON_CLOSE" };

/** String: setTransferHandler. */
static char str_set_transfer_handler[] = { "setTransferHandler" };

/** String: setToolTipText. */
static char str_set_tool_tip_text[] = { "setToolTipText" };

/** String: jLayoutFontChange. */
static char str_jlayoutfontchange[] = { "jLayoutFontChange" };

/** String: setAccessibleContext. */
static char str_set_accessible_description[] = { "setAccessibleDescription" };

/** String: getAccessibleContext. */
static char str_get_accessible_context[] = { "getAccessibleContext" };

/** String: Horizontal. */
static char str_horizontal[] = { "Horizontal" };

/** String: Vertical. */
static char str_vertical[] = { "Vertical" };

/** String: Alignment. */
static char str_alignment[] = { "Alignment" };

/** String: TextPosition. */
static char str_text_position[] = { "TextPosition" };

/** String: SwingConstants. */
static char str_swing_constants[] = { "SwingConstants" };

/** String: set. */
static char str_set[] = { "set" };

/** String: Minimum. */
static char str_minimum[] = { "Minimum" };

/** String: Preferred. */
static char str_preferred[] = { "Preferred" };

/** String: Maximum. */
static char str_maximum[] = { "Maximum" };

/** String: Size. */
static char str_size[] = { "Size" };

/** String: setConstraints. */
static char str_set_constraints[] = { "setConstraints" };

/** String: true. */
static char str_true[] = { "true" };

/** String: false. */
static char str_false[] = { "false" };

/** String: jLayoutGbc. */
static char str_jlayout_gbc[] = { "jLayoutGbc" };

/** String: "?" . */
static char str_unknown_string[] = { "\"?\"" };

/** String if(jLayoutIcon1 != null) { . */
static char str_p1[] = { "if(jLayoutIcon1 != null) {" };

/** String: } else { . */
static char str_p2[] = { "} else {" };

/** String: } . */
static char str_p3[] = { "}" };

/** String: GridBagLayout gbl = new GidBagLayout(); . */
static char str_new_grid_bag_layout[] = {
  "GridBagLayout gbl = new GridBagLayout();"
};

/** String: jLayoutCreateDistancePanel. */
static char str_jlayout_create_distance_panel[] = {
  "jLayoutCreateDistancePanel"
};
/* static char str_gbc_for_border_panels[] = { */
/*   "1, 1, GridBagConstraints.BOTH, GridBagConstrains.CENTER, 0.5, 0.5" */
/* }; */
/* static char str_half[] = { "0.5" }; */

/** String: 0 dot 0. */
static char str_zero_zero[] = { "0.0" };

/** String: dot 0. */
static char str_dot_zero[] = { ".0" };

/** String: JPnael dp = null; . */
static char str_jpanel_dp[] = { "JPanel dp = null;" };

/** String: dp. */
static char str_dp[] = { "dp" };

/** String: GridBagConstraints. */
static char str_grid_bag_constraints[] = { "GridBagConstraints" };

/** String: gbl. */
static char str_gbl[] = { "gbl" };

/** String: gbc. */
static char str_gbc[] = { "gbc" };

/** String: Box createRigidArea. */
static char str_box_create_rigid_area[] = { "Box.createRigidArea" };

/** String: 0. */
static char str_null[] = { "0" };

/** String: null. */
static char str_ref_null[] = { "null" };

/** String: 1. */
static char str_one[] = { "1" };

/** String: Listener. */
static char str_listener[] = { "Listener" };

/** String: Dimension. */
static char str_dimension[] = { "Dimension" };

/** String: X_AXIS. */
static char str_x_axis[] = { "X_AXIS" };

/** String: Y_AXIS. */
static char str_y_axis[] = { "Y_AXIS" };

/** String: PAGE_AXIS. */
static char str_page_axis[] = { "PAGE_AXIS" };

/** String: LINE_AXIS. */
static char str_line_axis[] = { "LINE_AXIS" };

/** String: BoxLayout. */
static char str_box_layout[] = { "BoxLayout" };

/** String: CardLayout. */
static char str_card_layout[] = { "CardLayout" };

/** String: GridLayout. */
static char str_grid_layout[] = { "GridLayout" };

/** String: FlowLayout. */
static char str_flow_layout[] = { "FlowLayout" };

/** String: BorderLayout. */
static char str_border_layout[] = { "BorderLayout" };

/** String: setLayout. */
static char str_set_layout[] = { "setLayout" };

/** String: remove. */
static char str_remove[] = { "remove" };

/** String: Box createHorizontalGlue(). */
static char str_box_create_horizontal_glue[] = { "Box.createHorizontalGlue()" };

/** String: Box createVerticalGlue(). */
static char str_box_create_vertical_glue[] = { "Box.createVerticalGlue()" };

/** String: setJMenuBar. */
static char str_set_j_menu_bar[] = { "setJMenuBar" };

/** String: Container cp = null; . */
static char str_container_cp[] = { "Container cp = null;" };

/** String: cp. */
static char str_cp[] = { "cp" };

/** String: getContentPane(). */
static char str_get_content_pane[] = { "getContentPane()" };

/** String: Start of Java comment. */
static char comment_start[] = { "//" };

/** String: Space. */
static char str_space[] = { " " };

/** String: Newline. */
static char str_nl[] = { "\n" };

/** String: Equal sign. */
static char str_equal[] = { "=" };

/** String: new. */
static char str_new[] = { "new" };

/** String: Opening bracket. */
static char str_bropen[] = { "(" };

/** String: Closing bracket. */
static char str_brclose[] = { ")" };

/** String: setTitle. */
static char str_set_title[] = { "setTitle" };

/** String: Opening curly bracket. */
static char str_block_open[] = { "{" };

/** String: Closing curly bracket. */
static char str_block_close[] = { "}" };

/** String: jLayoutImageIconFromName. */
static char str_find_image_icon_from_name[] = { "jLayoutImageIconFromName" };

/** String dkt.findImageIcon. */
static char str_find_image_icon_from_dktool[] = { "dkt.findImageIcon" };

/** String: dot. */
static char str_dot[] = { "." };

/** String: comma. */
static char str_comma[] = { "," };

/** String: setActionCommand. */
static char str_set_action_cmd[] = { "setActionCommand" };

/** String: setIcon. */
static char str_set_icon[] = { "setIcon" };

/** String: setText. */
static char str_set_text[] = { "setText" };

/** String: import. */
static char str_import[] = { "import" };

/** String: semicolon. */
static char str_semicolon[] = { ";" };

/** String: ButtonGroup. */
static char str_button_group[] = { "ButtonGroup" };

/** String: add. */
static char str_add[] = { "add" };

/** String: this. */
static char str_this[] = { "this" };

/** String: jLayoutResetSizes. */
static char str_jlayout_reset_sizes[] = { "jLayoutResetSizes" };

/** String: jLayoutGetSize. */
static char str_jlayout_get_size[] = { "jLayoutGetSize" };

/** String: jLayoutSetSize. */
static char str_jlayout_set_size[] = { "jLayoutSetSize" };

/** String: setBorder. */
static char str_set_border[] = { "setBorder" };

/** String: ImageIcon jLayoutIcon1; . */
static char str_image_icon[] = { "ImageIcon jLayoutIcon1;" };

/** String: jLayoutIcon1. */
static char str_jlayout_icon[] = { "jLayoutIcon1" };

/** String: BorderFactory. */
static char str_border_factory[] = { "BorderFactory" };

/** String: create. */
static char str_create[] = { "create" };

/** String: Etched. */
static char str_etched[] = { "Etched" };

/** String: Bevel. */
static char str_bevel[] = { "Bevel" };

/** String: Matte. */
static char str_matte[] = { "Matte" };

/** String: Line. */
static char str_line[] = { "Line" };

/** String: Titled. */
static char str_titled[] = { "Titled" };

/** String: Border. */
static char str_border[] = { "Border" };

/** String: Empty. */
static char str_empty[] = { "Empty" };

/** Configuration file name jlclass.conf. */
static char str_conf_file_name[] = { "jlclass.conf" };

/** String: setFont. */
static char str_setfont[] = { "setFont" };

/** String: setForeground. */
static char str_setforeground[] = { "setForeground" };

/** String: setBackground. */
static char str_setbackground[] = { "setBackground" };

/** String: Start of vim editor fold. */
static char str_fold_open_vim[] = { "{{{" };

/** String: End of vim editor fold. */
static char str_fold_close_vim[] = { "}}}" };

/** String: start of NetBeans editor fold. */
static char str_fold_open_netbeans[] = {
"<editor-fold defaultstate=\"collapsed\" desc=\"jlayout generated code\">"
};

/** String: End of NetBeans editor fold. */
static char str_fold_close_netbeans[] = { "</editor-fold>" };

/** Commands for lowered or raised edges. */
static char *cmd_lowered_or_raised[] = {
  "LOWERED", "RAISED", NULL
};

/** Keywords for lowered or raised edges. */
static char *cmd_bevel_lowered_or_raised[] = {
  "Lowered", "Raised", NULL
};

/** Option dklibsj=on. */
static char str_opt_dklibsj_on[] = { "@option:dklibsj=on" };

/** Option dklibsj=off. */
static char str_opt_dklibsj_off[] = { "@option:dklibsj=off" };

/**	Variables for font sizes.
*/
static char *str_variables_for_sizes[] = {
"private int jLayoutSizeMinX = 0;",
"private int jLayoutSizeMinY = 0;",
"private int jLayoutSizePrefX = 0;",
"private int jLayoutSizePrefY = 0;",
"private int jLayoutSizeMaxX = 0;",
"private int jLayoutSizeMaxY = 0;",
NULL
};



/**	Start of a jlayout generated section.
*/
static char *start_of_jlayout_section[] = {
"// START OF JLAYOUT SECTION",
"// This section was generated automatically by the jlayout program.",
"// You should not modify this section.",
"// Any changes applied to this section will be lost if jlayout is run again.",
NULL
};



/**	End of jlayout section.
*/
static char *end_of_jlayout_section[] = {
"// END OF JLAYOUT SECTION",
NULL
};



/**	Tool functions for gridbag layout.
*/
static char *str_functions_for_gridbag_layout[] = {
"/**",
" * Create GridBag constraints for a component.",
" *",
" * @param x",
" * The start grid column of the component.",
" *",
" * @param y",
" * The start grid line of the component.",
" *",
" * @param w",
" * The width of the component in grid columns.",
" *",
" * @param h",
" * The height of the component in grid lines.",
" *",
" * @param f",
" * The fill information (horizontal, vertical, both).",
" *",
" * @param wx",
" * The weight factor to grow the component horizontally",
" * when the container grows.",
" *",
" * @param wy",
" * The weight factor to grow the component vertically",
" * when the container grows.",
" */",
"private GridBagConstraints jLayoutGbc(",
"  int x, int y, int w, int h, int f, int a, double wx, double wy",
") {",
"  GridBagConstraints back = new GridBagConstraints();",
"  back.gridx = x;",
"  back.gridy = y;",
"  back.gridwidth = w;",
"  back.gridheight = h;",
"  back.fill = f;",
"  back.anchor = a;",
"  back.weightx = wx;",
"  back.weighty = wy;",
"  return back;",
"}",
NULL
};


/**	Functions to set component size.
*/
static char *str_functions_for_sizes[] = {
"/**",
" * Reset the stored size informatio.",
" */",
"private void jLayoutResetSizes()",
"{",
"  jLayoutSizeMinX = 0; jLayoutSizeMinY = 0;",
"  jLayoutSizeMaxX = 0; jLayoutSizeMaxY = 0;",
"  jLayoutSizePrefX = 0; jLayoutSizePrefY = 0;",
"}",
"/**",
" * Get component size.",
" * @param c Component to retrieve size from.",
" */",
"private void jLayoutGetSize(JComponent c)",
"{",
"  Dimension d;",
"  d = c.getMinimumSize();",
"  if(d.width > jLayoutSizeMinX) { jLayoutSizeMinX = d.width; }",
"  if(d.height > jLayoutSizeMinY) { jLayoutSizeMinY = d.height; }",
"  d = c.getPreferredSize();",
"  if(d.width > jLayoutSizePrefX) { jLayoutSizePrefX = d.width; }",
"  if(d.height > jLayoutSizePrefY) { jLayoutSizePrefY = d.height; }",
"  d = c.getMaximumSize();",
"  if(d.width > jLayoutSizeMaxX) { jLayoutSizeMaxX = d.width; }",
"  if(d.height > jLayoutSizeMaxY) { jLayoutSizeMaxY = d.height; }",
"}",
"/**",
" * Set a components size to a given minimum (save as other",
" * components).",
" *",
" * @param c",
" * The component to change.",
" */",
"private void jLayoutSetSize(JComponent c)",
"{",
"  c.setMinimumSize(new Dimension(jLayoutSizeMinX, jLayoutSizeMinY));",
"  c.setPreferredSize(new Dimension(jLayoutSizePrefX, jLayoutSizePrefY));",
"  c.setMaximumSize(new Dimension(jLayoutSizeMaxX, jLayoutSizeMaxY));",
"}",
NULL
};



/**	Special comments to mark jlayout sections.
*/
static char *special_comments[] = {
  "jlayout.variables.begin",
  "jlayout.variables.end",
  "jlayout.setup.begin",
  "jlayout.setup.end",
  "jlayout.cleanup.begin",
  "jlayout.cleanup.end",
  "jlayout.import.begin",
  "jlayout.import.end",
  NULL
};


/**	Function jLayoutImageIconFromName().
*/
static char *str_fct_find_image_for_name[] = {
"/**",
" * Find an image icon.",
" *",
" * @param n",
" * The name of the image icon file.",
" *",
" * @return",
" * The image icon.",
" */",
"protected ImageIcon jLayoutImageIconFromName(String n)",
"{",
"  ImageIcon back = null;",
"  try {",
"    back = new ImageIcon(getClass().getResource(n));",
"  } catch(Exception e) { back = null; }",
"  if(back == null) {",
"    if(!n.startsWith(\"/\")) {",
"      try {",
"        back = new ImageIcon(getClass().getResource(\"/\" + n));",
"      } catch(Exception e) { back = null; }",
"    }",
"  }",
"  if(back == null) {",
"    try {",
"      back = new ImageIcon(n);",
"    } catch(Exception e) { back = null; }",
"  }",
"  return back;",
"}",
NULL
};


/**	Distance panel for gridbag layout.
*/
static char *str_for_gridbag[] = {
"/**",
" * Create a dummy panel to ensure minimum distances.",
" *",
" * @param b",
" * The panels width and hight.",
" *",
" * @param rms",
" * A flag to indicate whether or not the panel must",
" * be set exactly to the given size (The panel can not grow).",
" */",
"protected JPanel jLayoutCreateDistancePanel(int b, boolean rms)",
"{",
"  JPanel back = new JPanel();",
"  Dimension d = new Dimension(b, b);",
"  back.setMinimumSize(d);",
"  back.setPreferredSize(d);",
"  if(rms) {",
"    back.setMaximumSize(d);",
"  }",
"  return back;",
"}",
NULL
};


/**	Function for font changes.
*/
static char *str_for_font_change[] = {
"/**",
" * Apply font changes.",
" *",
" * @param c",
" * The component to change the font.",
" *",
" * @param m",
" * Style changes. This parameter contains an OR-combination",
" * of bits constants: 1=bold on, 2=bold off, 4=italic on,",
" * 8=italic off.",
" */",
"void jLayoutFontChange(",
"  JComponent c, boolean bon, boolean boff, boolean ion, boolean ioff",
") {",
"  boolean bMustModify = false;",
"  Font f = c.getFont();",
"  String n = f.getName();",
"  int sz = f.getSize();",
"  int st = f.getStyle();",
"  if(bon) {		// bold on",
"    if((st & Font.BOLD) == 0) {",
"      st |= Font.BOLD;",
"      bMustModify = true;",
"    }",
"  } else {",
"    if(boff) {		// bold off",
"      if((st & Font.BOLD) != 0) {",
"        st &= (~(Font.BOLD));",
"        bMustModify = true;",
"      }",
"    }",
"  }",
"  if(ion) {		// italic on",
"    if((st & Font.ITALIC) == 0) {",
"      st |= Font.ITALIC;",
"      bMustModify = true;",
"    }",
"  } else {",
"    if(ioff) {		// italic off",
"      if((st & Font.ITALIC) != 0) {",
"        st &= (~(Font.ITALIC));",
"        bMustModify = true;",
"      }",
"    }",
"  }",
"  if(bMustModify) {",
"    c.setFont(new Font(n, st, sz));",
"  }",
"}",
NULL
};

/** Start of jLayoutSetup() function. */
static char *str_fct_setup_begin[] = {
"/**",
" * Create GUI components.",
" */",
"private void jLayoutSetup()",
"{",
NULL
};

/** Start of jLayoutCleanup function. */
static char *str_fct_cleanup_begin[] = {
"private void jLayoutCleanup()",
"{",
NULL
};

/** End of jLayoutCleanup()/jLayoutSetup() function. */
static char *str_fct_setup_end[] = {
"}",
NULL
};

/** File open mode write. */
static char str_w[] = { "w" };



/**	Compare two JSRC_LINE structures by line number.
	@param	l	Left line.
	@param	r	Right line.
	@param	cr	Comparison criteria (ignored).
	@return	Comparison result.
*/
static
int
compare_by_lineno DK_P3(void *,l, void *,r, int,cr)
{
  int back = 0;
  JSRC_LINE *pl = NULL, *pr = NULL;
  pl = (JSRC_LINE *)l; pr = (JSRC_LINE *)r;
  if(pl) {
    if(pr) {
      if(pl->lineno > pr->lineno) {
        back = 1;
      } else {
        if(pl->lineno < pr->lineno) {
	  back = -1;
	}
      }
    } else {
      back = 1;
    }
  } else {
    if(pr) {
      back = -1;
    } else {
    }
  }
  return back;
}



void
jsrc_line_delete DK_P1(JSRC_LINE *,l)
{
  char *x;
  if(l) {
    if(l->t) {
      x = l->t; dk_delete(x);
    }
    l->t = NULL; l->lineno = 0UL;
    dk_delete(l);
  }
}



JSRC_LINE *
jsrc_line_new DK_P2(char *,s, unsigned long,l)
{
  JSRC_LINE *back = NULL;
  if((s) && (l > 0UL)) {
    back = dk_new(JSRC_LINE,1);
    if(back) {
      back->t = dkstr_dup(s);
      if(back->t) {
        back->lineno = l;
      } else {
        dk_delete(back); back = NULL;
      }
    }
  }
  return back;
}



/**	Prepare JLJ to read Java source.
	@param	j	Jlayout job.
	@return	1 on success, 0 on error.
*/
int
jsrc_setup DK_P1(JLJ *,j)
{
  int back = 0;
  
  if(j) {
    j->l_st = dksto_open(0);
    if(j->l_st) {
      dksto_set_comp(j->l_st, compare_by_lineno, 0);
      j->l_it = dksto_it_open(j->l_st);
      if(j->l_it) {
        back = 1;
      }
    }
  }
  
  return back;
}



/**	Cleanup JLJ after Java source is processed.
	@param	j	Jlayout job.
*/
void
jsrc_cleanup DK_P1(JLJ *,j)
{
  JSRC_LINE *l;
  
  if(j) {
    if(j->l_st) {
      if(j->l_it) {
        dksto_it_reset(j->l_it);
	while((l = (JSRC_LINE *)dksto_it_next(j->l_it)) != NULL) {
	  jsrc_line_delete(l);
	}
        dksto_it_close(j->l_it);
	j->l_it = NULL;
      }
      dksto_close(j->l_st);
      j->l_st = NULL;
    }
  }
  
}



/**	Check whether or not the input line contains one of the
	controlling comments.
	@param	j	Jlayout job.
	@param	l	Input line.
*/
static
int
classify_input_line DK_P2(JLJ *,j, char *,l)
{
  size_t sl, xl;
  int back = -1; int i; char *ptr1, **xptr;
  
  sl = strlen(l);
  if(sl > 2) {		
    if(strncmp(l,comment_start,2) == 0) {
      ptr1 = &(l[2]);	
      ptr1 = dkstr_start(ptr1, NULL);
      if(ptr1) {	
        xl = strlen(l);
        i = 0; xptr = special_comments;
	while((back == -1) && (*xptr)) {
	  
	  sl = strlen(*xptr);
	  if(xl >= sl) {
	    if(strncmp(*xptr, ptr1, sl) == 0) {
	      back = i;
	    }
	  }
	  i++; xptr++;
	}
      }
    }
  }
  switch(back) {
    case -1: {
      size_t lgt_import;
      if(!(j->freeze)) {
        lgt_import = strlen(comment_start);
	if(strlen(l) >= lgt_import) {
	  if(strncmp(l, comment_start, lgt_import) == 0) {
	    if(strstr(l, str_opt_dklibsj_on)) {
	      j->use_dklibsj = 0x01;
	      j->freeze = 0x01;
	    } else {
	      if(strstr(l, str_opt_dklibsj_off)) {
	        j->use_dklibsj = 0x00;
		j->freeze = 0x01;
	      }
	    }
	  }
	}
        lgt_import = strlen(str_import);
        if(strlen(l) > lgt_import) {
          if(strncmp(l, str_import, lgt_import) == 0) {
	    xptr = dklibs_kw;
	    while((*xptr) && (!(j->use_dklibsj))) {
	      if(strstr(l, *xptr)) {
	        j->use_dklibsj = 0x01;
	      }
	      xptr++;
	    }
	  }
        }
      }
    } break;
    case 0: case 2: case 4: case 6: {
      j->freeze = 0x01;
    } break;
  } 
  return back;
}



/**	Read the existing Java source file.
	@param	j	Jlayout job.
	@return	1 on success, 0 on error.
*/
int
jsrc_read DK_P1(JLJ *,j)
{
  int back = 0;
  unsigned long lineno, olineno;
  FILE *input_file = NULL;
  FILE *opened_file = NULL;
  JSRC_LINE *lp;
  char *ptr1 = NULL;
  int  state;
  int  must_print;
  
  if(j) {
    if(j->ofname) {
      input_file = opened_file = dkapp_fopen(j->a, j->ofname, "r");
    } else {
      opened_file = NULL; input_file = stdin;
    }
    if(input_file) {
      back = 1; lineno = olineno = 0UL; state = 0;
      while(fgets(j->input_buffer, j->sz_input_buffer, input_file)) {
        lineno++; must_print = 0;
	ptr1 = dkstr_start(j->input_buffer, NULL);
	if(ptr1) {
	  dkstr_chomp(ptr1, NULL);
	  switch(classify_input_line(j, ptr1)) {
	    case -1: {	/* normal line */
	      if(state == 0) {
	        must_print = 1;
	      }
	    } break;
	    case 0: {	/* variables begin */
	      if(state != 0) {
	        /* ##### Warning */
	      }
	      state = 1;
	      j->ln_variables = olineno;
	    } break;
	    case 1: {	/* variables end */
	      if(state != 1) {
	        /* ##### Warning */
	      }
	      state = 0;
	    } break;
	    case 2: {	/* setup begin */
	      if(state != 0) {
	        /* ##### Warning */
	      }
	      state = 2;
	      j->ln_setup = olineno;
	    } break;
	    case 3: {	/* setup end */
	      if(state != 2) {
	        /* ##### Warning */
	      }
	      state = 0;
	    } break;
	    case 4: {	/* cleanup begin */
	      if(state != 0) {
	        /* ##### Warning */
	      }
	      state = 3;
	      j->ln_cleanup = olineno;
	    } break;
	    case 5: {	/* cleanup end */
	      if(state != 3) {
	        /* ##### Warning */
	      }
	      state = 0;
	    } break;
	    case 6: {
	      if(state != 0) {
	        /* ##### Warning */
	      } state = 4;
	      j->ln_import = olineno;
	    } break;
	    case 7: {
	      if(state != 4) {
	        /* ##### Warning */
	      } state = 0;
	    } break;
	  }
	} else {
	  if(state == 0) {
	    (j->input_buffer)[0] = '\0';
	    must_print = 1;
	  }
	}
	if(must_print) {
	  olineno++;
	  lp = jsrc_line_new(j->input_buffer, olineno);
	  if(lp) {
	    if(!dksto_add(j->l_st, (void *)lp)) {
	      jsrc_line_delete(lp); lp = NULL;
	      back = 0;
	      /* ERROR: Memory */
	      jlmsg_error_memory(j);
	    }
	  } else {
	    /* ERROR: Memory */
	    jlmsg_error_memory(j);
	    back = 0;
	  }
	}
      }
    } else {
      
      /* ERROR: Failed to open input file */
      /* jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 51); */
      if(j->ofname) {
        jlmsg_log3(j, DK_LOG_LEVEL_ERROR, 58, j->ofname, 59);
      }
    }
    if(opened_file) {
      fclose(opened_file); opened_file = NULL;
    }
  }
  
  return back;
}



/**	Write an array of strings to the output file.
	@param	f	Output file.
	@param	a	String array.
	@param	tab	Flag: Write tabulator before each string.
*/
static
void
print_str_array DK_P3(FILE *,f, char **,a, int,tab)
{
  char **x;
  x = a;
  while(*x) {
    if(tab) {
      fputc('\t', f);
    }
    fputs(*x, f);
    fputs(str_nl, f);
    x++;
  }
}



/**	Start fold marker.
	@param	f	Output file.
	@param	j	Jlayout job.
*/
static
void
start_fold_marker DK_P2(FILE *,f, JLJ *,j) {
  int is_not_first = 0;
  if(j->foldmt) {
    fputs(comment_start, f);
    fputs(str_space, f);
    if((j->foldmt) & FOLD_MARKER_NETBEANS) {
      fputs(str_fold_open_netbeans, f);
      is_not_first = 1;
    }
    if((j->foldmt) & FOLD_MARKER_VIM) {
      if(is_not_first) fputs(str_space, f);
      fputs(str_fold_open_vim, f);
      is_not_first = 1;
    }
    fputs(str_nl, f);
  }
}



/**	End a fold marker
	@param	f	Output file.
	@param	j	Jlayout job.
*/
static void
end_fold_marker DK_P2(FILE *,f, JLJ *,j) {
  int is_not_first = 0;
  if(j->foldmt) {
    fputs(comment_start, f);
    fputs(str_space, f);
    if((j->foldmt) & FOLD_MARKER_VIM) {
      fputs(str_fold_close_vim, f);
      is_not_first = 1;
    }
    if((j->foldmt) & FOLD_MARKER_NETBEANS) {
      if(is_not_first) fputs(str_space, f);
      fputs(str_fold_close_netbeans, f);
      is_not_first = 1;
    }
    fputs(str_nl, f);
  }
}



/**	Write import section.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	sourcelineno	Source line number.
*/
static
void
write_section_import DK_P3(JLJ *,j, FILE *,f, unsigned long,sourcelineno)
{
  char *l1, *l2, *p1;
  size_t mpl;
  dk_stream_t *cfgfile;
  int done;
  char *oldfilename; unsigned long oldlineno;
  
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[6], f);
  fputs(str_nl, f);
  start_fold_marker(f, j);
  print_str_array(f, start_of_jlayout_section, 0);
  mpl = (size_t)dksf_get_maxpathlen();
  l1 = dk_new(char,mpl); l2 = dk_new(char,mpl);
  if((l1) && (l2)) {
    l1[0] = l2[0] = '\0';
    oldfilename = dkapp_get_source_filename(j->a);
    oldlineno = dkapp_get_source_lineno(j->a);
    if(j->ofname) {
      dkapp_set_source_filename(j->a, j->ofname);
      dkapp_set_source_lineno(j->a, sourcelineno + 1UL);
    }
    cfgfile = dkapp_read_file(j->a, str_conf_file_name);
    if(cfgfile) {
      while(dkstream_gets(cfgfile, l2, mpl)) {
        dkstr_delcomm(l2, '#');
	p1 = dkstr_start(l2, NULL);
	if(p1) {
	  dkstr_chomp(l2, NULL);
	  if((l2[0] == ' ') || (l2[0] == '\t')) {
	    if(!done) {
	      if((j->uc_st) && (j->uc_it)) {
	        if(dksto_it_find_like(j->uc_it, (void *)p1, 0)) {
		  done = 1;
		  fputs(str_import, f);
		  fputs(str_space, f);
		  fputs(l1, f);
		  fputs(str_semicolon, f);
		  fputs(str_nl, f);
		}
	      }
	    }
	  } else {
	    strcpy(l1, p1); done = 0;
	  }
	}
      }
      dkstream_close(cfgfile);
    }
    dkapp_set_source_filename(j->a, oldfilename);
    dkapp_set_source_lineno(j->a, oldlineno);
  } else { jlmsg_error_memory(j); }
  if(l1) { dk_delete(l1); }
  if(l2) { dk_delete(l2); }
  print_str_array(f, end_of_jlayout_section, 0);
  end_fold_marker(f, j);
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[7], f);
  fputs(str_nl, f);
  
}



/**	Write variables declarations to the output file.
	@param	j	Jlayout job.
	@param	f	Output file.
*/
static
void
write_section_variables DK_P2(JLJ *,j, FILE *,f)
{
  JLO *o;
  JSG *jsg;
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[0], f);
  fputs(str_nl, f);
  start_fold_marker(f, j);
  print_str_array(f, start_of_jlayout_section, 0);
  if((j->f1) & F1_HAVE_SIZE_GROUP) {
    print_str_array(f, str_variables_for_sizes, 1);
  }
  dksto_it_reset(j->o_it);
  while((o = (JLO *)dksto_it_next(j->o_it)) != NULL) {
    
    if((o->on) && (jlo_get_classname(o))) {
      fprintf(f, "\t%s %s = null;\n", jlo_get_classname(o), o->on);
    } else {
      if(!(o->on)) {
        
      }
      if(!jlo_get_classname(o)) {
        
      }
    }
  }
  if((j->f1) & F1_HAVE_BUTTON_GROUPS) {
    dksto_it_reset(j->b_it);
    while((jsg = (JSG *)dksto_it_next(j->b_it)) != NULL) {
      fprintf(f, "\tButtonGroup %s = null;\n", jsg->n);
    }
  }
  print_str_array(f, end_of_jlayout_section, 0);
  end_fold_marker(f, j);
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[1], f);
  fputs(str_nl, f);
}



/**	Write tabulator and some space on start of line.
	@param	f	Output file.
	@param	l	Number of spaces.
*/
static
void
line_start DK_P2(FILE *,f, int,l)
{
  int i;
  fputc('\t', f); fputs(str_space, f); fputs(str_space, f);
  for(i = 0; i < l; i++) {
    fputs(str_space, f); fputs(str_space, f);
  }
}



/**	Write call to image icon search method.
	@param	j	Jlayout job.
	@param	n	Icon file name.
	@param	f	Output file.
*/
static
void
write_image_icon_search DK_P3(JLJ *,j, char *,n, FILE *,f) {
  if(j->use_dklibsj) {
    fputs(str_find_image_icon_from_dktool, f);
    fputs(str_bropen, f);
    fputs(str_this, f);
    fputs(str_comma, f);
    fputs(str_space, f);
    if(n) { fputs(n, f); }
    else { fputs(str_unknown_string, f); }
    fputs(str_brclose, f);
  } else {
    fputs(str_find_image_icon_from_name, f);
    fputs(str_bropen, f);
    fputs(n, f);
    fputs(str_brclose, f);
  }
}



/**	Write initialization code for one variable.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	con	Constructor argument(s).
	@param	l	Line starts with \a l spaces.
*/
static
void
variable_declaration_constructor DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, char *,con, int,l)
{
  int x;
  x = l;
  line_start(f, 0);
  while(x-- > 0) {
    fputs(str_space, f); fputs(str_space, f);
  }
  fputs(jlo->on, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(jlo_get_classname(jlo), f);
  fputs(str_bropen, f);
  if(con) {
    fputs(con, f);
  }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}


/**	Write initialization code for one variable.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	con	Constructor arguments.
*/
static
void
variable_declaration_icon DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, char *,con)
{
  line_start(f, 0);
  fputs(str_jlayout_icon, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  write_image_icon_search(j, con, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 0);
  fputs(str_p1, f);
  fputs(str_nl, f);
  line_start(f, 0);
  fputs(str_space, f); fputs(str_space, f);
  fputs(jlo->on, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(jlo_get_classname(jlo), f);
  fputs(str_bropen, f);
  fputs(str_jlayout_icon, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 0);
  fputs(str_p2, f);
  fputs(str_nl, f);
  fputs(str_space, f); fputs(str_space, f);
  variable_declaration_constructor(j, f, jlo, NULL, 1);
  line_start(f, 0);
  fputs(str_p3, f);
  fputs(str_nl, f);
}



/**	Write initialization code for one variable.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	con	Constructor arguments.
	@param	at	Constructor type.
*/
static
void
variable_declaration DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, char *,con, int,at)
{
  /*
  line_start(f, 0);
  fputs(jlo->on, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(jlo_get_classname(jlo), f);
  fputs(str_bropen, f);
  if(con) {
    switch(at) {
      case OBJ_ATT_ICON_FILE: {
	write_image_icon_search(j, con, f);
      } break;
      default: {
        fputs(con, f);
      } break;
    }
  }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  */
  switch(at) {
    case OBJ_ATT_ICON_FILE: {
      variable_declaration_icon(j, f, jlo, con);
    } break;
    default: {
      variable_declaration_constructor(j, f, jlo, con, 0);
    } break;
  }
}



/**
 * Write initialization code for one variable.
 * This function is used only for text fields if we provide
 * the start text and a text field width.
 * No image icons involved.
 * @param	j	jLayout conversion job.
 * @param	f	Output file.
 * @param	jlo	Current object.
 * @param	con	Constructor arguments.
 * @param	at	Constructor type.
 * @param	c2	Second constructor argument.
 * @param	a2	Seconed constructor argument type.
 */
static
void
variable_declaration_2 DK_P7(\
  JLJ *,j, FILE *,f, JLO *,jlo, char *,con, int,at, char *,c2, int,a2)
{
  line_start(f, 0);
  fputs(jlo->on, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(jlo_get_classname(jlo), f);
  fputs(str_bropen, f);
  if(con) {
    switch(at) {
      case OBJ_ATT_ICON_FILE: {
        /* Never used. */
	write_image_icon_search(j, con, f);
      } break;
      default: {
        fputs(con, f);
      } break;
    }
    fputs(str_comma, f);
    fputs(str_space, f);
    switch(a2) {
      case OBJ_ATT_ICON_FILE: {
        /* Never used. */
	write_image_icon_search(j, c2, f);
      } break;
      default: {
        fputs(c2, f);
      } break;
    }
  }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Set sizes.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	cr	Decision which size to set: 2=max, 1=preferred, 3=min.
*/
static
void
set_sizes DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int,cr)
{
  long x, y;
  switch(cr) {
    case 1: {
      x = jlo->s_pref_x; y = jlo->s_pref_y;
    } break;
    case 2: {
      x = jlo->s_max_x; y = jlo->s_max_y;
    } break;
    default: {
      x = jlo->s_min_x; y = jlo->s_min_y;
    } break;
  }
  if((x >= 0L) && (y >= 0L)) {
    line_start(f, 0);
    if(jlo->on) {
      fputs(jlo->on, f); fputs(str_dot, f);
    }
    fputs(str_set, f);
    switch(cr) {
      case 2: {
        fputs(str_maximum, f);
      } break;
      case 1: {
        fputs(str_preferred, f);
      } break;
      default: {
        fputs(str_minimum, f);
      } break;
    }
    fputs(str_size, f);
    fputs(str_bropen, f);
    fputs(str_new, f);
    fputs(str_space, f);
    fputs(str_dimension, f);
    fputs(str_bropen, f);
    fprintf(f, "%ld, %ld", x, y);
    fputs(str_brclose, f);
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
}



/**	Add accessible description.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	t	Accessible description text.
*/
static
void
add_accessible_description DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, char *, t)
{
  line_start(f, 0);
  if(jlo->on) {
    fputs(jlo->on, f); fputs(str_dot, f);
  }
  fputs(str_get_accessible_context, f);
  fputs(str_bropen, f);
  fputs(str_brclose, f);
  fputs(str_dot, f);
  fputs(str_set_accessible_description, f);
  fputs(str_bropen, f);
  fputs(str_nl, f);
  line_start(f, 1);
  fputs(t, f);
  fputs(str_nl, f);
  line_start(f, 0);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Set one single property.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	mn	Method name (Method to set property).
	@param	arg	Property value.
*/
static
void
set_single_property DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, char *,mn, char *,arg)
{
  line_start(f, 0);
  if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
  fputs(mn, f);
  fputs(str_bropen, f);
  if(arg) { fputs(arg, f); }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Set alignment.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	hv	Flag for horizontal (0) or vertical (1) contents align.
	@param	al	Alignment.
*/
static
void
set_alignment DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, int,hv, int,al)
{
  line_start(f, 0);
  if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
  fputs(str_set, f);
  switch(hv) {
    case 1: {
      fputs(str_vertical, f);
    } break;
    default: {
      fputs(str_horizontal, f);
    } break;
  }
  fputs(str_alignment, f);
  fputs(str_bropen, f);
  fputs(str_swing_constants, f);
  fputs(str_dot, f);
  switch(hv) {
    case 1: {
      fputs(v_align_types[al], f);
    } break;
    default: {
      fputs(h_align_types[al], f);
    } break;
  }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Write code to set text position.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	hv	Flag: Horizontal (0) or vertical (1) object alignment.
	@param	al	Text position.
*/
static
void
set_text_position DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, int,hv, int,al)
{
  line_start(f, 0);
  if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
  fputs(str_set, f);
  switch(hv) {
    case 1: {
      fputs(str_vertical, f);
    } break;
    default: {
      fputs(str_horizontal, f);
    } break;
  }
  fputs(str_text_position, f);
  fputs(str_bropen, f);
  fputs(str_swing_constants, f);
  fputs(str_dot, f);
  switch(hv) {
    case 1: {
      fputs(v_align_types[al], f);
    } break;
    default: {
      fputs(h_text_positions[al], f);
    } break;
  }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}


/**	Write code to set additional properties after an object
	is created.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	af	Attributes specified for object.
	@param	au	Attributes already written for object.
*/
static
void
set_attributes DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, int,af, int,au)
{
  
  if(jlo->title) {
    line_start(f, 0);
    if(jlo->on) {
      fputs(jlo->on, f); fputs(str_dot, f);
    }
    fputs(str_set_title, f);
    fputs(str_bropen, f);
    fputs(jlo->title, f);
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  if(af & OBJ_ATT_ICON_FILE) {
    if(!(au & OBJ_ATT_ICON_FILE)) {
      
      line_start(f, 0);
      fputs(str_jlayout_icon, f);
      fputs(str_space, f);
      fputs(str_equal, f);
      fputs(str_space, f);
      write_image_icon_search(j, jlo->icon_file, f);
      fputs(str_semicolon, f);
      fputs(str_nl, f);
      line_start(f, 0);
      fputs(str_p1, f);
      fputs(str_nl, f);
      line_start(f, 0);
      fputs(str_space, f); fputs(str_space, f);
      if(jlo->on) {
        fputs(jlo->on, f); fputs(str_dot, f);
      }
      fputs(str_set_icon, f);
      fputs(str_bropen, f);
      fputs(str_jlayout_icon, f);
      fputs(str_brclose, f);
      fputs(str_semicolon, f);
      fputs(str_nl, f);
      line_start(f, 0);
      fputs(str_p3, f);
      fputs(str_nl, f);
    }
  }
  if(af & OBJ_ATT_TEXT) {
    if(!(au & OBJ_ATT_TEXT)) {
      
      set_single_property(j, f, jlo, str_set_text, jlo->text);
      au &= (~(OBJ_ATT_ACTION_CMD));
    }
  }
  if(jlo->font) {
    line_start(f, 0);
    if(jlo->on) {
      fputs(jlo->on, f); fputs(str_dot, f);
    }
    fputs(str_setfont, f);
    fputs(str_bropen, f);
    fputs(jlo->font, f);
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  if(jlo->foreground) {
    line_start(f, 0);
    if(jlo->on) {
      fputs(jlo->on, f); fputs(str_dot, f);
    }
    fputs(str_setforeground, f);
    fputs(str_bropen, f);
    fputs(jlo->foreground, f);
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  if(jlo->background) {
    line_start(f, 0);
    if(jlo->on) {
      fputs(jlo->on, f); fputs(str_dot, f);
    }
    fputs(str_setbackground, f);
    fputs(str_bropen, f);
    fputs(jlo->background, f);
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  if(af & OBJ_ATT_ACTION_CMD) {
    if((!(au & OBJ_ATT_ACTION_CMD)) || (af & OBJ_ATT_TEXT)) {
      
      set_single_property(
        j, f, jlo, str_set_action_cmd, jlo->action_cmd
      );
    }
  }
  set_sizes(j, f, jlo, 0);
  set_sizes(j, f, jlo, 1);
  set_sizes(j, f, jlo, 2);
  if(jlo->tt_text) {
    set_single_property(j, f, jlo, str_set_tool_tip_text, jlo->tt_text);
  }
  if(jlo->acd_text) {
    add_accessible_description(j, f, jlo, jlo->acd_text);
  } else {
    if(j->f1 & F1_USE_TTT_AS_AD) {
      if(jlo->tt_text) {
        add_accessible_description(j, f, jlo, jlo->tt_text);
      }
    }
  }
  if(jlo->h_align != HALIGN_NOT_SPECIFIED) {
    set_alignment(j, f, jlo, 0, jlo->h_align);
  }
  if(jlo->v_align != VALIGN_NOT_SPECIFIED) {
    set_alignment(j, f, jlo, 1, jlo->v_align);
  }
  if(jlo->h_textpos != HTEXTPOS_NOT_SPECIFIED) {
    set_text_position(j, f, jlo, 0, jlo->h_textpos);
  }
  if(jlo->v_textpos != VTEXTPOS_NOT_SPECIFIED) {
    set_text_position(j, f, jlo, 1, jlo->v_textpos);
  }
  if(jlo->def_close != CLOSE_OPERATION_NOT_SPECIFIED) {
    line_start(f, 0);
    if(jlo->on) {
      fputs(jlo->on, f); fputs(str_dot, f);
    }
    fputs(str_set_default_close_operation, f);
    fputs(str_bropen, f);
    switch(jlo->def_close) {
      case CLOSE_OPERATION_HIDE: {
        fputs(str_window_constants, f);
	fputs(str_dot, f);
	fputs(str_close_operation[1], f);
	fputs(str_on_close, f);
      } break;
      case CLOSE_OPERATION_DISPOSE: {
        fputs(str_window_constants, f);
	fputs(str_dot, f);
	fputs(str_close_operation[2], f);
	fputs(str_on_close, f);
      } break;
      case CLOSE_OPERATION_EXIT: {
        fputs(str_jframe, f);
	fputs(str_dot, f);
	fputs(str_close_operation[3], f);
	fputs(str_on_close, f);
      } break;
      default: {	/* CLOSE_OPERATION_NOTHING */
        fputs(str_window_constants, f);
	fputs(str_dot, f);
	fputs(str_close_operation[0], f);
	fputs(str_on_close, f);
      } break;
    }
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  if(jlo->enab != ENABLED_NOT_SPECIFIED) {
    line_start(f, 0);
    if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
    fputs(str_set_enabled, f);
    fputs(str_bropen, f);
    switch(jlo->enab) {
      case ENABLED_YES: { fputs(str_true, f); } break;
      default: { fputs(str_false, f); } break;
    }
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  if(jlo->font_changes) {
    line_start(f, 0);
    fputs(str_jlayoutfontchange, f);
    fputs(str_bropen, f);
    if(jlo->on) {
      fputs(jlo->on, f);
    } else {
      fputs(str_this, f);
    }
    fputs(str_comma, f);
    fputs(str_space, f);
    if((jlo->font_changes) & 0x01) {
      fputs(str_font_on, f);
    } else {
      if((jlo->font_changes) & 0x02) {
        fputs(str_font_off, f);
      } else {
        fputs(str_font_unchanged, f);
      }
    }
    fputs(str_comma, f); fputs(str_space, f);
    if((jlo->font_changes) & 0x04) {
      fputs(str_font_on, f);
    } else {
      if((jlo->font_changes) & 0x08) {
        fputs(str_font_off, f);
      } else {
        fputs(str_font_unchanged, f);
      }
    }
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
  
}



/**	Write code to start and end a border definition.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
start_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  line_start(f, 0);
  if(jlo->on) {
    fputs(jlo->on, f);
    fputs(str_dot, f);
  }
  fputs(str_set_border, f);
  fputs(str_bropen, f);
  fputs(str_nl, f);
}



/**	Write code to end a border definition.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
end_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  line_start(f, 0);
  fputs(str_brclose, f); fputs(str_semicolon, f); fputs(str_nl, f);
}



/**	Write code t o create etched border.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	ind	Border index (lowered or raised).
*/
static
void
etched_border DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int, ind)
{
  start_border(j, f, jlo);
  line_start(f, 1);
  fputs(str_border_factory, f);
  fputs(str_dot, f);
  fputs(str_create, f);
  fputs(str_etched, f);
  fputs(str_border, f);
  fputs(str_bropen, f);
  fputs(str_etched, f);
  fputs(str_border, f);
  fputs(str_dot, f);
  fputs(cmd_lowered_or_raised[ind ? 1 : 0], f);
  fputs(str_brclose, f);
  fputs(str_nl, f);
  end_border(j, f, jlo);
}



/**	Write code to create titled border
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
titled_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  start_border(j, f, jlo);
  line_start(f, 1);
  fputs(str_border_factory, f);
  fputs(str_dot, f);
  fputs(str_create, f);
  fputs(str_titled, f);
  fputs(str_border, f);
  fputs(str_bropen, f);
  fputs(jlo->border_text, f);
  fputs(str_brclose, f);
  fputs(str_nl, f);
  end_border(j, f, jlo);
}



/**	Write code to create matte border.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
matte_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  switch(jlo->border_subtype) {
    case 1: {	/* color */
      start_border(j, f, jlo);
      line_start(f, 1);
      fputs(str_border_factory, f);
      fputs(str_dot, f);
      fputs(str_create, f);
      fputs(str_matte, f);
      fputs(str_border, f);
      fputs(str_bropen, f);
      fputs(str_nl, f);
      line_start(f, 2);
      fprintf(
        f, "%d, %d, %d, %d,",
        jlo->bo_top, jlo->bo_left, jlo->bo_bottom, jlo->bo_right
      );
      fputs(str_space, f); fputs(jlo->border_text, f);
      fputs(str_nl, f);
      line_start(f, 1);
      fputs(str_brclose, f);
      fputs(str_nl, f);
      end_border(j, f, jlo);
    } break;
    default: {	/* icon */
      line_start(f, 0);
      fputs(str_jlayout_icon, f);
      fputs(str_space, f);
      fputs(str_equal, f);
      fputs(str_space, f);
      write_image_icon_search(j, jlo->border_text, f);
      fputs(str_semicolon, f);
      fputs(str_nl, f);
      line_start(f, 0);
      fputs(str_p1, f);
      fputs(str_nl, f);
      start_border(j, f, jlo);
      line_start(f, 1);
      fputs(str_border_factory, f);
      fputs(str_dot, f);
      fputs(str_create, f);
      fputs(str_matte, f);
      fputs(str_border, f);
      fputs(str_bropen, f);
      fputs(str_nl, f);
      line_start(f, 2);
      fprintf(
        f, "%d, %d, %d, %d,",
        jlo->bo_top, jlo->bo_left, jlo->bo_bottom, jlo->bo_right
      );
      fputs(str_space, f); fputs(str_jlayout_icon, f);
      fputs(str_nl, f);
      line_start(f, 1);
      fputs(str_brclose, f);
      fputs(str_nl, f);
      end_border(j, f, jlo);
      fputs(str_p3, f);
      fputs(str_nl, f);

    } break;
  }
}



/**	Write code to create line border.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
line_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  start_border(j, f, jlo);
  line_start(f, 1);
  fputs(str_border_factory, f);
  fputs(str_dot, f);
  fputs(str_create, f);
  fputs(str_line, f);
  fputs(str_border, f);
  fputs(str_bropen, f);
  fputs(jlo->border_text, f);
  if(jlo->bo_top > -1) {
    fputs(str_comma, f);
    fputs(str_space, f);
    fprintf(f, "%d", jlo->bo_top);
  }
  fputs(str_brclose, f);
  fputs(str_nl, f);
  end_border(j, f, jlo);
}



/**	Write code to create empty border.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
empty_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  start_border(j, f, jlo);
  line_start(f, 1);
  fputs(str_border_factory, f);
  fputs(str_dot, f);
  fputs(str_create, f);
  fputs(str_empty, f);
  fputs(str_border, f);
  fputs(str_bropen, f);
  fprintf(
    f, "%d, %d, %d, %d",
    jlo->bo_top, jlo->bo_left, jlo->bo_bottom, jlo->bo_right
  );
  fputs(str_brclose, f);
  fputs(str_nl, f);
  end_border(j, f, jlo);
}



/**	Write code to create bevel border.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	ind	Index of command for lowered or raised.
*/
static
void
bevel_border DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int,ind)
{
  start_border(j, f, jlo);
  line_start(f, 1);
  fputs(str_border_factory, f);
  fputs(str_dot, f);
  fputs(str_create, f);
  fputs(cmd_bevel_lowered_or_raised[ind ? 1 : 0], f);
  fputs(str_bevel, f);
  fputs(str_border, f);
  fputs(str_bropen, f);
  fputs(str_brclose, f);
  fputs(str_nl, f);
  end_border(j, f, jlo);
}



/**	Write code to add border to object.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
set_border DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  switch(jlo->border_type) {
    case BORDER_EMPTY: {
      empty_border(j, f, jlo);
    } break;
    case BORDER_ETCHED_LOWERED: {
      etched_border(j, f, jlo, 0);
    } break;
    case BORDER_ETCHED_RAISED: {
      etched_border(j, f, jlo, 1);
    } break;
    case BORDER_BEVEL_LOWERED: {
      bevel_border(j, f, jlo, 0);
    } break;
    case BORDER_BEVEL_RAISED: {
      bevel_border(j, f, jlo, 1);
    } break;
    case BORDER_LINE: {
      if(jlo->border_text) {
        line_border(j, f, jlo);
      }
    } break;
    case BORDER_MATTE: {
      if(jlo->border_text) {
        matte_border(j, f, jlo);
      }
    } break;
    case BORDER_TITLED: {
      if(jlo->border_text) {
        titled_border(j, f, jlo);
      }
    } break;
  }
}



/**	Write code to create one object.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
create_object DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  char buffer[32];
  int atts_found, atts_used;
  JCN *jcn; char *cptr;
  int cc;
  atts_found = atts_used = 0;
  if(jlo->constructor) {
    atts_found |= OBJ_ATT_CONSTRUCTOR;
  }
  if(jlo->action_cmd) {
    atts_found |= OBJ_ATT_ACTION_CMD;
  }
  if(jlo->text) {
    atts_found |= OBJ_ATT_TEXT;
  }
  if(jlo->icon_file) {
    atts_found |= OBJ_ATT_ICON_FILE;
  }
  if(jlo->tf_width) {
    atts_found |= OBJ_ATT_TEXTFIELD_WIDTH;
  }
  line_start(f, 0);
  fputs(comment_start, f);
  fputs(str_space, f);
  sprintf(buffer, "%lu", jlo->l_creation);
  if(jlo->on) {
    fputs(jlo->on, f);
  } else {
    fputs("MAIN", f);
  }
  fputs(str_space, f);
  fputs(buffer, f);
  fputs(str_nl, f);
  switch(jlo->ocn) {
    case CLASS_NOT_SPECIFIED: {
      if(jlo->parent) {	/* no warning for root object */
        /* ERROR: No class specified for object */
        jlmsg_log3(j, DK_LOG_LEVEL_ERROR, 3, jlo->on, 4);
	
      }
      set_attributes(j, f, jlo, atts_found, atts_used);
    } break;
    case CLASS_EXISTING_OBJECT: {
      /* Nothing to do here, object already exists */
    } break;
    case CLASS_JTEXTFIELD: {
      if(jlo->on) {
        if(atts_found & OBJ_ATT_CONSTRUCTOR) {
	  variable_declaration(j,f,jlo,jlo->constructor,OBJ_ATT_CONSTRUCTOR);
	  atts_used |= OBJ_ATT_CONSTRUCTOR;
	} else {
	  if(atts_found & OBJ_ATT_TEXTFIELD_WIDTH) {
	    if(atts_found & OBJ_ATT_TEXT) {
	      variable_declaration_2(
	        j,f,jlo,jlo->text,OBJ_ATT_TEXT,
		jlo->tf_width,OBJ_ATT_TEXTFIELD_WIDTH
	      );
	      atts_used |= (OBJ_ATT_TEXTFIELD_WIDTH | OBJ_ATT_TEXT);
	    } else {
	      variable_declaration(
	        j,f,jlo,jlo->tf_width,OBJ_ATT_TEXTFIELD_WIDTH
	      );
	      atts_used |= OBJ_ATT_TEXTFIELD_WIDTH;
	    }
	  } else {
	    if(atts_found & OBJ_ATT_TEXT) {
	      variable_declaration(
	        j,f,jlo,jlo->text, OBJ_ATT_TEXT
	      );
	      atts_used |= OBJ_ATT_TEXT;
	    } else {
	      variable_declaration(j, f, jlo, NULL, 0);
	    }
	  }
	}
      }
      set_attributes(j, f, jlo, atts_found, atts_used);
    } break;
    case CLASS_JBUTTON:
    case CLASS_JCHECKBOX:
    case CLASS_JRADIOBUTTON:
    case CLASS_JCHECKBOXMENUITEM:
    case CLASS_JRADIOBUTTONMENUITEM:
    case CLASS_JTOGGLEBUTTON:
    case CLASS_JLABEL:
    case CLASS_JMENUITEM: {
      if(jlo->on) {
        if(atts_found & OBJ_ATT_CONSTRUCTOR) {
          variable_declaration(j, f, jlo, jlo->constructor, OBJ_ATT_CONSTRUCTOR);
	  atts_used |= OBJ_ATT_CONSTRUCTOR;
        } else {
          if(atts_found & OBJ_ATT_ICON_FILE) {
	    variable_declaration(j, f, jlo, jlo->icon_file, OBJ_ATT_ICON_FILE);
	    atts_used |= OBJ_ATT_ICON_FILE;
	  } else {
	    if(atts_found & OBJ_ATT_ACTION_CMD) {
	      variable_declaration(j, f, jlo,jlo->action_cmd,OBJ_ATT_ACTION_CMD);
	      atts_used |= OBJ_ATT_ACTION_CMD;
	    } else {
	      if(atts_found & OBJ_ATT_TEXT) {
	        variable_declaration(j, f, jlo, jlo->text, OBJ_ATT_TEXT);
	        atts_used = OBJ_ATT_TEXT;
	      } else {
	        variable_declaration(j, f, jlo, NULL, 0);
	      }
	    }
	  }
        }
      }
      set_attributes(j, f, jlo, atts_found, atts_used);
    } break;
    case CLASS_JSCROLLPANE: {
      cptr = NULL;
      if((jlo->c_st) && (jlo->c_it)) {
        dksto_it_reset(jlo->c_it);
	cc = 1;
	while(cc) {
	  jcn = (JCN *)dksto_it_next(jlo->c_it);
	  if(jcn) {
	    switch(jcn->t) {
	      case JCN_OBJECT: {
	        if(jcn->o) {
		  cptr = (jcn->o)->on;
		  cc = 0;
		}
	      } break;
	    }
	  } else {
	    cc = 0;
	  }
	}
      }
      variable_declaration(j, f, jlo, cptr, OBJ_ATT_CONSTRUCTOR);
    } break;
    default: {
      if(jlo->on) {
        variable_declaration(
          j, f, jlo, jlo->constructor,
	  ((jlo->constructor) ? OBJ_ATT_CONSTRUCTOR : 0)
        );
        if(jlo->constructor) { atts_used |= OBJ_ATT_CONSTRUCTOR; }
      }
      set_attributes(j, f, jlo, atts_found, atts_used);
    } break;
  }
  set_border(j, f, jlo);
}



/**	Write code to create a button group and add
	all buttons.
	@param	j	Jlayout job.
	@param	jsg	Size group.
	@param	f	Output file.
*/
static
void
create_button_group DK_P3(JLJ *,j, JSG *,jsg, FILE *,f)
{
  JLO *jlo;
  line_start(f, 0);
  fputs(jsg->n, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(str_button_group, f);
  fputs(str_bropen, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  dksto_it_reset(jsg->i);
  while((jlo = (JLO *)dksto_it_next(jsg->i)) != NULL) {
    line_start(f, 0);
    fputs(jsg->n, f);
    fputs(str_dot, f);
    fputs(str_add, f);
    fputs(str_bropen, f);
    if(jlo->on) {
      fputs(jlo->on, f);
    } else {
      fputs(str_this, f);
    }
    fputs(str_brclose, f);
    fputs(str_semicolon, f);
    fputs(str_nl, f);
  }
}



/**	Write code to retrieve and set sizes for all
	objects of a size group.
	@param	j	Jlayout job.
	@param	jsg	Size group.
	@param	f	Output file.
*/
static
void
create_size_group DK_P3(JLJ *,j, JSG *,jsg, FILE *,f)
{
  JLO *jlo;
  line_start(f, 0);
  fputs(str_jlayout_reset_sizes, f);
  fputs(str_bropen, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  dksto_it_reset(jsg->i);
  while((jlo = (JLO *)dksto_it_next(jsg->i)) != NULL) {
    if(jlo->on) {
      line_start(f, 0);
      fputs(str_jlayout_get_size, f);
      fputs(str_bropen, f);
      fputs(jlo->on, f);
      fputs(str_brclose, f);
      fputs(str_semicolon, f);
      fputs(str_nl, f);
    }
  }
  dksto_it_reset(jsg->i);
  while((jlo = (JLO *)dksto_it_next(jsg->i)) != NULL) {
    if(jlo->on) {
      line_start(f, 0);
      fputs(str_jlayout_set_size, f);
      fputs(str_bropen, f);
      fputs(jlo->on, f);
      fputs(str_brclose, f);
      fputs(str_semicolon, f);
      fputs(str_nl, f);
    }
  }
}



/**	Write object identifier for which we create a
	new Layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
add_object_or_this_or_cp DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  if((jlo->f2) & F2_USE_CONTENT_PANE) {
    fputs(str_cp, f);
  } else {
    if(jlo->on) {
      fputs(jlo->on, f);
    } else {
      fputs(str_this, f);
    }
  }
}



/**	Write name (identidifer, empty string or
	content pane) of object for add() method.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
add_object_to_add  DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  if((jlo->f2) & F2_USE_CONTENT_PANE) {
    fputs(str_cp, f); fputs(str_dot, f);
  } else {
    if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
  }
}



/**	Write start of .add() instruction.
	This is either a method of the current object,
	a specified object or the content pane of the
	current object.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
add_add DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  add_object_to_add(j, f, jlo);
  fputs(str_add, f);
}



/**	Write code to add horizontal or vertical glue.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	hv	Flag to add space horizontally (0) or vertically (1).
*/
static
void
add_glue DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int,hv)
{
  line_start(f, 1);
  add_add(j, f, jlo);
  fputs(str_bropen, f);
  if(hv) {
    fputs(str_box_create_vertical_glue, f);
  } else {
    fputs(str_box_create_horizontal_glue, f);
  }
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Write code to add a rigid area.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	hv	Flag to add space horizontally (0) or vertically (1).
*/
static
void
add_space DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int,hv)
{
  line_start(f, 1);
  add_add(j, f, jlo);
  fputs(str_bropen, f);
  fputs(str_box_create_rigid_area, f);
  fputs(str_bropen, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(str_dimension, f);
  fputs(str_bropen, f);
  if(hv) {
    fputs(str_null, f);
    fputs(str_comma, f);
    fputs(str_space, f);
    if(jlo->border) {
      fputs(jlo->border, f);
    } else {
      fputs(str_one, f);
    }
  } else {
    if(jlo->border) {
      fputs(jlo->border, f);
    } else {
      fputs(str_one, f);
    }
    fputs(str_comma, f);
    fputs(str_space, f);
    fputs(str_null, f);
  }
  fputs(str_brclose, f);
  fputs(str_brclose, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Set contraints and add an object.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	on	Name of object to add.
*/
static
void
set_constraints_and_add DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, char *,on)
{
  line_start(f, 1);
  fputs(str_gbl, f);
  fputs(str_dot, f);
  fputs(str_set_constraints, f);
  fputs(str_bropen, f);
  fputs(on, f);
  fputs(str_comma, f);
  fputs(str_space, f);
  fputs(str_gbc, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 1);
  add_add(j, f, jlo);
  fputs(str_bropen, f);
  fputs(on, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Write double value to file.
	@param	f	Output file.
	@param	d	Value to write.
*/
static
void
double_to_file DK_P2(FILE,*f, double,d) {
  char buffer[32];
  sprintf(buffer, "%lg", d);
  fputs(buffer, f);
  if(!dkstr_chr(buffer, '.')) {
    fputs(str_dot_zero, f);
  }
}



/**	Create distance panel for borders.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	posx	X position of the panel.
	@param	posy	Y position of the panel.
*/
static
void
border_distance_panel DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, long,posx, long,posy)
{
  double d;
  line_start(f, 1);
  fputs(str_dp, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_jlayout_create_distance_panel, f);
  fputs(str_bropen, f);
  if(jlo->border) {
    fputs(jlo->border, f);
  } else {
    fputs(str_null, f);
  }
  fputs(str_comma, f);
  fputs(str_space, f);
  fputs(str_false, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 1);
  fputs(str_gbc, f);
  fputs(str_space , f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_jlayout_gbc, f);
  fputs(str_bropen, f);
  fputs(str_nl, f);
  line_start(f, 2);
  fprintf(f, "%ld, %ld", posx, posy);
  fputs(str_comma, f); fputs(str_space, f);
  fputs(str_one, f); fputs(str_comma, f); fputs(str_space, f);
  fputs(str_one, f); fputs(str_comma, f); fputs(str_nl, f);
  line_start(f, 2);
  fputs(str_grid_bag_constraints, f);
  fputs(str_dot, f);
  fputs(fill_kw[3], f);
  fputs(str_comma, f);
  fputs(str_space, f);
  fputs(str_grid_bag_constraints, f);
  fputs(str_dot, f);
  fputs(anchor_kw[25], f);
  fputs(str_comma, f);
  fputs(str_nl, f);
  line_start(f, 2);
  /* weight x */
  d = ((posx) ? jlo->weightr : jlo->weightl);
  double_to_file(f, d); fputs(str_comma, f); fputs(str_space, f);
  /* weight y */
  d = ((posy) ? jlo->weightb : jlo->weightt);
  double_to_file(f, d); fputs(str_nl, f);
  line_start(f, 1);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  set_constraints_and_add(j, f, jlo, str_dp);
}



/**	Write code to create a distance panel.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	posx	X position of the panel.
	@param	posy	Y position of the panel.
*/
static
void
normal_distance_panel DK_P5(JLJ *,j, FILE *,f, JLO *,jlo, long,posx, long,posy)
{
  line_start(f, 1);
  fputs(str_dp, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_jlayout_create_distance_panel, f);
  fputs(str_bropen, f);
  if(jlo->border) {
    fputs(jlo->border, f);
  } else {
    fputs(str_null, f);
  }
  fputs(str_comma, f);
  fputs(str_space, f);
  fputs(str_true, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 1);
  fputs(str_gbc, f);
  fputs(str_space , f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_jlayout_gbc, f);
  fputs(str_bropen, f);
  fputs(str_nl, f);
  line_start(f, 2);
  fprintf(f, "%ld, %ld", posx, posy);
  fputs(str_comma, f); fputs(str_space, f);
  fputs(str_one, f); fputs(str_comma, f); fputs(str_space, f);
  fputs(str_one, f); fputs(str_comma, f); fputs(str_nl, f);
  line_start(f, 2);
  fputs(str_grid_bag_constraints, f);
  fputs(str_dot, f);
  fputs(fill_kw[0], f);
  fputs(str_comma, f);
  fputs(str_space, f);
  fputs(str_grid_bag_constraints, f);
  fputs(str_dot, f);
  fputs(anchor_kw[25], f);
  fputs(str_comma, f);
  fputs(str_nl, f);
  line_start(f, 2);
  fputs(str_zero_zero, f); fputs(str_comma, f); fputs(str_space, f);
  fputs(str_zero_zero, f); fputs(str_nl, f);
  line_start(f, 1);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  set_constraints_and_add(j, f, jlo, str_dp);
}



/**	Write code to add objects in a gridbag layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
gridbag_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JCN *jcn; JLO *o;
  int fill, anchor;
  long x, y, xd, yd;
  
  line_start(f, 1);
  fputs(str_new_grid_bag_layout, f);
  fputs(str_nl, f);
  line_start(f, 1);
  add_object_to_add(j, f, jlo);
  fputs(str_set_layout, f);
  fputs(str_bropen, f);
  fputs(str_gbl, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 1);
  fputs(str_grid_bag_constraints, f);
  fputs(str_space, f);
  fputs(str_gbc, f);
  fputs(str_space, f);
  fputs(str_equal, f);
  fputs(str_space, f);
  fputs(str_ref_null, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  line_start(f, 1);
  fputs(str_jpanel_dp, f);
  fputs(str_nl, f);
  /* border distance panels */
  border_distance_panel(j, f, jlo, 0L, 0L);
  border_distance_panel(j, f, jlo, 0L, (1L + jlo->cor_my));
  border_distance_panel(j, f, jlo, (1L + jlo->cor_mx), 0L);
  border_distance_panel(j, f, jlo, (1L + jlo->cor_mx), (1L + jlo->cor_my));
  /* normal distance panels */
  if(jlo->distance_x) {
    x = xd = 0L;
    for(x = 0L; x <= jlo->max_x; x++) {
      if(dkbf_get(jlo->distance_x, x)) {
        if(x != 0L) {
	  normal_distance_panel(j, f, jlo, xd, 0L);
	}
	xd++;
      }
      xd++;
    }
  }
  if(jlo->distance_y) {
    y = yd = 0L;
    for(y = 0L; y <= jlo->max_y; y++) {
      if(dkbf_get(jlo->distance_y, y)) {
        if(y != 0L) {
	  normal_distance_panel(j, f, jlo, 0L, yd);
	}
	yd++;
      }
      yd++;
    }
  }
  /* objects */
  dksto_it_reset(jlo->c_it);
  while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
    switch(jcn->t) {
      case JCN_OBJECT: {
        if(jcn->o) {
	  o = jcn->o;
	  if(o->on) {
	    fill = o->fill; anchor = o->anchor;
	    line_start(f, 1);
	    fputs(str_gbc, f);
	    fputs(str_space, f);
	    fputs(str_equal, f);
	    fputs(str_space, f);
	    fputs(str_jlayout_gbc, f);
	    fputs(str_bropen, f);
	    fputs(str_nl, f);
	    line_start(f, 2);
	    fprintf(f, "%ld, %ld, %ld, %ld,\n",
	      o->cor_x, o->cor_y, o->cor_w, o->cor_h
	    );
	    if(fill < FILL_NONE) fill = 0;
	    if(fill > FILL_BOTH) fill = FILL_BOTH;
	    if(anchor < ANCHOR_CENTER) anchor = ANCHOR_CENTER;
	    if(anchor > ANCHOR_MAX) anchor = ANCHOR_MAX;
	    if(anchor == ANCHOR_CENTER) anchor = ANCHOR_MAX;
	    else anchor = anchor - 1;
	    line_start(f, 2);
	    fputs(str_grid_bag_constraints, f);
	    fputs(str_dot, f);
	    fputs(fill_kw[fill], f);
	    fputs(str_comma, f);
	    fputs(str_space, f);
	    fputs(str_grid_bag_constraints, f);
	    fputs(str_dot, f);
	    fputs(anchor_kw[anchor], f);
	    fputs(str_comma, f);
	    fputs(str_nl, f);
	    line_start(f, 2);
	    if(o->weightx >= 0.0) fprintf(f, "%lg", o->weightx);
	    else fputs(str_zero_zero, f);
	    fputs(str_comma, f);
	    fputs(str_space, f);
	    if(o->weighty >= 0.0) fprintf(f, "%lg", o->weighty);
	    else fputs(str_zero_zero, f);
	    fputs(str_nl, f);
	    line_start(f, 1);
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	    set_constraints_and_add(j, f, jlo, o->on);
	  } else {
	    
	  }
	}
      } break;
      default: {
        /* Error: dont know how to handle glue or separator here */
      } break;
    }
  } 
}



/**	Write code to add objects in a card layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
card_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JCN *jcn;
  unsigned long ul = 0UL;
  line_start(f, 1);
  add_object_to_add(j, f, jlo);
  fputs(str_set_layout, f);
  fputs(str_bropen, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(str_card_layout, f);
  fputs(str_bropen, f);
  fputs(str_brclose, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
  dksto_it_reset(jlo->c_it);
  while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
    switch(jcn->t) {
      case JCN_OBJECT: {
        if(jcn->o) {
	  if((jcn->o)->on) {
	    line_start(f, 1);

	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs((jcn->o)->on, f);
	    fputs(str_comma, f);
	    fputs(str_space, f);
	    if((jcn->o)->card_add_text) {
	      fputs((jcn->o)->card_add_text, f);
	    } else {
	      fprintf(f, "\"%lu\"", ul);
	    }
	    fputs(str_brclose, f);

	    fputs(str_semicolon, f);
	    fputs(str_nl, f);

	    ul++;
	  }
	}
      } break;
      case JCN_GLUE: {
        /* dont know how to add glue here */
      } break;
      case JCN_SEPARATOR: {
        /* dont know how to add separator here */
      } break;
    }
  }
}



/**	Write code to add objects in a border layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
border_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JCN *jcn; JLO *o;
  dksto_it_reset(jlo->c_it);
  while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
    switch(jcn->t) {
      case JCN_OBJECT: {
        o = jcn->o;
	if(o) {
	  if(o->on) {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs(o->on, f);
	    fputs(str_comma, f);
	    fputs(str_space, f);
	    fputs(str_border_layout, f);
	    fputs(str_dot, f);
	    switch(o->anchor) {
	      case ANCHOR_NORTH: {
	        fputs(anchor_kw[0], f);
	      } break;
	      case ANCHOR_NORTHEAST: {
	        fputs(anchor_kw[1], f);
	      } break;
	      case ANCHOR_EAST: {
	        fputs(anchor_kw[2], f);
	      } break;
	      case ANCHOR_SOUTHEAST: {
	        fputs(anchor_kw[3], f);
	      } break;
	      case ANCHOR_SOUTH: {
	        fputs(anchor_kw[4], f);
	      } break;
	      case ANCHOR_SOUTHWEST: {
	        fputs(anchor_kw[5], f);
	      } break;
	      case ANCHOR_WEST: {
	        fputs(anchor_kw[6], f);
	      } break;
	      case ANCHOR_NORTHWEST: {
	        fputs(anchor_kw[7], f);
	      } break;
	      case ANCHOR_PAGE_START: {
	        fputs(anchor_kw[8], f);
	      } break;
	      case ANCHOR_PAGE_END: {
	        fputs(anchor_kw[9], f);
	      } break;
	      case ANCHOR_LINE_START: {
	        fputs(anchor_kw[10], f);
	      } break;
	      case ANCHOR_LINE_END: {
	        fputs(anchor_kw[11], f);
	      } break;
	      case ANCHOR_FIRST_LINE_START: {
	        fputs(anchor_kw[12], f);
	      } break;
	      case ANCHOR_FIRST_LINE_END: {
	        fputs(anchor_kw[13], f);
	      } break;
	      case ANCHOR_LAST_LINE_START: {
	        fputs(anchor_kw[14], f);
	      } break;
	      case ANCHOR_LAST_LINE_END: {
	        fputs(anchor_kw[15], f);
	      } break;
	      case ANCHOR_BASELINE: {
	        fputs(anchor_kw[16], f);
	      } break;
	      case ANCHOR_BASELINE_LEADING: {
	        fputs(anchor_kw[17], f);
	      } break;
	      case ANCHOR_BASELINE_TRAILING: {
	        fputs(anchor_kw[18], f);
	      } break;
	      case ANCHOR_ABOVE_BASELINE: {
	        fputs(anchor_kw[19], f);
	      } break;
	      case ANCHOR_ABOVE_BASELINE_LEADING: {
	        fputs(anchor_kw[20], f);
	      } break;
	      case ANCHOR_ABOVE_BASELINE_TRAILING: {
	        fputs(anchor_kw[21], f);
	      } break;
	      case ANCHOR_BELOW_BASELINE: {
	        fputs(anchor_kw[22], f);
	      } break;
	      case ANCHOR_BELOW_BASELINE_LEADING: {
	        fputs(anchor_kw[23], f);
	      } break;
	      case ANCHOR_BELOW_BASELINE_TRAILING: {
	        fputs(anchor_kw[24], f);
	      } break;
	      default: {			/* center */
	        fputs(anchor_kw[25], f);
	      } break;
	    }
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	  }
	}
      } break;
      case JCN_GLUE: {
        /* dont know how to add glue here */
      } break;
      case JCN_SEPARATOR: {
        /* dont know how to add separator here */
      } break;
    }
  }
}



/**	Write code to add objects in a box layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	hv	Flag to add horizontal (0) or vertical (1) space.
*/
static
void
box_layout DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int,hv)
{
  JCN *jcn; JLO *o;
  dksto_it_reset(jlo->c_it);
  while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
    add_space(j, f, jlo, hv);
    switch(jcn->t) {
      case JCN_OBJECT: {
        if(jcn->o) {
	  o = jcn->o;
	  if(o->on) {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs(o->on, f);
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	  }
	}
      } break;
      case JCN_GLUE: {
        /* dont know how to add glue here */
	add_glue(j, f, jlo, hv);
      } break;
      case JCN_SEPARATOR: {
        /* dont know how to add a separator here */
      } break;
    }
  }
  add_space(j, f, jlo, hv);
}



/**	Write code to install a box layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
	@param	cr	Box axis: 0=x, 1=y, 2=page, 3=line.
*/
static
void
set_box_layout DK_P4(JLJ *,j, FILE *,f, JLO *,jlo, int,cr)
{
  line_start(f, 1);
  add_object_to_add(j, f, jlo);
  fputs(str_set_layout, f);
  fputs(str_bropen, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(str_box_layout, f);
  fputs(str_bropen, f);
  add_object_or_this_or_cp(j, f, jlo);
  fputs(str_comma, f);
  fputs(str_space, f);
  fputs(str_box_layout, f);
  fputs(str_dot, f);
  switch(cr) {
    case 1: {
      fputs(str_y_axis, f);
    } break;
    case 2: {
      fputs(str_page_axis, f);
    } break;
    case 3: {
      fputs(str_line_axis, f);
    } break;
    default: {
      fputs(str_x_axis, f);
    } break;
  }
  fputs(str_brclose, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Write code to set up flow layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
set_flow_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  line_start(f, 1);
  add_object_to_add(j, f, jlo);
  fputs(str_set_layout, f);
  fputs(str_bropen, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(str_flow_layout, f);
  fputs(str_bropen, f);
  fputs(str_brclose, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Write code to set up grid layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
set_grid_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  line_start(f, 1);
  add_object_to_add(j, f, jlo);
  fputs(str_set_layout, f);
  fputs(str_bropen, f);
  fputs(str_new, f);
  fputs(str_space, f);
  fputs(str_grid_layout, f);
  fputs(str_bropen, f);
  fprintf(f, "%ld, %ld", jlo->grid_layout_r, jlo->grid_layout_c);
  fputs(str_brclose, f);
  fputs(str_brclose, f);
  fputs(str_semicolon, f);
  fputs(str_nl, f);
}



/**	Write code to add objects in a flow layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
flow_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JCN *jcn;
  unsigned long ul = 0UL;
  
  dksto_it_reset(jlo->c_it);
  while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
    switch(jcn->t) {
      case JCN_OBJECT: {
        if(jcn->o) {
	  if((jcn->o)->on) {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs((jcn->o)->on, f);
	    switch(jlo->ocn) {
	      case CLASS_JTABBEDPANE: {
	        fputs(str_comma, f);
		fputs(str_space, f);
	        if((jcn->o)->card_add_text) {
		  fputs((jcn->o)->card_add_text, f);
		} else {
		  fprintf(f, "\"%lu\"", ul);
		}
	      } break;
	    }
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	    ul++;
	  }
	}
      } break;
      case JCN_GLUE: {
        switch(jlo->layout) {
	  case LAYOUT_BOX_Y:
	  case LAYOUT_BOX_PAGE:
	  {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs(str_box_create_vertical_glue, f);
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	  } break;
	  default:
	  {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs(str_box_create_horizontal_glue, f);
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	  } break;
	}
      } break;
      case JCN_SEPARATOR: {
        switch(jlo->ocn) {
	  case CLASS_JMENU:
	  case CLASS_JPOPUPMENU:
	  {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	  } break;
	}
      } break;
    }
  } 
}



/**	Write code to add components in a grid layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
grid_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JCN *jcn;
  unsigned long ul = 0UL;
  
  set_grid_layout(j, f, jlo);
  dksto_it_reset(jlo->c_it);
  while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
    switch(jcn->t) {
      case JCN_OBJECT: {
        if(jcn->o) {
	  if((jcn->o)->on) {
	    line_start(f, 1);
	    add_add(j, f, jlo);
	    fputs(str_bropen, f);
	    fputs((jcn->o)->on, f);
	    fputs(str_brclose, f);
	    fputs(str_semicolon, f);
	    fputs(str_nl, f);
	    ul++;
	  }
	}
      } break;
    }
  } 
}



/**	Write code to add components in a group layout.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
group_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  /* ERROR: GroupLayout not supported */
  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 36);
}



/**	Write code to add components in a spring layout.
	Note: jLayout does not support SpringLayout, so
	we print an error message here.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
spring_layout DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  /* ERROR: SpringLayout not supported */
  jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 37);
}



/**	Write code to add all contents objects to the
	current object.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
add_contents DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  int have_contents = 0;
  JCN *jcn;
  
  if(jlo->c_st) {
    if(jlo->c_it) {
      dksto_it_reset(jlo->c_it);
      while((jcn = (JCN *)dksto_it_next(jlo->c_it)) != NULL) {
        have_contents = 1;	
      }
    }
  }
  if(jlo->menubar) {
    if((jlo->menubar)->on) {
      have_contents = 1;	
    }
  }
  if((have_contents) && (jlo->ocn != CLASS_JSCROLLPANE)) { 
    line_start(f, 0);
    fputs(str_block_open, f);
    fputs(str_nl, f);
    if(jlo->menubar) {
      if((jlo->menubar)->on) {
        line_start(f, 1);
        if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
	fputs(str_set_j_menu_bar, f);
	fputs(str_bropen, f);
	fputs((jlo->menubar)->on, f);
	fputs(str_brclose, f);
	fputs(str_semicolon, f);
	fputs(str_nl, f);
      }
    }
    switch(jlo->ocn) {
      case CLASS_JSCROLLPANE: {
        /* contents already added */
      } break;
      default: {
        if((jlo->f2) & F2_USE_CONTENT_PANE) {
          line_start(f, 1);
          fputs(str_container_cp, f);
          fputs(str_nl, f);
          line_start(f, 1);
          fputs(str_cp, f); fputs(str_space, f);
          fputs(str_equal, f); fputs(str_space, f);
          if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
          fputs(str_get_content_pane, f);
          fputs(str_semicolon, f);
          fputs(str_nl, f);
        }
        switch(jlo->layout) {
          case LAYOUT_GRIDBAG: {
            gridbag_layout(j, f, jlo);
          } break;
          case LAYOUT_CARD: {
            card_layout(j, f, jlo);
          } break;
          case LAYOUT_BORDER: {
            border_layout(j, f, jlo);
          } break;
          case LAYOUT_BOX_X: {
	    set_box_layout(j, f, jlo, 0); box_layout(j, f, jlo, 0);
          } break;
          case LAYOUT_BOX_Y: {
	    set_box_layout(j, f, jlo, 1); box_layout(j, f, jlo, 1);
          } break;
          case LAYOUT_BOX_PAGE: {
	    set_box_layout(j, f, jlo, 2); box_layout(j, f, jlo, 1);
          } break;
          case LAYOUT_BOX_LINE: {
	    set_box_layout(j, f, jlo, 3); box_layout(j, f, jlo, 0);
          } break;
          case LAYOUT_FLOW:
          {
            set_flow_layout(j, f, jlo);
          }
          case LAYOUT_NOT_SPECIFIED:
          case LAYOUT_NONE:
          {
            flow_layout(j, f, jlo);
          } break;
          case LAYOUT_GRID: {
            grid_layout(j, f, jlo);
          } break;
          case LAYOUT_GROUP: {
            group_layout(j, f, jlo);
          } break;
          case LAYOUT_SPRING: {
            spring_layout(j, f, jlo);
          } break;
          default: {
          } break;
        }
        line_start(f, 0);
        fputs(str_block_close, f);
        fputs(str_nl, f);
      } break;
    }
  } 
}



/**	Write function to add all listeners.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
add_all_listeners DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JLISTENER *jl;
  
  if(jlo->l_st) {				
    if(jlo->l_it) {				
      dksto_it_reset(jlo->l_it);
      while((jl = (JLISTENER *)dksto_it_next(jlo->l_it)) != NULL) {
        
        if((jl->t) && (jl->tp >= 0) && (jl->tp <= LISTENER_MAX)) {
	  
	  line_start(f, 0);
	  switch(jl->tp) {
	    case LISTENER_TRANSFER_HANDLER: { 
	      if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
	      fputs(str_set_transfer_handler, f);
	      fputs(str_bropen, f);
	      fputs(jl->t, f);
	      fputs(str_brclose, f);
	      fputs(str_semicolon, f);
	      fputs(str_nl, f);
	    } break;
	    default: {		
	      if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
	      fputs(str_add, f);
	      fputs(listener_kw[jl->tp], f);
	      fputs(str_listener, f);
	      fputs(str_bropen, f);
	      fputs(jl->t, f);
	      fputs(str_brclose, f);
	      fputs(str_semicolon, f);
	      fputs(str_nl, f);
	    } break;
	  }
	}
      }
    }
  } 
}



/**	Write function to remove all listeners from object.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	jlo	Current object.
*/
static
void
remove_all_listeners DK_P3(JLJ *,j, FILE *,f, JLO *,jlo)
{
  JLISTENER *jl;
  if(jlo->li_st) {
    if(jlo->li_it) {
      dksto_it_reset(jlo->li_it);
      while((jl = (JLISTENER *)dksto_it_next(jlo->li_it)) != NULL) {
        if((jl->t) && (jl->tp >= 0) && (jl->tp <= LISTENER_MAX)) {
	  switch(jl->tp) {
	    case LISTENER_TRANSFER_HANDLER: {
	    } break;
	    default: {
	      line_start(f, 0);
	      if(jlo->on) { fputs(jlo->on, f); fputs(str_dot, f); }
	      fputs(str_remove, f);
	      fputs(listener_kw[jl->tp], f);
	      fputs(str_listener, f);
	      fputs(str_bropen, f);
	      fputs(jl->t, f);
	      fputs(str_brclose, f);
	      fputs(str_semicolon, f);
	      fputs(str_nl, f);
	    } break;
	  }
	}
      }
    }
  }
}


/**	Write the setup functions to the output file.
	@param	j	Jlayout job.
	@param	f	Output file.
*/
static
void
write_section_setup DK_P2(JLJ *,j, FILE *,f)
{
  unsigned long passno;
  JLO *jlo; JSG *jsg;
  
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[2], f);
  fputs(str_nl, f);
  start_fold_marker(f, j);
  print_str_array(f, start_of_jlayout_section, 0);
  if((j->f1) & F1_HAVE_SIZE_GROUP) {
    print_str_array(f, str_functions_for_sizes, 1);
  }
  if((j->f1) & F1_HAVE_ICON_FILE_NAMES) {
    if(!(j->use_dklibsj)) {
      print_str_array(f, str_fct_find_image_for_name, 1);
    }
  }
  if((j->f1) & F1_HAVE_GRIDBAG) {
    print_str_array(f, str_functions_for_gridbag_layout, 1);
    print_str_array(f, str_for_gridbag, 1);
  }
  if((j->f1) & F1_HAVE_FONT_STYLE) {
    print_str_array(f, str_for_font_change, 1);
  }
  
  print_str_array(f, str_fct_setup_begin, 1);
  
  /* create objects */
  if((j->f1) & F1_HAVE_ICON_FILE_NAMES) {
    line_start(f, 0); fputs(str_image_icon, f); fputc('\n', f);
  }
  for(passno = 1UL; passno <= j->max_level; passno++) {
    
    dksto_it_reset(j->o_it);
    while((jlo = (JLO *)dksto_it_next(j->o_it)) != NULL) {
      if(passno == jlo->obj_lvl) {
        create_object(j, f, jlo);
	add_contents(j, f, jlo);
      }
    }
  }
  create_object(j, f, j->o_main);
  add_contents(j, f, j->o_main);
  /* button groups */
  dksto_it_reset(j->b_it);
  while((jsg = (JSG *)dksto_it_next(j->b_it)) != NULL) {
    create_button_group(j, jsg, f);
  }
  /* size groups */
  dksto_it_reset(j->g_it);
  while((jsg = (JSG *)dksto_it_next(j->g_it)) != NULL) {
    create_size_group(j, jsg, f);
  }
  for(passno = 1UL; passno <= j->max_level; passno++) {
    dksto_it_reset(j->o_it);
    while((jlo = (JLO *)dksto_it_next(j->o_it)) != NULL) {
      if(passno == jlo->obj_lvl) {
        add_all_listeners(j, f, jlo);
      }
    }
  }
  add_all_listeners(j, f, j->o_main);
  
  print_str_array(f, str_fct_setup_end, 1);
  
  print_str_array(f, end_of_jlayout_section, 0);
  end_fold_marker(f, j);
  
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[3], f);
  fputs(str_nl, f);
  
}



/**	Write the cleanup function to the output file.
	@param	j	Jlayout job.
	@param	f	Output file.
*/
static
void
write_section_cleanup DK_P2(JLJ *,j, FILE *,f)
{
  unsigned long passno;
  JLO *jlo;
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[4], f);
  fputs(str_nl, f);
  start_fold_marker(f, j);
  print_str_array(f, start_of_jlayout_section, 0);
  print_str_array(f, str_fct_cleanup_begin, 1);
  for(passno = j->max_level; passno > 0UL; passno--) {
    dksto_it_reset(j->o_it);
    while((jlo = (JLO *)dksto_it_next(j->o_it)) != NULL) {
      if(passno == jlo->obj_lvl) {
        remove_all_listeners(j, f, jlo);
      }
    }
  }
  print_str_array(f, str_fct_setup_end, 1);
  print_str_array(f, end_of_jlayout_section, 0);
  end_fold_marker(f, j);
  fputs(comment_start, f);
  fputs(str_space, f);
  fputs(special_comments[5], f);
  fputs(str_nl, f);
}




/**	Check whether one of the special sections must follow
	the previous line. If necessary print the section.
	@param	j	Jlayout job.
	@param	f	Output file.
	@param	l	Current line number.
*/
static
void
special_sections DK_P3(JLJ *,j, FILE *,f, unsigned long,l)
{
  if(l == j->ln_import) {
    write_section_import(j, f, l);
  }
  if(l == j->ln_variables) {
    write_section_variables(j, f);
  }
  if(l == j->ln_setup) {
    write_section_setup(j, f);
  }
  if(l == j->ln_cleanup) {
    write_section_cleanup(j, f);
  }
}



int
jsrc_write DK_P1(JLJ *,j)
{
  int back = 0;
  FILE *output_file = NULL, *opened_file = NULL;
  JSRC_LINE *lp;
  
  if(j) {
    if(j->ofname) {
      if(j->a) {
        output_file = opened_file = dkapp_fopen(j->a, j->ofname, str_w);
      } else {
        output_file = opened_file = dksf_fopen(j->ofname, str_w);
      }
    } else {
      output_file = stdout;
    }
    if(output_file) {
      back = 1;
      dksto_it_reset(j->l_it);
      /* special_sections(j, output_file, 0UL); */
      while((lp = (JSRC_LINE *)dksto_it_next(j->l_it)) != NULL) {
        fputs(lp->t, output_file);
	fputc('\n', output_file);
	special_sections(j, output_file, lp->lineno);
      }
    } else {
      /* ERROR: Failed to open output file */
      jlmsg_log1(j, DK_LOG_LEVEL_ERROR, 52);
    }
    if(opened_file) {
      fclose(opened_file); opened_file = NULL;
    }
    output_file = NULL;
  }
  
  return back;
}




