/*
Copyright (c) 2007-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	dksdgdbm.c	Simple database API for GDBM databases.
*/



#include "dk.h"

#if DK_HAVE_STRING_H
#include <string.h>
#endif
#if DK_HAVE_FCNTL_H
#include <fcntl.h>
#endif
#if DK_HAVE_IO_H
#include <io.h>
#endif

#include "dkmem.h"
#include "dkstr.h"



/**	Inside the dksdgdbm module.
*/
#define DKSDBI_GDBM	1

#include "dksdbi.h"
#include "dksdbii.h"




#line 71 "dksdgdbm.ctr"




#if DK_HAVE_GDBM_H


#include <gdbm.h>



/**	Open database for writing.
	@param	p	Simple database API structure.
	@return	GDBM file opened for writing (success) or NULL (error).
*/
static
GDBM_FILE
open_writer DK_P1(dk_sdbi_db *,p)
{
  GDBM_FILE back = NULL;
  int wrmode;
  int b;
  
  wrmode = GDBM_WRCREAT;
  if((p->acc) & DK_SDBI_MODE_TRUNCATE) {
    if(!(p->did_truncate)) {
      p->did_truncate = 1;
      wrmode = GDBM_NEWDB;
    }
  }
  b = p->blksize;
  if(!b) { b = 512; }
  back = gdbm_open(p->fn, b, wrmode, p->tfm, NULL);
  
  return back;
}




/**	Open database for reading.
	@param	p	Simple database API structure.
	@return	GDBM file opened for reading (success) or NULL (error).
*/
static
GDBM_FILE
open_reader DK_P1(dk_sdbi_db *,p)
{
  GDBM_FILE back = NULL;
  int b;
  
  b = p->blksize;
  if(!b) { b = 512; }
  back = gdbm_open(p->fn, b, GDBM_READER, p->tfm, NULL);
  
  return back;
}



int
dksdbi_gdbm_close DK_P1(dk_sdbi_db *,p)
{
  int back = 0;
  GDBM_FILE dbf = NULL;
  
  if(p->acc != DK_SDBI_MODE_RDWR) {
    dbf = (GDBM_FILE)(p->dbptr);
    if(dbf) { gdbm_close(dbf); p->dbptr = NULL; }
  }
  
  return back;
}



int
dksdbi_gdbm_open DK_P1(dk_sdbi_db *,p)
{
  int back = 0;
  GDBM_FILE dbf;
  
  if(p->acc == DK_SDBI_MODE_RDWR) {
    dbf = open_writer(p);
    if(dbf) {
      gdbm_close(dbf);
      dbf = open_reader(p);
      if(dbf) {
        gdbm_close(dbf); back = 1;
      }
    }
  } else {
    if((p->acc) & DK_SDBI_MODE_WRITE) {
      dbf = open_writer(p);
      if(dbf) { p->dbptr = (void *)dbf; back = 1; }
    } else {
      if((p->acc) & DK_SDBI_MODE_READ) {
        dbf = open_reader(p);
	if(dbf) { p->dbptr = (void *)dbf; back = 1; }
      }
    }
  }
  
  return back;
}



int
dksdbi_gdbm_store DK_P6(dk_sdbi_db *,p, void *,kp, size_t,kl, void *,vp, size_t,vl, int,insmod)
{
  int back = 0;
  GDBM_FILE dbf = NULL;
  datum k, v; int ret;
  
  if(p->acc == DK_SDBI_MODE_RDWR) {
    dbf = open_writer(p);
  } else {
    dbf = (GDBM_FILE)(p->dbptr);
  }
  if(dbf) {
    DK_MEMRES(&k,sizeof(datum)) ; DK_MEMRES(&v,sizeof(datum)) ;
    k.dptr = kp; k.dsize = kl; v.dptr = vp; v.dsize = vl;
    ret = gdbm_store(dbf, k, v, GDBM_INSERT);
    if(ret == 0) {
      back = 1;
    } else {
      if(ret == 1) {
        DK_MEMRES(&k,sizeof(datum)) ; DK_MEMRES(&v,sizeof(datum)) ;
	k.dptr = kp; k.dsize = kl; v.dptr = vp; v.dsize = vl;
	ret = gdbm_store(dbf, k, v, GDBM_REPLACE);
	if(ret == 0) { back = 1; }
      }
    }
  }
  if(p->acc == DK_SDBI_MODE_RDWR) {
    if(dbf) { gdbm_close(dbf); }
  }
  
  return back;
}



int
dksdbi_gdbm_delete DK_P3(dk_sdbi_db *,p, char *,kp, size_t,kl)
{
  int back = 0;
  GDBM_FILE dbf;
  datum k; int ret;
  
  if((p->acc) == DK_SDBI_MODE_RDWR) {
    dbf = open_writer(p);
  } else {
    dbf = (GDBM_FILE)(p->dbptr);
  }
  if(dbf) {
    DK_MEMRES(&k,sizeof(datum)) ;
    k.dptr = kp; k.dsize = kl;
    ret = gdbm_delete(dbf, k);
    if(ret == 0) { back = 1; }
  }
  if((p->acc) == DK_SDBI_MODE_RDWR) {
    if(dbf) { gdbm_close(dbf); }
  }
  
  return back;
}



int
dksdbi_gdbm_fetch DK_P5(dk_sdbi_db *,p, void *,kp, size_t,kl, void *,vp, size_t *,vl)
{
  int back = 0;
  datum k, v;
  GDBM_FILE dbf;
  
  if(p->acc == DK_SDBI_MODE_RDWR) {
    dbf = open_reader(p);
  } else {
    dbf = (GDBM_FILE)(p->dbptr);
  }
  if(dbf) {
    DK_MEMRES(&k,sizeof(datum)) ; DK_MEMRES(&v,sizeof(datum)) ;
    k.dptr = kp; k.dsize = kl;
    v = gdbm_fetch(dbf, k);
    if(v.dptr) {
      if(v.dsize <= (*vl)) {
        DK_MEMCPY(vp,v.dptr,v.dsize);
	*vl = v.dsize;
	back = 1;
      } else {
        DK_MEMCPY(vp,v.dptr,(*vl));
	back = -1;
      }
      free(v.dptr);
    } else {
    }
  }
  if(p->acc == DK_SDBI_MODE_RDWR) {
    if(dbf) { gdbm_close(dbf); }
  }
  
  return back;
}



int
dksdbi_gdbm_string_fetch  DK_P4(dk_sdbi_db *,p, char *,kp, char *,vp, size_t,s)
{
  int back = 0;
  GDBM_FILE dbf; datum k, v;
  
  if(p->acc == DK_SDBI_MODE_RDWR) {
    dbf = open_reader(p);
  } else {
    dbf = (GDBM_FILE)(p->dbptr);
  }
  if(dbf) {
    DK_MEMRES(&k,sizeof(datum)) ; DK_MEMRES(&v,sizeof(datum)) ;
    k.dptr = kp; k.dsize = (1 + strlen(kp));
    v = gdbm_fetch(dbf, k);
    if(v.dptr) {
      if(v.dsize) {
        if(v.dsize < s) {
	  DK_MEMCPY(vp,v.dptr,v.dsize) ;
	  vp[v.dsize] = '\0';
	  back = 1;
	} else {
	  DK_MEMCPY(vp,v.dptr,(s-1)) ;
	  vp[s-1] = '\0';
	  back = -1;
	}
      }
      free(v.dptr);
    }
  }
  if(p->acc == DK_SDBI_MODE_RDWR) {
    if(dbf) { gdbm_close(dbf); }
  }
  
  return back;
}



int
dksdbi_gdbm_traverse DK_P3(dk_sdbi_db *,p, void *,d, dk_sdbi_fct_t *,f)
{
  int back = 0;
  GDBM_FILE dbf;
  datum k, nk, v; int cc = 0, found = 0, is_first = 0, can_run = 0;
  
  if(p->acc == DK_SDBI_MODE_RDWR) {
    dbf = open_reader(p);
  } else {
    dbf = (GDBM_FILE)(p->dbptr);
  }
  if(dbf) {
    cc = 1; found = 0; back = 1; is_first = 1;
    DK_MEMRES(&k,sizeof(datum)) ; 
    DK_MEMRES(&nk,sizeof(datum)) ;
    DK_MEMRES(&v,sizeof(datum)) ;
    while(cc) {
      can_run = 0;
      if(is_first) {
        is_first = 0;
        k = gdbm_firstkey(dbf);
        if(k.dptr) {
          can_run = 1;
        } else {
          cc = 0;
        }
      } else {
        nk = gdbm_nextkey(dbf, k);
        free(k.dptr);
        k = nk; DK_MEMRES(&nk,sizeof(datum)) ;
        if(k.dptr) {
          can_run = 1;
        } else {
          cc = 0;
        }
      }
      if(can_run) {
        v = gdbm_fetch(dbf, k);
        if(v.dptr) {
          if(v.dsize) {
            found = 1;
            switch((*f)(d, k.dptr, k.dsize, v.dptr, v.dsize)) {
              case -1: { cc = 0; back = 0; } break;
            }
          }
          free(v.dptr);
        }
      }
    }
    if(k.dptr) {
      free(k.dptr);
    }
    if(!found) { back = 0; }
  }
  if(p->acc == DK_SDBI_MODE_RDWR) {
    if(dbf) { gdbm_close(dbf); }
  }
  
  return back;
}



int
dksdbi_gdbm_sync DK_P1(dk_sdbi_db *,p)
{
  int back = 0;
  
  if(p->acc == DK_SDBI_MODE_RDWR) { back = 1; }
  else {
    if((p->acc) & DK_SDBI_MODE_WRITE) {
      if(p->dbptr) {
        gdbm_sync((GDBM_FILE)(p->dbptr));
	back = 1;
      }
    }
  }
  
  return back;
}



#endif

