/*
Copyright (c) 2004-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file dkfont.h Font handling module.

	This module provides information about the 35 well-known
	PS fonts. For each font we can retrieve LaTeX and PS font name
	and a list of features. use the functions
	dkfont_get_tex_name(),
	dkfont_get_ps_name(),
	dkfont_get_svg_font_id(),
	dkfont_get_svg_family_name(),
	dkfont_get_gs_alias_name(),
	dkfont_get_gs_file_name() and
	dkfont_get_features() for this purpose.

	Since version 1.11.0 the module can read a font configuration
	file for more comfortable and flexible font substitutions.

	First we have to read the configuration file:
@code
	  dk_font_mapping_t *fm = NULL;
	  char fn[] = { "font.cfg" };

	  fm = dkfont_mapping_open();
	  if(fm) {
	    dk_stream_t *is;
	    is = dkstream_openfile(fn, "r", 0, NULL);
	    if(is) {
	      if(dkfont_mapping_add_stream(fm, is)) {
	        ... Here we have successfully read the configuration file ...
	      }
	      dkstream_close(is);
	    }
	    dkfont_mapping_close();
	  }
@endcode
	Once we have read the configuration file we can find font
	mappings:
@code
	  dk_one_font_mapping_t *of;
	  dk_font_replacement_t	*rf;

	  of = dkfont_get_one_font(fm, 25);
	  if(of) {
	    dkfont_one_font_reset(of);
	    while((rf = dkfont_one_font_get(of)) != NULL) {
	      ... Now we have rf pointing to one replacement candidate ...
	    }
	  }
@endcode	
	When we have a replacement candidate rf we can use
	dkfont_rep_get_name(),
	dkfont_rep_get_family(),
	dkfont_rep_get_location(),
	dkfont_rep_get_source_name(),
	dkfont_rep_get_pfb_location(),
	dkfont_rep_get_pfb_source_name(),
	dkfont_rep_get_ttf_location(),
	dkfont_rep_get_ttf_source_name(),
	dkfont_rep_get_font_type(),
	dkfont_rep_get_tex_name(),
	dkfont_rep_get_features(),
	dkfont_rep_get_source_type() and
	dkfont_rep_check_driver()
	to obtain detailed information about the replacement candidate.

	See
	<a href="../../font.html">"dklibs - Font configuration"</a>
	for details about font configuration files.
*/

#ifndef DK_FONT_INC
#define DK_FONT_INC

#include <dk.h>
#include <dktypes.h>
#include <dkfont.h>

#if defined(EXTERN)
#undef EXTERN
#endif
#ifndef DKFONT_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif


/**	Find TeX name of a font.
	@param	number	Index of the font (0-34).
	@return	Pointer to name.
*/
EXTERN char *
dkfont_get_tex_name DK_PR((size_t number));



/**	Find PS name of a font.
	@param	number	Index of the font (0-35).
	@return	Pointer to name.
*/
EXTERN char *
dkfont_get_ps_name DK_PR((size_t number));



/**	Get SVG ID for font.
	@param	number	Index of the font (0-35).
	@return	Pointer to name.
*/
EXTERN char *
dkfont_get_svg_font_id DK_PR((size_t number));



/**	Get SVG font family name.
	@param	number	Index of the font (0-35).
	@return	Pointer to font family name.
*/
EXTERN char *
dkfont_get_svg_family_name DK_PR((size_t number));



/**	Get GhostScript alias name for font.
	@param	number	Index of the font (0-34).
	@return	Pointer to name.
*/
EXTERN char *
dkfont_get_gs_alias_name DK_PR((size_t number));



/**	Get GhostScript font file name.
	@param	number	Index of the font (0-34).
	@return	Pointer to name.
*/
EXTERN char *
dkfont_get_gs_file_name DK_PR((size_t number));


/**	Get new GhostScript font file name.
	@param	number	Index of the font (0-34).
	@return	Pointer to name.
*/
EXTERN char *
dkfont_get_new_gs_file_name DK_PR((size_t number));


/**	Get font features.
	@param	number	Index of the font (0-34).
	@return	Or-combined font features.
*/
EXTERN int
dkfont_get_features DK_PR((size_t number));



/**	Get all PS font descriptions.
	@return	Pointer to an array (35 elements) of descriptions for PS fonts.
*/
EXTERN dk_font_t *
dkfont_get_ps_fonts DK_PR((void));



/**	Create a new font mapping structure.
	The font mapping structure is created in dynamically allocated
	memory, use dkfont_mapping_close() to destroy the structure
	and release the memory if the structure is no longer needed.
	@return	Pointer to new font mapping structure on success, NULL on error.
*/
EXTERN dk_font_mapping_t *
dkfont_mapping_open DK_PR((void));



/**	Destroy font mapping structure and release memory.
	@param	p	Pointer to font mapping structure to destroy.
*/
EXTERN void
dkfont_mapping_close DK_PR((dk_font_mapping_t *p));



/**	Add information from an input stream to the font mapping structure.
	@param	f	Pointer to font mapping structure.
	@param	s	Pointer to stream.
	@return	Flag to indicate success.
*/
EXTERN int
dkfont_mapping_add_stream DK_PR((dk_font_mapping_t *f, dk_stream_t *s));




/**	Retrieve last error code.
	@param	f	Pointer to font mapping structure
	@return	Error code of last error occured.
*/
EXTERN int
dkfont_get_error_code DK_PR((dk_font_mapping_t *f));



/**	Retrieve line number of input in which the last error occured.
	@param	f	Pointer to font mapping structure.
	@return	The line number in which the last error occured.
*/
EXTERN unsigned long
dkfont_get_error_lineno DK_PR((dk_font_mapping_t *f));



/**	Retrieve font substitution data for one font.
	@param	f	Pointer to font mapping structure.
	@param	s	Index of the font (0-34).
	@return	Pointer to font substitution structure.
*/
EXTERN dk_one_font_mapping_t *
dkfont_get_one_font DK_PR((dk_font_mapping_t *f, size_t s));



/**	Mark one font in the font mapping structure as used.
	@param	fm	Pointer to font mapping structure.
	@param	sz	Index of the font (0-34).
	@param	fl	Flag (1=font used, 0=font not used).
*/
EXTERN void
dkfont_set_used DK_PR((dk_font_mapping_t *fm, size_t sz, int fl));



/**	Chech whether or not a font is marked as used.
	@param	f	Pointer to font mapping structure.
	@param	s	Index of the font (0-34).
	@return	Flag to indicate whether the font is marked as used.
*/
EXTERN int
dkfont_get_used DK_PR((dk_font_mapping_t *f, size_t s));



/**	Reset a font substution structure (the internal pointer
	is set to point to the first substitution candidate
	structure).
	@param	s	Pointer to font substution structure.
*/
EXTERN void
dkfont_one_font_reset DK_PR((dk_one_font_mapping_t *s));



/**	Get next font substitution candidate structure in font
	substitution structure.
	@param	s	Pointer to font substitution structure.
	@return	Pointer to font substitution candidate structure on success,
		NULL on error.
*/
EXTERN dk_font_replacement_t *
dkfont_one_font_get DK_PR((dk_one_font_mapping_t *s));



/**	Get font name of font substitution candidate structure.
	@param	r	Font substitution candidate structure.
	@return	The name.
*/
EXTERN char *
dkfont_rep_get_name DK_PR((dk_font_replacement_t *r));



/**	Get font family name of font substitution candidate structure.
	@param	r	Font substitution candidate structure.
	@return	Font family name.
*/
EXTERN char *
dkfont_rep_get_family DK_PR((dk_font_replacement_t *r));



/**	Get location (directory part of download URL or directory name)
	of font file for a font substitution
	candidate structure.
	@param	r	Font substitution candidate structure.
	@return	Pointer to directory name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_location DK_PR((dk_font_replacement_t *r));



/**	Get file name of font file for a font substitution candidate
	structure.
	@param	r	Font substitution candidate structure.
	@return	Pointer to file name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_source_name DK_PR((dk_font_replacement_t *r));



/**	Get location (directory part of download URL or directory name)
	of PFB font file for a font substitution
	candidate structure.
	@param	r	Font substitution candidate structure.
	@return	Pointer to directory name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_pfb_location DK_PR((dk_font_replacement_t *r));



/**	Get file name of PFB font file for a font substitution candidate
	structure.
	@param	r	Font substitution candidate structure.
	@return	Pointer to file name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_pfb_source_name DK_PR((dk_font_replacement_t *r));



/**	Get location (directory part of download URL or directory name)
	of TTF font file for a font substitution
	candidate structure.
	@param	r	Font substitution candidate structure.
	@return	Pointer to directory name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_ttf_location DK_PR((dk_font_replacement_t *r));



/**	Get file name of TTF font file for a font substitution candidate
	structure.
	@param	r	Font substitution candidate structure.
	@return	Pointer to file name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_ttf_source_name DK_PR((dk_font_replacement_t *r));



/**	Get font type.
	@param	r	Font substitution candidate structure.
	@return	Pointer to type name or NULL on error.
*/
EXTERN char *
dkfont_rep_get_font_type DK_PR((dk_font_replacement_t *r));



/**	Get TeX name for substitution candidate.
	@param	r	Font substitution candidate structure.
	@return	Pointer to TeX font name on success, NULL on error.
*/
EXTERN char *
dkfont_rep_get_tex_name DK_PR((dk_font_replacement_t *r));



/**	Get features of substitution candidate.
	@param	r	Font substitution candidate structure.
	@return	Or-combined font features.
*/
EXTERN int
dkfont_rep_get_features DK_PR((dk_font_replacement_t *r));



/**	Get source type for font substitution candidate.
	@param	r	Font substitution candidate structure.
	@return	DK_FONT_RP_TYPE_SYSTEM for system fonts or
		DK_FONT_RP_TYPE_DOWNLOAD for fonts to download.
*/
EXTERN int
dkfont_rep_get_source_type DK_PR((dk_font_replacement_t *r));



/**	Check whether a font substitution candidate can be used with a driver.
	@param	r	Font substitution candidate structure.
	@param	d	Driver name.
	@return	Flag to indicate whether the font substitution candidate
		can be used.
*/
EXTERN int
dkfont_rep_check_driver DK_PR((dk_font_replacement_t *r, char *d));

#if defined(__cplusplus)
}
#endif

/* two last significant bits=0: use default font, no explicit switching */

/** Font feature: roman font. */
#define DK_FONT_FEATURE_RM	1

/** Font feature: sans-serif font. */
#define DK_FONT_FEATURE_SF	2

/** Font feature: typewriter font. */
#define DK_FONT_FEATURE_TT	3

/** Font feature mask: Font family. */
#define DK_FONT_FEATURE_FAMILY	3

/** Font feature: bold. */
#define DK_FONT_FEATURE_BD	4

/** Font feature: italic. */
#define DK_FONT_FEATURE_IT	8

/** Font replacement type: System. */
#define DK_FONT_RP_TYPE_SYTEM		0

/** Font replacement type: Download. */
#define DK_FONT_RP_TYPE_DOWNLOAD	1

#endif

