/*
Copyright (c) 2008-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file	chimgsize.c
The chimgsize program.

This file contains the code of the chimgsize application.
*/

/*
Options for target size:
	-w <maximum>
	-w <minimum>:<maximum>
	--width=<maximum>
	--wdith=<minimum>:<maximum>

Options for original size
	width height
	image
*/


#include <dk.h>

#if DK_HAVE_STRING_H
#include <string.h>
#endif
#if DK_HAVE_STRINGS_H
#include <strings.h>
#endif
#if DK_HAVE_UNISTD_H
#include <unistd.h>
#endif
#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if DK_HAVE_PROCESS_H
#include <process.h>
#endif

#include <dklogc.h>
#include <dkmem.h>
#include <dkstr.h>
#include <dkapp.h>
#include <dklic.h>
#include <dkbif.h>
#include <dksf.h>
#include <dkma.h>

#include "dktools-version.h"




#line 88 "chimgsize.ctr"




/**	Image size job.
*/
typedef struct {
  int		exval;	/**< Returned by main program to the caller. */
  dk_app_t	*app;	/**< Application structure. */
  int		argc;	/**< Number of command line arguments. */
  char		**argv;	/**< Array of command line arguments. */
  int		cmd;	/**< Command, what to do. */
  char		**msg;	/**< Message texts for errors, warnings... */
  char		*arg1;	/**< First command line argument. */
  char		*arg2;	/**< Second command line argument. */
  unsigned long	w;	/**< Original width. */
  unsigned long	h;	/**< Original height. */
  unsigned long	minw;	/**< Minimum output width. */
  unsigned long maxw;	/**< Maximum output width. */
  char		*fn;	/**< File name. */
} ISJ;



/**	Short notation for function header. */
#define ISJ_FCT(x)	static void x DK_P1(ISJ *,i)



/**	Run normally.
*/
#define CMD_RUN		0

/**	Show help.
*/
#define CMD_HELP	1

/**	Show version.
*/
#define CMD_VERSION	2

/**	Report error.
*/
#define CMD_ERROR	128



/**	Program name, used in output messages.
*/
static char program_name[] = { "chimgsize" };

/**	Package name.
*/
static char packagename[] = { "dktools" };

/**	System configuration directory.
*/
static char sysconfdir[] = { DK_SYSCONFDIR };

/**	Name of help text file.
*/
static char help_file_name[] = { "chimgsize.txt" };



/**	Default help text shown if help text file is not found.
*/
static char *help_text[] = {
"chimgsize - Calculate resize factors for image",
"============================================",
"",
"Usage",
"-----",
"chimgsize -w [<minwidth>:]<maxwidth> <width> <height>",
"chimgsize -w [<minwidth>:]<maxwidth> <image>",
""
"width:    Image width before resizing",
"height:   Image height before resizing",
"image:    Image file name",
"minwidth: Minimum width of image after resizing",
"maxwidth: Maximum width of image after resizing",
"",
NULL
};



/**	License terms
*/
static char *license_terms[] = {
"Redistribution and use in source and binary forms, with or without",
"modification, are permitted provided that the following conditions are met:",
"* Redistributions of source code must retain the above copyright notice, this",
"  list of conditions and the following disclaimer.",
"* Redistributions in binary form must reproduce the above copyright notice,",
"  this list of conditions and the following disclaimer in the documentation",
"  and/or other materials provided with the distribution.",
"* Neither the name of the Dirk Krause nor the names of other contributors may",
"  be used to endorse or promote products derived from this software without",
"  specific prior written permission.",
"THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS \"AS IS\"",
"AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE",
"IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE",
"ARE DISCLAIMED.",
"IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY",
"DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES",
"(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;",
"LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND",
"ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT",
"(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS",
"SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.",
NULL
};



/**	Long options keywords.
*/
static char *long_options[] = {
  "h$elp",
  "v$ersion",
  "w$idth",
  NULL
};


/**	Version number.
*/
static char versnumb[] = { VERSNUMB };



/**	Key/value pairs for text messages.
*/
static dk_key_value_t kv[] = {
  { "/m/00", "\"" },
  { "/m/01", "\" is not a numeric value!" },
  { "/m/02", "The string \"" },
  { "/m/03", "\" is too long!" },
  { "/m/04", "A numeric value is needed to specify the width!" },
  { "/m/05", "Too many arguments!" },
  { "/m/06", "Either width and height or a file name is needed!" },
  { "/m/07", "Failed to read image header!" },
  { "/m/08", "Failed to open file \"" },
  { "/m/09", "Failed to allocate memory for dkbif structure to read image!" },
  { "/m/10", "Failed to create name for temporary file!" },
  { "/m/11", "The dkbif library can not handle this image file type!" },
  { "/m/12", "The file name suffix type is unknown to the dkbif library!" },
  { "/m/13", "Pattern \"" },
  { "/m/14", "\" must match exactly one file!" },
  { "/m/15", "Failed to create pattern expander for \"" },
  { "/m/16", "\"!" },
  { "/m/17", "\" for reading!" },

};

/**	Size of the \a kv array.
*/
static size_t szkv = sizeof(kv)/sizeof(dk_key_value_t);


static
void
silence_check DK_P4(int,argc, char **,argv, int *,rs, int *,rf) {
  int	i;		/**< Index to traverse arguments. */
  int	filenames;	/**< Number of filenames found. */
  char	*ptr;		/**< Pointer to argument currently processed. */
  char	**lfdptr;	/**< Used to traverse arguments. */
  
  i = 1; filenames = 0; lfdptr = argv; lfdptr++;
  while(i < argc) {
    ptr = *lfdptr;
    if(*ptr == '-') {
      ptr++;
      switch(*ptr) {
	case 's': {
	  if(rs) { *rs = 1; }
	} break;
	case 'w': {
	  ptr++;
	  if(!(*ptr)) { lfdptr++; i++; }
	} break;
      }
    } else {
      filenames++;
    }
    lfdptr++; i++;
  }
  if(rf) { *rf = 1; }
  if(dkapp_silence_check(argc, argv)) {
    if(rs) { *rs = 1; }
  }
  
}



/**	Write message, 1 part.
	@param	i	Imagesize job.
	@param	ll	Log level.
	@param	n1	Index of message in text array.
*/
static
void
isj_msg_1 DK_P3(ISJ *,i, int,ll, size_t,n1) {
  if(n1 < szkv) {
    dkapp_log_msg(i->app, ll, &((i->msg)[n1]), 1);
  }
}



/**	Write message, 3 parts.
	@param	i	Imagesize job.
	@param	ll	Log level.
	@param	n1	Index of first part in text array.
	@param	n2	Index of third part in text array.
	@param	t	Second part.
*/
static
void
isj_msg_3 DK_P5(ISJ *,i, int,ll, size_t,n1, size_t,n2, char *,t) {
  char *msg[3];
  if((n1 < szkv) && (n2 < szkv)) {
    msg[0] = (i->msg)[n1];
    msg[1] = t;
    msg[2] = (i->msg)[n2];
    dkapp_log_msg(i->app, ll, msg, 3);
  }
}



/**	Initialize imagesize job.
	@param	i	Image size job to initialize.
*/
ISJ_FCT(isj_init) {
  
  i->exval = 0;
  i->app = NULL;
  i->argc = 0;
  i->argv = NULL;
  i->cmd = CMD_RUN;
  i->msg = NULL;
  i->arg1 = NULL;
  i->arg2 = NULL;
  i->w = i->h = i->minw = i->maxw = 0UL;
  i->fn = NULL;
  
}



/**	Clean up image size job.
	@param	i	Imagesize job.
*/
ISJ_FCT(isj_cleanup) {
  
  if(i->app) {
    dkapp_close(i->app);
  } i->app = NULL;
  if(i->msg) {
    char **xptr;
    xptr = i->msg; dk_delete(xptr);
  } i->msg = NULL;
  i->argc = 0;
  i->argv = NULL;
  
}



static
void
set_widths DK_P2(ISJ *,i, char *,t) {
  char	buffer[128];		/**< Private copy of t for modifications. */
  char	*p1;			/**< Pointer to start of buffer. */
  char	*p2;			/**< Pointer to second number. */
  unsigned long ul;		/**< Used for sscanf() operations. */
  if(strlen(t) < sizeof(buffer)) {
    strcpy(buffer, t);
    p1 = buffer;
    p2 = strchr(buffer, '-');
    if(p2) {
      *(p2++) = '\0';
      if(sscanf(buffer, "%lu", &ul) == 1) {
        i->minw = ul;
	if(sscanf(p2, "%lu", &ul) == 1) {
	  i->maxw = ul;
	} else {
	  i->exval = 1;
	  /* ERROR: Not a numeric value */
	  isj_msg_3(i, DK_LOG_LEVEL_ERROR, 0, 1, p2);
	}
      } else {
        i->exval = 1;
	/* ERROR: Not a numeric value */
	isj_msg_3(i, DK_LOG_LEVEL_ERROR, 0, 1, buffer);
      }
    } else {
      if(sscanf(buffer, "%lu", &ul) == 1) {
        i->maxw = ul;
      } else {
        i->exval = 1;
	/* ERROR: Not a numeric value */
	isj_msg_3(i, DK_LOG_LEVEL_ERROR, 0, 1, buffer);
      }
    }
  } else {
    i->exval = 1;
    /* ERROR: String too long */
    isj_msg_3(i, DK_LOG_LEVEL_ERROR, 2, 3, t);
  }
}



static
void
process_long_argument DK_P3(ISJ *,i, char *,b, char *,v) {
  int	w;		/**< What to do. */
  w = dkstr_array_abbr(long_options, b, '$', 0);
  switch(w) {
    case 0: {
      i->cmd |= CMD_HELP;
    } break;
    case 1: {
      i->cmd |= CMD_VERSION;
    } break;
    case 2: {
      if(v) {
        set_widths(i, v);
      } else {
        i->exval = 1;
	/* ERROR: Numeric argument needed */
	isj_msg_1(i, DK_LOG_LEVEL_ERROR, 4);
      }
    } break;
  }
}



ISJ_FCT(check_arguments) {
  char		buffer[512];	/**< Buffer for long arguments */
  char		*ptr;		/**< Current argument to process. */
  char		**lfdptr;	/**< Pointer to traverse arguments array. */
  int		argi;		/**< Index of current argument. */
  char		*vptr;		/**< Pointer to argument value. */
  
  lfdptr = i->argv; lfdptr++; argi = 1;
  while(argi < i->argc) {
    ptr = *lfdptr;
    if(*ptr == '-') {		
      ptr++;
      switch(*ptr) {
        case '-': {
	  ptr++;
	  if(strlen(ptr) < sizeof(buffer)) {
	    strcpy(buffer, ptr);
	    vptr = strchr(buffer, '=');
	    if(vptr) { *(vptr++) = '\0'; }
	    process_long_argument(i, buffer, vptr);
	  } else {
	    i->exval = 1;
	    /* ERROR: Long argument too long for buffer */
	    isj_msg_3(i, DK_LOG_LEVEL_ERROR, 2, 3, ptr);
	  }
	} break;
	case 'h': {
	  i->cmd |= CMD_HELP;
	} break;
	case 'v': {
	  i->cmd |= CMD_VERSION;
	} break;
	case 'w': {
	  ptr++;
	  if(!(*ptr)) {
	    lfdptr++; argi++; ptr = NULL;
	    if(argi < i->argc) { ptr = *lfdptr; }
	  }
	  if(ptr) {
	    set_widths(i, ptr);
	  } else {
	    i->cmd |= CMD_ERROR;
	    /* ERROR: -w needs numeric argument */
	    isj_msg_1(i, DK_LOG_LEVEL_ERROR, 4);
	  }
	} break;
      }
    } else {			
      if(i->arg1) {
        if(i->arg2) {
	  i->cmd |= CMD_ERROR;
	  /* ERROR: Too many arguments */
	  isj_msg_1(i, DK_LOG_LEVEL_ERROR, 5);
	} else {
	  i->arg2 = ptr;
	}
      } else {
        i->arg1 = ptr;
      }
    }
    argi++; lfdptr++;
  }
  if((!(i->arg1)) && (!((i->cmd) & (CMD_HELP | CMD_VERSION)))) {
    i->cmd |= CMD_ERROR;
    /* ERROR: At least one argument needed */
    isj_msg_1(i, DK_LOG_LEVEL_ERROR, 6);
  }
  
}



ISJ_FCT(print_version) {
  char	**ptr;		/**< Pointer into array of texts. */
  printf("\nchimgsize (part of the dktools collection, version %s)\n", versnumb);
  printf("Copyright (C) 2008-2010 Dipl.-Ing. D. krause\n");
  printf("http://dktools.sourceforge.net\n\n");
  ptr = license_terms;
  while(*ptr) { printf("%s\n", *(ptr++)); }
  printf("\nLibraries used:\n\n");
  ptr = dklic_get();
  while(*ptr) { printf("%s\n", *(ptr++)); }
  printf("\n");
}



static
int
is_power2 DK_P1(unsigned long,d) {
  int back = 0;		/**< Result to return. */
  int test;		/**< Number of set bits in d. */
  unsigned long b;	/**< Bit pattern for test. */
  size_t i;		/**< Loop counter. */
  size_t loopmax;	/**< Number of loops. */
  

  b = 1UL; test = 0; loopmax = 8 * sizeof(unsigned long);
  for(i = 0; i < loopmax; i++) {
    if(d & b) { test++; }
    b = 2UL * b;
  }
  if(test == 1) { back = 1; }
  return back;
}


ISJ_FCT(run_with_sizes) {
  char wb[512];			/**< Buffer for width. */
  char hb[512];			/**< Buffer for height. */
  char fw[512];			/**< Buffer for format string for width. */
  char fh[512];			/**< Buffer for format string for height. */
  unsigned long	gcd = 0UL;	/**< Greatest common divisor.*/
  unsigned long	sw = 0UL;	/**< Smallest width. */
  unsigned long	sh = 0UL;	/**< Smallest height. */
  unsigned long	w = 0UL;	/**< Width. */
  unsigned long	h = 0UL;	/**< Height. */
  unsigned long	f = 0UL;	/**< Factor. */
  unsigned long	d = 0UL;	/**< Divisor. */
  int		can_continue=0;	/**< Flag: Can continue. */
  char		c = '\0';	/**< Start of line. */
  size_t	maxlw = 0;	/**< Maximum length of wb. */
  size_t	maxlh = 0;	/**< Maximum length of wh. */
  if(i->maxw == 0UL) { i->maxw = i->w; }
  gcd = dkma_ul_gcd(i->w, i->h);
  sw = i->w / gcd; sh = i->h / gcd;
  can_continue = 1; f = 1UL;
  while(can_continue) {
    w = f * sw; h = f * sh;
    if(w >= i->minw) {
      if(w <= i->maxw) {
        sprintf(wb, "%lu", w);
	sprintf(hb, "%lu", h);
	maxlw = strlen(wb);
	maxlh = strlen(hb);
      } else {
        can_continue = 0;
      }
    }
    f++;
  }
  sprintf(fw, "%%%lulu", (unsigned long)maxlw);
  sprintf(fh, "%%%lulu", (unsigned long)maxlh);
  
  can_continue = 1; f = 1UL;
  while(can_continue) {
    w = f * sw; h = f * sh;
    if(w >= i->minw) {
      if(w <= i->maxw) {
        c = '-';
	if(i->w >= w) {
	  if(((i->w) % w) == 0UL) {
	    c = '+';
	    d = (i->w) / w;
	    if(is_power2(d)) {
	      c = '*';
	    }
	  }
	} else {
	  if(w % (i->w) == 0UL) {
	    c = '+';
	    d = w / i->w;
	    if(is_power2(d)) {
	      c = '*';
	    }
	  }
	}
	sprintf(wb, fw, w);
	sprintf(hb, fh, h);
	printf("%c\t%s\t%s\n", c, wb, hb);
      } else {
        can_continue = 0;
      }
    }
    f++;
  }
}



ISJ_FCT(run_for_sizes) {
  unsigned long	ul;	/**< Used in sscanf() operations. */
  if(sscanf(i->arg1, "%lu", &ul) == 1) {
    i->w = ul;
    if(sscanf(i->arg2, "%lu", &ul) == 1) {
      i->h = ul;
      run_with_sizes(i);
    } else {
      i->exval = 1;
      /* ERROR: Not numeric */
      isj_msg_3(i, DK_LOG_LEVEL_ERROR, 0, 1, i->arg2);
    }
  } else {
    i->exval = 1;
    /* ERROR: Not numeric */
    isj_msg_3(i, DK_LOG_LEVEL_ERROR, 0, 1, i->arg1);
  }
}



ISJ_FCT(run_with_file) {
  int		it;	/**< Image type. */
  dk_bif_ptr	image;	/**< Structure to read image information. */
  size_t	mpl;	/**< Maximum path length. */
  char		*tmpb;	/**< Buffer for temporary file name. */
  FILE		*fipo;	/**< Used to read the image file. */
  it = dkbif_get_type(i->fn);
  if(it != DKBIF_TYPE_UNKNOWN) {
    if(dkbif_can_handle(it)) {
      mpl = (size_t)dksf_get_maxpathlen();
      if(mpl < 512) mpl = 512;
      tmpb = dk_new(char,mpl);
      if(tmpb) {
        if(dkapp_tmpnam(i->app, tmpb, mpl)) {
          image = dkbif_open();
          if(image) {
            dkbif_set_width_height_sufficient(image, 1);
	    fipo = dksf_fopen(i->fn, "rb");
	    if(fipo) {
	      if(dkbif_read_header(image, fipo, i->fn, it, tmpb)) {
	        i->w = dkbif_get_width(image);
		i->h = dkbif_get_height(image);
		run_with_sizes(i);
	      } else {
	        i->exval = 1;
		/* ERROR: Failed to read image header */
		isj_msg_1(i, DK_LOG_LEVEL_ERROR, 7);
	      }
	      fclose(fipo);
	    } else {
	      i->exval = 1;
	      /* ERROR: Failed to open image file for reading */
	      isj_msg_3(i, DK_LOG_LEVEL_ERROR, 8, 17, i->fn);
	    }
            dkbif_close(image);
          } else {
            i->exval = 1;
            /* ERROR: Memory */
	    isj_msg_1(i, DK_LOG_LEVEL_ERROR, 9);
          }
	} else {
	  i->exval = 1;
	  /* ERROR: Failed to create name for temporary file. */
	  isj_msg_1(i, DK_LOG_LEVEL_ERROR, 10);
	}
        dk_delete(tmpb);
      } else {
        i->exval = 1;
        /* ERROR: Memory */
	dkapp_err_memory(i->app, 1, mpl);
      }
    } else { 
      i->exval = 1;
      /* ERROR: Can not handle file type. */
      isj_msg_1(i, DK_LOG_LEVEL_ERROR, 11);
    }

  } else {
    i->exval = 1;
    /* ERROR: File name suffix unknown to dkbif  */
    isj_msg_1(i, DK_LOG_LEVEL_ERROR, 12);
  }
}



ISJ_FCT(run_for_file) {
  dk_fne_t	*fneo;		/**< File name expander .*/
  if(dksf_must_expand_filename(i->arg1)) {
    fneo = dkfne_open(i->arg1, 1, 0);
    if(fneo) {
      i->fn = dkfne_get_one(fneo);
      if(i->fn) {
        run_with_file(i);
      } else {
        i->exval = 1;
	/* ERROR: Pattern must match exactly one file. */
	isj_msg_3(i, DK_LOG_LEVEL_ERROR, 13, 14, i->arg1);
      }
      dkfne_close(fneo);
    } else {
      i->exval = 1;
      /* ERROR: Failed to create file name expander */
      isj_msg_3(i, DK_LOG_LEVEL_ERROR, 15, 16, i->arg1);
    }
  } else {
    i->fn = i->arg1;
    run_with_file(i);
  }
}



/**	Run either on specified sizes or files.
	@param	i	Imagesize job.
*/
ISJ_FCT(run_normally) {
  if(i->arg1) {
    if(i->arg2) {
      run_for_sizes(i);
    } else {
      run_for_file(i);
    }
  } else {
    i->exval = 1;
    /* ERROR: No arguments */
    isj_msg_1(i, DK_LOG_LEVEL_ERROR, 6);
  }
}



/**	Run the program.
	@param	i	Imagesize job.
*/
ISJ_FCT(run) {
  
  check_arguments(i);
  if(i->exval == 0) {
    if(i->cmd == CMD_RUN) {
      run_normally(i);
    } else {
      if((i->cmd) & (CMD_HELP | CMD_ERROR | CMD_VERSION)) {
        print_version(i);
      }
      if((i->cmd) & (CMD_HELP | CMD_ERROR)) {
        dkapp_help(i->app, help_file_name, help_text);
      }
    }
  }
  if((i->cmd) & CMD_ERROR) { i->exval = 1; }
  
}


#ifdef DK_HAVE_PROTOTYPES
int main(int argc, char *argv[])
#else
int main(argc, argv) int argc; char *argv[];
#endif
{
  int	exval	=	0;	/**< Returned to caller. */
  int	rs	=	0;	/**< Flag: Run silently. */
  int	rf	=	1;	/**< Flag: Run as filter. */
  ISJ	isj;			/**< Main data structure. */
  
#line 777 "chimgsize.ctr"

  
  isj_init(&isj);
  silence_check(argc, argv, &rs, &rf);
  isj.app = dkapp_open_ext1(argc, argv, packagename, sysconfdir, rs, rf);
  if(isj.app) {
    isj.argc = dkapp_get_argc(isj.app);
    isj.argv = dkapp_get_argv(isj.app);
    isj.msg = dkapp_find_key_value(isj.app, kv, szkv, program_name);
    if(isj.msg) {
    } else {
    }
    run(&isj);
  } else {
    if(!dkapp_silence_check(argc, argv)) {
      fprintf(
        stderr,
        "%s: ERROR: Not enough memory (RAM/swap space)!\n",
        program_name
      );
    }
  }
  isj_cleanup(&isj);
  exval = isj.exval;
  
  
#line 802 "chimgsize.ctr"

  exit(exval); return exval;
}


