package net.psammead.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

public final class ListUtil {
	private ListUtil() {}
	
	public static <T> List<T> create(T... in) {
		List<T> out	= new ArrayList<T>();
		for (T element : in)	out.add(element);
		return out;
	}
	
	public static <T> List<T> create(Iterable<T> in) {
		List<T> out	= new ArrayList<T>();
		for (T element : in)	out.add(element);
		return out;
	}
	
	public static <T,X extends T> List<T> single(X element) {
		List<T> out	= new ArrayList<T>();
		out.add(element);
		return out;
	}
	
	public static <T> List<T> filter(List<T> in, Function<? super T,Boolean> filter) {
		List<T> out	= new ArrayList<T>();
		for (T element : in) {
			if (filter.apply(element)) {
				out.add(element);
			}
		}
		return out;
	}
	
	public static <S,T> List<T> map(List<S> in, Function<? super S,? extends T> transformer) {
		List<T> out	= new ArrayList<T>();
		for (S element : in) {
			out.add(transformer.apply(element));
		}
		return out;
	}
	 
	public static <T> List<T> concat(List<T> a, List<T> b) {
		List<T> out	= new ArrayList<T>();
		out.addAll(a);
		out.addAll(b);
		return out;
	}
	
	public static <T> List<T> intersection(List<T> a, List<T> b) {
		List<T> out	= new ArrayList<T>();
		out.addAll(a);
		out.retainAll(b);
		return out;
	}
	
	public static <T> List<T> without(List<T> a, List<T> b) {
		List<T> out	= new ArrayList<T>();
		out.addAll(a);
		out.removeAll(b);
		return out;
	}
	
	public static <T> List<T> difference(List<T> a, List<T> b) {
		return without(concat(a,b), intersection(a,b));
	}
	
	//-------------------------------------------------------------------------
	
	public static <T> List<T> reverse(List<T> in) {
		List<T>	out	= new ArrayList<T>(in);
		Collections.reverse(out);
		return out;
	}
	
	public static <T extends Comparable<? super T>> List<T> sort(List<T> in) {
		List<T> out	= new ArrayList<T>(in);
		Collections.sort(out);
		return out;
	}
	
	public static <T> List<T> sort(List<T> in, Comparator<? super T> comparator) {
		List<T> out	= new ArrayList<T>(in);
		Collections.sort(out, comparator);
		return out;
	} 
	
	public static <T> T first(List<T> list) {
		return list.get(0);
	}
	
	public static <T> T last(List<T> list) {
		return list.get(lastIndex(list));
	}
	
	public static int lastIndex(List<?> list) {
		return list.size()-1;
	} 
	
	public static boolean containsIndex(List<?> list, int index) {
		return index >= 0 && index < list.size();
	}
	
	public static <T> T getOrElse(List<T> list, int index, T elseValue) {
		if (containsIndex(list, index))	return list.get(index);
		else							return elseValue;
	}
	
//	@SuppressWarnings("unchecked")
//	public static <T> T[] toArray(List<T> list, Class<? extends T> clazz) {
//		T[]	out	= (T[])Array.newInstance(clazz, list.size());
//		return list.toArray(out);
//	}
}
