package net.psammead.mwapi.yurik;

import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.Map;

import net.psammead.mwapi.MediaWikiException;
import net.psammead.mwapi.NameSpace;
import net.psammead.mwapi.ui.MethodException;
import net.psammead.mwapi.ui.UnexpectedAnswerException;
import net.psammead.mwapi.yurik.data.*;
import net.psammead.mwapi.yurik.data.list.*;
import net.psammead.mwapi.yurik.data.prop.*;
import net.psammead.mwapi.yurik.json.*;
import net.psammead.util.Logger;
import net.psammead.util.Throttle;
import net.psammead.util.json.JSONDecodeException;
import net.psammead.util.json.JSONDecoder;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.StatusLine;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.httpclient.methods.GetMethod;

public class YurikAPI {
	private final Logger		logger;
	private final HttpClient	client;
	private final Throttle		throttle;
	
//	private final String		wiki;
	private final String		apiURL;
	private final String		charSet;
	private final String		userAgent;
	
	
	private final JSONConverterContext	ctx;

	public YurikAPI(
			Logger		logger,
			HttpClient	client,
			Throttle	throttle,
			String		apiURL,
			String		userAgent,
			String		wiki,
			String		charSet
	) {
		this.logger		= logger;
		this.client		= client;
		this.throttle	= throttle;
		this.apiURL		= apiURL;
		this.userAgent	= userAgent;
//		this.wiki		= wiki;
		this.charSet	= charSet;
		
		ctx	= new JSONConverterContext(wiki);
	}

	//==============================================================================
	//## prop
	
/* prop=categories (cl) *
  List all categories the page(s) belong to
Parameters:
  clprop         - Which additional properties to get for each category.
                   Values (separate with '|'): sortkey
*/
	public CategoriesResult categories(String title) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"categories");
		query.string("titles",	title);
		query.string("clprop",	"sortkey");
		
		return (CategoriesResult)apiJSON(
				query, CATEGORIES_CONVERTER);
	}
	
	public static final JSONConverter	CATEGORIES_CONVERTER	= 
		new CreateObj(CategoriesResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(Categories_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("categories",
								new CreateObj(Categories_categories.class,
									new ListIter(
										new CreateObj(Categories_cl.class,
											new MapEntry("title",	new TitleToLocation()),
											new MapEntry("sortkey",	new Copy()))))))))));
	
/* prop=extlinks (el) *
  Returns all external urls (not interwikies) from the given page(s)
*/
	public ExtLinksResult extLinks(String title) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"extlinks");
		query.string("titles",	title);
		
		return (ExtLinksResult)apiJSON(
				query, EXTLINKS_CONVERTER);
	}
	
	public static final JSONConverter	EXTLINKS_CONVERTER	= 
		new CreateObj(ExtLinksResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(ExtLinks_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("extlinks",
								new CreateObj(ExtLinks_extlinks.class,
									new ListIter(
										new CreateObj(ExtLinks_el.class,
											new MapEntry("*",	new Copy()))))))))));

/* prop=imageinfo (ii) *
  Returns image information and upload history
Parameters:
  iiprop         - What image information to get.
                   Values (separate with '|'): timestamp, user, comment, url, size
                   Default: timestamp|user
  iihistory      - Include upload history
*/
	public ImageInfoResult imageInfo(String title) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"imageinfo");
		query.string("titles",	title);
		query.string("iiprop",	"timestamp|user|comment|url|size");
		query.bool("iihistory",	true);
		
		return (ImageInfoResult)apiJSON(
				query, IMAGEINFO_CONVERTER);
	}
	
	public static final JSONConverter	IMAGEINFO_CONVERTER	= 
		new CreateObj(ImageInfoResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(ImageInfo_page.class,
							new MapEntry("title",			new TitleToLocation()),
							new MapEntry("pageid",			new Copy()),
							new MapEntry("imagerepository",	new Copy()),
							new MapEntry("imageinfo",
								new CreateObj(ImageInfo_imageinfo.class,
									new ListIter(
										new CreateObj(ImageInfo_ii.class,
											new MapEntry("timestamp",	new IsoToDate()),
											new MapEntry("user",		new Copy()),
											new MapEntry("size",		new Copy()),
											new MapEntry("width",		new LongToInt()),
											new MapEntry("height",		new LongToInt()),
											new MapEntry("url",			new Copy()),
											new MapEntry("comment",		new Copy()),
											new MapEntry("content",		new Copy()))))))))));
	
/* prop=images (im) *
  Returns all images contained on the given page(s)
*/
	public ImagesResult images(String title) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"images");
		query.string("titles",	title);
		
		return (ImagesResult)apiJSON(
				query, IMAGES_CONVERTER);
	}
	
	public static final JSONConverter	IMAGES_CONVERTER	= 
		new CreateObj(ImagesResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(Images_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("images",
								new CreateObj(Images_images.class,
									new NullSafe(	// BH
										new ListIter(
											new CreateObj(Images_im.class,
												new MapEntry("title",	new TitleToLocation())))))))))));

/* prop=info (in) *
  Get basic page information such as namespace, title, last touched date, ...
Parameters:
  inprop         - Which additional properties to get:
                    "protection"   - List the protection level of each page
                   Values (separate with '|'): protection
*/
	public InfoResult info(String title) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"info");
		query.string("titles",	title);
		query.string("inprop",	"protection");
		
		return (InfoResult)apiJSON(
				query, INFO_CONVERTER);
	}
	
	public static final JSONConverter	INFO_CONVERTER	= 
		new CreateObj(InfoResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(Info_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("lastrevid",	new Copy()),
							new MapEntry("touched",		new IsoToDate()),
							new MapEntry("counter",		new LongToInt()),
							new MapEntry("length",		new LongToInt()),
							new MapEntry("redirect",	new ExistsToBool()))))));

		/*
		TODO
		MapToObj	converting: class net.psammead.mwapi.yurik.data.prop.Info_page
		MapToObj	not mapped: [protection, ns]
		 
		"protection": [
		]
		
		new MapParam("protection",
			new ListIter( ???
		*/
	
/* prop=langlinks (ll) *
  Returns all interlanguage links from the given page(s)
*/		
	public LangLinksResult langLinks(String title) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"langlinks");
		query.string("titles",	title);
		
		return (LangLinksResult)apiJSON(
				query, LANGLINKS_CONVERTER);
	}
	
	public static final JSONConverter	LANGLINKS_CONVERTER	= 
		new CreateObj(LangLinksResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(LangLinks_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("langlinks",
								new CreateObj(LangLinks_langlinks.class,
									new ListIter(
										new CreateObj(LangLinks_li.class,
											new MapEntry("lang",	new Copy()))))))))));
											//new MapParam("*",	new Copy())

/* prop=links (pl) *
  Returns all links from the given page(s)
Parameters:
  plnamespace    - Show links in this namespace(s) only
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
*/
	public LinksResult links(String title, List<NameSpace> nameSpaces) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"links");
		query.string("titles",	title);
		query.nameSpaces("plnamespace",	nameSpaces);
		
		return (LinksResult)apiJSON(
				query, LINKS_CONVERTER);
	}
	
	public static final JSONConverter	LINKS_CONVERTER	= 
		new CreateObj(LinksResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(Links_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("links",
								new CreateObj(Links_links.class,
									new ListIter(
										new CreateObj(Links_pl.class,
											new MapEntry("title",	new TitleToLocation()))))))))));
	
/* prop=revisions (rv) *
Get revision information.
This module may be used in several ways:
 1) Get data about a set of pages (last revision), by setting titles or pageids parameter.
 2) Get revisions for one given page, by using titles/pageids with start/end/limit params.
 3) Get data about a set of revisions by setting their IDs with revids parameter.
All parameters marked as (enum) may only be used with a single page (#2).
Parameters:
rvprop         - Which properties to get for each revision.
                 Values (separate with '|'): ids, flags, timestamp, user, comment, content
                 Default: ids|timestamp|flags|comment|user
rvlimit        - limit how many revisions will be returned (enum)
                 No more than 50 (500 for bots) allowed.
rvstartid      - from which revision id to start enumeration (enum)
rvendid        - stop revision enumeration on this revid (enum)
rvstart        - from which revision timestamp to start enumeration (enum)
rvend          - enumerate up to this timestamp (enum)
rvdir          - direction of enumeration - towards "newer" or "older" revisions (enum)
                 One value: newer, older
                 Default: older
rvuser         - only include revisions made by user
rvexcludeuser  - exclude revisions made by user
*/
	public static final int REVISIONS_MAX_LIMIT	= 50;
	
	public RevisionsResult revisions(
			String title,
			Long startId, Long endId,
			Date start, Date end, boolean newer,
			String user, String excludeUser,
			int limit) throws MediaWikiException {
		
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"revisions");
		query.string("titles",	title);
		query.string("rvprop",			"ids|flags|timestamp|user|comment");	// TODO add |content
		query.number("rvstartid",		startId);
		query.number("rvendid",			endId);
		query.date("rvstart",			start);
		query.date("rvend",				end);
		query.direction("rvdir",		newer);
		query.string("rvuser",			user);
		query.string("rvexcludeuser",	excludeUser);
		query.limit("rvlimit", limit, REVISIONS_MAX_LIMIT);
		
		return (RevisionsResult)apiJSON(
				query, REVISIONS_CONVERTER);
	}
	
	public static final JSONConverter	REVISIONS_CONVERTER	= 
		new CreateObj(RevisionsResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(Revisions_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("revisions",
								new CreateObj(Revisions_revisions.class,
									new ListIter(
										new CreateObj(Revisions_rev.class,
											new MapEntry("revid",		new Copy()),
											new MapEntry("user",		new Copy()),
											new MapEntry("timestamp",	new IsoToDate()),
											new MapEntry("comment",		new Copy()),
											new MapEntry("minor",		new ExistsToBool()),
											new MapEntry("anon",		new ExistsToBool()),
											new MapEntry("content",		new Copy()))))))))));
	
/* prop=templates (tl) *
  Returns all templates from the given page(s)
Parameters:
  tlnamespace    - Show templates in this namespace(s) only
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
*/
	public TemplatesResult templates(String title, List<NameSpace> nameSpaces) throws MediaWikiException {
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("prop",	"templates");
		query.string("titles",	title);
		query.nameSpaces("tlnamespace",	nameSpaces);
		
		return (TemplatesResult)apiJSON(
				query, TEMPLATES_CONVERTER);
	}
	
	public static final JSONConverter	TEMPLATES_CONVERTER	= 
		new CreateObj(TemplatesResult.class,
			new MapEntry("query", 
				new MapEntry("pages",
					new MapValuesIter(
						new CreateObj(Templates_page.class,
							new MapEntry("title",		new TitleToLocation()),
							new MapEntry("pageid",		new Copy()),
							new MapEntry("templates",
								new CreateObj(Templates_templates.class,
									new ListIter(
										new CreateObj(Templates_tl.class,
											new MapEntry("title",	new TitleToLocation()))))))))));
		
	//==============================================================================
	//## list
	
/* list=allpages (ap) *
  Enumerate all pages sequentially in a given namespace
Parameters:
  apfrom         - The page title to start enumerating from.
  apprefix       - Search for all page titles that begin with this value.
  apnamespace    - The namespace to enumerate.
				   One value: 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
				   Default: 0
  apfilterredir  - Which pages to list.
				   One value: all, redirects, nonredirects
				   Default: all
  aplimit        - How many total pages to return.
				   No more than 500 (5000 for bots) allowed.
				   Default: 10
*/
	public static final int	ALLPAGES_MAX_LIMIT	= 500;
	
	public AllPagesResult allPages(
			String from, String prefix, List<NameSpace> nameSpaces, String filterRedir,
			int limit) throws MediaWikiException {
		
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"allpages");
		query.string("apfrom",			from);
		query.string("apprefix",		prefix);
		query.nameSpaces("apnamespace",	nameSpaces);
		query.filterRedir("apfilterredir",	filterRedir);
		query.limit("aplimit", limit, ALLPAGES_MAX_LIMIT);
		
		return (AllPagesResult)apiJSON(
				query, ALLPAGES_CONVERTER);
	}
	
	public static final JSONConverter	ALLPAGES_CONVERTER	= 
		new CreateObj(AllPagesResult.class,
			new MapEntry("query", 
				new CreateObj(AllPages_allpages.class,
					new MapEntry("allpages", 
						new ListIter(
							new CreateObj(AllPages_p.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("allpages",
						new MapEntry("apfrom",	new Copy())))));

/* list=alllinks (al) *
  Enumerate all links that point to a given namespace
Parameters:
  alfrom         - The page title to start enumerating from.
  alprefix       - Search for all page titles that begin with this value.
  alunique       - Only show unique links. Cannot be used with generator or prop=ids
  alprop         - What pieces of information to include
                   Values (separate with '|'): ids, title
                   Default: title
  alnamespace    - The namespace to enumerate.
                   One value: 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
                   Default: 0
  allimit        - How many total links to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	ALLLINKS_MAX_LIMIT	= 500;
	
	public AllLinksResult allLinks(
			String from, String prefix, List<NameSpace> nameSpaces, /*boolean unique,*/
			int limit) throws MediaWikiException {
		
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"alllinks");
		query.string("alprop",	"ids|title");
		query.string("alfrom",			from);
		query.string("alprefix",		prefix);
		query.nameSpaces("alnamespace",	nameSpaces);
		query.bool("alunique",			false);	// unique and  alprop=id do not work together
		query.limit("allimit", limit, ALLLINKS_MAX_LIMIT);
		
		return (AllLinksResult)apiJSON(
				query, ALLLINKS_CONVERTER);
	}
	
	public static final JSONConverter	ALLLINKS_CONVERTER	= 
		new CreateObj(AllLinksResult.class,
			new MapEntry("query", 
				new CreateObj(AllLinks_alllinks.class,
					new MapEntry("alllinks", 
						new ListIter(
							new CreateObj(AllLinks_l.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("fromid",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("alllinks",
						new MapEntry("alfrom",	new Copy())))));
	
/* list=allusers (au) *
  Enumerate all registered users
Parameters:
  aufrom         - The user name to start enumerating from.
  auprefix       - Search for all page titles that begin with this value.
  augroup        - Limit users to a given group name
                   One value: bot, sysop, bureaucrat, checkuser, steward, boardvote, import, developer, oversight
  auprop         - What pieces of information to include.
                   `groups` property uses more server resources and may return fewer results than the limit.
                   Values (separate with '|'): editcount, groups
  aulimit        - How many total user names to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	ALLUSERS_MAX_LIMIT	= 500;
	
	public AllUsersResult allUsers(
			String from, String prefix, List<NameSpace> nameSpaces, String group,
			int limit) throws MediaWikiException {
		
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"allusers");
		query.string("auprop",	"editcount|groups");
		query.string("aufrom",			from);
		query.string("auprefix",		prefix);
		query.nameSpaces("alnamespace",	nameSpaces);
		query.group("augroup",			group);
		query.limit("aulimit", limit, ALLUSERS_MAX_LIMIT);
		
		return (AllUsersResult)apiJSON(
				query, ALLUSERS_CONVERTER);
	}
	
	public static final JSONConverter	ALLUSERS_CONVERTER	= 
		new CreateObj(AllUsersResult.class,
			new MapEntry("query", 
				new CreateObj(AllUsers_allusers.class,
					new MapEntry("allusers", 
						new ListIter(
							new CreateObj(AllUsers_u.class,
								new MapEntry("name",		new Copy()),
								new MapEntry("editcount",	new LongToInt())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("allusers",
						new MapEntry("aufrom",	new Copy())))));
	
/* list=backlinks (bl) *
  Find all pages that link to the given page
Parameters:
  bltitle        - Title to search. If null, titles= parameter will be used instead, but will be obsolete soon.
  blcontinue     - When more results are available, use this to continue.
  blnamespace    - The namespace to enumerate.
				   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  blredirect     - If linking page is a redirect, find all pages that link to that redirect (not implemented)
  bllimit        - How many total pages to return.
				   No more than 500 (5000 for bots) allowed.
				   Default: 10
*/
	public static final int	BACKLINKS_MAX_LIMIT	= 500;
	
	public BackLinksResult backLinks(
			String title, List<NameSpace> nameSpaces, boolean redirect,
			int limit, String continueKey) throws MediaWikiException {
		
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"backlinks");
		query.string("bltitle",			title);
		query.nameSpaces("blnamespace",	nameSpaces);
		query.bool("blredirect",		redirect);
		query.string("blcontinue",		continueKey);
		query.limit("bllimit", limit, BACKLINKS_MAX_LIMIT);
		
		return (BackLinksResult)apiJSON(
				query, BACKLINKS_CONVERTER);
	}
	
	public static final JSONConverter	BACKLINKS_CONVERTER	= 
		new CreateObj(BackLinksResult.class,
			new MapEntry("query", 
				new CreateObj(BackLinks_backlinks.class,
					new MapEntry("backlinks", 
						new ListIter(
							new CreateObj(BackLinks_bl.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("backlinks",
						new MapEntry("blcontinue",	new Copy())))));
	
/* list=categorymembers (cm) *
  List all pages in a given category
Parameters:
  cmcategory     - Which category to enumerate (required)
  cmprop         - What pieces of information to include
                   Values (separate with '|'): ids, title, sortkey
                   Default: ids|title
  cmnamespace    - Only include pages in these namespaces
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  cmcontinue     - For large categories, give the value retured from previous query
  cmlimit        - The maximum number of pages to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	CATEGORYMEMBERS_MAX_LIMIT	= 500;
	
	public CategoryMembersResult categoryMembers(
			String category, List<NameSpace> nameSpaces, 
			int limit, String continueKey) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"categorymembers");
		query.string("cmcategory",		category);
		query.string("cmprop",			"ids|title|sortkey");
		query.nameSpaces("cmnamespace",	nameSpaces);
		query.string("cmcontinue",		continueKey);
		query.limit("cmlimit", limit, CATEGORYMEMBERS_MAX_LIMIT);
		
		return (CategoryMembersResult)apiJSON(
				query, CATEGORYMEMBERS_CONVERTER);
	}
	
	public static final JSONConverter	CATEGORYMEMBERS_CONVERTER	= 
		new CreateObj(CategoryMembersResult.class,
			new MapEntry("query", 
				new CreateObj(CategoryMembers_categorymembers.class,
					new MapEntry("categorymembers",
						new ListIter(
							new CreateObj(CategoryMembers_cm.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy()),
								new MapEntry("sortkey",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("categorymembers",
						new MapEntry("cmcontinue",	new Copy())))));
	
/* list=embeddedin (ei) *
  Find all pages that embed (transclude) the given title
Parameters:
  eititle        - Title to search. If null, titles= parameter will be used instead, but will be obsolete soon.
  eicontinue     - When more results are available, use this to continue.
  einamespace    - The namespace to enumerate.
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  eiredirect     - If linking page is a redirect, find all pages that link to that redirect (not implemented)
  eilimit        - How many total pages to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/		
	public static final int	EMBEDDEDIN_MAX_LIMIT	= 500;
	
	public EmbeddedInResult embeddedIn(
			String title, List<NameSpace> nameSpaces, boolean redirect,
			int limit, String continueKey) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"embeddedin");
		query.string("eititle",			title);
		query.nameSpaces("einamespace",	nameSpaces);
		query.bool("eiredirect",		redirect);
		query.string("eicontinue",		continueKey);
		query.limit("eilimit", limit, EMBEDDEDIN_MAX_LIMIT);
		
		return (EmbeddedInResult)apiJSON(
				query, EMBEDDEDIN_CONVERTER);
	}
	
	public static final JSONConverter	EMBEDDEDIN_CONVERTER	= 
		new CreateObj(EmbeddedInResult.class,
			new MapEntry("query",
				new CreateObj(EmbeddedIn_embeddedin.class,
					new MapEntry("embeddedin",
						new ListIter(
							new CreateObj(EmbeddedIn_ei.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("embeddedin",
						new MapEntry("eicontinue",	new Copy())))));
	
/* list=exturlusage (eu) *
  Enumerate pages that contain a given URL
Parameters:
  euprop         - What pieces of information to include
                   Values (separate with '|'): ids, title, url
                   Default: ids|title|url
  euoffset       - Used for paging. Use the value returned for "continue"
  euprotocol     - Protocol of the url
                   One value: http, https, ftp, irc, gopher, telnet, nntp, worldwind, mailto, news
                   Default: http
  euquery        - Search string without protocol. See [[Special:LinkSearch]]
  eunamespace    - The page namespace(s) to enumerate.
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  eulimit        - How many entries to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	EXTURLUSAGE_MAX_LIMIT	= 500;
	
	public ExtUrlUsageResult extUrlUsage(
			String protocol, String search, List<NameSpace> nameSpaces,
			int limit, String continueKey) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"exturlusage");
		query.string("euquery",			search);
		query.protocol("euprotocol",	protocol);
		query.string("euprop",			"ids|title|url");
		query.nameSpaces("eunamespace",	nameSpaces);
		query.string("euoffset",		continueKey);	// NOTE: not "continue" this time // TODO not a number??
		query.limit("eulimit", limit, EXTURLUSAGE_MAX_LIMIT);
		
		return (ExtUrlUsageResult)apiJSON(
				query, EXTURLUSAGE_CONVERTER);
	}
	
	public static final JSONConverter	EXTURLUSAGE_CONVERTER	= 
		new CreateObj(ExtUrlUsageResult.class,
			new MapEntry("query", 
				new CreateObj(ExtUrlUsage_exturlusage.class,
					new MapEntry("exturlusage",
						new ListIter(
							new CreateObj(ExtUrlUsage_eu.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy()),
								new MapEntry("url",			new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("exturlusage",
						new MapEntry("euoffset",	new LongToInt())))));
	
/* list=imageusage (iu) *
  Find all pages that use the given image title.
Parameters:
  iutitle        - Title to search. If null, titles= parameter will be used instead, but will be obsolete soon.
  iucontinue     - When more results are available, use this to continue.
  iunamespace    - The namespace to enumerate.
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  iuredirect     - If linking page is a redirect, find all pages that link to that redirect (not implemented)
  iulimit        - How many total pages to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	IMAGEUSAGE_MAX_LIMIT	= 500;
	
	public ImageUsageResult imageUsage(
			String title, List<NameSpace> nameSpaces, boolean redirect,
			int limit, String continueKey) throws MediaWikiException {
		
		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"imageusage");
		query.string("iutitle",			title);
		query.nameSpaces("iunamespace",	nameSpaces);
		query.bool("iuredirect",		redirect);
		query.string("iucontinue",		continueKey);
		query.limit("iulimit", limit, IMAGEUSAGE_MAX_LIMIT);
		
		return (ImageUsageResult)apiJSON(
				query, IMAGEUSAGE_CONVERTER);
	}
	
	public static final JSONConverter	IMAGEUSAGE_CONVERTER	= 
		new CreateObj(ImageUsageResult.class,
			new MapEntry("query", 
				new CreateObj(ImageUsage_imageusage.class,
					new MapEntry("imageusage",
						new ListIter(
							new CreateObj(ImageUsage_iu.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("imageusage",
						new MapEntry("iucontinue",	new Copy())))));
	
/* list=logevents (le) *
  Get events from logs.
Parameters:
  leprop         - 
                   Values (separate with '|'): ids, title, type, user, timestamp, comment, details
                   Default: ids|title|type|user|timestamp|comment|details
  letype         - Filter log entries to only this type(s)
                   Can be empty, or Values (separate with '|'): block, protect, rights, delete, upload, move, import, patrol, renameuser, newusers, makebot
  lestart        - The timestamp to start enumerating from.
  leend          - The timestamp to end enumerating.
  ledir          - In which direction to enumerate.
                   One value: newer, older
                   Default: older
  leuser         - Filter entries to those made by the given user.
  letitle        - Filter entries to those related to a page.
  lelimit        - How many total event entries to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	LOGEVENTS_MAX_LIMIT	= 500;
	
	public LogEventsResult logEvents(
			Date start, Date end, boolean newer, 
			String user, String title,
			int limit) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"logevents");
		query.string("leprop",		"ids|title|type|user|timestamp|comment|details");
		query.string("letype",		"block|protect|rights|delete|upload|move|import|patrol|renameuser|newusers|makebot");
		query.date("lestart",		start);
		query.date("leend",			end);
		query.direction("ledir",	newer);
		query.string("leuser",		user);
		query.string("letitle",		title);
		query.limit("lelimit", limit, LOGEVENTS_MAX_LIMIT);
		
		return (LogEventsResult)apiJSON(
				query, LOGEVENTS_CONVERTER);
	}
	
	public static final JSONConverter	LOGEVENTS_CONVERTER	= 
		new CreateObj(LogEventsResult.class,
			new MapEntry("query", 
				new CreateObj(LogEvents_logevents.class,
					new MapEntry("logevents",
						new ListIter(
							new CreateObj(LogEvents_item.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy()),
								new MapEntry("logid",		new LongToInt()),
								new MapEntry("timestamp",	new IsoToDate()),
								new MapEntry("type",		new Copy()),
								new MapEntry("action",		new Copy()),
								new MapEntry("user",		new Copy()),
								new MapEntry("comment",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("logevents",
						new MapEntry("lestart",	new IsoToDate())))));
		/*
		TODO
		MapToObj	converting: class net.psammead.mwapi.yurik.data.list.LogEvents_item
		MapToObj	not mapped: [ns, move]
		"move": {
			"new_ns": 0,
			"new_title": "Johannes Garcaeus der \u00c4ltere"
		},
		*/
	
	
/* list=recentchanges (rc) *
  Enumerate recent changes
Parameters:
  rcstart        - The timestamp to start enumerating from.
  rcend          - The timestamp to end enumerating.
  rcdir          - In which direction to enumerate.
                   One value: newer, older
                   Default: older
  rcnamespace    - Filter log entries to only this namespace(s)
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  rcprop         - Include additional pieces of information
                   Values (separate with '|'): user, comment, flags, timestamp, title, ids, sizes
                   Default: title|timestamp|ids
  rcshow         - Show only items that meet this criteria.
                   For example, to see only minor edits done by logged-in users, set show=minor|!anon
                   Values (separate with '|'): minor, !minor, bot, !bot, anon, !anon
  rclimit        - How many total pages to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
*/
	public static final int	RECENTCHANGES_MAX_LIMIT	= 500;
	
	public RecentChangesResult recentChanges(
			List<NameSpace> nameSpaces, Date start, Date end, boolean newer, 
			int limit) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"recentchanges");
		query.string("rcprop",			"user|comment|flags|timestamp|title|ids|sizes");
		query.nameSpaces("rcnamespace",	nameSpaces);
		query.date("rcstart",			start);
		query.date("rcend",				end);
		query.direction("rcdir",		newer);
		/*
		rcshow         - Show only items that meet this criteria.
                   For example, to see only minor edits done by logged-in users, set show=minor|!anon
                   Values (separate with '|'): minor, !minor, bot, !bot, anon, !anon
		*/
		query.limit("rclimit", limit, RECENTCHANGES_MAX_LIMIT);
		
		return (RecentChangesResult)apiJSON(
				query, RECENTCHANGES_CONVERTER);
	}
	
	public static final JSONConverter	RECENTCHANGES_CONVERTER	= 
		new CreateObj(RecentChangesResult.class,
			new MapEntry("query", 
				new CreateObj(RecentChanges_recentchanges.class,
					new MapEntry("recentchanges",
						new ListIter(
							new CreateObj(RecentChanges_rc.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy()),
								new MapEntry("revid",		new Copy()),
								new MapEntry("old_revid",	new Copy()),
								new MapEntry("rcid",		new Copy()),
								new MapEntry("timestamp",	new IsoToDate()),
								new MapEntry("new",			new ExistsToBool()),
								new MapEntry("minor",		new ExistsToBool()),
								new MapEntry("anon",		new ExistsToBool()),
								new MapEntry("type",		new Copy()),
								new MapEntry("user",		new Copy()),
								new MapEntry("oldlen",		new LongToInt()),
								new MapEntry("newlen",		new LongToInt()),
								new MapEntry("comment",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("recentchanges",
						new MapEntry("rcstart",	new IsoToDate())))));
	
/* list=usercontribs (uc) *
  Get all edits by a user
Parameters:
  uclimit        - The maximum number of contributions to return.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
  ucstart        - The start timestamp to return from.
  ucend          - The end timestamp to return to.
  ucuser         - The user to retrieve contributions for.
  ucdir          - The direction to search (older or newer).
                   One value: newer, older
                   Default: older
  ucnamespace    - Only list contributions in these namespaces
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  ucprop         - Include additional pieces of information
                   Values (separate with '|'): ids, title, timestamp, comment, flags
                   Default: ids|title|timestamp|flags|comment
  ucshow         - Show only items that meet this criteria, e.g. non minor edits only: show=!minor
                   Values (separate with '|'): minor, !minor
*/
	public static final int	USERCONTRIBS_MAX_LIMIT	= 500;
	
	public UserContribsResult userContribs(
			String user, List<NameSpace> nameSpaces, Date start, Date end, boolean newer, 
			int limit) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"usercontribs");
		query.string("ucuser",			user);
		query.string("ucprop",			"ids|title|timestamp|comment|flags");
		query.nameSpaces("ucnamespace",	nameSpaces);
		query.date("ucstart",			start);
		query.date("ucend",				end);
		query.direction("ucdir",		newer);
		/*
		show         - Show only items that meet this criteria, e.g. non minor edits only: show=!minor
                 		Values (separate with '|'): minor, !minor
		*/
		query.limit("uclimit", limit, USERCONTRIBS_MAX_LIMIT);
		
		return (UserContribsResult)apiJSON(
				query, USERCONTRIBS_CONVERTER);
	}
	
	public static final JSONConverter	USERCONTRIBS_CONVERTER	= 
		new CreateObj(UserContribsResult.class,
			new MapEntry("query", 
				new CreateObj(UserContribs_usercontribs.class,
					new MapEntry("usercontribs",
						new ListIter(
							new CreateObj(UserContribs_item.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy()),
								new MapEntry("revid",		new Copy()),
								new MapEntry("timestamp",	new IsoToDate()),
								new MapEntry("new",			new ExistsToBool()),
								new MapEntry("minor",		new ExistsToBool()),
								new MapEntry("comment",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("usercontribs",
						new MapEntry("ucstart",	new IsoToDate())))));
	
/* list=watchlist (wl) *

Parameters:
  wlallrev       - Include multiple revisions of the same page within given timeframe.
  wlstart        - The timestamp to start enumerating from.
  wlend          - The timestamp to end enumerating.
  wlnamespace    - Filter changes to only the given namespace(s).
                   Values (separate with '|'): 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 100, 101
  wldir          - In which direction to enumerate pages.
                   One value: newer, older
                   Default: older
  wllimit        - How many total pages to return per request.
                   No more than 500 (5000 for bots) allowed.
                   Default: 10
  wlprop         - Which additional items to get (non-generator mode only).
                   Values (separate with '|'): ids, title, flags, user, comment, timestamp, patrol, sizes
                   Default: ids|title|flags
*/
	public static final int	WATCHLIST_MAX_LIMIT	= 500;
	
	public WatchListResult watchList(
			List<NameSpace> nameSpaces, Date start, Date end, boolean newer, 
			int limit) throws MediaWikiException {

		YurikQuery	query	= new YurikQuery();
		query.string("action",	"query");
		query.string("format",	"json");
		query.string("list",	"watchlist");
		query.string("wlprop",			"ids|title|flags|user|comment|timestamp|sizes");	// TODO patrol (not available sometimes)
		query.bool("wlallrev",			true);
		query.nameSpaces("wlnamespace",	nameSpaces);
		query.date("wlstart",			start);
		query.date("wlend",				end);
		query.direction("wldir",		newer);
		query.limit("wllimit", limit, WATCHLIST_MAX_LIMIT);
		
		return (WatchListResult)apiJSON(
				query, WATCHLIST_CONVERTER);
	}
	
	public static final JSONConverter	WATCHLIST_CONVERTER	= 
		new CreateObj(WatchListResult.class,
			new MapEntry("query", 
				new CreateObj(WatchList_watchlist.class,
					new MapEntry("watchlist",
						new ListIter(
							new CreateObj(WatchList_item.class,
								new MapEntry("title",		new TitleToLocation()),
								new MapEntry("pageid",		new Copy()),
								new MapEntry("revid",		new Copy()),
								new MapEntry("timestamp",	new IsoToDate()),
								new MapEntry("new",			new ExistsToBool()),
								new MapEntry("minor",		new ExistsToBool()),
								new MapEntry("anon",		new ExistsToBool()),
								new MapEntry("type",		new Copy()),
								new MapEntry("user",		new Copy()),
								new MapEntry("oldlen",		new LongToInt()),
								new MapEntry("newlen",		new LongToInt()),
								new MapEntry("comment",		new Copy())))))),
			new MapEntry("query-continue",
				new NullSafe(
					new MapEntry("watchlist",
						new MapEntry("wlstart",	new IsoToDate())))));
	
	//==============================================================================
	//## private helper
	
	public static final JSONConverter	ERROR_CONVERTER	=
		new MapEntry("error",
			new CreateObj(ErrorResult.class,
				new MapEntry("code",	new Copy()),
				new MapEntry("info",	new Copy()),
				new MapEntry("*",		new Copy())));
	
	/** fetch api.php, convert to JSON and handle error tags */
	private Object apiJSON(YurikQuery query, JSONConverter resultConverter) throws MediaWikiException {
		String	raw	= apiGET(query.toQueryString(charSet));
		Object	json;
		try { 
			json = JSONDecoder.decode(raw);
		}
		catch (JSONDecodeException e) { 
			throw new YurikJSONException("json decoding failed", e)
					.addFactoid("content",	raw); 
		}
		
		if (!(json instanceof Map)) {
			throw new YurikJSONException("expected a Map at top level")
					.addFactoid("content",	raw)
					.addFactoid("json",		json);
		}
		Map<?,?>	data	= (Map<?,?>)json;

		try { 
			if (data.get("error") != null) {
				throw new YurikErrorException(
						(ErrorResult)ERROR_CONVERTER.convert(ctx, json)); 
			}
			return resultConverter.convert(ctx, json); 
		}
		catch (JSONConverterException e) { 
			throw new YurikJSONException("json conversion failed", e)
					.addFactoid("content",	raw)
					.addFactoid("json",		json);
		}
	}

	// TODO use POST ?
	private String apiGET(String query) throws MediaWikiException {
		HttpMethod	method	= null;
		try {
			throttle.gate();
			
			// execute method
			method	= new GetMethod(apiURL);
			method.setFollowRedirects(false);
			method.addRequestHeader("User-Agent", userAgent);
			method.setQueryString(query);
			
			int			responseCode	= client.executeMethod(method);
			String		responseBody	= method.getResponseBodyAsString();
			StatusLine	statusLine		= method.getStatusLine();
			debug(method);
			
			// handle response
			if (responseCode != 200) {
				throw new UnexpectedAnswerException("unexpected response code (YurikAPI)")
						.addFactoid("status",	statusLine)
						.addFactoid("content",	responseBody);
			}
			
			return responseBody;
		}
		catch (HttpException		e) { throw new MethodException("method failed",  e); }
		catch (IOException			e) { throw new MethodException("method failed",  e); }
		catch (InterruptedException	e) { throw new MethodException("method aborted", e); }
		finally { if (method != null) method.releaseConnection(); }
	}
	
	/** print debug info for a HTTP-request */
	protected void debug(HttpMethod method) throws URIException {
		logger.debug(
				"HTTP " + method.getName() + 
				" " + method.getURI().toString() + 
				" " + method.getStatusLine());
	}
}
