//-----------------------------------------------------------------------------
// World
//-----------------------------------------------------------------------------

#include "world.h"
#include "console.h"
#include "timer.h"
#include "vars.h"
#include "overlay.h"
#include "render.h"
#include "math.h"
#include "sound.h"
#include "entity.h"
#include "system.h"
#include "client.h"

Var v_showclientbox("v_showclientbox", 0);

World::World(void)
{
  bsp = NULL;

  gravity = 800.f;
  maxvspeed = 2000.f;

  gVars->RegisterVar(v_showclientbox);
}

World::~World(void)
{
  gVars->UnregisterVar(v_showclientbox);
}

Q3BSP* World::GetBSP(void)
{
  return bsp;
}

int World::AddMap(const char *name)
{
  int result;

  memset(mapname, '\0', 32);
  strncpy(mapname, name, 32);

  char mappath[1024] = { '\0' };
  strncpy(mappath, gVars->StringForKey("r_mapsubdir"), 1023);
  if (mappath[strlen(mappath)-1] != '/') sprintf(mappath, "%s/", mappath);
  strcat(mappath, name);

  // replace extension
  f_StripExtension(mappath);
  strcat(mappath, ".bsp");

  // Stop and delete music and sounds
  freeBGMusic();

  bsp = new Q3BSP(mappath, this, &result);

  if (!result) bsp = NULL;

  return result;
}

// FIXME: heap gets corrupted here:
void World::Init(void)
{ 
  bsp->DisplayLoadingMessage("Loading... shaders");

  // load shader that have been referenced, and load also his textures
  // (shaders must be loaded before any other object)
  gConsole->Insertln("Updating world shaders...");
  shaders.Update();

  // TODO: every object should get the current pointer to his shaders in the
  //     initialization instead of calling GetShader every frame.
  bsp->Init();
}

void World::Shut(void)
{
  gConsole->Insertln("Shuting world...");

  // Stop and delete music
  freeBGMusic();

  if (bsp) delete bsp;
  bsp = NULL;

  shaders.ResetAll();
  shaders.Update();
}

void World::Render(Client *clients, int numclients)
{
  int i;
  bool waterstate;

  // All clients are updated before rendering
  for (i = 0; i < numclients; ++i)
  {
    if (clients[i].active_client)
    {
      clients[i].Update(bsp, this);           // update client
    }
  }

  // Render the clients
  for (i = 0; i < numclients; ++i)
  {
    gRender->InitializeViewPort();

    gRender->current_camera = &clients[i].cam;
    waterstate = gRender->current_camera->inWater;

    clients[i].cam.UpdateRotation();
    clients[i].cam.UpdatePosition();
    clients[i].cam.UpdateProjection();
    clients[i].cam.UpdateViewport();
    clients[i].cam.UpdateFrustum(); // set up frustum planes

    // TODO: Only update entities that are in current PVS
    bsp->GetEntities()->UpdateEntities(&clients[i]);

    bsp->Update();                      // update the scene (moving objects, visibility determination)

    if (clients[i].isCurrent)
      SoundUpdate(clients[i].cam.pos, clients[i].cam.forward);  // update sound settings for current client

    // FIXME: Put this in a better place
    if (!waterstate && gRender->current_camera->inWater) clients[i].Water(0);
    else if (waterstate && !gRender->current_camera->inWater) clients[i].Water(1);

    bsp->Render();

    gRender->ForceFlush();

    // Draw the camera box
    if (v_showclientbox.ivalue)
    {
      colour_t colorline = { 255, 255, 255, 255 };
      colour_t colorface;
      bboxf_t clbox;
      for (int j = 0; j < numclients; ++j)
      {
        VectorAdd(clients[j].mins, clients[j].cam.pos, &clbox[0]);
        VectorAdd(clients[j].maxs, clients[j].cam.pos, &clbox[3]);
        if (i == j) ColorClear(colorface);
        else ColorCopy(clients[j].color, colorface);
        Q3BSP::DrawBoundingBox(clbox, colorline, colorface);
      }
      gRender->ForceFlush();
    }
  }
}

// Find the first spawn point in the entities list
void World::SetStartPos(Client *c, int n)
{
  if (!bsp || !c) return;

  EntityManager *entman = bsp->GetEntities();
  int i = entman->Find_entity(ET_INFO_PLAYER_DEATHMATCH, n);
  if (i < 0) return;

  Entity* ent = entman->entities+i;

  c->ResetMovement();
  c->cam.rot[PITCH] = 90;
  c->cam.rot[YAW] = ent->angle-90;
  c->cam.rot[ROLL] = 0;
  c->TeleportTo(ent->origin, ent->angle, 0);
}

// TODO: Optimize by caching the diffrent elements of entities and sorting the different entities
int World::GetNumStartPos(void)
{
  if (!bsp) return 0;
  int i, res = 0;
  EntityManager *entman = bsp->GetEntities();
  if (!entman) ThrowException(NULL_POINTER, "World::GetNumStartPos.entman");
  for (i = 0; i < entman->num_entities; ++i)
  {
    if (entman->entities[i].enttype == ET_INFO_PLAYER_DEATHMATCH) ++res;
  }
  return res;
}
