//-----------------------------------------------------------------------------
// Types
// Defines base types
//-----------------------------------------------------------------------------

/**
 * This file contents the declaration of some types of base. These types
 * are used everywhere in the engine.
 */

#ifndef __TYPES_H__
#define __TYPES_H__

#include "definitions.h"
#include "shaderflags.h"
#include <string.h>         // for NULL pointer + memset

class Messages;
class Var;                  // Variable
class Vars;                 // Variables list
class Commands;
class LogFile;              // Log file support
class Console;              // Console
class FrameWork;            // FrameWork
class Timer;                // Timer
class Render;               // Render
class Camera;               // Camera
class Frustum;              // Frustum
class Entity;               // Entity
class Surface;
class Overlay;
class TextureManager;
class LayerManager;
class ShaderManager;
class EntityManager;
class World;
class Client;
class Alias;
class Q3BSP;
class BezierPatch;
class App;                  // Application

typedef float vec_t;
typedef vec_t vec2_t[2];
typedef vec_t vec3_t[3];
typedef vec_t vec4_t[4];
typedef vec_t vec5_t[5];

typedef float texcoord_t[2];        /**< Texture s&t coordinates */
typedef int bbox_t[6];              /**< Integer bounding box (mins, maxs) */
typedef float bboxf_t[6];           /**< Float bounding box */
typedef unsigned char colour_t[4];  /**< RGBA */

// Undo any Windows defines.
#undef BYTE
#undef WORD
#undef DWORD
#undef INT
#undef UINT
#undef FLOAT
#undef MAXBYTE
#undef MAXWORD
#undef MAXDWORD
#undef MAXINT
#undef CDECL
#undef BOOL

// Unsigned base types.
typedef unsigned char     BYTE;     /**< 8-bit  unsigned. */
typedef unsigned int      UINT;
typedef unsigned short    WORD;     /**< 16-bit unsigned. */
typedef unsigned long     DWORD;    /**< 32-bit unsigned. */
#ifdef WIN32
typedef unsigned __int64  QWORD;    /**< 64-bit unsigned. */
#endif

// Signed base types.
typedef signed char       SBYTE;    /**< 8-bit  signed. */
typedef signed short      SWORD;    /**< 16-bit signed. */
typedef signed int        INT;      /**< 32-bit signed. */
#ifdef WIN32
typedef signed __int64    SQWORD;   // 64-bit signed. */
#endif

// small letter version:
typedef unsigned char     byte;
typedef unsigned short    word;
typedef unsigned int      dword;

// other types.
typedef float       FLOAT;
typedef double      DOUBLE;         /**< 64-bit IEEE double. */
typedef INT         BOOL;
typedef long        LONG;

/**
 * Engine states.
 * The states engines are used to mark different steps in engine execution.
 * Depending on the current state, some functions or some features are
 * enabled or disabled.
 */
typedef enum
{
  INACTIVE = 0,               /**< the engine is inactive */
  INITIALIZING = 1,           /**< the engine is in initialization phase */
  RUNNING = 2,                /**< the engine is running */
  SHUTING = 3                 /**< the engine is shuting down */
} EngineState;

//-----------------------------------------------------------------------------

// content masks
#define MASK_ALL          (-1)
#define MASK_SOLID        (CONTENTS_SOLID)
#define MASK_PLAYERSOLID  (CONTENTS_SOLID|CONTENTS_PLAYERCLIP|CONTENTS_BODY)
#define MASK_DEADSOLID    (CONTENTS_SOLID|CONTENTS_PLAYERCLIP)
#define MASK_MONSTERSOLID (CONTENTS_SOLID|CONTENTS_MONSTERCLIP|CONTENTS_WINDOW|CONTENTS_MONSTER)
#define MASK_WATER        (CONTENTS_WATER|CONTENTS_LAVA|CONTENTS_SLIME)
#define MASK_OPAQUE       (CONTENTS_SOLID|CONTENTS_SLIME|CONTENTS_LAVA)
#define MASK_SHOT         (CONTENTS_SOLID|CONTENTS_BODY|CONTENTS_CORPSE)
#define MASK_CURRENT      (CONTENTS_CURRENT_0|CONTENTS_CURRENT_90|CONTENTS_CURRENT_180|CONTENTS_CURRENT_270|CONTENTS_CURRENT_UP|CONTENTS_CURRENT_DOWN)

//-----------------------------------------------------------------------------

/**
 * Interleaved vertex info.
 * The vertexes lump stores lists of vertices used to describe faces. There is a
 * position, texture and lightmap coordinates, the vertex normal and color. There
 * are a total of length / sizeof(vertex) records in the lump, where length is
 * the size of the lump itself, as specified in the lump directory.
 */
typedef struct
{
  vec3_t v_point;     /**< vertex position */
  texcoord_t tex_st;  /**< surface texture coords */
  texcoord_t lm_st;   /**< lightmap texture coords */
  vec3_t v_norm;      /**< vertex normal */
  colour_t colour;    /**< colour used for vertex lighting (RGBA) */
} vertex_t;

/**
 * Faces (or surfaces).
 * The faces lump stores information used to render the surfaces of the map.
 * There are a total of length / sizeof(faces) records in the lump, where length
 * is the size of the lump itself, as specified in the lump directory.<br><br>
 * 
 * There are four types of faces:
 *   - polygons, which are described by a loop of vertices
 *   - biquadratic Bezier patches, which are described by a two-dimensional grid
 *     of control vertices
 *   - triangle meshes, which contain independent triangles described by a set
 *     of vertices and a list of vertex indices
 *   - billboards for flares, which are represented using a single vertex.
 *
 * Note: Several components have different meanings depending on the face type. 
 * Note: This struct isn't used in the code (another is used instead)<br><br>
 * 
 * The lm_ variables are primarily used to deal with lightmap data. A face that
 * has a lightmap has a non-negative lm_index. For such a face, lm_index is the
 * index of the image in the lightmaps lump that contains the lighting data for
 * the face. The data in the lightmap image can be located using the rectangle
 * specified by lm_start and lm_size.<br><br> 
 * 
 * For type 1 faces (polygons) only, lm_origin and lm_vecs can be used to compute
 * the world-space positions corresponding to lightmap samples. These positions
 * can in turn be used to compute dynamic lighting across the face.<br><br> 
 * 
 * None of the lm_ variables are used to compute texture coordinates for indexing
 * into lightmaps. In fact, lightmap coordinates need not be computed. Instead,
 * lightmap coordinates are simply stored with the vertices used to describe each
 * face.<br><br> 
 * 
 * For type 1 faces (polygons), meshvert and n_meshverts describe a valid polygon
 * triangulation. This is in addition to the loop of vertices described by vertex
 * and n_vertexes. For certain polygons, the meshverts describe a triangulation
 * built entirely from the original polygon vertices. For other polygons, the
 * meshverts describe a triangulation built by adding a central vertex and
 * building a fan around that vertex.
 */
typedef struct
{
  int shader;           /**< shader reference (texture index) */
  int effect;           /**< index into lump 12 (Effects), or -1 if n/a */
  int facetype;         /**< face type: 1=polygon, 2=patch, 3=mesh, 4=billboard */
  int firstvert,        /**< index of first vertex  */
      numverts;         /**< number of vertices */
  int firstelem,        /**< index of first meshvert */
      numelems;         /**< number of meshverts */
  int lm_texnum;        /**< lightmap index */
  int lm_offset[2];     /**< corner of this face's lightmap image in lightmap */
  int lm_size[2];       /**< size of this face's lightmap image in lightmap */
  vec3_t v_orig;        /**< world space origin of lightmap (FACETYPE_NORMAL only) */
  bboxf_t bbox;         /**< world space lightmap s and t unit vectors (FACETYPE_MESH only) */
  vec3_t v_norm;        /**< surface normal (FACETYPE_NORMAL only) */
  int mesh_cp[2];       /**< patch dimensions (mesh control point dimensions) */
} face_t;

/**
 * Texture info.
 */
typedef struct
{
  int num;              /**< tex index */
  byte *mem;            /**< data */
  int width;            /**< texture width */
  int height;           /**< texture height */
  int bpp;              /**< texture bits per pixel */
} texinfo;

typedef struct _texrefs
{
  char *name;
  int count;            /**< @todo count could be used to give them priorities */
  int flags;
  texinfo *info;        /**< each texref has an unique texinfo */
} TexRefs;

/**
 * List of functions for sucesive vertex or coordinates modification.
 */
typedef struct _funclist
{
  int func;
  int parameters;
  float p[MAX_PARAMS];
  _funclist *next;
} funclist;

typedef struct Node_s
{
  int     lightmap;
  vertex_t *  firstvert;
  int     numverts;
  int *   firstelem;
  int     numelems;
} Node;

/**
 * Layers are used for multiple passes on shaders.
 */
typedef struct
{
  int       flags;              /**< layer flags */

  float     anim_speed;         /**< speed of the animation */
  int       num_maps;           /**< number of texures (for animated layers) */
  texinfo   *map[MAX_ANIMFRAMES]; /**< array of the maps */

  funclist  *tcMod;             /**< texture coordinates modifiers */
  funclist  *aGen;              /**< alpha color generation */

  TCGEN     tcGen;

  RGBGEN    rgbGen;
  float     rgbGenParams[5];

  ALPHAGEN  alphaGen;           /**< alpha channel generation */
  float     alphaGenParams[5];  /**< alpha channel parameter */

  int       alphafunc;          /**< alpha function */
  float     alphafuncref;

  int       blendsrc;           /**< blend function */
  int       blenddst;

  int       depthFunc;          /**< depth func */
  int       depthWrite;         /**< touch depth buffer */
} Layer;

/**
 * Deformed vertex.
 * @see DEFORMVERTEXES
 */
typedef struct
{
  DEFORMVERTEXES    type;
  float       params[8];
} DeformVertexes;

/**
 * Sky parameters.
 * @see SkyBox
 */
typedef struct
{
  int num_layers;         /**< 2 layers max */
  Layer* layer[2][6];     /**< layers used by farbox and nearbox */
  int cloudheight;        /**< apparent curvature of the cloud layers (default is 128) */
} SkyParms;

/**
 * Fog parameters.
 */
typedef struct
{
  Layer* layer;
  float params[5];
} FogParms;

/**
 * A shader describes how a surface is rendered.
 */
typedef struct
{
  char  shadername[64];   /**< shader name - used for debug only */

  int   surface_flags;    /**< surface_flags (described in surface_flags, come from the bsp, 
                               and not from the shader definition) */
  int   content_flags;    /**< contents of the shader ? */
  int   cull;             /**< culling type */

  int   sortvalue;        /**< @todo Carmack changed int sortvalues to float, why? */

  /**
   * Flags to determine misc properties of the shader, like polygonoffset,
   * multitexture, mipmap, nopicmip, haslightmap, portal... and to determine
   * which info of the datasource we need to render the surface
   */
  int   flags;

  int   num_layers;           /**< number of layers */
  Layer * layer[MAX_LAYERS];  /**< layers of stages (passes) table */

  int         num_deforms;            /**< number of deformations for the shader */
  DeformVertexes    deformVertexes[MAX_DEFORMS];  /**< vertex deformation */

  SkyParms      sky_params;           /**< sky parameters */
  FogParms      fog_params;           /**< fog parameters */
} Shader;

typedef struct
{
  int   num_nodes;
  Node* list;           /**< list of elements asociated with this shader */

  int   ref_count;      /**< number of times this shader has been referenced (for cache issues) */
  Shader* shader;       /**< the shader itself */

  char  name[64];       /**< name of the shader */
  int   flags;          /**< surface flags */
  int   contents;       /**< content flags */
  int   type;           /**< ? */
} ShaderHash;

typedef struct
{
  char *keyword;
  int minargs, maxargs;
  void (* func)(Shader *shader, Layer *layer);
} shaderkey_t;

//-----------------------------------------------------------------------------
// BSP Data types
//-----------------------------------------------------------------------------

/**
 * BSP lumps in the order they appear in the header
 */
typedef enum
{
  ENTITIES,         /**< Game-related object descriptions */
  SHADERREFS,       /**< Shader descriptions */
  _PLANES,          /**< Planes used by map geometry */
  NODES,            /**< BSP tree nodes */
  LEAFS,            /**< BSP tree leaves */
  LFACES,           /**< Leaf faces - Lists of leaf's indices into faces, one list per leaf */
  LBRUSHES,         /**< Leaf brushes - Lists of leaf's indices into brushes, one list per leaf */
  MODELS,           /**< Descriptions of rigid world geometry in map */
  BRUSHES,          /**< Convex polyhedra used to describe solid space */
  BRUSH_SIDES,      /**< Brush surfaces info */
  VERTS,            /**< Vertices used to describe faces */
  ELEMS,            /**< Lists of offsets, one list per mesh */
  EFFECTS,          /**< List of special map effects */
  FACES,            /**< Surface geometry */
  LIGHTMAPS,        /**< Packed lightmap data */
  LIGHTVOLS,        /**< Local illumination data */
  VISIBILITY,       /**< Cluster-cluster visibility data */
  NUM_LUMPS         /**< Number of lumps */
} lumps;

//-----------------------------------------------------------------------------

/**
 * Shader references (indexed from faces).
 * The shaders lump stores information about surfaces and volumes, which are in
 * turn associated with faces, brushes, and brushsides. There are a total of
 * length / sizeof(texture) records in the lump, where length is the size of the
 * lump itself, as specified in the lump directory.
 */
typedef struct
{
  char name[64];          /**< Shader name */
  int surface_flags;      /**< Surface flags */
  int content_flags;      /**< Content flags */
} shaderref_t;

/** 
 * Planes.
 * The planes lump stores a generic set of planes that are in turn referenced by
 * nodes and brushsides. There are a total of length / sizeof(plane) records in
 * the lump, where length is the size of the lump itself, as specified in the lump
 * directory.
 * Note that planes are paired. The pair of planes with indices i and i ^ 1 are
 * coincident planes with opposing normals.
 * @see cplane_t
 */
typedef struct
{
  vec3_t normal;        /**< Plane normal */
  float dist;           /**< Distance from origin to plane along normal */
} plane_t;

/**
 * Nodes in the BSP tree.
 * The nodes lump stores all of the nodes in the map's BSP tree. The BSP tree is
 * used primarily as a spatial subdivision scheme, dividing the world into convex
 * regions called leafs. The first node in the lump is the tree's root node. The
 * BSP is not used for rendering so much in Quake3, but for collision detection.
 * The node holds the splitter plane index, the front and back index, aloong with
 * the bounding box for the node. If the front or back indices are negative, then
 * it's an index into the leafs array. Since negative numbers can't constitute an
 * array index, you need to use the ~ operator or -(index + 1) to find the correct
 * index. This is because 0 is the starting index. There are a total of
 * length / sizeof(node) records in the lump, where length is the size of the lump
 * itself, as specified in the lump directory.
 * @see cnode_t
 */
typedef struct
{
  int plane;            /**< Dividing plane index (index into planes array) */
  int children[2];      /**< Left and right node. Negatives are leafs indices : -(leaf+1) */
  bbox_t bbox;          /**< Integer bounding box (order : 3 min + 3 max) */
} node_t;

/**
 * Leafs in BSP tree.
 * The leafs lump stores the leaves of the map's BSP tree. Each leaf is a convex
 * region that contains, among other things, a cluster index (for determining the
 * other leafs potentially visible from withing the leaf), a list of faces (for
 * rendering), and a list of brushes (for collision detection). There are a total
 * of length / sizeof(leaf) records in the lump, where length is the size of the
 * lump itself, as specified in the lump directory.
 * @see cleaf_t
 */
typedef struct
{
  int cluster;        /**< Visibility cluster number */
  int area;           /**< area minus one ??? */
  bbox_t bbox;        /**< Integer bounding box (order : 3 min + 3 max) */
  int firstface,      /**< First leafface for leaf */
  numfaces;           /**< Number of leaffaces for leaf */
  int firstbrush,     /**< First leafbrush for leaf */
  numbrushes;         /**< Number of leafbrushes for leaf */
} leaf_t;

/**
 * Leaffaces lump.
 * The leaffaces lump stores lists of face indices, with one list per leaf. The leaf
 * faces are used to index into faces array. You might at first think this is strange
 * to have leaf structure have an index into the lface array, which in turn is just an
 * index into the faces array. This is because it's set up to start with a starting
 * point (leafface) and a count to go from there for each face (number of leaffaces).
 * The faces array is not contiguous (in a row) according to each leaf. That is where
 * the leaffaces array comes into play. It's kinda like the same concept of model
 * loaders where they store the vertices and then havefaces that store the indices
 * into the vertex array for that face. There are a total of length / sizeof(leafface)
 * records in the lump, where length is the size of the lump itself, as specified in
 * the lump directory.<br>
 * Note : this is directly used as int in the code
 */
typedef struct
{
  int face;         /**< Face index */
} lface_t;

/**
 * Leafbrushes lump.
 * The leafbrushes lump stores lists of brush indices, with one list per leaf.
 * There are a total of length / sizeof(leafbrush) records in the lump, where
 * length is the size of the lump itself, as specified in the lump directory.<br>
 * Note : this is directly used as int in the code
 */
typedef struct
{
  int brush;          /**< Brush index */
} lbrush_t;

/**
 * Model 0 is the main map, others are doors, gates, buttons, etc.
 * The models lump describes rigid groups of world geometry. The first model
 * correponds to the base portion of the map while the remaining models correspond 
 * movable portions of the map, such as the map's doors, platforms, and buttons.
 * Each model has a list of faces and list of brushes; these are especially
 * important for the movable parts of the map, which (unlike the base portion of
 * the map) do not have BSP trees associated with them. There are a total of
 * length / sizeof(models) records in the lump, where length is the size of the
 * lump itself, as specified in the lump directory.
 * @see cmodel_t
 */
typedef struct
{
  bboxf_t bbox;         /**< Bonunding box (order : 3 min + 3 max) */
  int firstface;        /**< First face for model */
  int numfaces;         /**< Number of faces for model */
  int firstbrush;       /**< First brush for model */
  int numbrushes;       /**< Number of brushes for model */
} model_t;

/**
 * Brushes in BSP tree.
 * The brushes lump stores a set of brushes, which are in turn used for collision
 * detection. Each brush describes a convex volume as defined by its surrounding
 * surfaces, the brush sides. There are a total of length / sizeof(brushes) records
 * in the lump, where length is the size of the lump itself, as specified in the
 * lump directory.
 * @see cbrush_t
 */
typedef struct
{
  int firstside;      /**< First brushside for brush */
  int numsides;       /**< Number of brushsides for brush */
  int shader;         /**< Texture index */
} brush_t;

/**
 * Brushes sides in BSP tree.
 * The brushsides lump stores descriptions of brush bounding surfaces. There are a
 * total of length / sizeof(brushsides) records in the lump, where length is the
 * size of the lump itself, as specified in the lump directory.
 */
typedef struct
{
  int planeidx;       /**< Plane index */
  int shader;         /**< Texture index (shader) */
} brushside_t;

/**
 * Meshverts lump.
 * The meshverts lump stores lists of vertex offsets, used to describe generalized
 * triangle meshes. There are a total of length / sizeof(meshvert) records in the
 * lump, where length is the size of the lump itself, as specified in the lump
 * directory.<br>
 * Note : this is directly used as int in the code
 */
typedef struct
{
  int offset;         /**< Vertex index offset, relative to first vertex of corresponding face */
} meshvert_t;

/**
 * Effect lump.
 * The effects lump stores references to volumetric shaders (typically fog) which
 * affect the rendering of a particular group of faces. There are a total of
 * length / sizeof(effect) records in the lump, where length is the size of the
 * lump itself, as specified in the lump directory.
 * @see ceffect_t
 */
typedef struct
{
  char name[64];      /**< Effect shader */
  int brush;          /**< Brush that generated this effect */
  int visibleside;    /**< The brush side that ray tests need to clip against (-1 == none) */
} effect_t;

/**
 * Lightmaps lump.
 * The lightmaps lump stores the light map textures used make surface lighting
 * look more realistic. There are a total of length / sizeof(lightmap) records in
 * the lump, where length is the size of the lump itself, as specified in the lump
 * directory.<br>
 * Note : this is directly used as BYTE in the code
 */
typedef struct
{
  BYTE map[128][128][3];    /**< Lightmap color data (RGB) */
} lightmap_t;

/**
 * Lightvols lump.
 * The lightvols lump stores a uniform grid of lighting information used to
 * illuminate non-map objects. There are a total of length / sizeof(lightvol)
 * records in the lump, where length is the size of the lump itself, as specified
 * in the lump directory.
 * <br>
 * Lightvols make up a 3D grid whose dimensions are: 
 * <pre>
  nx = floor(models[0].maxs[0] / 64) - ceil(models[0].mins[0] / 64) + 1
  ny = floor(models[0].maxs[1] / 64) - ceil(models[0].mins[1] / 64) + 1
  nz = floor(models[0].maxs[2] / 128) - ceil(models[0].mins[2] / 128) + 1
 * </pre>
 */
typedef struct
{
  BYTE ambient[3];      /**< Ambient color component. RGB. */
  BYTE directional[3];  /**< Directional color component. RGB. */
  BYTE dir[2];          /**< Direction to light. 0=phi, 1=theta. */
} lightvols_t;

/**
 * Visibility info.
 * The visdata lump stores bit vectors that provide cluster-to-cluster visibility
 * information. There is exactly one visdata record, with a length equal to that
 * specified in the lump directory.<br>
 * Cluster x is visible from cluster y if the <code>(1 << y % 8)</code> bit of
 * <code>vecs[x * sz_vecs + y / 8]</code> is set. (cf. BSP_TESTVIS macro)<br>
 * Note that clusters are associated with leaves. 
 */
typedef struct
{
    int numclusters;    /**< Number of PVS clusters (vectors) */
    int rowsize;        /**< Size of each vector, in bytes */
    BYTE data[1];       /**< Visibility data. One bit per cluster per vector */
                        /**<  => size : BYTE data[numclusters*rowsize] */
} visibility_t;

//-----------------------------------------------------------------------------
// Data of the bsp (internal structure of the engine)
//-----------------------------------------------------------------------------

/**
 * Faces (or surfaces).
 */
typedef struct
{
  int shader;           /**< Shader reference */
  int facetype;         /**< FACETYPE enum */
  vertex_t *firstvert;
  int numverts;
  int *firstelem;
  int numelems;
  int lm_texnum;
  vec3_t v_orig;        /**< FACETYPE_NORMAL only */
  vec3_t v_norm;        /**< FACETYPE_NORMAL only */
  float planedist;
} cnormalface_t;

/**
 * Planes.
 * @see plane_t
 */
typedef struct cplane_s
{
  vec3_t  normal;     /**< plane normal */
  float dist;
  BYTE  type;         /**< for fast side tests: 0,1,2 = axial, 3 = nonaxial */
  BYTE  signbits;     /**< signx + (signy<<1) + (signz<<2), used as lookup during collision */
  BYTE  pad[2];       /**< 2 bytes for padding puspouses */
} cplane_t;

/**
 * Brushes sides in BSP tree.
 * The brushsides lump stores descriptions of brush bounding surfaces. There are a
 * total of length / sizeof(brushsides) records in the lump, where length is the
 * size of the lump itself, as specified in the lump directory.
 */
typedef struct
{
  int planeidx;       /**< Plane index */
  int shader;         /**< Texture index (shader) */
} cbrushside_t;

/**
 * Model structure.
 * @see model_t
 */
typedef struct
{
  bboxf_t bbox;         /**< Bonunding box (order : 3 min + 3 max) */
  int headnode;         /**< First node for model */
  int firstface;        /**< First face for model */
  int numfaces;         /**< Number of faces for model */
  int firstbrush;       /**< First brush for model */
  int numbrushes;       /**< Number of brushes for model */

  Entity* entity;       /**< Corresponding entity */
  bool render;          /**< Is model renderable ? */
} cmodel_t;

/**
 * Brushes in BSP tree.
 * @see brush_t
 */
typedef struct
{
  int firstbrushside;   /**< side index in r_brushsides */
  int numsides;         /**< number of sides */
  int shader;           /**< shader */
  int effect;           /**< effect index (-1 if none) */
  int contents;         /**< brush content (determined after having loaded all brushes and leafs) */
  int checkcount;       /**< to avoid repeated testings */
} cbrush_t;

typedef struct
{
  vec3_t absmins, absmaxs;

  int numbrushes;
  int firstbrush;

  int shader;           /**< patch shader */
  int checkcount;       /**< to avoid repeated testings */
} cpatch_t;

/**
 * Nodes in the BSP tree.
 * @see node_t
 */
typedef struct cnode_s
{
  cnode_s * parent;
  int     framenum;
  int     planeidx;     /**< dividing plane (corresponding variable is r_clipplanes) */
  int     children[2];  /**< left and right node. Negatives are leafs */
  bbox_t  bbox;         /**< bounding box */
} cnode_t;

/**
 * Leafs in BSP tree.
 * @see leaf_t
 */
typedef struct
{
  cnode_s * parent;
  int     framenum;
  int     contents;   /**< leaf content (determined after having loaded all brushes and leafs) */
  int     cluster;    /**< visibility cluster number */
  int     area;       /**< area minus one ??? */
  bbox_t  bbox;
  int     firstleafface;  /**< index of leafface in the faces data (get leafface i by using r_leaffaces[firstleafface+i]) */
  int     numleaffaces;
  int     firstleafbrush; /**< index of leafbrush in the brush data (get leafbrush i by using r_leafbrushes[firstleafbrush+i]) */
  int     numleafbrushes;
  int     firstleafpatch; /**< index of leafpatch in the patch data (get leafpatch i by using r_leafpatches[firstleafpatch+i]) */
  int     numleafpatches;
} cleaf_t;

typedef struct
{
  Surface *face;
  int sort_value;
} render_face_s;

/**
 * Effect.
 * @see effect_t
 */
typedef struct
{
  int shader;         /**< shader reference */
  int brush;
  int visibleside;    /**< the brush side that ray tests need to clip against (-1 == none) */
  int planeidx;       /**< the plane where the effect (fog) starts */
} ceffect_t;

/**
 * Surface class.
 * A surface is a form of face that have following properties:
 * - the surface is composed with more subfaces, having all the same shader
 * - the surface can have different tesselation levels (this is used for
 *   progressive curves subdivisions for isntance)
 * The advantage of surface is that texture is linked once for each subfaces.
 * Subfaces can be tied up (cf. skybox) or not (cf. some autosprites).
 * @todo Try to write an optimization for tied up faces by using a triangle
 * strip function.
 * In the case of bezier patches, level 0 stores bsp informations (control
 * points) as for other faces type and higher levels get patch subdivided faces.
 */
class Surface
{
  public:
    Surface(void);
    ~Surface(void);

    /**
     * Surface initialization.
     * This initialize the surface and creates the levels tables. If
     * surfaces was already initialized, old settings and data are lost.
     * @param lvl the level for vertices and elems
     */
    void Init(int lvl = 1);

    /**
     * Destroys the surface content.
     * The function intialize the surface content and frees the used memory.
     */
    void Shut(void);

    int     levels;       /**< tesselation level */
    int     currentlevel;

    int     indextype;    /**< surface index in facetype array */

    int     model;        /**< model that uses this face */

    int     framenum;

    int     facetype;     /**< FACETYPE enum */

    int     shader;       /**< Shader reference in world shaders manager */
    int     lm_texnum;    /**< lightmap info */

    vertex_t ** firstvert;    /**< @todo Should be 'void *' + 'int vertex_id' */
    int *   numverts;
    int **    firstelem;
    int *   numelems;

    short   dist;
    vec3_t    v_norm;
    vec3_t    v_orig;

    bboxf_t   bbox;

    int     patch_cp[2];

    int     autosprite2_axis;

    int     effect;
#if MODIF_FOG
    cplane_t *  fogplane;   /**< fog plane */
#else
    float     fog[5];       /**< fog parameters */
#endif
};

/**
 * Movement structure used for collision detection.
 * A trace is returned when a box is swept through the world.
 */
typedef struct
{
  float   fraction;       /**< Movement fraction (1.0 = didn't hit anything) */
  
  vec3_t    start;        /**< Movement start position */
  vec3_t    end;          /**< Movement end position */
  
  vec3_t    size;         /**< Half size of bounding box */
  vec3_t    mins, maxs;
  vec3_t    absmins, absmaxs;

  bool    allSolid;
  bool    startSolid;

  cplane_t  plane;

  int     tracemask;      /**< Collide against these types of surfaces */
  int     contents;       /**< Content mask */
  int     shader;

//  Entity *  ent;

  bool    isPoint;        /**< The trace is a point */
} trace_t;

/*
 * link_t is only used for entity area links now  
 */
typedef struct link_s
{
  struct link_s *prev, *next;
} link_t;

#endif  /* __TYPES_H__ */
