//-----------------------------------------------------------------------------
// SkyBox
//-----------------------------------------------------------------------------

#ifndef __SKYBOX_H__
#define __SKYBOX_H__

#include "cake.h"
#include "types.h"

/**
 * Variable status.
 * Status defines variable type and domain in which variable can be
 * modified.
 */
enum
{
  SKYBOX_FRONT  = 0,  /**< the front cube plane */
  SKYBOX_LEFT   = 1,  /**< the left cube plane */
  SKYBOX_TOP    = 2,  /**< the trop cube plane */
  SKYBOX_BACK   = 3,  /**< the back cube plane */
  SKYBOX_RIGHT  = 4,  /**< the right cube plane */
  SKYBOX_BOTTOM = 5   /**< the bottom cube plane */
};

enum
{
  SKYMINMAX_MINX = 0,
  SKYMINMAX_MAXX = 1,
  SKYMINMAX_MINY = 2,
  SKYMINMAX_MAXY = 3
};

/**
 * SkyBox class.
 * The sky box is a cube used as a half-sphere to simulate the sky effect.
 * The sky has 3 levels:
 * - the far box: this is a cube environment mapping that represents distant
 *                objects (planets, deep space, etc.)
 * - the near box: this is a second cube environment mapping that represent
 *                 nearer objects (moon for example)
 * - the sky box: this is the cube with clouds
 * Skybox faces are defined as following :
 * - front : (1, 0, 0)
 * - left : (0, 1, 0)
 * - top : (0, 0, 1)
 * - back : (-1, 0, 0)
 * - right : (0, -1, 0)
 * - bottom : (0, 0, -1)
 * I believe that quake 3 uses another order, but final result is the same.
 * @bug The q3dm16 map has problems with displaying the scene bounding box.
 *      See it in debug mode so the clear color isn't black, or disable the
 *      screen clear.
 *      In fact only one cube plane is displayed, depending on the camera
 *      position in the axis system. For example, if camera is at (+,-,+)
 *      position for (x,y,z), only planes that are (-,+,-) are displayed. A
 *      similar problem occurs with map q3dm17: there is simply no bounding
 *      box to the scene. Try in debug mode or with screen clear disabled.
 *      In fact, this has certainly nothing to do with skybox !
 */
class SkyBox
{
  public:
    /**
     * Sky surface
     * The sky surface has 2 levels. First level correspond to clouds
     * and second level correspond to boxes (farbox and nearbox). The
     * second level has a simplier tesselation.
     */
    Surface *skyface;

    /**
     * Table of cloud surfaces.
     * The clouds form a half sphere and we only need 5 surfaces for it
     * (there is no bottom surface).
     */
    vertex_t *clouds[6];  /**< clouds coordinates */
    int clouds_shader;    /**< clouds shader index */

    /**
     * Table of cube environment surfaces.
     * The cube environment mapping effect uses the 6 planes of a
     * environment cube (bottom is used).
     */
    vertex_t *box[6];   /**< box texture coords (used for farbox and nearbox) */
    Shader *box_shader[6];  /**< the shader indexes for environment mapping */

    int num_elems;      /**< number of elems for a surface */
    int *clouds_elems;    /**< table of elems of one cube plane (clouds) */
    int *box_elems;     /**< table of elems of one cube plane (farbox and nearbox) */
    int *framenum[6];   /**< frame number for each face */
    bool usedskysides[6]; /**< defines if a sky side is used in frame */
    float minmax[6][4];   /**< min and max position of a used element in sky side */

    float skyboxsize;   /**< size of sky box (used for rendering scale) */

    SkyBox(void);
    ~SkyBox(void);

    /**
     * Initialize the skybox and creates all data tables.
     * @param height Height of clouds for texture coordinates generation.
     */
    void Init(int height);

    /**
     * Destroys all data tables created during initialization.
     */
    void Shut(void);

    /**
     * Array containing pointers of sky surfaces of the bsp. The array is
     * made during bsp loading phase.
     */
    Surface **surf_array;
    int nsurfs,       /**< size of the surf_array array */
      ncurr;        /**< current size of the surf_array array
                     (the size is growing progressively
                   during initialization phase) */

    /**
     * Creates the surf_array pointer array with specified size.
     * @param n the number of entries of the array
     */
    void MakeSkyList(int n);

    /**
     * Creates the markedPoints automatically.
     * The function will get the surface with the bigger number of
     * vertices, and create the table with that number. So, the
     * surf_array table MUST BE created and it must not change after
     * this. If a surface with more that vertices than markedPoints size,
     * the engine will crash during execution.
     */
    void MakeMarkedPointsTable(void);

    /**
     * Adds a surface to the surf_array array.
     * @param surf the pointer to the surface to add
     */
    void AddToList(Surface *surf);

    /**
     * Makes the optimization of sky rendering.
     * This function is used for sky rendering optimization. We are
     * going to eliminate invisible sky quads. To do this, we need to
     * check if the quad is visible and the to mark visible quads
     * with a certain value. We generally give the frame number for
     * this value.
     * @param camera a pointer to the current camera
     * @param viscount the current visibility counter
     */
    void AutoFill(Camera *camera, int viscount);

    /**
     * Fills a sky side.
     * This function is for debug purpose only.
     */
    void FillSide(int side);

    /**
     * Specify we are beginning a new frame. The function will initialize
     * the framenum array with a given number.
     * @param v the initialization value for framenum array
     * @see framenum
     */
    void NewFrame(int v);

  private:
    float slope;    /**< constant slope used for transformation function */
    float h;      /**< constant value used for transformation function */

    int cloudsheight; /**< clouds height */

    /**
     * Number of subdivisions of the skybox.
     * The 6 faces of the sky cube are divided into small elements. These
     * elements are in fact quads (two collapsed triangles) with sky shader.
     * The faces are divided because of visible distorsions in the sky effect.
     * The SIDE_SIZE value must be odd.
     * @see surface
     */
    int side_size;

    int points_len;
    int elem_len;

    /**
     * Generates the order of elements. These elements are in fact vertices
     * of the faces of the surface of the cube planes.
     * <pre>
     *                          |
     *   /______________________|____
     *   \  u                   |
     *                  2    0  |
     *     -+---+---+--3+---+-  |
     *      |\  |\  |\ 6|\  |   |
     *      | \ | \ | \ | \ |   |
     *      |  \|  \|  \|5 \|1  |
     *     -+---+---+---+--4+-  |
     *      |\  |\  |\  |\  |   |
     *      | \ | \ | \ | \ |   |
     *      |  \|  \|  \|  \|   | v
     *     -+---+---+---+---+-  |
     *                         \|/
     * </pre>
     * The elems are positionned following the schema. There are 6 elems
     * for each quad.
     */
    void GenElems(void);

    /**
     * Generates the box. The function computes the coordinates for each
     * side of the cube.
     */
    void GenBox(void);

    /**
     * Generates the coordinates for each vertex of the surface. The
     * method is mainly taken from Aftershock sky generation algorithm.
     * It consists in tiling texture onto the inside of a large sphere,
     * and putting the camera near the top of the sphere. We then place
     * the box around the camera, and cast rays through the box verts to
     * the sphere to find the texture coordinates.
     * @param side the side the function has to treat
     * @param orig the position of the first vertice of the surface
     * @param drow a vector giving the direction and size of the face row
     * @param dcol a vector giving the direction and size of the face col
     */
    void GenBoxSide(int side, vec3_t orig, vec3_t drow, vec3_t dcol);

    /**
     * Marks the quad that covers the (x,y) point of the plane.
     * The function has to convert cube-relative coordinates to
     * side-relative coordinates and mark corresponding elements.
     * <pre>
     *                ^
     *                |
     *          + - - + - - + SIDE_SIZE-1
     *          |     |    /|
     *          |     |   / |
     *          |     |  /  |
     *          |     | /   |
     *          |     |/    |
     *          |     X     |
     *          |    /|     |
     *          |   / |     |
     *          |  /  |     |
     *          | /   |     |
     *          |/    |0    |
     *  --------+-----+-----+-------->
     *     -CUBE_SIZE | CUBE_SIZE
     *                |
     * </pre>
     * The function uses a constant precalculated slope and transformation
     * defined above.
     * @param x the x coordinate (cube-relative)
     * @param y the y coordinate (cube-relative)
     * @param plane the number of concerned plane
     */
    void MarkElem(float x, float y, int plane);

    /**
     * Adds an element in the minmax table.
     * The element is compared to existing elements and only added if it
     * bigger than maxs or smaller than mins.
     * @param x the x coordinate (cube-relative)
     * @param y the y coordinate (cube-relative)
     * @param p the number of concerned plane
     */
    void AddMinMax(float x, float y, int p);

    /**
     * 
     */
    void DetermineMinMax(vec3_t *verts, int num_verts);

    /**
     * Clips the "skypolygons" in a way that they don't span two or more skysides.
     * This is adapted from Diesel source (which takes it from Q2 source ;)
     * @param verts the sky polygons vertices.
     * @param num_verts the size of verts array
     * @param planenumber the clipping plane number
     */
    void ClipSkyPolygon(vec3_t *verts, int num_verts, int planenumber);
};

#endif  /* __SKYBOX_H__ */
