//-----------------------------------------------------------------------------
// Entities
//-----------------------------------------------------------------------------

#ifndef __ENTITIES_H__
#define __ENTITIES_H__

#include "client.h"
#include "world.h"

typedef enum 
{
  ET_UNKNOWN,
  ET_WORLDSPAWN,
  ET_INFO_PLAYER_DEATHMATCH,

  ET_TARGET_SPEAKER,
  ET_TARGET_PRINT,
  ET_TARGET_LOCATION,
//  ET_TARGET_POSITION,
//  ET_TARGET_DELAY,
//  ET_TARGET_GIVE,
//  ET_TARGET_KILL,
//  ET_TARGET_LASER,
//  ET_TARGET_PUSH,
//  ET_TARGET_RELAY,
//  ET_TARGET_REMOVE_POWERUPS,
//  ET_TARGET_SCORE,
  ET_TARGET_TELEPORTER,

//  ET_TRIGGER_ALWAYS,
//  ET_TRIGGER_HURT,
  ET_TRIGGER_MULTIPLE,
  ET_TRIGGER_PUSH,
  ET_TRIGGER_TELEPORT,

  ET_FUNC_TIMER,
  ET_FUNC_BOBBING,
  ET_FUNC_PENDULUM,
  ET_FUNC_ROTATING,
  ET_FUNC_TRAIN,
  ET_PATH_CORNER,
//  ET_FUNC_BUTTON,
  ET_FUNC_DOOR,
//  ET_FUNC_GROUP,
//  ET_FUNC_PLAT,
//  ET_FUNC_STATIC,

  ET_MISC_TELEPORTER_DEST,

  ET_ITEM
} ENTTYPE;

// NOTE: may not have more than 16
typedef enum
{
  PW_NONE,

  PW_QUAD,
  PW_BATTLESUIT,
  PW_HASTE,
  PW_INVIS,
  PW_REGEN,
  PW_FLIGHT,

  PW_REDFLAG,
  PW_BLUEFLAG,
  PW_NEUTRALFLAG,

  PW_SCOUT,
  PW_GUARD,
  PW_DOUBLER,
  PW_AMMOREGEN,
  PW_INVULNERABILITY,

  PW_NUM_POWERUPS

} powerup_t;

typedef enum
{
  HI_NONE,

  HI_TELEPORTER,
  HI_MEDKIT,
  HI_KAMIKAZE,
  HI_PORTAL,
  HI_INVULNERABILITY,

  HI_NUM_HOLDABLE
} holdable_t;

typedef enum
{
  WP_NONE,

  WP_GAUNTLET,
  WP_MACHINEGUN,
  WP_SHOTGUN,
  WP_GRENADE_LAUNCHER,
  WP_ROCKET_LAUNCHER,
  WP_LIGHTNING,
  WP_RAILGUN,
  WP_PLASMAGUN,
  WP_BFG,
  WP_GRAPPLING_HOOK,
  WP_NAILGUN,
  WP_PROX_LAUNCHER,
  WP_CHAINGUN,

  WP_NUM_WEAPONS
} weapon_t;

// gitem_t->type
typedef enum
{
  IT_UNKNOWN,
  IT_BAD,
  IT_WEAPON,        // EFX: rotate + upscale + minlight
  IT_AMMO,        // EFX: rotate
  IT_ARMOR,       // EFX: rotate + minlight
  IT_HEALTH,        // EFX: static external sphere + rotating internal
  IT_POWERUP,       // instant on, timer based
              // EFX: rotate + external ring that rotates
  IT_HOLDABLE,      // single use, holdable item
              // EFX: rotate + bob
  IT_PERSISTANT_POWERUP,
  IT_TEAM
} ITEMTYPE;

#define MAX_ITEM_MODELS 4

// Constants for door spawnflag
#define DOOR_START_OPEN   1
#define DOOR_REVERSE    2
#define DOOR_CRUSHER    4
#define DOOR_NOMONSTER    8
#define DOOR_TOGGLE     32
#define DOOR_X_AXIS     64
#define DOOR_Y_AXIS     128

typedef struct 
{
  char    *classname;   // spawning name
  char    *pickup_sound;
  char    *world_model[MAX_ITEM_MODELS];

  char    *icon;
  char    *pickup_name; // for printing on pickup

  int     quantity;   // for ammo how much, or duration of powerup
  int     giType;     // IT_* flags

  int     giTag;

  char    *precaches;   // string of all models and images this item will use
  char    *sounds;    // string of all sounds this item will use} gitem_t;
} gitem_t;

class Entity
{
  Client *      curr_client;      // Current client for entity interraction

  public:
    Entity();
    ~Entity();

    Q3BSP*      pBSP;

    int       enttype;
    int       itemtype;
    int       spawnflags;

    char*     dst_target;       // name of destination target
    char*     my_target;        // name of current entity as target

    char*     team;         // name of entity team

    char*     message;
    char*     noise;          // sound filename
    int       noise_index;      // index of sound channel
    int       ambient;
    float     light;
    vec3_t      _color;
    vec3_t      origin;

    int       model;
    cmodel_t    **models;       // pointer on associated model
    int       num_lods;       // number of levels of detail
    int       numchildren;
    Entity      **children;       // pointer on child entities
    Entity      *parent;        // pointer on parent entity
    Entity      **targets;        // pointer on target entities
    int       numtargets;       // number of targets

    int       health;
    int       dmg;
    float     height;
    float     phase;
    float     speed;
    float     accel;
    float     decel;
    float     angle;
    float     wait;
    float     random;
    int       lip;

    float     nexttime;

    vec_t     transformation[16];   // transformation matrix
    bboxf_t     bbox;

    void LoadSound();           // load the entity sounds

    void UpdateEntity(Client *client);

    void AddChild(Entity *child_ent);
    void RemoveChild(Entity *child_ent);

    void AddTarget(Entity *target_ent);
    void RemoveTarget(Entity *target_ent);

    void SetModel(cmodel_t *model = NULL, int lod = 0);
    cmodel_t *GetModel(int lod = 0);

    vec3_t      translation;      // translation vector
    vec3_t      rotation;       // rotation vector
    vec3_t      scaling;        // scaling vector
    void UpdateTransformMatrix(void);   // update the transformation matrix

    void UpdateItem(void);
    void UpdateFuncTimer(void);
    void UpdateFuncBobbing(void);
    void UpdateFuncPendulum(void);
    void UpdateFuncRotating(void);
    void UpdateFuncTrain(void);
    void UpdateFuncDoor(void);
    
    void UpdateTriggerPush(void);
    void UpdateTriggerTeleport(void);
    void UpdateTriggerMultiple(void);

    void UpdateTarget(void);
    void UpdateTargetSpeaker(void);
    void UpdateTargetPrint(void);
    void UpdateTargetLocation(void);
    void UpdateTargetTeleporter(void);
};

class EntityManager
{
    void ProcessWorldspawn(int i);
    void ProcessInfoPlayerDeathmatch(int i);
    
    void ProcessTargetSpeaker(int i);
    void ProcessTargetLocation(int i);
    void ProcessTargetPrint(int i);
    void ProcessTargetTeleporter(int i);

    void ProcessTriggerPush(int i);
    void ProcessTriggerTeleport(int i);
    void ProcessTriggerMultiple(int i);

    void ProcessMiscTeleporterDest(int i);

    void ProcessFuncTimer(int i);
    void ProcessFuncBobbing(int i);
    void ProcessFuncPendulum(int i);
    void ProcessFuncRotating(int i);
    void ProcessFuncTrain(int i);
    void ProcessPathCorner(int i);
    void ProcessFuncDoor(int i);

    void ProcessItem(int i);

    static gitem_t bg_itemlist[];

  public:
    Entity * entities;
    int num_entities;

    EntityManager(void);
    ~EntityManager(void);

    int   Load(char *ent, int len);
    
    void  Init(void);

    int   ProcessEntities(Q3BSP *pBSP);

    /**
     * Gets the value corresponding to a key in an entity.
     * @param entity The entity number.
     * @param key The key of entity value.
     * @return A pointer to the entity value string.
     */
    const char* Entity_value(int entity, const char *key);
    char* Entity_value_ex(int entity, const char *key);

    /**
     * Gets the value corresponding to a key in an entity.
     * @param entity The entity number.
     * @param key The key of entity value as a float value.
     * @return The requested float value.
     */
    float   Entity_float(int entity, const char *key);

    /**
     * Gets the value corresponding to a key in an entity.
     * @param entity The entity number.
     * @param key The key of entity value as a int value.
     * @return The requested int value.
     */
    int     Entity_int(int entity, const char *key);

    /**
     * Gets the value corresponding to a key in an entity.
     * @param entity The entity number.
     * @param key The key of entity value
     * @param vec The result value as a vector value.
     */
    void    Entity_vec3(int entity, const char *key, vec3_t vec);

    /**
     * Gets the index of an entity having specified type.
     * @param type The searched type in entities list.
     * @param number The number of entities to skip.
     * @return The index of entity having specified classname.
     */
    int     Find_entity(ENTTYPE type, int number = -1);

    /**
     * Write an entity report.
     * This is used for debug.
     */
    void    Report(void);

    void    UpdateEntities(Client *client);
};

#endif  /* __ENTITIES_H__ */
