//-----------------------------------------------------------------------------
// Demo Header
//-----------------------------------------------------------------------------

#include "demo.h"
#include "console.h"
#include "vars.h"
#include "timer.h"
#include "math.h"
#include "mem.h"

#define DEMO_VERSION        "0.1.1"
#define FRAMES_ALLOCATION_BLOC    64
#define SPLINE_LINEAR_LIMIT     0.1f

Demo::Demo()
{
  numframes = 0;
  numframesAllocated = 0;
  frames = NULL;
  cam_positions = NULL;
  cam_rotations = NULL;
  cam_positions_tangents = NULL;
  cam_rotations_tangents = NULL;
  timings = NULL;
  isLooping = false;

  noclipstate = false;

  currclient = NULL;

  isRecording = false;
  isPlaying = false;
}

Demo::~Demo()
{
  if (frames) cake_free(frames);
  if (cam_positions) cake_free(cam_positions);
  if (cam_rotations) cake_free(cam_rotations);
  if (cam_positions_tangents) cake_free(cam_positions_tangents);
  if (cam_rotations_tangents) cake_free(cam_rotations_tangents);
  if (timings) cake_free(timings);
  numframes = 0;
  numframesAllocated = 0;
  frames = NULL;
  cam_positions = NULL;
  cam_rotations = NULL;
  cam_positions_tangents = NULL;
  cam_rotations_tangents = NULL;
  timings = NULL;
  currclient = NULL;
}

int Demo::StartDemoRecording(Client *client, const char *mapname, float framesinterval)
{
  if (!client || isRecording || isPlaying) return 0;

  currclient = client;
  frame_interval = max(framesinterval, 0);
  last_frame_time = start_demo_time = (float) Timer::fTime;
  memset(currmapname, '\0', 32);
  strcpy(currmapname, mapname);

  // Allocate memory for frames
  frames = (DemoFrame*) cake_malloc(FRAMES_ALLOCATION_BLOC*sizeof(DemoFrame), "Demo::StartDemoRecording.frames");
  if (!frames) { gConsole->Insertln("^1Demo::StartDemoRecording: Cannot allocate required size"); return 0; }
  
  // generate first frame
  frames[0].timing = 0;
  VectorCopy(currclient->cam.pos, frames[0].campos);
  VectorCopy(currclient->cam.rot, frames[0].camrot);
  numframes = 1;
  numframesAllocated = FRAMES_ALLOCATION_BLOC;

  isRecording = true;

  return 1;
}

int Demo::StopDemoRecording(const char* destfile)
{
  if (isPlaying || !isRecording) return 0;

  // Generate last frame
  AddFrame();

  FILE *fp = fopen(destfile, "wb");
  if (!fp)
  {
    gConsole->Insertln("Unable to create demo file \"%s\".", destfile);
    return 0;
  }

  // Write header
  DemoHeader head;
  head.num_frames = numframes;
  strcpy(head.version, DEMO_VERSION);
  memset(head.map, '\0', 32);
  strcpy(head.map, currmapname);
  head.capture_interval = frame_interval;
  fwrite(&head, sizeof(DemoHeader), 1, fp);

  // Write frames lump
  fwrite(frames, sizeof(DemoFrame), numframes, fp);

  fclose(fp);

  // Unallocate memory
  if (frames) cake_free(frames);
  frames = NULL;
  numframes = 0;
  numframesAllocated = 0;
  currclient = NULL;

  isRecording = false;

  return 1;
}

void Demo::AddFrame(void)
{
  if (numframes == numframesAllocated)
  {
    // Reallocate memory for frames
    frames = (DemoFrame*) cake_realloc(frames, (numframesAllocated+FRAMES_ALLOCATION_BLOC)*sizeof(DemoFrame), "Demo::AddFrame");
    if (!frames) { gConsole->Insertln("^1Demo::AddFrame: Cannot allocate required size"); return; }
    numframesAllocated += FRAMES_ALLOCATION_BLOC;
  }

  frames[numframes].timing = (float) Timer::fTime - start_demo_time;
  VectorCopy(currclient->cam.pos, frames[numframes].campos);
  VectorCopy(currclient->cam.rot, frames[numframes].camrot);
  ++numframes;
}

char* Demo::LoadDemo(const char* demofile, enum_InterpolationMode mode)
{
  if (isRecording || isPlaying || currclient) return "";

  FILE *fp = fopen(demofile, "rb");
  if (!fp)
  {
    gConsole->Insertln("^1Unable to open demo file \"%s\".", demofile);
    return "";
  }

  DemoHeader head;
  fread(&head, sizeof(DemoHeader), 1, fp);
  if (strcmp(head.version, DEMO_VERSION))
  {
    gConsole->Insertln("^1Unsupported demo version (file version: %s - supported: %s).", head.version, DEMO_VERSION);
    fclose(fp);
    return "";
  }

  // Read frames from file
  numframes = numframesAllocated = head.num_frames;
  frames = (DemoFrame*) cake_malloc(numframes*sizeof(DemoFrame), "Demo::LoadDemo.frames");
  fread(frames, sizeof(DemoFrame), numframes, fp);
  fclose(fp);

  // Allocate memory for frames
  cam_positions = (vec3_t*) cake_malloc(numframes*sizeof(vec3_t), "Demo::LoadDemo.cam_positions");
  cam_rotations = (vec3_t*) cake_malloc(numframes*sizeof(vec3_t), "Demo::LoadDemo.cam_rotations");
  timings = (float*) cake_malloc(numframes*sizeof(float), "Demo::LoadDemo.timings");
  if (!cam_positions || !cam_rotations || !timings)
  {
    gConsole->Insertln("^1Demo::LoadDemo: Cannot allocate required size");
    return "";
  }
  for (int i = 0; i < numframes; ++i)
  {
    VectorCopy(frames[i].campos, cam_positions[i]);
    VectorCopy(frames[i].camrot, cam_rotations[i]);
    timings[i] = frames[i].timing;
  }

  // Free unused memory
  free(frames);
  frames = NULL;

  // Copy map name
  memset(currmapname, '\0', 32);
  strcpy(currmapname, head.map);

  interpolationmode = mode;
  if (interpolationmode == AUTO_INTERPOLATION)
  {
    if (head.capture_interval < SPLINE_LINEAR_LIMIT)
      interpolationmode = LINEAR_INTERPOLATION;
    else
      interpolationmode = SPLINE_INTERPOLATION;
  }

  if (interpolationmode == SPLINE_INTERPOLATION)
  {
    // Computes the tangents at keyframes
    cam_positions_tangents = (vec3_t*) cake_malloc(numframes*sizeof(vec3_t), "Demo::LoadDemo.cam_rotations_tangents");
    cam_rotations_tangents = (vec3_t*) cake_malloc(numframes*sizeof(vec3_t), "Demo::LoadDemo.cam_rotations_tangents");
    if (!cam_positions_tangents || !cam_rotations_tangents)
    {
      gConsole->Insertln("^1Demo::LoadDemo_spline: Cannot allocate required size");
      return "";
    }
    VectorClear(cam_positions_tangents[0]);
    VectorClear(cam_rotations_tangents[0]);
    VectorClear(cam_positions_tangents[numframes-1]);
    VectorClear(cam_rotations_tangents[numframes-1]);
    vec3_t a, b;
    int i, j, k;
    for (j = 1; j < numframes-1; ++j)
    {
      i = max(j-1, 0);
      k = min(j+1, numframes-1);
      VectorSub(cam_positions[j], cam_positions[i], a);
      VectorSub(cam_positions[k], cam_positions[j], b);
      VectorScale(a, 0.5f, a);
      VectorScale(b, 0.5f, b);
      VectorAdd(a, b, cam_positions_tangents[j]);

      VectorSub(cam_rotations[j], cam_rotations[i], a);
      VectorSub(cam_rotations[k], cam_rotations[j], b);
      VectorScale(a, 0.5f, a);
      VectorScale(b, 0.5f, b);
      VectorAdd(a, b, cam_rotations_tangents[j]);
    }
  }

  return currmapname;
}

void Demo::StartDemoPlaying(Client *client, bool loop)
{
  if (!client || isPlaying || isRecording) return;
  Timer::Refresh();
  start_demo_time = (float) Timer::fTime;
  currframenum = 1;
  isPlaying = true;
  isLooping = loop;

  // set client flying and noclipping
  currclient = client;
  currclient->flying = true;
  noclipstate = client->noclip;
  client->noclip = true;
}

void Demo::StopDemoPlaying(void)
{
  if (!currclient || !isPlaying || isRecording) return;

  // restore noclip value
  currclient->noclip = noclipstate;

  // Unallocate memory
  if (cam_positions) cake_free(cam_positions); cam_positions = NULL;
  if (cam_rotations) cake_free(cam_rotations); cam_rotations = NULL;
  if (cam_positions_tangents) cake_free(cam_positions_tangents); cam_positions_tangents = NULL;
  if (cam_rotations_tangents) cake_free(cam_rotations_tangents); cam_rotations_tangents = NULL;

  if (timings) cake_free(timings); timings = NULL;
  numframes = numframesAllocated = 0;
  currclient = NULL;
  isPlaying = false;
}

float Demo::NewFrame(void)
{
  if (!currclient) return -1;

  if (isRecording)
  {
    if (Timer::fTime - last_frame_time < frame_interval) return -1;

    AddFrame();

    last_frame_time = (float) Timer::fTime;
  }
  else if (isPlaying)
  {
    while (timings[currframenum] < (Timer::fTime - start_demo_time))
    {
      if (++currframenum >= numframes)
      { 
        // demo is over
        if (isLooping)
        {
          start_demo_time = (float) Timer::fTime;
          currframenum = 1;
        }
        else
        {
          StopDemoPlaying();
          gConsole->Insertln("Demo ended.");
          return -1;
        }
      }
    }

    float timeframe_a, timeframe_b;
    timeframe_b = timings[currframenum];
    timeframe_a = timings[currframenum-1];

    float currtime = (float) Timer::fTime - start_demo_time;    // elapsed time since demo has started
    float u = (currtime - timeframe_a)/(timeframe_b - timeframe_a); // current progress in the frame (0 -> 1)

    switch (interpolationmode)
    {
      case LINEAR_INTERPOLATION:
      {
        // Set client interpolated position and rotation
        vec3_t interpol_vect;
        VectorSub(cam_positions[currframenum], cam_positions[currframenum-1], interpol_vect);
        VectorMA(cam_positions[currframenum-1], u, interpol_vect, currclient->cam.pos);

        VectorSub(cam_rotations[currframenum], cam_rotations[currframenum-1], interpol_vect);
        VectorMA(cam_rotations[currframenum-1], u, interpol_vect, currclient->cam.rot);
      }
      break;
      case SPLINE_INTERPOLATION:
      {
        // Based on gamedev article "Smooth interpolation of irregularly spaced keyframes"
        vec3_t interpol_vect;
        float a = 2*u*u*u-3*u*u+1;
        float b = 3*u*u-2*u*u*u;
        float c = u*u*u-2*u*u+u;
        float d = u*u*u-u*u;

        // need tangents at keyframes
        VectorScale(cam_positions[currframenum-1], a, interpol_vect);
        VectorMA(interpol_vect, b, cam_positions[currframenum], interpol_vect);
        VectorMA(interpol_vect, c, cam_positions_tangents[currframenum-1], interpol_vect);
        VectorMA(interpol_vect, d, cam_positions_tangents[currframenum], currclient->cam.pos);

        VectorScale(cam_rotations[currframenum-1], a, interpol_vect);
        VectorMA(interpol_vect, b, cam_rotations[currframenum], interpol_vect);
        VectorMA(interpol_vect, c, cam_rotations_tangents[currframenum-1], interpol_vect);
        VectorMA(interpol_vect, d, cam_rotations_tangents[currframenum], currclient->cam.rot);
      }
      break;
      default:
        break;
    }
  }

  return 100.f*(float)currframenum/(float)numframes;
}

void Demo::DemoDump(char *demofile, int n)
{
  FILE *fp = fopen(demofile, "rb");
  if (!fp)
  {
    gConsole->Insertln("^1Unable to open demo file \"%s\".", demofile);
    return;
  }

  DemoHeader head;
  fread(&head, sizeof(DemoHeader), 1, fp);

  DemoFrame *tmp_frames = (DemoFrame*) cake_malloc(head.num_frames*sizeof(DemoFrame), "Demo::DemoDump.tmp_frames");
  fread(tmp_frames, sizeof(DemoFrame), head.num_frames, fp);
  fclose(fp);

  gConsole->Insertln("Demo file \"%s\":", demofile);
  gConsole->Insertln("\tHeader:");
  gConsole->Insertln("\t\tVersion: %s", head.version);
  gConsole->Insertln("\t\tUsed map: %s", head.map);
  gConsole->Insertln("\t\tNumber of frames: %d", head.num_frames);
  gConsole->Insertln("\t\tRecord interval: %f", head.capture_interval);
  if (n > 0) gConsole->Insertln("\tContent:");
  for (int i = 0; i < head.num_frames; ++i)
  {
    if (i == n) break;
    gConsole->Insertln("\t\tFrame %d:", i);
    gConsole->Insertln("\t\t\tTiming: %f", tmp_frames[i].timing);
    gConsole->Insertln("\t\t\tCamera position: %f %f %f", tmp_frames[i].campos[0], tmp_frames[i].campos[1], tmp_frames[i].campos[2]);
    gConsole->Insertln("\t\t\tCamera rotation: %f %f %f", tmp_frames[i].camrot[0], tmp_frames[i].camrot[1], tmp_frames[i].camrot[2]);
  }

  cake_free(tmp_frames);

  return;
}
