//-----------------------------------------------------------------------------
// Client
//-----------------------------------------------------------------------------

#include "client.h"
#include "cake.h"
#include "commands.h"
#include "console.h"
#include "camera.h"
#include "timer.h"
#include "vars.h"
#include "math.h"
#include "world.h"
#include "sound.h"

void InitPlayerSounds(PlayerSounds *s);
void FreePlayerSounds(PlayerSounds *s);
int LoadPlayerSound(const char *path, const char *soundname);

Client::Client(void)
{
  life = 100;
  armor = 100;

  flying = true;            // default value (currently true for debug purpose)
  falling = false;          // initialisation (updated in world.CheckCollision())
  swimming = false;

  active_client = true;
  isCurrent = false;

  fov = 90;
  zoomlimit = 20;
  max_speed = 300;
  acceleration = 4096;
  friction = 1024;
  jump_height = 286;

  pitchLimit = true;          // pitch limit enabled by default

  noclip = false;

  VectorClear(velocity);        // initialization
  VectorSet(mins, -15, -15, -48);
  VectorSet(maxs,  15,  15,  8);
  step_size = 18;           // default value
  viewheight = mins[2];

  VectorSet(extSpeed, 0, 0, 0);   // initialization

  nzoom = pzoom = 1;

  progress = 1;           // initialization - all interpolations are complete

  color[0] = rand()%255;
  color[1] = rand()%255;
  color[2] = rand()%255;
  color[3] = 64;

  InitPlayerSounds(&sounds);
  sprintf(name, "");
}

Client::~Client(void)
{
  FreePlayerSounds(&sounds);
}

void Client::Init(int left, int top, int width, int height)
{
  cam.viewport_width = (float) width;
  cam.viewport_height = (float) height;
  cam.viewport_top = (float) top;
  cam.viewport_left = (float) left;

  if (cam.viewport_width < 0) cam.viewport_width = (float) gVars->IntForKey("v_width");
  if (cam.viewport_height < 0) cam.viewport_height = (float) gVars->IntForKey("v_height");

  if (cam.viewport_width == 0 || cam.viewport_height == 0)  // there may be an error, use default res
  {
    gConsole->Insertln("^5WARNING: --- uh, uh, something is going wrong");

    cam.viewport_width = 640;
    cam.viewport_height = 480;
  }
  
  cmd.forwardmove = 0;
  cmd.sidemove = 0;
  cmd.upmove = 0;

  noclip = false;

  UpdateCam();
}

void Client::LoadPlayer(const char *playername)
{
  if (!playername)
  {
    gConsole->Insertln("^5WARNING: LoadPlayer received empty player name. No player loaded.");
    return;
  }

  // free current sounds
  FreePlayerSounds(&sounds);

  memset(name, '\0', MAX_PLAYERNAME_LENGTH*sizeof(char));
  strcpy(name, playername);
  
  // get gender (FIXME: get gender from player)
  gender = rand()%3+1;

  // load specific sounds
  char soundpath[256] = { '\0' };
  sprintf(soundpath, "sound/player/%s/", playername);

  sounds.death1 = LoadPlayerSound(soundpath, "death1.wav");
  sounds.death2 = LoadPlayerSound(soundpath, "death2.wav");
  sounds.death3 = LoadPlayerSound(soundpath, "death3.wav");
  sounds.drown = LoadPlayerSound(soundpath, "drown.wav");
  sounds.fall1 = LoadPlayerSound(soundpath, "fall1.wav");
  sounds.falling1 = LoadPlayerSound(soundpath, "falling1.wav");
  sounds.gasp = LoadPlayerSound(soundpath, "gasp.wav");
  sounds.jump1 = LoadPlayerSound(soundpath, "jump1.wav");
  sounds.pain25_1 = LoadPlayerSound(soundpath, "pain25_1.wav");
  sounds.pain50_1 = LoadPlayerSound(soundpath, "pain50_1.wav");
  sounds.pain75_1 = LoadPlayerSound(soundpath, "pain75_1.wav");
  sounds.pain100_1 = LoadPlayerSound(soundpath, "pain100_1.wav");
  sounds.taunt = LoadPlayerSound(soundpath, "taunt.wav");

  // load common sounds
  sounds.fry = load3DSound("sound/player/fry.wav");
  sounds.gibimp1 = load3DSound("sound/player/gibimp1.wav");
  sounds.gibimp2 = load3DSound("sound/player/gibimp2.wav");
  sounds.gibimp3 = load3DSound("sound/player/gibimp3.wav");
  sounds.gibsplt1 = load3DSound("sound/player/gibsplt1.wav");
  sounds.gurp1 = load3DSound("sound/player/gurp1.wav");
  sounds.gurp2 = load3DSound("sound/player/gurp2.wav");
  sounds.land1 = load3DSound("sound/player/land1.wav");
  sounds.talk = load3DSound("sound/player/talk.wav");
  sounds.watr_in = load3DSound("sound/player/watr_in.wav");
  sounds.watr_out = load3DSound("sound/player/watr_out.wav");
  sounds.watr_un = load3DSound("sound/player/watr_un.wav");
}

int LoadPlayerSound(const char *path, const char *soundname)
{
  char soundpathname[256] = { '\0' };
  sprintf(soundpathname, "%s%s", path, soundname);
  return load3DSound(soundpathname);
}

void InitPlayerSounds(PlayerSounds *s)
{
  s->death1 = s->death2 = s->death3 =
  s->drown = s->fall1 = s->falling1 =
  s->gasp = s->jump1 = s->taunt =
  s->pain25_1 = s->pain50_1 = s->pain75_1 = s->pain100_1 =

  s->fry = s->gibimp1 = s->gibimp2 = s->gibimp3 = s->gibsplt1 =
  s->gurp1 = s->gurp2 = s->land1 = s->talk = 
  s->watr_in = s->watr_out = s->watr_un = -1;
}

void FreePlayerSounds(PlayerSounds *s)
{
  if (s->death1 >= 0) freeSound(s->death1);
  if (s->death2 >= 0) freeSound(s->death2);
  if (s->death3 >= 0) freeSound(s->death3);
  if (s->drown >= 0) freeSound(s->drown);
  if (s->fall1 >= 0) freeSound(s->fall1);
  if (s->falling1 >= 0) freeSound(s->falling1);
  if (s->gasp >= 0) freeSound(s->gasp);
  if (s->jump1 >= 0) freeSound(s->jump1);
  if (s->pain25_1 >= 0) freeSound(s->pain25_1);
  if (s->pain50_1 >= 0) freeSound(s->pain50_1);
  if (s->pain75_1 >= 0) freeSound(s->pain75_1);
  if (s->pain100_1 >= 0) freeSound(s->pain100_1);
  if (s->taunt >= 0) freeSound(s->taunt);

  if (s->fry >= 0) freeSound(s->fry);
  if (s->gibimp1 >= 0) freeSound(s->gibimp1);
  if (s->gibimp2 >= 0) freeSound(s->gibimp2);
  if (s->gibimp3 >= 0) freeSound(s->gibimp3);
  if (s->gibsplt1 >= 0) freeSound(s->gibsplt1);
  if (s->gurp1 >= 0) freeSound(s->gurp1);
  if (s->gurp2 >= 0) freeSound(s->gurp2);
  if (s->land1 >= 0) freeSound(s->land1);
  if (s->talk >= 0) freeSound(s->talk);
  if (s->watr_in >= 0) freeSound(s->watr_in);
  if (s->watr_out >= 0) freeSound(s->watr_out);
  if (s->watr_un >= 0) freeSound(s->watr_un);
  InitPlayerSounds(s);
}

void Client::ResetMovement(void)
{
  falling = false;          // initialisation (updated in world.CheckCollision())
  swimming = false;

  // Interpolation stuff
  nzoom = pzoom = 1;
  progress = 1;           // initialization - all interpolations are complete

  VectorClear(extSpeed);
  VectorClear(velocity);

  cmd.forwardmove = 0;
  cmd.sidemove = 0;
  cmd.upmove = 0;
  cmd.move_mouse_x = 0;
  cmd.move_mouse_y = 0;
}

void Client::UpdateCam(void)
{
  float real_width = (float) cam.viewport_width*cam.viewport_size;
  float real_height = (float) cam.viewport_height*cam.viewport_size;

  cam.Init(fov, real_width, real_height);
}

/**
 * @todo Consider water -> other friction and other acceleration
 * @todo Consider jumps -> cannot change direction while jumping
 */
#define FALLING_FRICTION_REDUCTION    8.f   // reduction of friction if falling
#define FALLING_ACCELERATION_REDUCTION  4.f   // reduction of acceleration if falling
#define SWIMMING_FRICTION_REDUCTION   1.4f  // reduction of friction if swimming
#define SWIMMING_ACCELERATION_REDUCTION 1.2f  // reduction of acceleration if swimming
void Client::Update(Q3BSP *bsp, World *wld)
{
  vec3_t temp, dir = { 0.0f, 0.0f, 0.0f }, versor = { 0.0f, 0.0f, 0.0f };

  curr_bsp = bsp;                     // update current bsp
  curr_world = wld;                   // update current world

  swimming = cam.inWater;

  if (flying || swimming || noclip) versor[1] += cmd.upmove;
  versor[0] += cmd.sidemove;
  versor[2] -= cmd.forwardmove;


  // Update rotation
  cam.rot[PITCH] += cmd.move_mouse_y;
  cam.rot[YAW] += cmd.move_mouse_x;


  // Get the direction vector using camera rotation
  if (flying || swimming || noclip)
  {
    cam.UpdateRotation();           // we need the rotation matrix
    MultVect3x3(cam.rotation, versor, dir);
    VectorCopy(cam.forward, forward);
    VectorCopy(cam.right, right);
    VectorCopy(cam.up, up);
  }
  else
  {
    float rad_yaw = DEG2RAD(cam.rot[YAW]);
    dir[0] = versor[0]*FastCos(rad_yaw) + versor[2]*FastSin(rad_yaw);
    dir[1] = versor[0]*FastSin(rad_yaw) - versor[2]*FastCos(rad_yaw);

    forward[0] = cam.forward[0];
    forward[1] = cam.forward[1];
    forward[2] = 0;

    right[0] = cam.right[0];
    right[1] = cam.right[1];
    right[2] = 0;

    VectorSet(up, 0, 0, 1);

    VectorNormalize(forward);
    VectorNormalize(right);
  }

  // Acceleration
  // TODO: Manage acceleration and friction for water and air movements
  if (VectorLength(versor))
  {
    float r_acceleration = (float)Timer::frametime * acceleration;
    if (swimming) r_acceleration /= SWIMMING_ACCELERATION_REDUCTION;
    else if (falling) r_acceleration /= FALLING_ACCELERATION_REDUCTION;

    // Apply acceleration
    velocity[0] += dir[0]*r_acceleration;
    velocity[1] += dir[1]*r_acceleration;
    velocity[2] += dir[2]*r_acceleration;
  }

  VectorCopy(velocity, temp);
  float vel = VectorNormalize(temp);

  // Friction
  float r_friction = (float)Timer::frametime * friction;
  if (swimming) r_friction /= SWIMMING_FRICTION_REDUCTION;
  else if (falling) r_friction /= FALLING_FRICTION_REDUCTION;

  velocity[0] -= temp[0]*r_friction;
  velocity[1] -= temp[1]*r_friction;
  velocity[2] -= temp[2]*r_friction;

  if ((temp[0] > 0 && velocity[0] < 0) || (temp[0] < 0 && velocity[0] > 0)) velocity[0] = 0;
  if ((temp[1] > 0 && velocity[1] < 0) || (temp[1] < 0 && velocity[1] > 0)) velocity[1] = 0;
  if ((temp[2] > 0 && velocity[2] < 0) || (temp[2] < 0 && velocity[2] > 0)) velocity[2] = 0;

  
  // Speed
  if (vel > max_speed)
  {
    float ivel = 1/vel*max_speed;
    VectorScale(velocity, ivel, velocity);
  }

  // dont go out of bounds
  if (pitchLimit)
  {
    if (cam.rot[PITCH] < 0.0f) cam.rot[PITCH] = 0.0f;
    else if (cam.rot[PITCH] > 180.0f) cam.rot[PITCH] = 180.0f;
  }

  // Reinit movement for next frame
  cmd.forwardmove = 0;
  cmd.sidemove = 0;
  cmd.upmove = 0;
  cmd.move_mouse_x = 0;
  cmd.move_mouse_y = 0;


  // Position update
  if (noclip)
  {
    VectorMA(cam.pos, (float) Timer::frametime, velocity, cam.pos);
  }
  else
  {
    vec3_t pos;
    VectorMA(cam.pos, (float) Timer::frametime, velocity, pos);
    CheckCollision(pos);
  }

  // subwater time update
  // FIXME: should it be done here ?
  if (life > 0 &&
    swimming &&
    diveStart + 10 < Timer::fTime &&
    Timer::fTime - lastgurptime > 1)
  {
    lastgurptime = Timer::fTime;
    Pain((int) (Timer::fTime-diveStart));
    Gurp(gender);

    if (life <= 0)
    {
      // restore life
      life = 100;
      diveStart = Timer::fTime;
    }
  }
}

void Client::SetAngle(float p, float y, float r)
{
  VectorSet(cam.rot, p, y, r);
}

void Client::SetAngle(vec3_t rot)
{
  VectorCopy(rot, cam.rot);
}

void Client::SetPos(float x, float y, float z)
{
  VectorSet(cam.pos, x, y, z-viewheight);
}

void Client::SetPos(vec3_t pos)
{
  VectorCopy(pos, cam.pos);
  cam.pos[2] -= viewheight;
}

void Client::AddAngle(float p, float y, float r)
{
  cam.rot[PITCH] += p;
  cam.rot[YAW] += y;
  cam.rot[ROLL] += r;
}

void Client::AddPos(float x, float y, float z)
{
  cam.pos[0] += x;
  cam.pos[1] += y;
  cam.pos[2] += z;
}

void Client::MoveForward(float val)
{
  cmd.forwardmove += val;
}

void Client::MoveBackward(float val)
{
  cmd.forwardmove -= val;
}

void Client::MoveRight(float val)
{
  cmd.sidemove += val;
}

void Client::MoveLeft(float val)
{
  cmd.sidemove -= val;
}

void Client::MoveUp(float val)
{
  cmd.upmove += val;
}

void Client::MoveDown(float val)
{
  cmd.upmove -= val;
}

void Client::MoveMouseX(float val)
{
  // the rotation is proportionnal to camera fov
  cmd.move_mouse_x = val*cam.fov/M_TWO_PI;
}

void Client::MoveMouseY(float val)
{
  // the rotation is proportionnal to camera fov
  cmd.move_mouse_y = val*cam.fov/M_TWO_PI;
}

void Client::MoveMouseXY(float xval, float yval)
{
  // the rotation is proportionnal to camera fov
  cmd.move_mouse_x = xval*cam.fov/M_TWO_PI;
  cmd.move_mouse_y = yval*cam.fov/M_TWO_PI;
}

void Client::Jump(void)
{
  // No jump when already jumping (or falling)
  if (!flying && !falling && !noclip)
  {
    extSpeed[2] = jump_height;
    if (!flying) falling = true;

    // play jump sound
    playSound(sounds.jump1, cam.pos);
  }
}

void Client::Land(void)
{
  playSound(sounds.land1, cam.pos);
}

void Client::Fall(void)
{
  playSound(sounds.fall1, cam.pos);
}

void Client::Falling(void)
{
  playSound(sounds.falling1, cam.pos);
}

void Client::Die(int depthnum)
{
  switch (depthnum)
  {
    case 1: 
      playSound(sounds.death1, cam.pos);
      break;
    case 2:
      playSound(sounds.death2, cam.pos);
      break;
    case 3:
    default:
      playSound(sounds.death3, cam.pos);
      break;
  }
  life = 0;
}

void Client::Gasp(void)
{
  playSound(sounds.gasp, cam.pos);
}

void Client::Drown(void)
{
  playSound(sounds.drown, cam.pos);
}

void Client::Pain(int amount)
{
  life -= amount;
  gConsole->Insertln("client life: %d", life);

  if (life <= 0) Die(rand()%3+1);
  else if (life > 0 && life < 25) playSound(sounds.pain25_1, cam.pos);
  else if (life >= 25 && life < 50) playSound(sounds.pain50_1, cam.pos);
  else if (life >= 50 && life < 75) playSound(sounds.pain75_1, cam.pos);
  else playSound(sounds.pain100_1, cam.pos);
}

void Client::Taunt(void)
{
  playSound(sounds.taunt, cam.pos);
}

void Client::Fry(void)
{
  playSound(sounds.fry, cam.pos);
  Pain(25);
}

void Client::Gibimp(int type)
{
  switch (type)
  {
    case 1:
      playSound(sounds.gibimp1, cam.pos);
      break;
    case 2:
      playSound(sounds.gibimp2, cam.pos);
      break;
    case 3:
      playSound(sounds.gibimp3, cam.pos);
      break;
    case 0:
    default:
      playSound(sounds.gibsplt1, cam.pos);
      break;
  }
}

void Client::Gurp(int type)
{
  switch (type)
  {
    case 2:
      playSound(sounds.gurp2, cam.pos);
      break;
    case 1:
    default:
      playSound(sounds.gurp1, cam.pos);
      break;
  }
}

void Client::Talk(void)
{
  playSound(sounds.talk, cam.pos);
}

void Client::Water(int type)
{
  /**< type -> 0 = in, 1 = out, 2 = un */
  switch (type)
  {
    case 1:
      playSound(sounds.watr_out, cam.pos);
      break;
    case 2:
      playSound(sounds.watr_un, cam.pos);
      break;
    case 0:
    default:
      playSound(sounds.watr_in, cam.pos);
      diveStart = Timer::fTime;
      lastgurptime = diveStart;
      break;
  }
}

bool Client::JumpTo(vec3_t dest)
{
  if (!flying && !falling && !noclip)
  {
    VectorClear(velocity);

    /*
    ** The bumper target is the top of the trajectory curve:
    **
    **               /    target
    ** start speed  /   __--X--__
    **             /_-""_-"      ""-_
    **            /" _-'             "_
    **     start X_-'-----------------X end
    **    --------.    <- range ->   .---------
    **    ________|                  |_________
    **
    **
    ** So we'll have to solve the motion equation:
    **
    **  x(t) = x0 + Vx.t
    **  y(t) = y0 + Vy.t
    **  z(t) = z0 + Vz.t - 1/2.g.t^2
    **
    ** when the motion reaches its end point, we're on z = z0:
    ** so the motion range is:
    **
    **  z(t) = z0 + Vz.t_end - 1/2.g.t_end^2
    **  z(t) = z0;
    **
    **  Vz.t_end - 1/2.g.t_end^2 = 0
    **  this equation has two solutions:
    **  t_end0 = 0
    **  t_end1 = 2.Vz / g
    **
    **  we're only interested in the second one, and the maximum
    **  motion height is achieved when the horizontal distance
    **  equals half the range:
    **
    **  t_max = Vz / g
    **
    ** We just have to plug that in the equations:
    **
    **  x_target = x0 + Vx.Vz / g
    **  y_target = y0 + Vy.Vz / g
    **  z_target = z0 + Vz.Vz / g - 1/2.g.(Vz / g)^2
    **
    ** so we finally get:
    **
    **  Vz = sqrt(2.g.(z_target - z0))
    **  Vx = (x_target - x0).g / Vz
    **  Vy = (y_target - y0).g / Vz
    **
    ** and we have our accelerator pad speed vector: {Vx, Vy, Vz}
    **
    */

    extSpeed[2] = sqrtf(2*curr_world->gravity*(dest[2]-cam.pos[2]-viewheight));
    extSpeed[0] = (dest[0] - cam.pos[0])*curr_world->gravity / extSpeed[2];
    extSpeed[1] = (dest[1] - cam.pos[1])*curr_world->gravity / extSpeed[2];
  
    swimming = false;
    falling = true;

    progress = 1;           // initialization - all interpolations are complete

    // play jump sound
    playSound(sounds.jump1, cam.pos);

    return true;
  }
  else return false;
}

bool Client::TeleportTo(vec3_t dest, float angle, float startwalk)
{
  cam.pos[0] = dest[0];
  cam.pos[1] = dest[1];
  cam.pos[2] = dest[2]-viewheight;

  // Reset movement
  falling = false;          // initialisation (updated in world.CheckCollision())
  swimming = false;

  // Interpolation stuff
  progress = 1;           // initialization - all interpolations are complete

  VectorClear(extSpeed);
  VectorClear(velocity);

  cam.rot[PITCH] = 90;
  cam.rot[YAW] = angle-90;
  cam.rot[ROLL] = 0;

  MoveForward(startwalk);       // make the client moving forward

  return true;
}

void Client::Zoom(float val)
{
  if (val < 0 && !nzoom || val > 0 && !pzoom) return;

  float camzoom = cam.GetFov();
  float zoomval = val*(float)Timer::frametime;

  if (zoomval > 0)
  {
    if (camzoom+zoomval < fov)
    {
      cam.SetFov(camzoom+zoomval);
      pzoom = true;       // there is variation for val > 0
    }
    else
    {
      cam.SetFov(fov);
      pzoom = false;        // there is no variation for val > 0
    }
    nzoom = true;
  }
  else if (zoomval < 0)
  {
    if (camzoom+zoomval > zoomlimit)
    {
      cam.SetFov(camzoom+zoomval);
      nzoom = true;       // there is variation for val < 0
    }
    else
    {
      cam.SetFov(zoomlimit);
      nzoom = false;        // there is no variation for val < 0
    }
    pzoom = true;
  }
}

int Client::TouchBounds(bboxf_t bbox)
{
  if (noclip) return false;

  vec3_t cmins, cmaxs;
  VectorAdd(mins, cam.pos, cmins);
  VectorAdd(maxs, cam.pos, cmaxs);

  return BoundsIntersect(bbox, cmins, cmaxs);
}

#define MIN_STEP_NORMAL 0.7f    // can't step up onto very steep slopes
#define DELTA_DOWN 1
#define INTERPOLATION_SPEED 0.125f  // Speed used for interpolations (proportional to height difference)
void Client::CheckCollision(vec3_t targetpos)
{
  trace_t trace;
  vec3_t depl, pos, endPoint = { 0, 0, 0 };
  float time_frame = (float) Timer::frametime * 100;
  bool jumping = (!flying && !swimming && extSpeed[2] > 0);

  VectorCopy(cam.pos, pos);
  VectorSub(targetpos, pos, depl);

  if (depl[0] || depl[1] || depl[2])
  {
    float stepdiff = 0;
    if (progress < 1)
    {
      stepdiff = progress*interpolation;
      depl[2] -= stepdiff;
    }
    StepSlideMove(depl);
    VectorAdd(pos, depl, cam.pos);

    endPoint[2] = cam.pos[2] - DELTA_DOWN;
    trace = curr_bsp->CheckMove(cam.pos, mins, maxs, endPoint);

    // Step climbing movement
    // In step case, plane normal is not perpendicular to displacement
    //   => dot product not null
    if (!flying && depl[2]+stepdiff > 1 && depl[2] < 20 &&
      DotProduct(trace.plane.normal, depl) != 0)
    {
      if (progress < 1)
      {
        // another interpolation is already in progress
        startheight += (progress * interpolation);
        interpolation = depl[2] + pos[2] - startheight + stepdiff;
      }
      else
      {
        // start new interpolation
        startheight = pos[2];
        interpolation = depl[2];
      }
      
      progress = 0; // start interpolation
    }
  }
  else
  {
    // create an empty trace
    trace.contents = 0;
    trace.fraction = 1;
    trace.shader = -1;
  }

  // Check triggered evenments
  //  if (trace.shader >= 0)
  //    gConsole->Insertln("^6trace.shader.content: %d  |  trace.shader.surface: %d",
  //      shaders.GetShader(trace.shader)->content_flags,shaders.GetShader(trace.shader)->surface_flags);
  //  gConsole->Insertln("^5trace.contents: %d", trace.contents );

  // Check if falling, jumping or anyth-ing ;)
  endPoint[2] = cam.pos[2] - DELTA_DOWN;
  trace = curr_bsp->CheckMove(cam.pos, mins, maxs, endPoint);
  falling = (!flying && trace.fraction >= 0.01);

  if (progress < 1)
  {
    // Process interpolation (proportional to frametime)
    progress += (INTERPOLATION_SPEED * time_frame);

    if (progress > 1) progress = 1;

    cam.pos[2] = startheight + progress * interpolation;
  }
  else if (falling)
  {
    // if the client is falling (actually if he is not on
    // the ground, for instance when jumping), only add gravity
    extSpeed[2] -= curr_world->gravity * (float) Timer::frametime;

    progress = 1; // reset interpolation

    // clamp vertical speed
    if (extSpeed[2] < -curr_world->maxvspeed) extSpeed[2] = -curr_world->maxvspeed;
  }
/*  else if (trace.plane.normal[2] < MIN_STEP_NORMAL)
  {
#if 0
    VectorNormalize(trace.normal);

    // Slide with speed relative to ground slope
    vec3_t slide = { 0, 0, -(curr_world->gravity * (float) Timer::frametime) * (1 - trace.normal[2] / MIN_STEP_NORMAL) };

    // Remaining to move
    vec3_t remain;
    VectorMA(slide, -trace.fraction, slide, remain);

    float a = DotProduct(remain, trace.normal);
    VectorScale(trace.normal, a, slide);
    VectorSub(remain, slide, slide);

    VectorAdd(cam.pos, slide, endPoint);
    trace = curr_bsp->checkMove(cam.pos, mins, maxs, endPoint);

    if (trace.fraction >= 0.01) VectorCopy(trace.end, cam.pos);
#else
    extSpeed[2] -= (curr_world->gravity * (float) Timer::frametime) * (1 - trace.plane.normal[2] / MIN_STEP_NORMAL);
#endif
  }*/

  if (!jumping && !falling && progress >= 1)
  { // -> if progress < 1 is treated below
    VectorClear(extSpeed);
  }

  // jumping or falling
  if (extSpeed[2] && !flying)
  {
    vec3_t sp;
    VectorScale(extSpeed, (float)Timer::frametime, sp);
    VectorCopy(sp, depl);
    VectorCopy(cam.pos, pos);

    trace = StepSlideMove(depl);

    progress = 1; // reset interpolation

    VectorAdd(pos, depl, cam.pos);

    if (extSpeed[2] > 0 && trace.fraction < 1)
      VectorClear(extSpeed);
  }
}

trace_t Client::StepSlideMove(vec3_t vel)
{
  // Dtag method
  trace_t trace;
  vec3_t start_o, start_v;
  vec3_t down_o, down_v;
  vec3_t up, down;

  VectorCopy(cam.pos, start_o); // backup of start position
  VectorCopy(vel, start_v);       // backup of wished end position

  VectorCopy(cam.pos, down_o);
  VectorCopy(vel, down_v);

  StepSlideMove_(down_o, down_v, mins, maxs);

  up[0] = start_o[0];
  up[1] = start_o[1];
  up[2] = start_o[2] + step_size;

  // check if up pos is in a solid
  trace = curr_bsp->CheckMove(up, mins, maxs, up);
  if (trace.allSolid && progress >= 1)
  {
    VectorCopy(down_o, cam.pos);
    VectorCopy(down_v, vel);
    return trace; // can't step up
  }

  // Try sliding above
  VectorCopy(up, cam.pos);
  VectorCopy(start_v, vel);
  StepSlideMove_(cam.pos, vel, mins, maxs);

  // push down the final amount
  down[0] = cam.pos[0];
  down[1] = cam.pos[1];
  down[2] = cam.pos[2] - step_size;
  trace = curr_bsp->CheckMove(cam.pos, mins, maxs, down);
  if (!trace.allSolid)
  {
    VectorCopy(trace.end, cam.pos);
  }

#if 0
  VectorSub(cam.pos, up, delta);
  up_dist = DotProduct(delta, start_v);

  VectorSub(down_o, start_o, delta);
  down_dist = DotProduct(delta, start_v);
#else
  VectorCopy(cam.pos, up);

  // Decide which one went farther
  float down_dist = (down_o[0] - start_o[0])*(down_o[0] - start_o[0]) +
    (down_o[1] - start_o[1])*(down_o[1] - start_o[1]);
  float up_dist   = (up[0] - start_o[0])*(up[0] - start_o[0]) +
    (up[1] - start_o[1])*(up[1] - start_o[1]);
#endif

  if (down_dist > up_dist || trace.plane.normal[2] < MIN_STEP_NORMAL)
  {
    VectorCopy(down_v, vel);
    VectorCopy(down_o, cam.pos);
    return trace;
  }

  VectorSub(up, start_o, vel);
  return trace;
}

#define STOP_EPSILON  0.1
void Client::ClipVelocity(vec3_t in, vec3_t normal, vec3_t out, float overbounce)
{
  float change, backoff = DotProduct(in, normal) * overbounce;

  change = normal[0] * backoff;
  out[0] = in[0] - change;
  if (out[0] > -STOP_EPSILON && out[0] < STOP_EPSILON) out[0] = 0;
  change = normal[1] * backoff;
  out[1] = in[1] - change;
  if (out[1] > -STOP_EPSILON && out[1] < STOP_EPSILON) out[1] = 0;
  change = normal[2] * backoff;
  out[2] = in[2] - change;
  if (out[2] > -STOP_EPSILON && out[2] < STOP_EPSILON) out[2] = 0;
}

#define MAX_CLIP_PLANES 5
void Client::StepSlideMove_(vec3_t pos, vec3_t vel, vec3_t mins, vec3_t maxs)
{
#if 1
  // Dtag method
  trace_t trace;
  trace.fraction = 0;

  vec3_t primalVel, end;
  VectorCopy(vel, primalVel);

  vec3_t planes[MAX_CLIP_PLANES];

  int i, j, numbumps = 4;
  float d, time_left = 1;

  vec3_t dir;

  int numplanes = 0;
  for (int index = 0; index < numbumps; ++index)
  {
    time_left -= time_left * trace.fraction;
    VectorMA(pos, time_left, vel, end);

    trace = curr_bsp->CheckMove(pos, mins, maxs, end);

    if (trace.allSolid)
    { // entity is trapped in another solid
      vel[2] = 0; // don't build up falling damage
      break;
    }

    if (trace.fraction > 0)
    { // actually covered some distance
      VectorCopy(trace.end, pos);
      numplanes = 0;
    }

    if (trace.fraction == 1) break;   // moved the entire distance

    // slide along this plane
    if (numplanes >= MAX_CLIP_PLANES)
    { // this shouldn't really happen
      VectorClear(vel);
      break;
    }

    VectorCopy(trace.plane.normal, planes[numplanes]);
    ++numplanes;

#if 0
    float rub;
    // Modify velocity so it parallels all of the clip planes
    if (numplanes == 1)
    { // go along this plane
      VectorCopy(vel, dir);
      VectorNormalize(dir);
      rub = 1.0 + .05 * DotProduct(dir, planes[0]);

      // slide along the plane
      clipVelocity(vel, planes[0], vel, 10.1f);

      // rub some extra speed off on xy axis
      // not on Z, or you can scrub down walls
      VectorScale(vel, rub, vel);
    }
    else if (numplanes == 2)
    { // go along the crease
      VectorCopy(vel, dir);
      VectorNormalize(dir);
      rub = 1.0 + 0.5 * DotProduct(dir, planes[0]);

      // slide along the plane
      CrossProduct(planes[0], planes[1], dir);
      d = DotProduct(dir, vel);
      VectorScale(dir, d, vel);

      // rub some extra speed off
      VectorScale(vel, rub, vel);
    }
    else
    {
      VectorClear(vel);
      break;
    }
#else
    // Modify original velocity so it parallels all of the clip planes
    for (i = 0; i < numplanes; ++i)
    {
      ClipVelocity(vel, planes[i], vel, 1.01f);
      for (j = 0; j < numplanes; ++j)
        if (j != i)
        {
          if (DotProduct(vel, planes[j]) < 0) break;
        }
        if (j == numplanes) break;
    }

    if (i != numplanes)
    {
      // Go along this plane
    }
    else
    {
      // Go along the crease
      if (numplanes != 2)
      {
        VectorClear(vel);
        break;
      }

      CrossProduct(planes[0], planes[1], dir);
      d = DotProduct(dir, vel);
      VectorScale(dir, d, vel);
    }
#endif

    // If velocity is against the original velocity, stop dead
    // to avoid tiny occilations in slopping corners
    if (DotProduct(vel, primalVel) <= 0)
    {
      VectorClear(vel);
      break;
    }
  }
#else
  // Q2 method
  trace_t trace;
  trace.fraction = 0;

  vec3_t primalVel, end;
  VectorCopy(vel, primalVel);

  vec3_t planes[MAX_CLIP_PLANES];

  int i, j, numbumps = 4;
  float d, time_left = (float) Timer::frametime;

  vec3_t dir;

  int numplanes = 0;
  for (int index = 0; index < numbumps; ++index)
  {
    VectorMA(pos, time_left, vel, end);

    trace = curr_bsp->checkMove(pos, mins, maxs, end);

    if (trace.allSolid)
    { // entity is trapped in another solid
      vel[2] = 0; // don't build up falling damage
      break;
    }

    if (trace.fraction > 0)
    { // actually covered some distance
      VectorCopy(trace.end, pos);
      numplanes = 0;
    }

    if (trace.fraction == 1)
      break;    // moved the entire distance

    time_left -= time_left * trace.fraction;

    // slide along this plane
    if (numplanes >= MAX_CLIP_PLANES)
    { // this shouldn't really happen
      VectorClear(vel);
      break;
    }

    VectorCopy(trace.plane.normal, planes[numplanes]);
    ++numplanes;

#if 0
    float rub;
    // Modify velocity so it parallels all of the clip planes
    if (numplanes == 1)
    { // go along this plane
      VectorCopy(vel, dir);
      VectorNormalize(dir);
      rub = 1.0 + .05 * DotProduct(dir, planes[0]);

      // slide along the plane
      clipVelocity(vel, planes[0], vel, 10.1f);

      // rub some extra speed off on xy axis
      // not on Z, or you can scrub down walls
      VectorScale(vel, rub, vel);
    }
    else if (numplanes == 2)
    { // go along the crease
      VectorCopy(vel, dir);
      VectorNormalize(dir);
      rub = 1.0 + 0.5 * DotProduct(dir, planes[0]);

      // slide along the plane
      CrossProduct(planes[0], planes[1], dir);
      d = DotProduct(dir, vel);
      VectorScale(dir, d, vel);

      // rub some extra speed off
      VectorScale(vel, rub, vel);
    }
    else
    {
      VectorClear(vel);
      break;
    }
#else
    // Modify original velocity so it parallels all of the clip planes
    for (i = 0; i < numplanes; ++i)
    {
      clipVelocity(vel, planes[i], vel, 1.01f);
      for (j = 0; j < numplanes; ++j)
        if (j != i)
        {
          if (DotProduct(vel, planes[j]) < 0)
            break;  // not ok
        }
        if (j == numplanes)
          break;
    }

    if (i != numplanes)
    { // go along this plane
    }
    else
    { // go along the crease
      if (numplanes != 2)
      {
        VectorClear(vel);
        break;
      }

      CrossProduct(planes[0], planes[1], dir);
      d = DotProduct(dir, vel);
      VectorScale(dir, d, vel);
    }
#endif

    // If velocity is against the original velocity, stop dead
    // to avoid tiny occilations in slopping corners
    if (DotProduct(vel, primalVel) <= 0)
    {
      VectorClear(vel);
      break;
    }
  }
#endif
}
