var MyCloudPortalModule = Class.create(
{
	initialize: function initialize()
	{
		this.currentSiteId			= 0;
		this.currentlyDisplayed		= 0;
		this.currentHelpContent		= '';
		this.templates				= {};
		this.currentSite			= {};
		this.noSites				= false;
		this.showInfo				= false;
		this.showNoProjectWarn		= true;
		this.projectWarnShown		= false;
		this.svnInstalled			= false;
	},
	
	finishInit: function finishInit()
	{
		this.extendSelf();
		this.registerObservers();
		this.subscribeToChannels();
		
		// we set this because we don't load this portal content through the 
		// standard API for doing so (Portal.API.tabs.loadPortalContent)
		Portal.API.tabs.switchingTabs = false;
		
		if (!this.hasRunningSites() || !Portal.Modules.OnlineStatus.isOnline) 
		{
			// console.log('unregister in finishInit');
			// Portal.API.tabs.unregisterTab('my_cloud');
			// return;
		}
		// get everything running...
		this._init();
	},
	
	loadComplete: function loadComplete()
	{
		// nothing here
	},
	
	dispatchEvent: function dispatchEvent(msg)
	{
		// handle delete responses
		if (msg.data.response == 'deleteSite')
		{
			this.deleteSite(msg);
		}
		if (msg.data.response == 'get') 
		{
			if (msg.data.name == 'portal.cloud.noprojectwarn') 
			{
				this.showNoProjectWarn = (msg.data.value == 'false') ? false : true;
			}
		}
		if (msg.data.response == 'svnteamprovider')
		{
			this.svnInstalled = msg.data.installed;
			// don't think we need to do anything since this is its own tab now...
			// this.renderSiteAccess();
		}
	},
	
	registerObservers: function registerObservers()
	{
		// triggers site list updates...
        Portal.Data.siteList.observe(['siteAdded', 'siteDeleted', 'siteUpdated', 'siteTeamUpdated'], function(siteId)
        {
			this.fetchSiteList();
        }.bind(this));
		
		// triggers site HUD updates
        Portal.Data.siteList.observe(['siteUpdated'], function(siteId)
        {
			if(siteId == this.currentSite.id)
			{
				// this.renderSiteHud();
				this.renderCurrentSite();
			}
        }.bind(this));
		
		// triggers site services overview updates
		Portal.Data.siteList.observe(['siteServicesUpdated', 'siteServicesLoaded'], function(siteId)
		{
			if(siteId == this.currentSite.id)
			{
				this.renderSiteServicesSummary();
			}
		}.bind(this));
		
		// update the sync image
        Portal.Data.siteList.observe('syncEndpointUpdated', function(siteId)
        {
			if(siteId == this.currentSite.id)
			{
				if ($('sync_button_left')) 
				{
					if(this.currentSite.type == 'java' || this.currentSite.type == 'rails')
					{
						$('sync_button_left').replace('<img src="images_global/button_deploy_left_' + this.currentSite.endpoint + '.png" id="sync_button_left" />');
					}
					else
					{
						$('sync_button_left').replace('<img src="images_global/button_sync_left_' + this.currentSite.endpoint + '.png" id="sync_button_left" />');
					}
				}
			}
        }.bind(this));
		
		// update the team count...
        Portal.Data.siteList.observe('siteTeamUpdated', function(siteId)
        {
			if($('tab_team') && siteId == this.currentSiteId)
			{
				this.updateTeamCount();
			}
			
        }.bind(this));
		
		// show / hide this tab if need be...
        Portal.Data.siteList.observe(['siteAdded', 'siteUpdated', 'siteDeleted'], function(siteId)
        {
			var totalRunning = 0;
			
            Portal.Data.siteList.sites.each(function(site)
            {
				if(['deleted', 'deleting'].indexOf(site.value.status.toLowerCase) == -1)
				{
					totalRunning ++;
				}
            });
			
			if(totalRunning == 0 && this.siteToDelete == 0)
			{
				// console.log('unregister in siteList observer');
				Portal.API.tabs.unregisterTab('my_cloud');
			}
			else
			{
				Portal.API.tabs.reRegisterTab('my_cloud');
			}
			
        }.bind(this));
		
		Portal.Data.cloudProjects.observe(['undeployedListChanged', 'deployedListChanged', 'projectListUpdated', 'projectListChanged'], function()
        {
			if($('project_list'))
			{
				if($('project_list').style.display == '')
				{
					Portal.Modules.MyCloudPortal.showProjectList = true;
				}
				else
				{
					Portal.Modules.MyCloudPortal.showProjectList = false;
				}
				
				this.templates.siteMetaData.set('projectList', Portal.Data.cloudProjects.undeployedProjects);
			}
			
        }.bind(this));
		
	},
	
	extendSelf: function extendSelf()
	{
		Object.extend(this, MyCloudControllers);
		Object.extend(this, MyCloudViews);
		Object.extend(this, MyCloudPortalUtils);
		Object.extend(this, MyCloudViewportManager);
	},
	
	subscribeToChannels: function subscribeToChannels()
	{
		// we receive delete responses on this channel
		EventManager.subscribe('/portal/cloud/sites', { channelHook: 'MyCloudPortal' });
		
		EventManager.subscribe('/portal/preferences/', 
        {
            channelHook: 'MyCloudPortal',
			onComplete: function()
			{
				Portal.API.utils.getPref('portal.cloud.noprojectwarn');
			}
        });
		
        EventManager.subscribe('/portal/plugins/query/', {
            channelHook: 'MyCloudPortal',
            onComplete: function() {
				EventManager.publish('/portal/plugins/query', { request : 'svnteamprovider' });
            }
        });
	}
});

/*  ----------------------------------------------  */
/* |           CONTROLLERS FUNCTIONALITY          | */
/*  ----------------------------------------------  */

var MyCloudControllers = 
{
	/**
	 * This function should only be called by finishInit()
	 * 
	 * It is responsible for building the main UI and rendering all of the initial templates.  The chain of events needs to be:
	 * 
	 *  - Determine the site we need to show (either first in the list, or passed somewhere else)
	 *  - Fetch the site model for the site to show and store it in this.currentSite
	 *  - Load the main UI and render it
	 *  - Render the site list
	 *  - Render the HUD info (site name / IP / status)
	 *  - Render the main display area
	 *  
	 */
	_init: function _init()
	{
		try
		{
			this.setCurrentSite();
		
			this._initViews();
		}
		catch (e)
		{
			console.warn(e);
		}
	},
	
	/**
	 * Sets this.currentSiteId
	 * 
	 * If siteId is not passed to the function, it will check to see if there is a particular site to load.  If none is
	 * specified, we'll use the first site in the site list.  If the site list is empty, or contains only deleted items,
	 * we leave this.currentSiteId as 0
	 * 
	 * @param {int} siteId
	 */
	setCurrentSite: function setCurrentSite(siteId)
	{
		if(!siteId || siteId == 0)
		{
			// check for specific site to show
			if(Portal.Vars.siteToLoad != 0)
			{
				this.currentSiteId = Portal.Vars.siteToLoad;
				Portal.Vars.siteToLoad = 0;
			}
			// find the first item in the site list
			else
			{
				var siteIds = Portal.Data.siteList.sites.keys();
				
				if(siteIds.length > 0)
				{
					this.currentSiteId = siteIds[0];
				}
			}
		}
		else
		{
			this.currentSiteId = siteId;
		}
		
		// fetch the site model and then validate that it exists
		this.currentSite = Portal.Data.siteList.sites.get(this.currentSiteId);
		
		this.projectWarnShown = false;
		
		if(!this.currentSite)
		{
			this.currentSiteId 	= 0;
			this.currentSite	= {};	
		}
	},
	
	/**
	 * Triggers the rendering of a selected site in the cloud UI
	 * 
	 * @param {Int} siteId
	 */
	showSiteInCloud: function showSiteInCloud(siteId)
	{
		if(siteId == this.currentlyDisplayed)
		{
			return;
		}
		
		this.setCurrentSite(siteId);
		this.highlightCurrentSite();
		this.renderCurrentSite();
	},
	
	/**
	 * Examines the state of the site list and determines what should be displayed
	 * 
	 * Also checks to see if the model is loading, and sets up an observer if need be
	 * 
	 */
	fetchSiteList: function()
	{
		// make sure the model's loaded
		if(Portal.Data.siteList.loading)
		{
            Portal.Data.siteList.observeOnce('siteModelLoaded', function()
            {
                this.fetchSiteList();
            }.bind(this));
			
			return;
		}
		
		// a little tricky, but if we need to see how many sites will actually be displayed and
		// use that to determine what state to show the UI in...
		if(this.hasRunningSites() && typeof(this.templates.siteList) != 'undefined')
		{
			this.renderSiteList();
		}
		else if (typeof(this.templates.siteList) != 'undefined')
		{
			this.renderNoSites();
		}
	},
	
	/**
	 * Determines whether the user has any running sites
	 * 
	 */
	hasRunningSites: function hasRunningSites()
	{
		var totalRunning = 0;
		
        Portal.Data.siteList.sites.each(function(site)
        {
			if(['deleted', 'deleting'].indexOf(site.value.status.toLowerCase()) == -1)
			{
				totalRunning ++;
			}
        });
		
		if(totalRunning == 0 || Portal.Data.siteList.sites.size() == 0)
		{
			this.noSites = true;
			return false;
		}
		
		// if we didn't have sites, and we do now, that means we need to set this.currentSiteId = the
		// first of the sites (usually is only one... could be a delayed load of the model tho)
		if(this.noSites)
		{
			this.setCurrentSite();
		}
		
		this.noSites = false;
		return true;
	},
	
	/**
	 * Highlights the currently displayed site
	 */
	highlightCurrentSite: function highLightCurrentSite()
	{
		if($('list_' + this.currentSiteId) && $('list_' + this.currentSiteId).hasClassName('currentSite'))
		{
			return;
		}
		
        $A($$('#site_list li.currentSite')).each(function(item)
        {
            new Effect.Morph(item, 
            {
                style: { backgroundColor: '#ededed' },
                duration: 0.25,
				beforeStart: function()
				{
					$(item).removeClassName('currentSite');
					$(item).down().setStyle({ fontWeight: 'normal' });
				}
            });
        });
		
		if(!$('list_' + this.currentSiteId))
		{
			return;
		}
		
        try
		{
			new Effect.Morph('list_' + this.currentSiteId, 
	        {
	            style: { backgroundColor: '#ffffff' },
	            duration: 0.25,
				beforeStart: function()
				{
					if(!$('list_' + this.currentSiteId))
					{
						return;
					}
					
					$('list_' + this.currentSiteId).down().setStyle({ fontWeight: 'bold' });
				}.bind(this),
	            afterFinish: function()
	            {
	                if(!$('list_' + this.currentSiteId))
					{
						return;
					}
					
					$('list_' + this.currentSiteId).addClassName('currentSite');
	            }.bind(this)
	        });
		}
		catch (e)
		{
			console.warn(e);
		}
	},
	
	/**
	 * Determines whether or not the site tabs should be shown / hidden
	 */
	showTabs: function showTabs()
	{
		if(['Running', 'Warning'].indexOf(this.currentSite.status) != -1 && $('cloud_site_tabs'))
		{
			var teamSize = Portal.Data.siteList.sites.get(this.currentSiteId).memberships.memberships.size();
			
			if(teamSize == 0)
			{
				teamSize = 1;
			}
			
			$('tab_team').update('Team (' + teamSize + ')');
			
			$('cloud_site_tabs').show();
			
            this.siteTabs = new Control.Tabs('cloud_site_tabs', 
            {
                afterChange: function(newContainer)
                {
					this.loadTabContent(newContainer);
					
					$('cloud_site_tabs').setStyle({ height: $('cloud_site_tabs').getHeight() - 2 + 'px' });
					
                }.bind(this)
            });
		}
		else
		{
			$('cloud_site_tabs').hide();
		}
	},
	
	/**
	 * Used by the Control.Tabs as a callback for tab selection
	 * 
	 * This function examines the selected tab, sets the help content, then renders the appropriate tab UI
	 * 
	 * @param {String} containerId
	 */
	loadTabContent: function loadTabContent(containerId)
	{
		switch($(containerId).id)
		{
			case 'site_overview':
				Portal.Vars.currentHelpContent = 'My_Cloud_-_Overview';
				this.renderSiteOverview();
				break;
			case 'site_settings':
				Portal.Vars.currentHelpContent = 'My_Cloud_-_Settings';
				this.renderSiteSettings();
				break;
			case 'site_services':
				Portal.Vars.currentHelpContent = 'My_Cloud_-_Services';
				this.renderSiteServices();
				break;
			case 'site_stats':
				Portal.Vars.currentHelpContent = 'My_Cloud_-_Stats';
				this.renderSiteAnalytics();
				break;
			case 'site_users':
				Portal.Vars.currentHelpContent = 'My_Cloud_-_Team';
				this.renderSiteTeam();
				break;
			case 'site_logs':
				Portal.Vars.currentHelpContent = "My_Cloud_-_Logs";
				this.renderSiteLogs();
				break;
			case 'site_access':
				Portal.Vars.currentHelpContent = "My_Cloud_-_Access";
				this.renderSiteAccess();
				break;
		}
		
		this.currentHelpContent = Portal.Vars.currentHelpContent;
	}
}

/*  ----------------------------------------  */
/* |           VIEWS FUNCTIONALITY          | */
/*  ----------------------------------------  */

var MyCloudViews = 
{
	/**
	 * Fetches all the global view templates we'll need, and starts the rendering process.
	 * 
	 * There's a lot of moving parts, so this will be slightly confusing, but here's the order in 
	 * which we fetch / render things:
	 * 
	 *  - The main wrapper UI is fetched and rendered
	 *  - We then set up the viewport resize observer
	 *  - The site list is rendered
	 *  - We then determine what state of the UI to show:
	 *  	- If there are no sites, show the default "deploy a site" UI
	 *  	- If there are sites, show the Dashboard UI
	 *  - If we're showing the dashboard, we'll next render the site list
	 *  - Then we highlight the current site
	 *  - Next is the HUD (for lack of a better term), which is the status icon, the site name and the IP in the header
	 *  - Then, the main content area is rendered
	 *  - Determine whether the state of the site needs tabs, and show them if needed
	 */
	_initViews: function _initViews()
	{
		// fetch all the templates
        View.load(
        {
			// the main wrapper
			mainWrapper:
			{
				file: 'portlets/cloud/templates/portals/my_cloud/my_cloud_new.html',
				binding: function()
				{
					return $('content_tab_my_cloud');
				},
				behaviors: function()
				{
					this.notify('cloudPortalLoaded');
					
					// make sure the content is visible.. another fragment of not using the default portal module 
					// loading method (see notes in finishInit)
					$('content_tab_my_cloud').show();
					
					// set up our observers for the viewport
					this.resizeViewPort();
					this.boundResizeViewPort = this.resizeViewPort.bindAsEventListener(this);
					
					Event.stopObserving(window, 'resize', this.boundResizeViewPort);
					Event.observe(window, 'resize', this.boundResizeViewPort);
					
					// load the undeployed projects module while we're at it
					Portal.API.modules.loadModule('UndeployedProjects', UndeployedProjectsModule, false);
					
				}.bind(this)
			},
			// the site list
			siteList:
			{
				file: 'portlets/cloud/templates/my_cloud/site_list.html',
				binding: function()
				{
					return $('site_list_wrap');	
				},
				scope:
				{
					siteList: Portal.Data.siteList.sites
				},
				behaviors: function()
				{
					// set up the click handlers for the list items 
                    $$('a.siteListItem').each(function(element)
                    {
                        $(element).observe('click', function(event)
                        {
							var siteId = Event.element(event).id;
							
							this.showSiteInCloud(siteId);
                        }.bind(this));
                    }.bind(this));
					
					// set up the context menus
					$$('li.siteListItem').each(function(item)
                    {
						var siteId = item.id.replace('list_', '');
						var site = Portal.Data.siteList.sites.get(siteId);
						
						var context_menu = new Control.ContextMenu(item.id);
						
						if(['pending', 'provisioning', 'provisioned', 'initializing', 'initialized'].indexOf(site.status.toLowerCase()) == -1)
						{
							// visit site menu item
	                        context_menu.addItem(
	                        {
								label: 'Visit site',
								callback: function()
								{
									Portal.API.utils.openUrl('http://' + site.domain);
								}
							});
							
							// sync menu item
	                        context_menu.addItem(
	                        {
								label: (site.type == 'java' || site.type == 'rails') ? 'Deploy site...' : 'Synchronize site...',
								callback: function()
								{
									this.doSync(site.id);
								}.bind(this)
							});
							
							// browse remote menu item
	                        context_menu.addItem(
	                        {
								label: 'Browse remote',
								callback: function()
								{
									this.browseRemote(site.id)
								}.bind(this)
							});
							
							// browse local project menu item
	                        context_menu.addItem(
	                        {
								label: 'Browse local project',
								enabled: (site.project) ? true : false,
								callback: function()
								{
									this.browseProject(site.project);
								}.bind(this)
							});
							
							// delete site menu item
	                        context_menu.addItem(
	                        {
								label: 'Delete site...',
								enabled: (site.isOwner) ? true : false,
								callback: function()
								{
									this.deleteSiteById(site.id);
								}.bind(this)
							});
						}
					}.bind(this));
					
					
					// highlights the current site...
					this.highlightCurrentSite();
					
				}.bind(this)
			},
			// the empty site list
			emptyList:
			{
				file: 'portlets/cloud/templates/my_cloud/no_sites.html',
				binding: function()
				{
					return $('site_content');
				},
				behaviors: function()
				{
					// removes the tab
					// comment this out for the old behavior...
					if (this.siteToDelete == 0) 
					{
						// console.log('unregister in emptyList')
						Portal.API.tabs.unregisterTab('my_cloud');
					}
					
					return;
					
					this.currentSiteId = 0;
					this.currentlyDisplayed = 0;
					
					if(!$('site_content'))
					{
						return;
					}
					
					$('site_content').show();
					
					if ($('site_hud_cage')) 
					{
						$('site_hud_cage').hide();
					}
					
					if($('cloud_site_tabs'))
					{
						$('cloud_site_tabs').hide();
					}
					
					$('site_list_wrap').update('');
					
					Portal.API.modules.loadModule('DeployProjectMini', DeployProjectMiniModule, false);
					
				}.bind(this)
			},
			// the site HUD
			siteHud:
			{
				file: 'portlets/cloud/templates/my_cloud/site_hud.html',
				binding: function()
				{
					return $('site_hud_cage');
				},
				scope: 
				{
					currentSite: this.currentSite
				},
				behaviors: function()
				{
					if ($('site_hud_cage')) 
					{
						$('site_hud_cage').show();
					}
					
					// sets the site's status image...
					var siteStatusImage = '';
		
					switch (this.currentSite.status)
					{
						case 'Error':
							siteStatusImage = 'error';
							break;
						case 'Running':
							siteStatusImage = 'running';
							break;
						case 'Initializing':
						case 'Initialized':
						case 'Provisioned':
						case 'Pending':
						case 'Resizing':
							siteStatusImage = 'provisioning';
							break;
						case 'Warning':
							siteStatusImage = 'warning';
							break;
						case 'Rebooting':
							siteStatusImage = 'provisioning';
							break;
						default:
							siteStatusImage = 'error';
					}
					
					if ($('cloud_site_status')) 
					{
						$('cloud_site_status').setStyle(
						{
							background: 'url(images_global/img_status_' + siteStatusImage + '.png) no-repeat right 3px'
						});
					}
				}.bind(this)
			},
			// the live site 
			liveSite:
			{
				file: 'portlets/cloud/templates/my_cloud/site_dashboard.html',
				binding: function()
				{
					return $('site_content');
				},
				scope:
				{
					currentSite: this.currentSite
				},
				behaviors: function()
				{
					this.showTabs();
				}.bind(this)
			},
			// you guessed it, provisioning site :)
			provisioningSite:
			{
				file: 'portlets/cloud/templates/my_cloud/site_provisioning.html',
				binding: function()
				{
					return $('site_content');
				},
				scope:
				{
					currentSite: this.currentSite
				},
				behaviors: function()
				{
					this.showTabs();
				}.bind(this)
			},
			errorSite:
			{
				file: 'portlets/cloud/templates/my_cloud/site_error.html',
				binding: function()
				{
					return $('site_content');
				},
				scope:
				{
					currentSite: this.currentSite
				},
				behaviors: function()
				{
					this.showTabs();
				}.bind(this)
			},
			resizingSite:
			{
				file: 'portlets/cloud/templates/my_cloud/site_resizing.html',
				binding: function()
				{
					return $('site_content');
				},
				scope:
				{
					currentSite: this.currentSite
				},
				behaviors: function()
				{
					this.showTabs();
				}.bind(this)
			},
			overview:
			{
				file: 'portlets/cloud/templates/my_cloud/site_overview.html',
				binding: function()
				{
					return $('site_overview');
				},
				scope:
				{
					currentSite: this.currentSite
				},
				behaviors: function()
				{
					// a render of the overview means we need to load some modules...
					Portal.API.modules.loadModule('NotificationList', NotificationListModule, false);
					Portal.API.modules.loadModule('SiteStats', SiteStatsModule, false);
					
                    $$('a.sync_endpoint').each(function(element)
                    {
						$(element).stopObserving('click');
                        $(element).observe('click', function(event)
                        {
							var newEndpoint = Event.element(event).id;
							
                            EventManager.publish('/portal/cloud/sync', 
                            {
								request: 'setEndpoint',
								id: this.currentSiteId,
								endpoint: newEndpoint
							});
							
							$('sync_button_left').replace('<img src="images_global/button_sync_left_' + newEndpoint + '.png" id="sync_button_left" />');
							
							var updateModel = function()
							{
                                EventManager.publish('/portal/cloud/sync', 
                                {
									request: 'getEndpoint',
									id: this.currentSiteId
								});
							}.bind(this);
							
							setTimeout(updateModel, 500);
							
							$('endpoint_list').hide();
							
                        }.bind(this));
                    }.bind(this));
				}.bind(this)
			},
			servicesSummary:
			{
				file: 'portlets/cloud/templates/my_cloud/site_overview_services_summary.html',
				binding: function()
				{
					return $('services_summary');
				},
				scope:
				{
					currentSite: this.currentSite
				}
			},
			siteMetaData:
			{
				file: 'portlets/cloud/templates/my_cloud/site_overview_site_metadata.html',
				binding: function()
				{
					return $('site_metadata');
				},
				scope:
				{
					currentSite: 	this.currentSite,
					currentUser:	Portal.Data.currentUser.username,
					projectList: 	Portal.Data.cloudProjects.undeployedProjects
				},
				behaviors: function()
				{
					if ($('import_site_to_project')) {
						$('import_site_to_project').observe('click', this.importSite);
					}
				}.bind(this)
			},
			siteAccess:
			{
				file: 'portlets/cloud/templates/my_cloud/site_access.html',
				binding: function()
				{
					return $('site_access');
				},
				scope:
				{
					currentSite: 	this.currentSite,
					currentUser:	Portal.Data.currentUser.username,
					projectList: 	Portal.Data.cloudProjects.undeployedProjects
				}
			},
			siteSettings:
			{
				file: 'portlets/cloud/templates/my_cloud/site_settings.html',
				binding: function()
				{
					return $('site_settings');
				},
				behaviors: function()
				{
					var isOwner = this.currentSite.isOwner;
					
					if (isOwner) 
					{
						Portal.API.modules.loadModule('SiteConfigurator', SiteConfiguratorModule, false);
						
                        $('delete_site_button').observe('click', function()
                        {
							this.deleteSite();
                        }.bind(this));
					}
					else
					{
						// $('deleteSiteDisabled').show();
					}
					
					Portal.API.modules.loadModule('LocalProjectSwitcher', LocalProjectSwitcherModule, false);
					
				}.bind(this)
			},
			siteAnalytics:
			{
				file: 'portlets/cloud/templates/my_cloud/site_analytics.html',
				binding: function()
				{
					return $('site_stats');
				},
				behaviors: function()
				{
					Portal.API.modules.loadModule('Cloudalytics', CloudalyticsClass, false);
				}
			},
			siteTeam:
			{
				file: 'portlets/cloud/templates/my_cloud/site_team.html',
				binding: function()
				{
					return $('site_users');
				},
				behaviors: function()
				{
					Portal.API.modules.loadModule('TeamManager', TeamManagerModule, false);
				}
			}
		},
		function (templates)
		{
			// store the templates...
			this.templates = templates;
			
			// start the UI build process...
			this.templates.mainWrapper.render();
			
			// continue the UI build process
			this.fetchSiteList();
			
			if(this.hasRunningSites())
			{
				this.renderCurrentSite();
			}
			
		}.bind(this));
	},
	
	/**
	 * Renders the site list
	 * 
	 * Updating the scope variable is enough to trigger a render
	 * 
	 */
	renderSiteList: function renderSiteList()
	{
		if (typeof(this.templates.siteList) != 'undefined') 
		{
			this.templates.siteList.set('siteList', Portal.Data.siteList.sites);
		}
	},
	
	/**
	 * Renders the UI in a "No sites" state
	 * 
	 * Basically all we're doing here is showing the same UI that exists on the cloud doormat under the "My Aptana" tab
	 * 
	 */
	renderNoSites: function renderNoSites()
	{
		if (typeof(this.templates.emptyList) != 'undefined' && Portal.API.tabs.currentTab == 'tab_my_cloud') 
		{
			this.templates.emptyList.render();
		}
	},
	
	/**
	 * Renders the current site
	 * 
	 * This function acts as a bit of a bootstrapper, determining which state of the UI to show based on
	 * the status of the current site
	 */
	renderCurrentSite: function renderCurrentSite()
	{
		// make sure we really should be displaying this site. Sometimes deleted sites can 
		// trigger this render function when they shouldn't (i.e. deleting the only site you 
		// have)
		if(!this.hasRunningSites())
		{
			this.renderNoSites();
			return;
		}
		
		this.renderSiteHud();
		
		var status = this.currentSite.status.toLowerCase();
		
		// figure out which state of the UI to render
		
		// running sites get the normal UI
		if(['running', 'warning'].indexOf(status) != -1)
		{
			// if the site isn't currently rendered in live state...
			if (!$('site_overview') || this.currentSiteId != this.currentlyDisplayed) 
			{
				this.showInfo = false;
				this.renderLiveSite();
			}
			// otherwise this is likely an update, so we can just call the renderSiteOverview function, as
			// a site update will only really apply when the person is viewing the overview
			else
			{
				this.renderSiteOverview();
			}
		}
		// provisioning sites get the provisioning UI
		else if (['pending','provisioning', 'provisioned', 'initializing', 'initialized'].indexOf(status) != -1)
		{
			this.renderProvisioningSite();
		}
		// site is resizing
		else if (['resizing'].indexOf(status) != -1)
		{
			this.renderResizingSite();
		}
		// anything else is an error state
		else
		{
			this.renderErrorSite();
		}
		
		this.currentlyDisplayed = this.currentSiteId;
	},
	
	/**
	 * Renders the live site UI (AKA the dashboard)
	 */
	renderLiveSite: function renderLiveSite()
	{
		if (typeof(this.templates.liveSite) != 'undefined') 
		{
			this.templates.liveSite.set('currentSite', this.currentSite);
			
			if(Portal.Vars.newSite == true && this.currentSite.id == Portal.Data.siteList.deployingSite)
			{
				Portal.Data.siteList.deployingSite = 0;
				Portal.Vars.newSite = false;
				this.showModalSiteComplete();
			}
		}
	},
	
	/**
	 * Renders the provisioning UI
	 */
	renderProvisioningSite: function renderProvisioningSite()
	{
		if (typeof(this.templates.provisioningSite) != 'undefined') 
		{
			this.templates.provisioningSite.set('currentSite', this.currentSite);
		}
	},
	
	/**
	 * Renders the error-state UI
	 */
	renderErrorSite: function renderErrorSite()
	{
		if (typeof(this.templates.errorSite) != 'undefined') 
		{
			this.templates.errorSite.set('currentSite', this.currentSite);
		}
	},
	
	/**
	 * Renders the resizing-state UI
	 */
	renderResizingSite: function renderResizingSite()
	{
		if (typeof(this.templates.resizingSite) != 'undefined') 
		{
			this.templates.resizingSite.set('currentSite', this.currentSite);
		}
	},
	
	/**
	 * Renders the HUD, or the stuff in the header above the tabs (site name, ip, etc)
	 */
	renderSiteHud: function renderSiteHud()
	{
		if (!$('site_hud_cage'))
		{
			return;
		}
		
		if (typeof(this.templates.siteHud) != 'undefined') 
		{
			this.templates.siteHud.set('currentSite', this.currentSite);
		}
	},
	
	/**
	 * Renders the site overview, or what most people would think of as the actual dashboard
	 */
	renderSiteOverview: function renderSiteOverview()
	{	
		// this function can be called by status updates as well as tab switch
		// events, so we want to make sure that the overview tab is visible
		// before we attempt to render any UI
		if(!this.siteTabs || this.currentSiteId != this.currentlyDisplayed || (this.siteTabs.activeContainer.id == 'site_overview' || this.siteTabs.activeContainer.id == ''))
		{
			// if the whole UI doesn't exist, we want to load it, otherwise we can
			// simply update the individual pieces
			if (!$('services_summary'))
			{
				this.templates.overview.set('currentSite', this.currentSite);
			}
			
			this.renderSiteServicesSummary();
			this.renderSiteMetaData();
			this.parseDiskUsage();
		}
	},
	
	/**
	 * Renders the services summary on the overview page 
	 */
	renderSiteServicesSummary: function renderSiteServicesSummary()
	{
		if ($('services_summary')) 
		{
			this.templates.servicesSummary.set('currentSite', this.currentSite);
		}
	},
	
	/**
	 * Renders the site metadata on the overview page (the last sync, etc.)
	 */
	renderSiteMetaData: function renderSiteMetaData()
	{
		this.templates.siteMetaData.set('currentSite', this.currentSite);
	},
	
	/**
	 * Renders the site settings tab
	 */
	renderSiteSettings: function renderSiteSettings()
	{
		// no scope vars yet...
		this.templates.siteSettings.render();
	},
	
	renderSiteAccess: function renderSiteAccess()
	{
		this.templates.siteAccess.set('currentSite', this.currentSite);
	},
	
	/**
	 * Renders the stie analytics tab
	 */
	renderSiteAnalytics: function renderSiteAnalytics()
	{
		// no scope vars yet...
		// this.templates.siteAnalytics.render();
		Portal.API.utils.setContent('site_stats', '');
		Portal.API.modules.loadModule('StatsManager', StatsManagerModule, false);
	},
	
	/**
	 * Renders the site services tab
	 */
	renderSiteServices: function renderSiteServices()
	{
		Portal.API.utils.setContent('site_services', '');
		
		Portal.API.modules.loadModule('SiteServices', ServicesManagerModule, false);
	},
	
	/**
	 * Renders the site team tab
	 */
	renderSiteTeam: function renderSiteTeam()
	{
		this.templates.siteTeam.render();
	},
	
	/**
	 * Renders the site logs tab
	 */
	renderSiteLogs: function renderSiteLogs()
	{
		Portal.API.utils.setContent('site_logs', '');
		
		Portal.API.modules.loadModule('SiteLogs', LogsManagerModule, false);
	},
	
	/**
	 * Updates the team count on the team tab
	 */
	updateTeamCount: function updateTeamCount()
	{
		try
		{
			var teamSize = Portal.Data.siteList.sites.get(this.currentSiteId).memberships.memberships.size();
				
			if(teamSize == 0)
			{
				teamSize = 1;
			}
			
			$('tab_team').update('Team (' + teamSize + ')');
		}
		catch (e) 
		{
			console.warn(e);
		}
	},
	
	/**
	 * Helps render the proper disk usage on the overview
	 * 
	 * This function may or may not be removed depending on the ultimate end-game of the overview page 
	 * and the services that go with everything... you'll know if it should be removed :)
	 * 
	 */
	parseDiskUsage: function parseDiskUsage()
	{
		// nothing here anymore...
	}
}

/*  --------------------------------------  */
/* |           UTILITY FUNCTIONS          | */
/*  --------------------------------------  */
var MyCloudPortalUtils = 
{
	showNoProjectWarning: function showNoProjectWarning()
	{
		if($('cloud_local_project_name').innerHTML != 'None' && !$('cloud_local_project_name').hasClassName('red'))
		{
			return;
		}
		
		var warningHtml = '<div style="height: 360px;" id="projectWarnDialog"><div class="size12 line14 bold">Your site does not have a local project associated with it.</div><div class="size11 line14 top5">You must associate a project before you can synchronize your site.</div>';
		warningHtml += '<div align="center" class="top5" style="height: 277px;"><img src="images_global/changeproject_anim.gif" style="border: 1px solid #a4a4a4;" /></div>';
		warningHtml += '<div class="top5"><input type="checkbox" value="false" id="noProjectWarnPref" name="noProjectWarnPref" onclick="if(this.checked == true) { Portal.Modules.MyCloudPortal.showNoProjectWarn = false; Portal.API.utils.setPref(\'portal.cloud.noprojectwarn\', \'false\'); } else { Portal.Modules.MyCloudPortal.showNoProjectWarn = true; Portal.API.utils.setPref(\'portal.cloud.noprojectwarn\', \'true\'); }" />';
		warningHtml += '<label for="noProjectWarnPref" class="bold">Don\'t warn me again.</label></div></div>';
		
		if(!$('projectWarnDialog') && Portal.API.tabs.currentTab == 'tab_my_cloud' && typeof(Portal.Modules.MyCloudPortal.siteTabs) != 'undefined' && $(Portal.Modules.MyCloudPortal.siteTabs.activeContainer).id == 'site_overview' && Portal.Modules.MyCloudPortal.projectWarnShown == false)
		{
			Portal.Modules.MyCloudPortal.projectWarnShown = true;
			Portal.API.dialogs.alert(warningHtml, 'No Local Project');
		}
	},
	
	createNewProject: function createNewProject()
	{
		EventManager.publish('/portal/wizard', 
        {
            id: 'com.aptana.ide.wizards.WebProjectWizard',
            request: 'new'
        });
		
		new Effect.BlindUp('project_list', { duration: 0.25 });
	},
	
	importSite: function importSite()
	{
		EventManager.publish('/portal/wizard', 
        {
            request: 'import',
            id: 'com.aptana.ide.server.cloud.ui.ImportApplicationWizard',
			data: Portal.Modules.MyCloudPortal.currentSiteId
        });
		
		new Effect.BlindUp('project_list', { duration: 0.25 });
	},
	
	switchProject: function switchProject(newProject)
	{
        Portal.API.dialogs.confirm(
        {
			message: 'Are you sure you want to use "' + newProject + '" as the local project for this site?',
			onConfirm: function()
			{
				Portal.Modules.MyCloudPortal.showProjectList = false;
				
                EventManager.publish('/portal/cloud/projects', 
                {
                    request: 'changeProject',
                    project: newProject,
                    site: Portal.Modules.MyCloudPortal.currentSiteId
                });
				
				$('project_list').hide();
				
				Portal.API.utils.sendTrackingData('c.psw');
				
				this.showSyncWindow.delay(0.45);
				
			}.bind(this),
			title: 'Confirm Project Association Change'
		});
	},
	
	showRootEnableConfirm: function showRootEnableConfirm()
	{
		var message = '<div class="line16 size11">I have requested that Aptana enable root access to my site. I acknowledge that:'
					+ '<ul style="list-style-type: disc; padding: 5px; margin-left: 15px;"><li>I am the owner of this site</li>'
					+ '<li>All system users will now have root access</li>'
					+ '<li>Aptana will no longer apply maintenance and other updates to my site</li>'
					+ '<li>Some management features that Aptana currently offers will no longer be available and other management features may not work as expected</li>'
					+ '<li>Aptana is no longer obligated to provide support for this site, regardless of whether or not I have purchased additional support services.</li>'
					+ '<li>The only remedy available to me should I wish to revert back to the state prior to my enabling root is to delete this site and redeploy my project to a new site as appropriate.</li></ul>'
					+ '<div class="top10" id="root_agree_cage"><input id="root_agree" name="root_agree" type="checkbox" value="1" /> <label for="root_agree">I agree to the terms above</label></div></div>';
		
		Portal.API.dialogs.confirm({
			title: 'Confirm Enable Root Access',
			message: message,
			onConfirm: function()
			{
				this.enableRoot()
			}.bind(this),
			beforeClose: function()
			{
				if($('root_agree').checked != true)
				{
					$('root_agree_cage').addClassName('red').setStyle({ fontWeight: 'bold' });
					return false;
				}
				
				return true;
			}
		})
	},
	
	enableRoot: function enableRoot()
	{
		try {
			// we'll use the current job polling stuff in the services module
			Portal.API.modules.loadModule('SiteServices', ServicesManagerModule, false);
			
			Portal.Modules.SiteServices.pollWorkingMessage = 'Please wait, we\'re enabling root access.';
			Portal.Modules.SiteServices.pollFailMessage = 'There was an error enabling root access or the process timed out.  Please try again later, or contact support if you continue to receive this error.';
			Portal.Modules.SiteServices.pollStartUrl = 'sites/' + Portal.Modules.MyCloudPortal.currentSiteId;
			Portal.Modules.SiteServices.pollMessage = {
				site: {
					sudo_enabled: 'true'
				}
			};
			Portal.Modules.SiteServices.pollCallBack = function()
			{
				this.currentSite.sudoEnabled = true;
				
				if(this.siteTabs.activeContainer.id == 'site_access')
				{
					this.renderSiteAccess();
				}				
			}.bind(this);
			
			Portal.Modules.SiteServices.confirmPollingJob(true);
		} 
		catch (e)
		{
			console.warn(e);
		}	
	},
	
	setMonitoringState: function setMonitoringState()
	{
		var currentState = this.currentSite.monitoring;
		var verb = (currentState == true) ? 'disable' : 'enable';
		
		Portal.API.dialogs.confirm({
			title: 'Confirm ' + verb.capitalize() + ' Monitoring',
			message: 'Are you sure you wish to ' + verb + ' monitoring for this site?',
			onConfirm: function()
			{
                EventManager.publish('/portal/cloud/model', {
                    url: 'sites/' + this.currentSiteId,
                    request: 'commit',
                    returnChannel: '/portal/cloud/sites/monitoring',
                    xmlData: {
                        site: {
                            monitoring_enabled: (currentState == true) ? false : true
                        }
                    }
                });
				
				this.currentSite.monitoring = (currentState == true) ? false : true;
				
				if(this.siteTabs.activeContainer.id == 'site_overview')
				{
					verb = (currentState == true) ? 'enable' : 'disable'; 
					
					$('monitoring_link').update('<img src="images_global/button_' + verb + '_monitoring.png" />');
				}
			}.bind(this)
		});
	},
	
	showSyncWindow: function showSyncWindow()
	{
		if (Portal.Modules.MyCloudPortal.currentSite.type == 'rails' || Portal.Modules.MyCloudPortal.currentSite.type == 'java') 
		{
			Portal.API.dialogs.alert('Now that you have changed your local project, would you like to deploy? <div class="top10 center"><a href="javascript: Portal.Modules.MyCloudPortal.doSync(); Portal.API.dialogs.closeAlert();"><img src="images_global/button_deploynow.png" align="absmiddle" class="inline link" /></a></div>', 'Would you like to deploy?');
		}
		else
		{
			Portal.API.dialogs.alert('Now that you have changed your local project, would you like to sync? <div class="top10 center"><a href="javascript: Portal.Modules.MyCloudPortal.doSync(); Portal.API.dialogs.closeAlert();"><img src="images_global/button_syncnow.png" align="absmiddle" class="inline link" /></a></div>', 'Would you like to sync?');
		}
	},
	
	deleteSiteById: function deleteSiteById(siteId)
	{
		this.siteToDelete = parseInt(siteId);
		
		this.deleteSite(false, true);
	},
	
	deleteSite: function deleteSite(msg, deleteById)
	{
		if(!msg)
		{
			var isOwner = Portal.Data.siteList.sites.get(this.currentSiteId).isOwner;
			
			if(!isOwner)
			{
				Portal.API.dialogs.alert('You do not have permission to delete this site.  Only the owner of the site may do so.', 'Cannot Delete Site');
				return;
			}
			
			if (!deleteById) 
			{
				this.siteToDelete = this.currentSiteId;
			}
			
            Portal.API.dialogs.confirm(
            {
				message: '<div class="gray48 line16">Are you sure you wish to delete the site "' + Portal.Data.siteList.sites.get(this.siteToDelete).name.truncate(30) + '"?</div><div class="gray87 size10 line14"><em>This cannot be undone, and we cannot restore your site once you delete it.</em></div>'
						 + '<div class="top10 gray48 size12 line16"><strong>Please help us out by telling us why you are deleting this site (optional):</strong><br /><textarea name="delete_reason" id="delete_reason" rows="3" cols="40" class="text3"></textarea></div>'
						 + '<div class="top10 size10 gray87 line14">Please also note that while the site will be deleted right away, you may not be able to reuse your domain for a while.  Domains usually become available again within 10 minutes.</div>',
				title: 'Confirm site deletion',
				onConfirm: function()
				{
					if($F('delete_reason') != '')
					{
						Portal.API.utils.sendTrackingData('c.dr', $F('delete_reason'));
					}
					
                    Portal.API.dialogs.observeOnce('confirmClosed', function()
                    {
						this.showDeleteWindow.delay(0.25);
                    }.bind(this));
				}.bind(this)
			});
		}
		else
		{
			Portal.Data.siteList.removeSiteFromModel(this.siteToDelete);
			
			var totalRunning = 0;
			var siteToShow = 0;
			
            Portal.Data.siteList.sites.each(function(site)
            {
				if(site.value.id != this.siteToDelete && siteToShow == 0 && ['deleting', 'deleted'].indexOf(site.value.status.toLowerCase()) == -1)
				{
					siteToShow = site.value.id;	
				}
				
				if(['deleted', 'deleting'].indexOf(site.value.status.toLowerCase) == -1)
				{
					totalRunning ++;
				}
            }.bind(this));
			
			if(totalRunning == 0 || siteToShow == 0)
			{
				if ($('cloud_site_status')) 
				{
					$('cloud_site_status').hide();
					$('cloud_site_ip').update('');
					$('cloud_site_name').update('');
				}
				
				this.siteToDelete = 0;
			
				this.deleteDialog.close();
				this.deleteDialog.destroy();
				
				var showConfirmation = function()
				{
					Portal.Modules.CloudTrialStatus.fetchTrialUpdate();
					Portal.API.dialogs.alert('Your site was deleted successfully', 'Site deleted');
				}
				
				setTimeout(showConfirmation, 1000);
				
				this.renderNoSites();
				
				return;
			}
			else if(this.currentSiteId == this.siteToDelete || this.currentlyDisplayed == this.siteToDelete)
			{
				this.currentSiteId = 0;
				this.currentlyDisplayed = 0;
				
				this.showSiteInCloud(siteToShow);
			}
			
			this.siteToDelete = 0;
			
			this.deleteDialog.close();
			this.deleteDialog.destroy();
			
			// just to be sure...
			this.fetchSiteList();
			
			var showConfirmation = function()
			{
				Portal.Modules.CloudTrialStatus.fetchTrialUpdate();
				Portal.API.dialogs.alert('Your site was deleted successfully', 'Site deleted');
			}
			
			setTimeout(showConfirmation, 1000);
		}
	},
	
	showDeleteWindow: function showDeleteWindow()
	{
		var deleteContent 	= '<div class="clean-yellow"><div class="bold line16 center top5"><img src="images_global/img_activity_deploy.gif" class="inline" align="absmiddle" />'
							+ ' Please wait, we\'re deleting your site</div><div class="size11 line14 top5 center"><em>This should only take a moment...</em></div></div>';
		
        Portal.Modules.MyCloudPortal.deleteDialog = new Control.Modal(deleteContent, 
        {
			closeOnClick: false,
			className: 'modal',
			width: 300,
			overlayOpacity: 0.75,
			fade: false,
			height: null,
			fadeDuration: 0.25,
			iframeshim: false,
			afterOpen: function()
			{
                EventManager.publish('/portal/cloud/sites', 
                {
                    request: 'deleteSite',
                    id: Portal.Modules.MyCloudPortal.siteToDelete
                });
				
			}.bind(this)
		});
		
		Portal.Modules.MyCloudPortal.deleteDialog.open();
	},
	
	browseRemote: function browseRemote(siteId, destination)
	{
		if(!siteId)
		{
			var siteId = this.currentSiteId;
		}
		
		if(!destination)
		{
			var destination = 'public';
		}
		
		EventManager.publish('/portal/cloud/site/browse', { request: 'showFile', id: siteId, destination: destination });
	},
	
	browseProject: function browseProject(projectName)
	{
		EventManager.publish('/portal/projects/show', { project: projectName });
	},
	
	showSyncExplorer: function showSyncExplorer()
	{
		EventManager.publish('/portal/cloud/site/browse', { request: 'showExplorer', id: this.currentSiteId });
	},
	
	showSqlBuddy: function showPhpMyAdmin()
	{
		Portal.API.utils.openUrl(this.currentSite.urls.sqlBuddy);
	},
	
	showPhpMyAdmin: function showPhpMyAdmin()
	{
		Portal.API.utils.openUrl(this.currentSite.urls.phpMyAdmin);
	},
	
	showDbManager: function()
	{
		Portal.API.utils.openPerspective('com.aptana.ide.db.PerspectiveFactory', 'zigen.plugin.db.ui.views.TreeView', { siteId: this.currentSiteId });
	},
	
	openTerminal: function openTerminal()
	{
		EventManager.publish('/portal/cloud/ssh', {request : 'ssh', id: this.currentSiteId });
	},
	
	showModalSiteComplete: function showModalSiteComplete()
    {
		// No project associated with the site.
		// Show the No Project Warnign dialog
		if (this.currentSite.project == null) {
			MyCloudPortalUtils.showNoProjectWarning();
			return;
		}
		
		var file = 'portlets/cloud/templates/my_cloud/modal_deployment_complete.html';
		var railsFile = 'portlets/cloud/templates/my_cloud/modal_deployment_complete_rails.html';

        new Ajax.Request((this.currentSite.type == 'rails') ? railsFile : file, 
        {
            onComplete: function(response)
            {
                this.siteCompleteDialog = new Control.Modal(response.responseText, 
                {
                    closeOnClick: false,
                    className: 'modal',
                    width: 400,
                    overlayOpacity: 0.5,
                    fade: true,
                    height: null,
                    fadeDuration: 0.25,
                    iframeshim: false,
                    afterOpen: Prototype.emptyFunction
                });
                
                this.siteCompleteDialog.open();
            }.bind(this)
        });
    },
	
	closeModalSiteComplete: function closeModalSiteComplete()
    {
        this.siteCompleteDialog.close();
        this.siteCompleteDialog.destroy();
    },
	
	browseRemoteFile: function(fileToOpen)
	{
        EventManager.publish('/portal/cloud/site/browse', 
        {
			request: 'showFile',
			id: this.currentSiteId,
			path: fileToOpen
		});
	},
	
	downloadRemoteFile: function downloadRemoteFile(location)
	{
		EventManager.publish('/portal/cloud/site/browse', 
        {
			request: 'downloadFile',
			id: this.currentSiteId,
			path: location
		});
	},
	
	tailRemoteFile: function tailRemoteFile(location, name)
	{
        EventManager.publish('/portal/cloud/site/browse', 
        {
			request:	'showTail',
			id: 		this.currentSiteId,
			tailName: 	name + ' (' + this.currentSite.name + ')',
			path: 		location
		});
	},
	
	openRemoteFile: function(fileToOpen)
	{
        EventManager.publish('/portal/cloud/site/browse', 
        {
			request: 'openFile',
			id: this.currentSiteId,
			path: fileToOpen
		});
	},
	
	doSync: function doSync(siteId)
	{
		if(!siteId)
		{
			var siteId = this.currentSiteId;
		}
		
		var currentSite = Portal.Data.siteList.sites.get(siteId);
		
		if(!currentSite.project)
		{
			var warningHtml = '<div style="height: 360px;" id="projectWarnDialog"><div class="size12 line14 bold">Your site does not have a local project associated with it.</div><div class="size11 line14 top5">You must associate a project before you can synchronize your site.</div>';
			warningHtml += '<div align="center" class="top5" style="height: 277px;"><img src="images_global/changeproject_anim.gif" style="border: 1px solid #a4a4a4;" /></div>';
			warningHtml += '</div>';
			
			Portal.API.dialogs.alert(warningHtml, 'No local project');
			return;
		}
		
		EventManager.publish('/portal/cloud/sync', { request: 'syncSite', id: siteId });
	},
	
	browseSvn: function browseSvn(location)
	{
		if(location == 'web')
		{
			Portal.API.utils.openUrl(Portal.Data.siteList.sites.get(this.currentSiteId).urls.svn);
		}
		else if (location == 'studio')
		{
			EventManager.publish('/portal/cloud/svn', { request: 'showView' });
		}
	},
	
	backups: function backups(action)
	{
		if (action == 'browse')
		{
			this.browseRemoteFile(Portal.Data.siteList.sites.get(this.currentSiteId).files.backups);
		}
		else if (action == 'backup')
		{
			EventManager.publish('/portal/cloud/site/dump', 
	        {
	            request: 'dumpData',
	            id: this.currentSiteId
	        });
			
			Portal.API.dialogs.alert('A backup of your site is being created on the remote server.  To see this backup file, use the Browse Backups link.  Then you can drag and drop the remote file to your local filesystem.', 'Backup started');
		}
	}
};

/*  -------------------------------------  */
/* |           VIEWPORT MANAGER          | */
/*  -------------------------------------  */
var MyCloudViewportManager = 
{
	resizeViewPort: function(recurse)
	{
		if(Portal.API.tabs.currentTab != 'tab_my_cloud')
		{
			return;
		}
		
		if(!this.previousWindowHeight)
		{
			this.previousWindowHeight = 0;
			this.previousWindowWidth = 0;
		}
		
		$('site_content').hide();
		$('site_list_wrap').hide();
		
		var windowHeight = document.viewport.getHeight();
		var windowWidth = document.viewport.getWidth();
		var headerHeight = 76;
		var legendHeight = 0; // 80;
		var utilsHeight = 76;
		var scrollBarSize = this.getScrollBarWidth();
				
		// set the height of the content area and site list
		// $('site_list_cage').setStyle({ height: windowHeight - headerHeight - legendHeight - utilsHeight - 2 + 'px', overflow: 'auto' });
		// $('site_list_wrap').setStyle({ height: $('site_list_cage').getHeight() + 'px', overflow: 'auto' });
		// $('site_content_cage').setStyle({height: windowHeight - headerHeight - utilsHeight - 1 + 'px' });
		// $('site_content').setStyle({
		// 	height: $('site_content_cage').getHeight() + 'px'
		// });
		
		$('content_tab_my_cloud').setStyle({ minHeight: (windowHeight - headerHeight) + 'px' });
		
		$('site_list_wrap').show();
		$('site_content').show();
		
		this.previousWindowHeight = windowHeight;
	},
	
		
	getScrollBarWidth: function()
	{
		var inner = document.createElement('p');
		inner.style.width = "100%";
		inner.style.height = "200px";
		
		var outer = document.createElement('div');
		outer.style.position = "absolute";
		outer.style.top = "0px";
		outer.style.left = "0px";
		outer.style.visibility = "hidden";
		outer.style.width = "200px";
		outer.style.height = "150px";
		outer.style.overflow = "hidden";
		outer.appendChild (inner);
		
		document.body.appendChild (outer);
		var w1 = inner.offsetWidth;
		outer.style.overflow = 'scroll';
		var w2 = inner.offsetWidth;
		if (w1 == w2) w2 = outer.clientWidth;
		
		document.body.removeChild (outer);
		
		return (w1 - w2);
	}
};

/*  ----------------------------------------  */
/* |           EVENT / PORTAL INIT          | */
/*  ----------------------------------------  */

Object.Event.extend(MyCloudPortalModule);

Portal.API.tabs.registerTab('my_cloud', 2, 
{
	name: 'my_cloud',
	moduleName: 'MyCloudPortal',
	moduleObj: MyCloudPortalModule,
	isDefault: false,
	title: 'My Sites',
	requireLogin: true,
	requireOnline: true
});