# Pantera - Web Pen-Test Proxy
#
# FILENAME      : panteraPlugins.py
# CODER         : Simon Roses Femerling
# DATE          : 06/05/2006
# LAST UPDATE   : 07/05/2006
# ABSTRACT      : Pantera Plugins Manager
#       
# - Roses Labs Innovations (RL+I)
# Roses Labs
# http://www.roseslabs.com
#
# Copyright (c) 2003-2006 Roses Labs.
#
# You may not distribute, transmit, repost this software for commercial 
# purposes without Roses Labs written permission. 
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, publish,
# distribute the Software, and to permit persons to whom the Software 
# is furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

'''
@author:       Simon Roses Femerling
@license:      GNU General Public License 2.0 or later
@contact:      pantera.proxy@gmail.com
@organization: OWASP / Roses Labs
'''

import re
import sys
import os
import imp
import sets

#############################################################################################
# Exceptions
#############################################################################################

#############################################################################################
# FUNC     : class ModuleException
# PARAMS   : Exception
# RETURN   : ...
# ABSTRACT : Module exception
class ModuleException(Exception):
    '''
    Module Exception Class.
    '''
    pass

#############################################################################################
# FUNC     : class FatalException
# PARAMS   : ModuleException
# RETURN   : ...
# ABSTRACT : Module Fatal Exception
class FatalException(ModuleException):
    '''
    Fatal Exception Class.
    '''
    pass

#############################################################################################
# PPA Plugins Classes
#############################################################################################

#############################################################################################
# FUNC     : class PPM
# PARAMS   : ...
# RETURN   : ...
# ABSTRACT : Passive plugins must use this class
class PPM:
    '''
    Pantera Passive Manager Class.
    '''
    
    #############################################################################################
    # FUNC     : def __init__
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Init 
    def __init__(self):
        '''
        Init class.
        '''
        self.result = 0
        self.result_data = sets.Set([])
        self.check_result_data = 0
        self.idata = sets.Set([]) # uniqueness
        self.type = ""
        self.set_have = ""
        self.obj = ""
        self.set_other_type = ""
        self.set_level = "info"
    # EOF: def __init__
        
    #############################################################################################
    # FUNC     : def GetResult
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return plugin result
    def GetResult(self):
        '''
        Return plugin result.
        
        @return: int
        '''
        return self.result
    # EOF: def GetResult
    
    #############################################################################################
    # FUNC     : def GetResultData
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return plugin result data
    def GetResultData(self):
        '''
        Return plugin result data.
        
        @return: list
        '''
        return self.result_data
    # EOF: def GetResultData
    
    #############################################################################################
    # FUNC     : def SetResult 
    # PARAMS   : value
    # RETURN   : ...
    # ABSTRACT : Set plugin result
    def SetResult(self, r, h):
        '''
        Set plugin result.
        
        @type r: int
        @param r: Set plugin result.
        @type h: string
        @param h: Set have.
        '''
        if not (r == 1 or r == 0):
            self.result = 0
        else:
            self.result = r
        self.set_have = h
    # EOF: def SetResult
    
    #############################################################################################
    # FUNC     : def SetResultData
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Set plugin result data
    def SetResultData(self, d):
        '''
        Set plugin result data.
        
        @type d: string
        @param d: Data.
        '''
        
        t = "%s(<>)%s"  % (self.set_level,d)
        self.result_data.add(t)
        self.check_result_data = 1
    # EOF: def SetResultData 
    
    #############################################################################################
    # FUNC     : def GetCheckResultData
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return result data
    def GetCheckResultData(self):
        '''
        Return result data.
        
        @return: string
        '''
        return self.check_result_data
    # EOF: def GetCheckResultData
    
    #############################################################################################
    # FUNC     : def SetType
    # PARAMS   : string
    # RETURN   : string...
    # ABSTRACT : Set type
    def SetType(self, t):
        '''
        Set type.
        
        @type t: string
        @param t: Type.
        '''
        self.type = t
    # EOF: def SetType
    
    #############################################################################################
    # FUNC     : def SetTypeData
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Set data type
    def SetTypeData(self, v):
        '''
        Set data type.
        
        @type v: string
        @param v: Data type.
        '''
        self.idata.add(v)
    # EOF: def SetTypeData
        
    #############################################################################################
    # FUNC     : def GetType
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return type
    def GetType(self):
        '''
        Return type.
        
        @return: string
        '''
        return self.type
    # EOF: def GetType
    
    #############################################################################################
    # FUNC     : def GetTypeData
    # PARAMS   : ...
    # RETURN   : list
    # ABSTRACT : Return type data
    def GetTypeData(self):
        '''
        Return type data.
        
        @return: list
        '''
        return self.idata
    # EOF: def GetTypeData
    
    #############################################################################################
    # FUNC     : def GetHave
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return have.
    def GetHave(self):
        '''
        Return have.
        
        @return: string
        '''
        return self.set_have
    # EOF: def GetHave
    
    #############################################################################################
    # FUNC     : def SetObjData
    # PARAMS   : obj
    # RETURN   : ...
    # ABSTRACT : Set object
    def SetObjData(self, o):
        '''
        Set object.
        
        @type o: obj
        @param o: Set object.
        '''
        self.obj = o
    # EOF: def SetObjData

    #############################################################################################
    # FUNC     : def SetOtherType
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Add dditional have
    def SetOtherType(self, h):
        '''
        Add additional have.
        
        @type h: string
        @param h: Additional have.
        '''
        self.set_other_type = h
    # EOF: def SetOtherType
        
    #############################################################################################
    # FUNC     : def GetOtherType
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return other type.
    def GetOtherType(self):
        '''
        Return other type.
        
        @return: string
        '''
        return self.set_other_type
    # EOF: def GetOtherType

    #############################################################################################
    # FUNC     : def SetLevel
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Set level 
    def SetLevel(self, l="info"):
        '''
        Set level.
        
        @type l: string
        @param l: Set level.
        '''
        if l == "":
            self.set_level = "info"
        elif l == "info":
            self.set_level = "info"
        elif l == "low":
            self.set_level = "low"
        elif l == "med":
            self.set_level = "med"
        elif l == "high":
            self.set_level = "high"
        elif l == "safe":
            self.set_level = "safe"
        else:
            self.set_level = "info"
    # EOF: def SetLevel

    ###################################################
    # Plugins Public Functions
    ###################################################
        
    #############################################################################################
    # FUNC     : def InitAnalyzer
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Init func inside plugin
    def InitAnalyzer(self):
        '''
        Init function inside plugin.
        '''
        pass
    # EOF: def InitAnalyzer
    
    #############################################################################################
    # FUNC     : def BeginAnalyzer
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Begin func inside plugin
    def BeginAnalyzer(self,obj):
        '''
        Begin function inside plugin.
        
        @type obj: obj
        @param obj: Set object.
        '''
        pass
    # EOF: def BeginAnalyzer
        
#############################################################################################
# FUNC     : PassivePluginsManager class
# PARAMS   : ...
# RETURN   : ...
# ABSTRACT : PPA Manager class
class PassivePluginsManager:
    '''
    PassivePluginsManager class
    '''
    
    #############################################################################################
    # FUNC     : def __init__
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Init func
    def __init__(self):
        '''
        Init function.
        '''
        self.plugin_path = "ppa_plugins"
        self.iClear()
    # EOF: def __init__
    
    #############################################################################################
    # FUNC     : def iClear
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Clear PPA internal variables
    def iClear(self):
        '''
        Clear PPA internal variables.
        '''
        self.DeletePlugins()
        self.modulename = None
        self.filename = None
    # EOF: def iClear

    #############################################################################################
    # FUNC     : def GetPlugins
    # PARAMS   : ...
    # RETURN   : list
    # ABSTRACT : Return plugin list
    def GetPlugins(self):
        '''
        Return plugin list.
        
        @return: list
        '''
        self.plugin_list.sort()
        return self.plugin_list
    # EOF: def GetPlugins
        
    #############################################################################################
    # FUNC     : def DeletePlugins
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Delete plugin list
    def DeletePlugins(self):
        '''
        Delete plugin list.
        '''
        self.plugin_list = []
    # EOF: def DeletePlugins
    
    #############################################################################################
    # FUNC     : def AddPlugin
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Add a plugin to list
    def AddPlugin(self,a):
        '''
        Add plugin to list.
        
        @type a: string
        @param a: Plugin to be added.
        '''
        self.plugin_list.append(a)
    # EOF: def AddPlugin
        
    #############################################################################################
    # FUNC     : def DeletePlugin
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Delete a plugin from list
    def DeletePlugin(self,d):
        '''
        Delete plugin from list.
        
        @type d: string
        @param d: Plugin to be deleted.
        '''
        remove = self.plugin_list.remove
        remove(c for c in self.plugin_list if c == d)
    # EOF: def DeletePlugin
        
    #############################################################################################
    # FUNC     : def LoadPlugins
    # PARAMS   : ...
    # RETURN   : ...
    # ABSTRACT : Load Plugins
    def LoadPlugins(self):
        '''
        Load plugins.
        '''
        adder = self.plugin_list.append
        for file in os.listdir(self.plugin_path):
            if file.endswith('.py'):
                self.modulename = file[:-3]
                adder(self.modulename)
    # EOF: def LoadPlugins
        
    #############################################################################################
    # FUNC     : def SetPluginPath
    # PARAMS   : string
    # RETURN   : ...
    # ABSTRACT : Set plugin path
    def SetPluginPath(self, p):
        '''
        Set plugin path.
        
        @type p: string
        @param p: plugin path.
        '''
        self.plugin_path = p
    # EOF: def SetPluginPath
    
    #############################################################################################
    # FUNC     : def GetPuglinPath
    # PARAMS   : ...
    # RETURN   : string
    # ABSTRACT : Return plugin path
    def GetPuglinPath(self):
        '''
        Return plugin path.
        
        @return: string
        '''
        return self.plugin_path
    # EOF: def GetPuglinPath
    
    #############################################################################################
    # FUNC     : def GetPluginData
    # PARAMS   : string
    # RETURN   : list or None
    # ABSTRACT : Get plugin info data
    def GetPluginData(self, fname):
        '''
        Get plugin info data.
        
        @type fname: string
        @param fname: Plugin.
        @return: list or None
        '''
        file, filename, description = imp.find_module(fname, [self.plugin_path])
        pluginmodule = imp.load_module(fname, file, filename, description)
        if hasattr(pluginmodule, 'plugin_info'):
            try:
                dat = pluginmodule.plugin_info
                return dat
            except Exception, e:
                    return None
        else:
            return None
    # EOF: def GetPluginData

    #############################################################################################
    # FUNC     : def GetPluginClass 
    # PARAMS   : string, string
    # RETURN   : plugin class or None
    # ABSTRACT : Get plugin class
    def GetPluginClass(self, fname, cn):
        '''
        Get plugin class.
        
        @type fname: string
        @param fname: Plugin.
        @type cn: string
        @param cn: Plugin.
        @return: plugin class or None
        '''
        file, filename, description = imp.find_module(fname, [self.plugin_path])
        pluginmodule = imp.load_module(fname, file, filename, description)
        if hasattr(pluginmodule, cn):
            try:
                return getattr(pluginmodule,cn)
            except Exception, e:
                return None
        else:
            return None
    # EOF: def GetPluginClass
    
    #############################################################################################
    # FUNC     : def CheckTag
    # PARAMS   : string
    # RETURN   : value
    # ABSTRACT : Check if plugin tag is valid    
    def CheckTag(self, tag):
        '''
        Check if plugin tag is valid.
        
        @type tag: string
        @param tag: Tag to be checked.
        @return: int. 1 on success, 0 on fail
        '''
        tag_list = ['recon','ssl','email','script','form','auth_form','cookie','session_id','external_link','comment','vuln','hidden','object','postauth','querystr','vuln_data','auth']
        if tag in tag_list:
            return 1
        else:
            return 0
    # EOF: def CheckTag

#############################################################################################
# Plugins Classes
#############################################################################################

#############################################################################################
# FUNC     : 
# PARAMS   : 
# RETURN   : 
# ABSTRACT : 
class PluginsManager:
    '''
    Plugin Manager class.
    '''

    #############################################################################################
    # FUNC     : def __init__
    # PARAMS   : 
    # RETURN   : 
    # ABSTRACT : 
    def __init(self):
        pass
    # EOF: def __init__

# RL+I EOF
