/***************************************************************************
 *   Copyright (C) 2004 by Juergen Thies                                   *
 *   layout@juergenthies.de                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef SETUP_H
#define SETUP_H

#include <qobject.h>
#include "layers.h"
#include <qcolor.h>
#include <qstring.h>
#include <qnamespace.h>
#include <QTime>
#include <QTranslator>
#include <QTextCodec>

#ifdef SCHEMATIC
class schematic;
#endif
#ifdef TEXTEDIT
class textEdit;
#endif
class QSettings;
/**
@author Juergen Thies
*/

class layout;
//! setup class
/*!
In this call you can modifie the basic setup.
*/
class setup : public QObject
{
Q_OBJECT
public:
    setup(QObject *parent = 0, const char *name = 0);

    ~setup();
//! mouse view
/*!
The type of the mouse in the drawing windows can be set:<br>
<table class="parameter">
<tr>
<th>value</th> <th>mouse</th>
<tr><td>0</td><td>@image "mousepoint.png" single point </td> </tr>
<tr><td>1</td><td>@image "mousecross.png" litle cross</td> </tr>
<tr><td>2</td><td>@image "mouselines.png" cross over drawing</td> </tr>
<tr><td>3</td><td>@image "mousedualcross.png" dual cross</td> </tr>
</table>
*/
 static int mouseView;
//! show reports
/*!
Situation when a report is shown automaticly.
<table  class="parameter">
<tr>
<th>value</th> <th>show report on</th>
<tr><td>-1</td><td>never</td> </tr>
<tr><td>0</td><td>fatal error</td> </tr>
<tr><td>1</td><td>error</td> </tr>
<tr><td>2</td><td>strong warnings</td> </tr>
<tr><td>3</td><td>warnings</td> </tr>
<tr><td>4</td><td>information</td> </tr>
<tr><td>4</td><td>always</td> </tr>
<tr><td>
</table>
*/
 static int showReport;
//! show grid
/*!
If true, a grid is painted in the drawing.
*/
 static bool showGrid;
//! font
/*!
Path and file of the fontfile for text to polygon conversion
@see @link "Setup Font"
*/
 static QString font;
//! font iteration
/*!
The iteration depth of bezier elements in fonts for convertion to polygons. The number of additional points can be calculated by (2^defaultBezierIteration)-1.
@see @link "Setup Font"
*/
 static int fontIteration;
//! font size
/*!
size in pixel of documenting text in the drawing, e.g. ruler text, etc.
@see @link "Setup Font"
*/
 static int fontSize;
//! display text
/*!
...
*/
 static bool displayText;
//! cell names
/*!
If true, the name of the cells are display in cellrefs.
@see @link "View Cellrefs"
*/
 static bool displayCellName;
//! display cells
/*!
If true, the complete cells are display in cellrefs.
@see @link "View Cellrefs"
*/
 static bool displayCells;
 //! display cell origin
/*!
If true, the origin of cell ref will be displayed.
@see @link "View Cellrefs"
*/
 static bool displayCellOrigin;
//! depth of cellrefs
/*!
The reference depth of the displayed cells. If is set to 0, all cellref are displayed.
@see @link "View Cellrefs"
*/
 static int depthOfCellrefs;
//! style of the layer dock
/*!
style of the layer dock, 0 for automatic
*/
 static int layerDockStyle;
//! show menu icon
/*!
If true, icon are shown next the menu entries.
@see @link "View Options"
*/
 static bool showMenuIcons;
 static QColor backgroundColor;
 static QColor mouseColor;
 static QColor gridColor;
 static QColor zeroColor;
 static QColor selectColor;
 static QColor highlightColor;
 static QColor wireColor;
 static QColor busColor;
 static QColor pinColor;
 static QColor textColor;
 static QColor deviceColor;
 static QColor instanceColor;
 static int highlightBrush;
//! set the backgroundcolor
/*!
The bachgroundcolor is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setBackgroundColor(int red, int green, int blue);
//! set the mousecolor
/*!
The mousecolor is set. The mousepoint is painted in this color.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setMouseColor(int red, int green, int blue);
//! set the gridcolor
/*!
The gridcolor is set. The gridpoint are painted in this color.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setGridColor(int red, int green, int blue);
//! set the zerocolor
/*!
The zerocolor is set. In the zerocolor the zeropoint of the coordinatessystem is painted.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setZeroColor(int red, int green, int blue);
//! set the selectcolor
/*!
The selectcolor is set. All select element/points are painted in this color.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setSelectColor(int red, int green, int blue);
//! set the highlightcolor
/*!
The highlightcolor is set. The highlichtcolor is used in the select and deselect modes to highlight the element/point to be selected with a mouse click. 
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setHighlightColor(int red, int green, int blue);
//! set the highlightbrush
/*!
The highlightbrush is set. 
The same brushes as for layer are possible
*/
 static void setHighlightBrush(int brush);
 static void saveSettings(layout *l);
 static void loadSettings(layout *l);
 static void loadSettings();
 static void toggleShowGrid();
// static void repaint();
//! default shape constraint
 static int defaultShapeConstraint;
//! circle default
/*!
The angle pro point for conversion of circles to polygons.
@see @link "Setup Angle Resolution"
*/
 static double circularDefault;
//! path width
/*!
The default width of path elements.
@see @link "Path Defaults"
*/
 static int defaultPathWidth;
//! path cap
/*!
The default cap of path elements.
<table  class="parameter">
<tr>
<th>value</th> <th>cap</th>
<tr><td>0</td><td>flat ended caps</td> </tr>
<tr><td>1</td><td>round caps</td> </tr>
<tr><td>2</td><td>square ended caps</td> </tr>
</table>
@see @link "Path Defaults"
*/
 static int defaultPathCap;
//! text presentation
/*!
The presentation of text elements
<table  class="parameter">
<tr>
<th>value</th> <th>position</th>
<tr><td>0</td><td>left aligned</td> </tr>
<tr><td>1</td><td>horizontal center</td> </tr>
<tr><td>2</td><td>right aligned</td> </tr>
<tr><td>4</td><td>bottom</td> </tr>
<tr><td>8</td><td>vertical centered</td> </tr>
<tr><td>16</td><td>top</td> </tr>
</table>
@see @link "Text Defaults"
*/
 static int defaultTextPresentation;
//! text width
/*!
The default width of text elements.
@see @link "Text Defaults"
*/
 static int defaultTextWidth;
//! bezier
/*!
The iteration depth of bezier elements. The number of additional points can be calculated by (2^defaultBezierIteration)-1.
@see @link "Setup Bezier Iteration"
*/
 static int defaultBezierIteration;
//! cif circle
/*!
If true, regular polygon are saved as circle in cif files.
@see @link "CIF: save regular polygon as circles"
*/
 static bool cifSaveCircle;
//! cif top cell
/*!
If true, one top cell is marked as the main cell of the design. If more than one top cell exists, the top cell with the most shapes is used. Some CIF viewer need this option to avoid a warnings.
*/
 static bool cifTopCell;
//! dxf circle
/*!
If true, regular polygon are saved as circle in dxf files.
@see @link "DXF: save regular polygon as circles"
*/
 static bool dxfSaveCircle;
//! oasis circle
/*!
If true, regular polygon are saved as circle in oasis files.
@see @link "OASIS: save regular polygon as circles"
*/
 static bool oasisSaveCircle;
//! oasis Ctrapezoid
/*!
If true, suitable polygon are saved as ctrapezoid in oasis files.
@see @link "using Ctrapezoids/Trapezoids"
*/
 static bool oasisSaveCtrapezoid;
//! oasis Trapezoid
/*!
If true, suitable polygon are saved as trapezoid in oasis files.
@see @link "using Ctrapezoids/Trapezoids"
*/
 static bool oasisSaveTrapezoid;
 //! oasis map layer
/*!
If true, the layers are mapped during OASIS file load/file save.
@see @link "Handling of Layers in OASIS"
*/
 static bool oasisMapLayer;
 //! oasis cblock
/*!
If true, compressed blocks are used to write oasis files
*/
 static bool oasisSaveCBlock;
 //! auto disable unused layers
/*!
If true, all layers not used in the current design are disabbled atfer a loading
*/
static bool autoDisableUnusedLayers;
 //! gds auto map layers / datatypes
/*!
If true, any layer/datatype combination in a gds file is maped to an other layer
*/
static bool gdsAutoMapDatatypes;
 //! oasis auto map layers / datatypes
/*!
If true, any layer/datatype combination in a oasis file is maped to an other layer
*/
static bool oasisAutoMapDatatypes;

 //! maximal number of points
/*!
The maximal number of points for polygons and path elements can be set. If elements have more points, they are splited during the gds save into two or more elements.
The technical limit of the GDS file format is 8191. In early GDS versions this number was limited to 200 points. 
@see @link "Set Maximal Number of Points in Polygons"
*/
 static int gdsMaxPoints;
 //! gds multi xy records
/*!
The gds file format is limited to 8191 point per record. So polygons are limited to the number of points as well. As a workaround it is possible to save multi record for the points. This is not part of the gds standard, but supported by some viewers. If false, all poygons are splitted into a couple of polygon to store all information.
*/
static bool gdsMultiXYRecord;
 //! gds padding
/*!
If true, the length of stored gds files is extended to N*2048 Bytes by add 0 to the end of the file. Some tape storages and old system may need this opention to handle the files.
@see @link "Padding the Filesize to N*2048"
*/
 static bool gdsPad2048;
 //! gds map layer
/*!
If true, the layers are mapped during GDSII file load/file save.
@see @link "Handling of Layers in GDS II"
*/
 static bool gdsMapLayer;
 //! gds box to polygon
/*!
If true, all boxes are saved as polygons
@see @link "Save Box as Polygon"
*/
 static bool gdsBoxToPolygon;
//! gds text to polygon
/*!

*/
 static bool gdsTextToPolygon;
//! gds text to polygon default width
 static int gdsTextToPolygonDefaultWidth;
//! openaccess callback path
/*!
see wiki doc
*/
static QString openAccessCallbackPath;
//! LEF Cell Boundary Layer
/*!
Layer on which a norminal bounding box of the cell will be added.  In LEF file the norminal bounding box must not be identical with the real box dimensions.
@see @link "Fileformat LEF" @link "Fileformat DEF"
*/
 static int lefCellBoundaryLayer;
//! Alliance physical format lamda value
/*!
Landa value for the preview of the Alliance physical format.
@see @link "Fileformat Alliance"
*/
 static double apLamda;
//! source database 
/*!
Database units for the source format
@see @link "Fileformat SOURCE"
*/
 static double sourceDatabase;
//! Pixel file import/export resolution
/*!
The resolution of pixel import/export in pixel pre meter. If zero the userunits are used as resolution. If -1, it try to read the resolution from the imported file.
*/
 static int pixelResolution;
//! Pixel file import change colors
/*!
If a pixel file is imported, boxes are added to the layer with the closest color to the original pixel color. By activating this option all layer colors above the entered value are adapted for a optimal display.
Set to -1 to disable any adaption.
*/
 static int pixelLayerMap;
//! color resolution for pixel
/*!
The color resolution for pixel import can be set with this value. Values from 1 to 8 bit are possible. A value of 4 bit equals (2^4 )^ 3 = 4096 colors.
Lower value will result in a better performance. Higher value in a better quality.
*/
static int pixelColorBits;

 //! Lasi format lamda value
/*!
Conversionfactor, if tld/tlc file are saved in lamda values. Value in m.
@see @link "Fileformat Lasi"
*/
 static double tldLamda;
 //! CVS format Field Delimiter
/*!
Field delimiter for the csv file format import/export.
@see @link "Fileformat CSV"
*/
 static QString csvFieldDelimiter;
 //! End of Line Chars
/*!
Character use for EOF in PS/EPS format: 0 for LFCR, 1 for LF
@see @link "Fileformat EPS"
*/
 static int epsEndOfLine;
 //! scaling for EPS
/*!
Additional scaling for PS/EPS files.
@see @link "Fileformat EPS"
*/
 static double epsScale;
 //! End of Line Chars
/*!
Character use for EOF in SVG format: 0 for LFCR, 1 for LF
@see @link "Fileformat SVG"
*/
 static int svgEndOfLine;
//! scroll direction
/*!
sets the scroll direction.
@see @link "View Options"
*/
 static bool scrollDirection;
//! update check
/*!
enables regular checks for updates
*/
 static bool updateCheck;
 static QDate lastUpdateCheck;
//! remove not needed points
/*!
Points which have no effect in a polygon are removed. e.g. three point in a line, the middle point is removed.
@see @link "Not needed Points"
*/
 static bool defaultPolygonRemoveNotNeededPoints;
//! fix selfintersection polygons
/*!
Selfintersecting polygons (polygons where two boaderlines cross) are fixed automaticly.
@see @link "Selfintersecting Polygons"
*/
 static bool defaultPolygonFixSelfintersectingPolygons;
//! paint engine
/*!
The paint engine defines how the elements are displayed on the screen. Modifications will take affect after program restart.
<table  class="parameter">
<tr>
<th>value</th> <th>engine</th>
<tr><td>0</td><td>overlay: Elements are drawn over each other.</td> </tr>
<tr><td>1</td><td>transparent: Elements are semi transparent.</td> </tr>
<tr><td>2</td><td>stacked: Elements are painted in the order as used for the 3D view (using layer.level3d).</td> </tr>
<tr><td>
</table>
@see @link "Paint Engine"
*/
 static int paintEngine;
 static bool transparentWhiteEngine(){ if ((backgroundColor.red()+backgroundColor.blue()+backgroundColor.green())>400) return true; return false;  };
//! netlist: auto trigger
/*!
If true, netlist are automaticly extracted after any modification of the design.
@see @link "Autotrigger of building Connections/Netlists"
*/
 static bool netlistAutotrigger;
//! netlist: unnamed nodes
/*!
Prefix for node names, if not named via the text element.
@see @link "Unnamed Nodes"
*/
 static QString netlistUnnamedNodes;
//! netlist: global nodes
/*!
Regular expression for nodes, which should be defected as global.
@see @link "Global Nodes"
*/
 static QString netlistGlobalNodes;
//! set the wirecolor
/*!
The wirecolor for the schematic is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setWireColor(int red, int green, int blue);
//! set the buscolor
/*!
The wirecolor for the schematic is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setBusColor(int red, int green, int blue);
//! set the buscolor
/*!
The default text color for the schematic is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setTextColor(int red, int green, int blue);
//! set the buscolor
/*!
The default device color for the schematic is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setDeviceColor(int red, int green, int blue);
//! set the buscolor
/*!
The default instance color for the schematic is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setInstanceColor(int red, int green, int blue);
//! set the buscolor
/*!
The default pin color for the schematic is set.
@param red	red channel
@param green	green channel
@param blue	blue channel
*/
 static void setPinColor(int red, int green, int blue);
//! set netlistformat
/*!
sets the decription of the netlistformat. This expressions will be processed:

@param -			<b>global parameters:</b>
@param $groundnode(name)	:how the ground node will be named. If not defined, 0 is used 
@param $model(type)	  	:all models of the sheet
@param $netlist(type)	     	:netlist of a sheet
@param $filename	     	:filename

@param -			<b>sheet specific:</b>
@param $node(port) 	     	:node with is connected with 'port' of each device
@param $ports(format)     	:output non global ports of this sheet $portname outputs its name.
@param $allports(format)     	:output all ports of this sheet $portname outputs its name.
@param $globalports(format)	:output all global ports of this sheet $portname outputs its name.
@param $sheetname	     	:name of the sheet

@param -			<b>placement spectific:</b>
@param $parameters(format)  	:output all component paremeter in that format $parametername and $parametervalue can be used
@param $devicename	     	:devicename
@param $name_of_the_parameter 	:value of "name_of_the_parameter"
@param $parametername_num   	:name of the parameter in position 'num'
@param $librarypath         	:path of the library the component belongs to
*/
 static void setNetlistFormat(QString type, QString value);
//! get netlistformat
/*!
@return the decription of the netlistformat
*/
 static QString getNetlistFormat(QString type);
//! netlist formats
/*!
@return a list with all defined netlist formats
*/
 static QStringList netlistFormats();
 //! set netlist sub format
/*!
sets the decription of hierarchical components in the netlist will be defined. 
@see set netlistformat
*/
 static void setNetlistSubFormat(QString type, QString value);
//! get netlist sub format
/*!
@return the decription of hierarchical components in the netlist
*/
 static QString getNetlistSubFormat(QString type);
 //! set netlist sub model format
/*!
sets the decription of hierarchical components in the model section of netlist will be defined. 
@see set netlistformat
*/
 static void setNetlistSubModelFormat(QString type, QString value);
//! get netlist sub model format
/*!
@return the decription of hierarchical components in the model section of netlist
*/
 static QString getNetlistSubModelFormat(QString type);
//! full license
/*!
@return true if a full license is installed
@nomacro
*/ 
 static bool fullLicense();
//! set license
/*!
sets the license to s
@nomacro
*/ 
 static void setLicense(QString s);
 static QString documentationDirectory;
 static QString macroDirectory;
 static QString windowStyle;
 static QString language;
 static QString readStyle();
 static QMap<QString,QVariant> netlistFormat;
 static QMap<QString,QVariant> netlistSubFormat;
 static QMap<QString,QVariant> netlistSubModelFormat;
 private:
 static bool loaded;
 static bool initDone;
 static void checkDirs();
 public:
 static QString version;
 static int numOpen;
 static int getViewMode();
 static void setViewMode(int mode);
 static QString getLicenseType();
 static void setLicenseType(QString mode);
 static int getNumOpen();
 static QTime centralTimer;
 static void preShowMessage();
 static void releaseNotes();
 static void setOrganization();
 static void setOldOrganization();
#ifdef SCHEMATIC
 static void saveSettings(schematic *l);
 static void loadSettings(schematic *l);
#endif
#ifdef TEXTEDIT
 static void saveSettings(textEdit *l);
 static void loadSettings(textEdit *l);
#endif
 static void init();
 static bool schematicSettingsLoaded;
 static void loadSchematicSettings(QSettings *s);
 
 static QTranslator t,tqt,tqsci;
 static QString currentLang;
 static void setLang(QString);
 static QString preferedLanguage;
 static QString languageToShort(QString);
 static QString languageToLong(QString);
 static QStringList translatorPath();
 static QString extraTranslator;
 // undocmunented for macros
 static void setCodec(QString s){ QTextCodec::setCodecForCStrings(QTextCodec::codecForName(s.toAscii()));};
  private:
 static void loadEarlySettings();
  public:
 static bool nonGPL;
 static void ngpl();
};

#endif
