#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <stdio.h>


//Work around windows gl.h define problem for msvc
#if !defined(APIENTRY) && defined(WIN32)
        #define APIENTRY __stdcall
#endif 
#if !defined(WINGDIAPI) && defined(WIN32)
        #define WINGDIAPI __declspec(dllimport)
#endif

#ifdef WIN32
        #include <float.h>  // for _finite
        #define finite _finite
#endif

#if  (__APPLE__) || defined(MACOSX)
#include <OpenGL/gl.h>
#include <GLUT/glut.h>
#else
#include <GL/gl.h>
#include <GL/glut.h>
#endif
#include "GL_Ldraw.h"

#define NUM_COLOURS 15
static GLfloat colours[NUM_COLOURS*3] = {
0.3, 0.3, 0.3, /* Ground color */
0.8, 0.4, 0.4, /* Red */
0.8, 0.8, 0.4, /* Yellow */
0.4, 0.8, 0.4, /* Green */
0.4, 0.8, 0.8, /* Cyan */
0.4, 0.4, 0.8, /* Blue */
0.8, 0.4, 0.8, /* Magenta */
0.2, 0.5, 0.2, /* Dark green */
0.2, 0.5, 0.5, /* Dark cyan */
0.2, 0.2, 0.5, /* Dark blue */
0.5, 0.2, 0.5, /* Dark magenta */
0.6, 0.2, 0.2, /* Brown */
0.5, 0.5, 0.5, /* Dark Grey */
0.7, 0.7, 0.7, /* Grey */
0.9, 0.9, 0.9, /* White */
};

#define R_MOD1 18000 /* could also use 30903 */ 
/* simple random number generating routine, taken from *
 * http://www.helsbreth.org/random/unbiased.html       */
#define F_RND(S) ((float) (((S) = R_MOD1*((S)&0xffff)+((S)>>16))&0xffff))
static float half_max_rnd = 0xffff/2.0;

unsigned int vertex_alloc_step = 3;

struct plist {
  struct plist *next;
  GLfloat *data;
  unsigned long size;
  unsigned long size_used;
};

struct defs {
  GLfloat scale;
  GLfloat length;
  GLfloat angle;
  GLfloat half_thick;
  GLfloat *colour;
};

void draw_stick(GLfloat half_width, GLfloat length) {
  static GLfloat n[4][3] = {
    {-0.707106781187f, -0.707106781187f, 0.0f},
    {0.707106781187f,  -0.707106781187f, 0.0f},
    {0.707106781187f,  0.707106781187f,  0.0f},
    {-0.707106781187f, 0.707106781187f,  0.0f}};
  static GLfloat v[8][3]= { /*just use 4.0f as a placeholder */
    {-4.0f, -4.0f, 0.0f}, {-4.0f, -4.0f, 1.0f},
    {4.0f,  -4.0f, 0.0f}, {4.0f,  -4.0f, 1.0f},
    {4.0f,   4.0f, 0.0f}, {4.0f,   4.0f, 1.0f},
    {-4.0f,  4.0f, 0.0f}, {-4.0f,  4.0f, 1.0f}};

  v[0][0] = v[0][1] = v[1][0] = v[1][1] = v[2][1] = v[3][1] = v[6][0] = v[7][0] = -half_width;
  v[2][0] = v[3][0] = v[4][0] = v[4][1] = v[5][0] = v[5][1] = v[6][1] = v[7][1] = half_width;
  v[1][2] = v[3][2] = v[5][2] = v[7][2] = length;

  glBegin(GL_QUAD_STRIP);
  glNormal3fv(n[0]); glVertex3fv(v[0]); glVertex3fv(v[1]);
  glNormal3fv(n[1]); glVertex3fv(v[2]); glVertex3fv(v[3]);
  glNormal3fv(n[2]); glVertex3fv(v[4]); glVertex3fv(v[5]);
  glNormal3fv(n[3]); glVertex3fv(v[6]); glVertex3fv(v[7]);
  glNormal3fv(n[0]); glVertex3fv(v[0]); glVertex3fv(v[1]);
  glEnd();
  glBegin(GL_QUADS);
  glNormal3fv(n[0]); glVertex3fv(v[0]);
  glNormal3fv(n[1]); glVertex3fv(v[2]);
  glNormal3fv(n[2]); glVertex3fv(v[4]);
  glNormal3fv(n[3]); glVertex3fv(v[6]);
  glNormal3fv(n[3]); glVertex3fv(v[7]);
  glNormal3fv(n[2]); glVertex3fv(v[5]);
  glNormal3fv(n[1]); glVertex3fv(v[3]);
  glNormal3fv(n[0]); glVertex3fv(v[1]);
  glEnd();
}

void record_bounds(GLfloat *m, GLfloat *bounds)
{
  if (m[12]<bounds[0]) bounds[0]=m[12];
  if (m[12]>bounds[1]) bounds[1]=m[12];
  if (m[13]<bounds[2]) bounds[2]=m[13];
  if (m[13]>bounds[3]) bounds[3]=m[13];
  if (m[14]<bounds[4]) bounds[4]=m[14];
  if (m[14]>bounds[5]) bounds[5]=m[14];
}

GLboolean push_pos_to_pstack(struct plist *p, GLfloat *colour, GLfloat *m)
{
  /* when calling this func, ensure that p is never NULL */
  GLfloat *tmp;
  
  /* check if more mem needed */
  if (p->size == p->size_used) {
    if ((tmp = (GLfloat *) realloc(p->data,
	        sizeof(GLfloat)*(p->size+vertex_alloc_step*6)))){
      p->data = tmp;
      p->size += vertex_alloc_step*6;
    } else {
      return GL_FALSE;
    };
  };

  p->data[p->size_used++]=m[12];/* m[12], m[13], m[14] are the         */
  p->data[p->size_used++]=m[13];/* co-ordinates of (0,0,0)             */
  p->data[p->size_used++]=m[14];/* transformed by the modelview matrix */
  p->data[p->size_used++]=colour[0];
  p->data[p->size_used++]=colour[1];
  p->data[p->size_used++]=colour[2];
	  
  return GL_TRUE;
}

GLboolean has_param(const char **c_pp, GLfloat *param) {
  //extract bracketed number from string
  //return GL_TRUE if there is a param, GL_FALSE if there isn't
  char *end_ptr;
  if ((*c_pp+1=='\0') || ((*(*c_pp+1)) != 40)) 
    return GL_FALSE; //next char is not '(' (char number 40)
  
  if (*c_pp+2=='\0') {
    *c_pp+=1;
    return GL_FALSE;
  };
  *param=strtod(*c_pp+2, &end_ptr);
  if (*c_pp+2 == end_ptr) {
    //no conversion: search for closing bracket
    *c_pp = strchr(end_ptr, 41); //set *c_pp to NULL if no closing bracket found
    return GL_FALSE;
  } else {
    //conversion: search for closing bracket
    *c_pp = strchr(end_ptr, 41);
    return GL_TRUE;
  };
}


int Ldraw_GL(const char *Lstring, GLfloat def_angle, GLfloat def_thick, unsigned long max_vertices, unsigned long* rseed, GLfloat *bounds)
{
  /* returns 0 on success, 1 for incomplete, -1 for out of memory, -2 for out of stack, or -3 for bad parameters */

  const char    *c_ptr;
  int   constrained=0, i, max, retval=0;
  unsigned long defrand=1;
  float scale_modifier;
  double  n;
  GLfloat readval, z_axis[3], tmp_z[3], tmp_x[3], vec[3], m[16], m_in[16], *v1, *v2, *v3;
  struct plist  *tmp_p, *current_poly=NULL;
  static GLint stack_depth=0;
  static struct defs *defs_stack=NULL;
  struct defs *current_defaults;
  long p1=0,p2=0;

  if (defs_stack == NULL) {
    glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, &stack_depth);
    defs_stack = (struct defs *) malloc(sizeof(struct defs) * (stack_depth));
    if (defs_stack == NULL)
      return -1;
  };
  
  if (max_vertices)
    constrained=1;

  if (rseed==NULL)
    rseed = &defrand;

  current_defaults               = defs_stack;
  current_defaults->scale        = 0.01f;
  scale_modifier                 = 1.0f;
  current_defaults->half_thick   = def_thick/2.0f;
  current_defaults->angle        = def_angle;
  current_defaults->colour       = colours;

   glMatrixMode(GL_MODELVIEW);
  glGetFloatv(GL_MODELVIEW_MATRIX, m); /* get the base matrix */  
  
  z_axis[0] = m[8];
  z_axis[1] = m[9];
  z_axis[2] = m[10];
  n = sqrt(z_axis[0]*z_axis[0]+z_axis[1]*z_axis[1]+z_axis[2]*z_axis[2]);
  z_axis[0]/=n; z_axis[1]/=n; z_axis[2]/=n; /* normalise */
  
  if (bounds) {
    bounds[0]=bounds[1]=m[13];
    bounds[2]=bounds[3]=m[14];
    bounds[4]=bounds[5]=m[15];
  };

  glClearStencil(0);
  glColor3fv(current_defaults->colour);
  c_ptr = Lstring;
  while (c_ptr++ && *c_ptr != '\0') {
    switch (*c_ptr) {
      /************ Movement Commands *************/
    case 70:  /* F */
      if ((constrained) && (max_vertices<8)) {
	c_ptr=NULL;
	retval=1;
      } else {
	max_vertices -= 8;
	if (has_param(&c_ptr, &readval)) {
	  draw_stick(current_defaults->scale*scale_modifier*current_defaults->half_thick, current_defaults->scale*scale_modifier*readval);
	  glTranslatef(0.0f, 0.0f, current_defaults->scale*scale_modifier*readval);
	} else {
	  draw_stick(current_defaults->scale*scale_modifier*current_defaults->half_thick, current_defaults->scale*scale_modifier*100.0f);
	  glTranslatef(0.0f, 0.0f, current_defaults->scale*scale_modifier*100.0f);
	};
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	if (bounds) record_bounds(m,bounds);
	if (current_poly)
	  if (push_pos_to_pstack(current_poly, current_defaults->colour, m)==GL_FALSE) {
	    retval = -1;
	    c_ptr = NULL;
	  };
      };
      break;
		
    case 102: /* f */
      if (has_param(&c_ptr, &readval)) {
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*readval);
      } else {
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*100.0f);
      };
      if (current_poly) {
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	if (bounds) record_bounds(m,bounds);
	if (push_pos_to_pstack(current_poly, current_defaults->colour, m) == GL_FALSE) {
	  retval = -1;
	  c_ptr = NULL;
	};
      };
      break;
      
    case 90:  /* Z */
      if ((constrained) && (max_vertices<8)) {
	c_ptr=NULL;
	retval=1;
      } else {
	max_vertices -= 8;
	if (has_param(&c_ptr, &readval)) {
	  draw_stick(current_defaults->scale*scale_modifier*current_defaults->half_thick, current_defaults->scale*scale_modifier*readval);
	  glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*readval);
	} else {
	draw_stick(current_defaults->scale*scale_modifier*current_defaults->half_thick, current_defaults->scale*scale_modifier*50.0f);
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*50.0f);
	};
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	if (bounds) record_bounds(m,bounds);
	if (current_poly) {
	  if (push_pos_to_pstack(current_poly, current_defaults->colour, m)==GL_FALSE) {
	    retval = -1;
	    c_ptr = NULL;
	  };
	};
      };
      break;
      
    case 122: /* z */
      if (has_param(&c_ptr, &readval)) {
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*readval);
      } else {
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*50.0f);
      };
      if (current_poly) {
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	if (bounds) record_bounds(m,bounds);
	if (push_pos_to_pstack(current_poly, current_defaults->colour, m)==GL_FALSE) {
	  retval = -1;
	  c_ptr = NULL;
	};
      };
      break;
      
    case 103: /* g */
      if (has_param(&c_ptr, &readval)) {
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*readval);
      } else {
	glTranslatef(0.0f,0.0f, current_defaults->scale*scale_modifier*100.0f);
      };
      break;
		
    case 46: /* . */
      if (current_poly) {
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	if (bounds) record_bounds(m,bounds);
	if (push_pos_to_pstack(current_poly, current_defaults->colour, m)==GL_FALSE) {
	  retval = -1;
	  c_ptr = NULL;
	};
      };
      break;

      /**************** Orientation Commands *************/
    case 43:  /* + */
      if (has_param(&c_ptr, &readval))
	glRotatef(readval, 0.0f, 1.0f, 0.0f);
      else
	glRotatef(current_defaults->angle, 0.0f, 1.0f, 0.0f);
      break;
    case 45:  /* - */
      if (has_param(&c_ptr, &readval) )
	glRotatef(-readval, 0.0f, 1.0f, 0.0f);
      else
	glRotatef(-current_defaults->angle, 0.0f, 1.0f, 0.0f);
      break;
    case 38:  /* & */
      if (has_param(&c_ptr, &readval))
	glRotatef(readval, 1.0f, 0.0f, 0.0f);
      else
	glRotatef(current_defaults->angle, 1.0f, 0.0f, 0.0f);
      break;
    case 94:  /* ^ */
      if (has_param(&c_ptr, &readval))
	glRotatef(-readval, 1.0f, 0.0f, 0.0f);
      else
	glRotatef(-current_defaults->angle, 1.0f, 0.0f, 0.0f);
      break;
    case 60:  /* < */
      if (has_param(&c_ptr, &readval))
	glRotatef(-readval, 0.0f, 0.0f, 1.0f);
      else
	glRotatef(-current_defaults->angle, 0.0f, 0.0f, 1.0f);
      break;
    case 62:  /* > */
      if (has_param(&c_ptr, &readval))
	glRotatef(readval, 0.0f, 0.0f, 1.0f);
      else
	glRotatef(current_defaults->angle, 0.0f, 0.0f, 1.0f);
      break;
      
      /*********** Increment/Decrement Commands  *************/
      
    case 34:  /* "  reduce the scale of everything */
      if (has_param(&c_ptr, &readval))
	current_defaults->scale = current_defaults->scale *= readval;
      else
	current_defaults->scale = current_defaults->scale *= 1.1f;
      break;
    case 39:  /* '  reduce the scale of everything */
      if (has_param(&c_ptr, &readval))
	current_defaults->scale = current_defaults->scale *= readval;
      else
	current_defaults->scale = current_defaults->scale *= 0.9f;
      break;
    case 58: /* : decrement default angle */
      if (has_param(&c_ptr, &readval))
		  current_defaults->angle *= readval;
      else
		  current_defaults->angle *= 0.9;
      break;
    case 59: /* ; increment default angle */
      if (has_param(&c_ptr, &readval))
	current_defaults->angle *= readval;
      else
	current_defaults->angle *= 1.1;
      break;
    case 63: /* ? increment thickness */ 
      if (has_param(&c_ptr, &readval))
	current_defaults->half_thick *= readval;
      else
	current_defaults->half_thick *= 1.4;
      break;
    case 33: /* ! */
      if (has_param(&c_ptr, &readval))
	current_defaults->half_thick *= readval;
      else
	current_defaults->half_thick *= 0.7;
      break;
      
      /************* Colour Command ************/
    case 99: /* c */
      if (has_param(&c_ptr, &readval)) {
	current_defaults->colour = colours+3*(((int) readval)%NUM_COLOURS);
      }  else {
	current_defaults->colour+=3;
	if ((current_defaults->colour - colours) >= NUM_COLOURS*3)
	  current_defaults->colour = colours;
      };
      glColor3fv(current_defaults->colour);
      break;
      

    case 64: /* @  - for fractional recursion levels */
      if (has_param(&c_ptr, &readval)) {
	scale_modifier=readval/100;
      } else {
	scale_modifier=1.0f;
      };

      /************* Special Orientation Commands ************/
    case 126: /* ~ */
      if (has_param(&c_ptr, &readval) == 0)
	readval=20;
      glRotatef((F_RND(*rseed)-half_max_rnd)*readval/half_max_rnd, 0, 0, 1);
      glRotatef((F_RND(*rseed)-half_max_rnd)*readval/half_max_rnd, 0, 1, 0);
      glRotatef((F_RND(*rseed)-half_max_rnd)*readval/half_max_rnd, 1, 0, 0);
      break;
    case 116: /* t     wilt towards horizontal */
      if (has_param(&c_ptr, &readval) == 0)
	readval=0.2;
      
      /* the normal to the base z_axis and the current z-axis gives the 
	 vector around which to rotate. It should have a z component
	 unchanged from the original base */
      glGetFloatv(GL_MODELVIEW_MATRIX, m);
      tmp_z[0] = m[8]; /* axis directions */
      tmp_z[1] = m[9];
      tmp_z[2] = m[10];
      
      /* cross product (z_axis x tmp_z). This finds the normal
	 to the two vectors */
      vec[0] = z_axis[1]*tmp_z[2] - z_axis[2]*tmp_z[1];
      vec[1] = z_axis[2]*tmp_z[0] - z_axis[0]*tmp_z[2];
      vec[2] = z_axis[0]*tmp_z[1] - z_axis[1]*tmp_z[0];
      
      /* dot product (gives the fraction of the amount to rotate) */
      /* normalise (z-axis should already be normalised) */
      n = sqrt(tmp_z[0]*tmp_z[0]+tmp_z[1]*tmp_z[1]+tmp_z[2]*tmp_z[2]);
      tmp_z[0]/=n; tmp_z[1]/=n; tmp_z[2]/=n;
      
      /* find dot product */
      n = tmp_z[0]*z_axis[0]+tmp_z[1]*z_axis[1]+tmp_z[2]*z_axis[2];
      
      /*angle between them is the arccos of the normalised dot product*/
      n=acos(n);
      
      /* if the vectors are coincident, don't do anything */
      if (finite(n))
	glRotatef(readval * n*57.29578, vec[0], vec[1], vec[2]);
      
      break;
      
    case 124: /* | */
      glRotatef(180.0f, 0.0f, 1.0f, 0.0f);
      break;
    case 37:  /* % */
      glRotatef(180.0f, 0.0f, 0.0f, 1.0f);
      break;
    case 36:  /* $ roll until horizontal*/
      glGetFloatv(GL_MODELVIEW_MATRIX, m);
      tmp_z[0] = m[8]; /* axis directions */
      tmp_z[1] = m[9];
      tmp_z[2] = m[10];
      tmp_x[0] = m[0];
      tmp_x[1] = m[1];
      tmp_x[2] = m[2];
      
      /* cross product (z_axis x tmp_z). This finds the vector of the
	 intersection between the base plane and the current base plane*/
      vec[0] = z_axis[1]*tmp_z[2] - z_axis[2]*tmp_z[1];
      vec[1] = z_axis[2]*tmp_z[0] - z_axis[0]*tmp_z[2];
      vec[2] = z_axis[0]*tmp_z[1] - z_axis[1]*tmp_z[0];
      
      /* Now just find the angle between the intersection vector
	 and the current x axis: this is the amount we should rotate */
      
      /* first normalise */
      n = sqrt(vec[0]*vec[0]+vec[1]*vec[1]+vec[2]*vec[2]);
      vec[0]/=n; vec[1]/=n; vec[2]/=n;
      
      n = sqrt(tmp_x[0]*tmp_x[0]+tmp_x[1]*tmp_x[1]+tmp_x[2]*tmp_x[2]);
      tmp_x[0]/=n; tmp_x[1]/=n; tmp_x[2]/=n;
      
      /* find dot product */
      n = tmp_x[0]*vec[0]+tmp_x[1]*vec[1]+tmp_x[2]*vec[2];
      
      /*angle between them is the arccos of the normalised dot product*/
      n=acos(n);
      
      if (finite(n))
	/* this should now rotate around the z axis (convert n to deg) */
	glRotatef(n*57.29578, 0.0f, 0.0f, 1.0f);
      break;
      
      /************* Structure Commands ************/
    case 91: /* [ */
      if (current_defaults-defs_stack < stack_depth) {
	glPushMatrix();
	current_defaults++;
	*current_defaults = *(current_defaults-1);
      } else {
	retval = -2;
	c_ptr = NULL;
      };
      break;
    case 93:  /* ] */
      if (current_defaults!=defs_stack) { 
	glPopMatrix();
	current_defaults--;
	glColor3fv(current_defaults->colour);
	/* def angle referenced each time => doesn't need setting here */
      };
      break;
      
    case 123: /* { */
      /* start recording vertices */
      tmp_p = (struct plist *) malloc(sizeof(struct plist));
      if (tmp_p == NULL) {
	c_ptr = NULL;
	retval = -1;
      } else {
	tmp_p->next = current_poly;
	tmp_p->size = 0;
	tmp_p->size_used = 0;
	tmp_p->data = NULL;
	current_poly = tmp_p;
      };
      break;
      
    case 125: /* } */
      if (current_poly) {
	if (current_poly->data) { /* draw the data, then free mem */
	  /* draw poly */
	  max = current_poly->size_used;
	  if (max >= 18) { /* must have at least 3 points */
	    if ((constrained) && (max_vertices < max/6 )) {
	      c_ptr=NULL;
	      retval=1;
	    } else {
	      max_vertices -= max/6;
	      glPushMatrix();         /* keep current matrix */
	      /* calculate inverse of the current modelview matrix
	       * - it should always be an orthonormal matrix
	       *   so we can use a simple transpose technique */
	      glGetFloatv(GL_MODELVIEW_MATRIX, m);
	      m_in[0]=m[0];
	      m_in[1]=m[4];
	      m_in[2]=m[8];
	      m_in[4]=m[1];
	      m_in[5]=m[5];
	      m_in[6]=m[9];
	      m_in[8]=m[2];
	      m_in[9]=m[6];
	      m_in[10]=m[10];
	      
	      m_in[12]=-(m_in[0]*m[12]+m_in[4]*m[13]+m_in[8]*m[14]);
	      m_in[13]=-(m_in[1]*m[12]+m_in[5]*m[13]+m_in[9]*m[14]);
	      m_in[14]=-(m_in[2]*m[12]+m_in[6]*m[13]+m_in[10]*m[14]);
	      
	      m_in[3]=m[3];
	      m_in[7]=m[7];
	      m_in[11]=m[11];
	      m_in[15]=m[15];
	      
	      //now transform the current matrix to get back to the origin
	      glMultMatrixf(m_in);
	    
	      glBegin(GL_TRIANGLE_STRIP);
	      /* first vertex is 3rd point in list */
	      v2=current_poly->data+12;
	      for (i=0; i < max; i+=6) {
		v1=v2;
	      v2=current_poly->data+i;
	      if (i+6 < max) 
		v3=current_poly->data+i+6;
	      else //last vertex
		v3=current_poly->data+i-12;
	      
	      /* calculate inverse of current modelview matrix */
	      
	      
	      
	      /* store vertex vectors in tmp_z & tmp_x, since they are available */
	      tmp_z[0]=v1[0]-v2[0]; tmp_z[1]=v1[1]-v2[1]; tmp_z[2]=v3[2]-v2[2];
	      tmp_x[0]=v3[0]-v2[0]; tmp_x[1]=v3[1]-v2[1]; tmp_x[2]=v3[2]-v2[2];
	      
	      /* cross product (tmp_z x tmp_x). This finds the normal to the two vectors */
	      vec[0] = tmp_z[1]*tmp_x[2] - tmp_z[2]*tmp_x[1];
	      vec[1] = tmp_z[2]*tmp_x[0] - tmp_z[0]*tmp_x[2];
	      vec[2] = tmp_z[0]*tmp_x[1] - tmp_z[1]*tmp_x[0];
	      
	      /* normalise */
	      n = sqrt(vec[0]*vec[0]+vec[1]*vec[1]+vec[2]*vec[2]);
	      if (n!=0) {
		vec[0]/=n; vec[1]/=n; vec[2]/=n;
	      } else {
		vec[0]=1; vec[1]=0; vec[2]=0; //arbitrary normal
	      };
	      
	      glColor3fv(v2+3);
	      glNormal3fv(vec);
	      glVertex3fv(v2);
	      };
	      glEnd();
	      glPopMatrix();          /* restore current matrix */
	    }; //end else
	  }; //end if max >=18
	  free(current_poly->data);
	};
	tmp_p = current_poly;
	current_poly = current_poly->next;
	free(tmp_p);
      };
      break;
      
    default:
      break;
    };
  }; //end while
  
  
  while (current_poly) {
    if (current_poly->data)
      free(current_poly->data);
    tmp_p = current_poly;
    current_poly = current_poly->next;
    free(tmp_p);
  }; 

  while (current_defaults != defs_stack) { 
    current_defaults--;
    glPopMatrix();
  };

  return retval;
}
