/*$************************************************************************************************
 **
 ** $Id: GraphicStoreFactory.java 1122 2007-11-24 18:49:16Z desruisseaux $
 **
 ** $URL: https://geoapi.svn.sourceforge.net/svnroot/geoapi/tags/2.1.0/geoapi/src/main/java/org/opengis/go/display/primitive/store/GraphicStoreFactory.java $
 **
 ** Copyright (C) 2003-2005 Open GIS Consortium, Inc.
 ** All Rights Reserved. http://www.opengis.org/legal/
 **
 *************************************************************************************************/
package org.opengis.go.display.primitive.store;

import java.io.IOException;
import java.net.URI;
import java.util.Map;
import org.opengis.util.InternationalString;


/**
 * The <code>GraphicStoreFactory</code> class/interface...
 *
 * @author <A HREF="http://www.opengis.org">OpenGIS&reg; consortium</A>
 * @author Jesse Crossley (SYS Technologies)
 */
public interface GraphicStoreFactory {
    /**
     * Ask for a {@code GraphicStore} connecting to the indicated provider or service.
     * The returned {@code GraphicStore} may have been previously cached.
     * <p>
     * Additional hints or configuration information may be provided according
     * to the metadata indicated by {@link #getParametersInfo}. This information
     * often includes security information such as username and password.
     *
     * @param provider Often a URI or JDBC URI locating the service to connect to.
     * @param params Map of hints or configuration information.
     * @return {@code GraphicStore} connected to the indicated provider or service.
     * @throws IOException if the {@code GraphicStore} cannot connect to its source.
     * @throws GraphicStoreException if anything else goes wrong while creating the {@code GraphicStore}.
     */
    GraphicStore createGraphicStore(URI provider, Map<String,Object> params) throws IOException, GraphicStoreException;

    /**
     * Ask for a new {@code GraphicStore} connecting to the indicated provider or service.
     * <p>
     * Additional hints or configuration information may be provided according
     * to the metadata indicated by {@link #getParametersInfo}. This information
     * often includes security information such as username and password.
     *
     * @param provider Often a URI or JDBC URI locating the service to connect to.
     * @param params Map of hints or configuration information.
     * @return {@code GraphicStore} connected to the newly created provider or service.
     * @throws IOException if the {@code GraphicStore} cannot connect to its source.
     * @throws GraphicStoreException if anything else goes wrong while creating the {@code GraphicStore}.
     */
    GraphicStore createNewGraphicStore(URI provider, Map<String,Object> params) throws IOException, GraphicStoreException;

    /**
     * Icon representing this category of <code>GraphicStore</code>s.
     *
     * @return the icon.
     * @todo Assumed to point to a 16x16 icon?
     */
    URI getIcon();

    /**
     * Display name used to communicate this type of {@code GraphicStore} to end users.
     */
    InternationalString getDisplayName();

    /**
     * Description of this type of {@code GraphicStore}.
     */
    InternationalString getDescription();

    /**
     * Gets an {@code Object} array relating to the parameters needed (beyond
     * the URI) to instantiate a {@code GraphicStore}.
     *
     * @todo Should be replaced with a <code>Param</code>[] based on ISO standards (ISO 19119).
     */
    Map<String,Class> getParametersInfo();

    /**
     * Indicates this {@code GraphicStoreFactory} communicate with the indicated provider or service.
     * <p>
     * This method should not fail, if a connection needs to be made
     * to parse a {@code GetCapabilities} file or negotiate WMS versions any
     * IO problems simply indicate the inability to process.
     * </p>
     * <p>
     * This method may be considered the same as:
     * <code>{@linkplain #canProcess(URI,Map) canProcess}(provider, hints)</code>
     * where hints was generated by using all the default values specified by the
     * {@link #getParametersInfo} method.
     * </p>
     * @param provider Provider or Server of spatial information.
     */
    boolean canProcess(URI provider);

    /**
     * Indicates this {@code GraphicStoreFactory} communicate with the indicated provider or service.
     * <p>
     * This method differs from {@link #canProcess(URI)} in that additional configuration
     * information may be supplied.
     * </p>
     * @param provider Provider or Server of spatial information.
     * @param params additional configuration information.
     * @return {@code true} if this factory can communicate with the provider.
     */
    boolean canProcess(URI provider, Map<String,Object> params);

    /**
     * Allows a {@code GraphicStoreFactory} to ensure all its preconditions are met,
     * such as the presense of required libraries.
     *
     * @return {@code true} if available
     */
    boolean isAvailable();
}
