/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.tools.fsa;

import java.lang.reflect.Method;
import java.util.*;

import de.upb.tools.fca.FEmptyIterator;


/**
 * cache "converter" methods from sourcetype to targettype
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.23 $
 */
public class FSATypeConverterCache
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String DEFAULT_CONVERTER_METHOD = "dcm";

   /**
    * the Singleton instance
    */
   private static FSATypeConverterCache theInstance = null;


   /**
    * Default private Constructor
    */
   private FSATypeConverterCache() { }


   /**
    * get the Singleton instance
    *
    * @return   the Singleton instance
    */
   public static FSATypeConverterCache get()
   {
      if (theInstance == null)
      {
         theInstance = new FSATypeConverterCache();
      }

      return theInstance;
   }


   /**
    * methodCache : TreeMap (<key>,<converterMethod>[]) <p>
    *
    * key:String = <sourceClass>.getName():<targetClass>.getName()[:<methodname>] method =
    * public static <targetClass> <methodName> ( <sourceClass> , <targetClass> )
    */
   private TreeMap methodCache = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param sourceClass  No description provided
    * @param targetClass  No description provided
    * @param methodName   No description provided
    * @return             No description provided
    */
   private Method[] findConverterMethod (Class sourceClass, Class targetClass, String methodName)
   {
      Method converterMethods[] = getConverterMethods (sourceClass, targetClass);
      int i = 0;
      int length = converterMethods.length;
      boolean found = false;
      Method converter[] = new Method[1];

      while (i < length && !found)
      {
         if (converterMethods[i].getName().equals (methodName))
         {
            found = true;
            converter[0] = converterMethods[i];
         }
         i++;
      }

      return converter;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param sourceClass  No description provided
    * @param targetClass  No description provided
    * @return             No description provided
    */
   private Method[] findConverterMethods (Class sourceClass, Class targetClass)
   {
      Class[] converterParameters =
         {
         sourceClass
         };
      Class[] modifierParameters =
         {
         sourceClass, targetClass
         };
      Method[] classMethods;
      Method[] converter = null;
      Vector converterMethods = new Vector();
      int i;

      // get all class methods
      try
      {
         classMethods = FSATypeConverterMethods.class.getMethods();
      }
      catch (SecurityException exception)
      {
         // no methods found
         classMethods = new Method[0];
      }

      // find methods
      for (i = 0; i < classMethods.length; i++)
      {
         if (classMethods[i].getReturnType().equals (targetClass) &&
             (Arrays.equals (classMethods[i].getParameterTypes(), converterParameters) ||
            Arrays.equals (classMethods[i].getParameterTypes(), modifierParameters)))
         {
            converterMethods.add (classMethods[i]);
         }
      }

      // TO DO : fix me i'm slow but safe
      // converter = (Method[]) converterMethods.toArray () ;
      converter = new Method[converterMethods.size()];
      for (i = 0; i < converterMethods.size(); i++)
      {
         converter[i] = (Method) converterMethods.elementAt (i);
      }

      return converter;
   }


   /**
    * get default converter method <br>
    * <targetClass> <[sourceClass.name]To[targetClass.name]> ( <sourceClass>)
    *
    * @param targetClass  the target/result class
    * @param sourceClass  the source/parameter class
    * @return             the default method which converts <sourceClass> to <targetClass>
    */
   public Method getDefaultConverterMethods (Class sourceClass,
                                             Class targetClass)
   {
      String key = createMethodCacheKey (sourceClass, targetClass, DEFAULT_CONVERTER_METHOD);

      Method converter[] = getFromMethodCache (key);

      if (converter == null)
      {
         // compute "default" converter method name
         String sourceClassName = FSAUtility.downFirstChar (FSAUtility.getClassName (sourceClass.getName()));
         String targetClassName = FSAUtility.getClassName (targetClass.getName());
         String methodName = sourceClassName + "To" + targetClassName;

         converter = findConverterMethod (sourceClass, targetClass, methodName);

         addToMethodCache (key, converter);
      }

      return converter[0];
   }


   /**
    * get all static methods from FSATypeConverterCache with result=<targetClass> and parameter=
    * <sourceClass>
    *
    * @param targetClass  the target/result class
    * @param sourceClass  the source/parameter class
    * @return             all static method from FSATypeConverterCache <targetClass> aMethodName
    *      ( <sourceClass> )
    */
   public Method[] getConverterMethods (Class sourceClass, Class targetClass)
   {
      String key = createMethodCacheKey (sourceClass,
         targetClass);
      Method converter[] = getFromMethodCache (key);

      if (converter == null)
      {
         converter = findConverterMethods (sourceClass, targetClass);

         addToMethodCache (key, converter);
      }

      return converter;
   }


   /**
    * compute key for methodCache
    *
    * @param methodName   the methodname
    * @param targetClass  the target Class
    * @param sourceClass  the source Class
    * @return             the key
    */
   public String createMethodCacheKey (Class sourceClass,
                                       Class targetClass,
                                       String methodName)
   {
      return sourceClass.getName() + ":" + targetClass.getName()
         + ":" + methodName;
   }


   /**
    * compute key for methodCache
    *
    * @param targetClass  the target Class
    * @param sourceClass  the source Class
    * @return             the key
    */
   public String createMethodCacheKey (Class sourceClass,
                                       Class targetClass)
   {
      return sourceClass.getName() + ":" + targetClass.getName();
   }


   /**
    * @param key  No description provided
    * @return     No description provided
    * @see        #methodCache
    */
   private boolean hasKeyInMethodCache (String key)
   {
      if (methodCache == null)
      {
         return false;
      }
      else
      {
         return methodCache.containsKey (key);
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator iteratorOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.values().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator keysOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.keySet().iterator();
      }
   }


   /**
    * @return   No description provided
    * @see      #methodCache
    */
   private Iterator entriesOfMethodCache()
   {
      if (methodCache == null)
      {
         return FEmptyIterator.get();
      }
      else
      {
         return methodCache.entrySet().iterator();
      }
   }


   /**
    * @param key  No description provided
    * @return     The fromMethodCache value
    * @see        #methodCache
    */
   private Method[] getFromMethodCache (String key)
   {
      Method elem[];

      if (this.hasKeyInMethodCache (key))
      {
         elem = (Method[]) this.methodCache.get (key);
      }
      else
      {
         elem = null;
      }

      return elem;
   }


   /**
    * @param key   The object added.
    * @param elem  The object added.
    * @see         #methodCache
    */
   private void addToMethodCache (String key, Method elem[])
   {
      if (key != null && elem != null)
      {
         if (methodCache == null)
         {
            methodCache = new TreeMap();
         }

         this.methodCache.put (key, elem);
      }
   }


   /**
    * @param key  No description provided
    * @see        #methodCache
    */
   private void removeKeyFromMethodCache (String key)
   {
      if (this.hasKeyInMethodCache (key))
      {
         this.methodCache.remove (key);
      }
   }


   /**
    * @see   #methodCache
    */
   private void removeAllFromMethodCache()
   {
      Iterator iter = keysOfMethodCache();
      String tmpKey = null;
      while (iter.hasNext())
      {
         tmpKey = (String) iter.next();
         iter.remove();
         removeKeyFromMethodCache (tmpKey);
      }
   }

}

/*
 * $Log: FSATypeConverterCache.java,v $
 * Revision 1.23  2004/10/20 17:50:34  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
