/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import javax.swing.*;
import javax.swing.border.BevelBorder;
import javax.swing.filechooser.FileFilter;

import de.uni_paderborn.fujaba.gui.comp.FujabaDialogPanel;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.views.*;


/**
 * Part of the FilterEditorDialog. This is the Panel that holds the editor for the filter
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.11 $
 */
public class FilterEditorPanel extends FujabaDialogPanel implements ActionListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JComboBox filterComboBox = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JScrollPane editorScrollPane = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton setButton = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton saveButton = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton createNewButton = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JButton revertButton = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean globalEditor = false;


   /**
    * Constructor for class FilterEditorPanel
    *
    * @param hostingDialog  No description provided
    */
   public FilterEditorPanel (FilterEditorDialog hostingDialog)
   {
      super();
      setHostingDialog (hostingDialog);
      globalEditor = false;

      guiBuild();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void guiBuild()
   {

      setLayout (new BorderLayout (0, 0));

      BevelBorder loweredBorder = new BevelBorder (BevelBorder.LOWERED);

      JPanel editorPanel = new JPanel();
      editorPanel.setBorder (loweredBorder);
      editorPanel.setLayout (new BoxLayout (editorPanel, BoxLayout.X_AXIS));

      editorScrollPane = new JScrollPane();
      editorScrollPane.setDoubleBuffered (true);
      editorScrollPane.getViewport().setScrollMode (JViewport.BLIT_SCROLL_MODE);

      editorPanel.add (editorScrollPane);
      add (BorderLayout.CENTER, editorPanel);

      JPanel editorButtonPanel = new JPanel();
      editorButtonPanel.setLayout (new FlowLayout (FlowLayout.CENTER, 5, 5));

      setButton = new JButton();
      setButton.setText ("Set");
      setButton.setActionCommand ("set");
      setButton.addActionListener (this);
      editorButtonPanel.add (setButton);

      saveButton = new JButton();
      saveButton.setText ("Save");
      saveButton.setActionCommand ("save");
      saveButton.addActionListener (this);
      editorButtonPanel.add (saveButton);

      createNewButton = new JButton();
      createNewButton.setText ("Create New");
      createNewButton.setActionCommand ("createNew");
      createNewButton.addActionListener (this);
      editorButtonPanel.add (createNewButton);
      createNewButton.setEnabled (globalEditor);

      revertButton = new JButton();
      revertButton.setText ("Revert");
      revertButton.setActionCommand ("revert");
      revertButton.addActionListener (this);
      editorButtonPanel.add (revertButton);

      add (BorderLayout.SOUTH, editorButtonPanel);

      JPanel filterChoosePanel = new JPanel();
      filterChoosePanel.setLayout (new FlowLayout (FlowLayout.LEFT, 5, 0));

      filterComboBox = new JComboBox();
      filterComboBox.setActionCommand ("filter");
      filterComboBox.addActionListener (this);
      filterChoosePanel.add (filterComboBox);

      add (BorderLayout.NORTH, filterChoosePanel);
      filterComboBox.setVisible (isGlobalEditor());

      setName ("Edit Current Filter");
   }


   /**
    * Sets the globalEditor attribute of the FilterEditorPanel object
    *
    * @param global  The new globalEditor value
    */
   public void setGlobalEditor (boolean global)
   {
      if (this.globalEditor != global)
      {
         this.globalEditor = global;
         unparse();
      }
   }


   /**
    * Get the globalEditor attribute of the FilterEditorPanel object
    *
    * @return   The globalEditor value
    */
   public boolean isGlobalEditor()
   {
      return globalEditor;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void unparse()
   {
      Filter filter = null;
      setEditor (null);
      if (getHostingDialog() != null)
      {
         filter = getHostingDialog().getCurrentFilter();
         setCurrentFilter (filter);
      }
      else
      {
         filter = getCurrentFilter();
      }
      if (filter != null)
      {
         filterComboBox.setSelectedItem (filter);
         if (filter instanceof ConfigurableFilter)
         {
            ConfigurableFilter cFilter = (ConfigurableFilter) filter;
            FilterEditor editor = cFilter.getEditor();
            setEditor (editor);
         }
      }
      else
      {
         if (filterComboBox.getComponentCount() > 0)
         {
            currentFilter = (Filter) filterComboBox.getItemAt (0);
            filterComboBox.setSelectedItem (currentFilter);
         }
      }

      filterComboBox.setVisible (isGlobalEditor());
      setButton.setEnabled (filter != null);
      saveButton.setEnabled (filter != null);
      createNewButton.setEnabled (filter != null && isGlobalEditor());
      revertButton.setEnabled (filter != null);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void refreshDialog()
   {
      Filter filter = getCurrentFilter();
      if (getHostingDialog() != null)
      {
         getHostingDialog().setCurrentFilter (filter);
      }
      unparse();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FilterEditorDialog hostingDialog = null;


   /**
    * the FilterEditorDialog this panel is nested in
    *
    * @param hostingDialog  The new hostingDialog value
    */
   protected void setHostingDialog (FilterEditorDialog hostingDialog)
   {
      this.hostingDialog = hostingDialog;
   }


   /**
    * the FilterEditorDialog this panel is nested in
    *
    * @return   The hostingDialog value
    */
   public FilterEditorDialog getHostingDialog()
   {
      return hostingDialog;
   }


   /**
    * the model for the filter combobox
    *
    * @param model  The new model value
    */
   public void setModel (ComboBoxModel model)
   {
      filterComboBox.setModel (model);
   }


   /**
    * the model for the filter combobox
    *
    * @return   The model value
    */
   public ComboBoxModel getModel()
   {
      return filterComboBox.getModel();
   }


   /**
    * the renderer for the filter combobox
    *
    * @param renderer  The new renderer value
    */
   public void setRenderer (ListCellRenderer renderer)
   {
      filterComboBox.setRenderer (renderer);
   }


   /**
    * the renderer for the filter combobox
    *
    * @return   The renderer value
    */
   public ListCellRenderer getRenderer()
   {
      return filterComboBox.getRenderer();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      String command = e.getActionCommand();
      if (command.equals ("filter"))
      {
         filterComboBoxAction();
      }
      else if (command.equals ("set"))
      {
         setAction();
      }
      else if (command.equals ("save"))
      {
         saveAction();
      }
      else if (command.equals ("createNew"))
      {
         createNewAction();
      }
      else if (command.equals ("revert"))
      {
         revertAction();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Filter currentFilter = null;


   /**
    * Get the currentFilter attribute of the FilterEditorPanel object
    *
    * @return   The currentFilter value
    */
   public Filter getCurrentFilter()
   {
      return currentFilter;
   }


   /**
    * Sets the currentFilter attribute of the FilterEditorPanel object
    *
    * @param filter  The new currentFilter value
    */
   public void setCurrentFilter (Filter filter)
   {
      if (this.currentFilter != filter)
      {
         this.currentFilter = filter;
         refreshDialog();
      }
   }


   /**
    * Sets the action attribute of the FilterEditorPanel object
    *
    * @return   No description provided
    */
   private boolean setAction()
   {
      Filter filter = getCurrentFilter();
      FilterEditor editor = getEditor();
      if (filter == null || ! (filter instanceof ConfigurableFilter) || editor == null)
      {
         return false;
      }

      int result = -1;
      if (filter.sizeOfViewDefinitions() > 0)
      {
         if (globalEditor)
         {
            result = JOptionPane.showOptionDialog (this,
               "This Filter is in use!\nChanging it might effect some of the " +
               "Views that use this Filter!\n\n" +
               "It would be safer to create a copy instead.", "Set values",
               JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE,
               null,
               new String[]
               {
               "Set anyway", "Cancel", "Create Copy"
               }
               , "Create Copy");
         }
         else
         {
            result = JOptionPane.showOptionDialog (this,
               "This Filter is in use!\nChanging it might effect some of the " +
               "Views that use this Filter!\n\n" +
               "Do you want to abort?.", "Set values",
               JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE,
               null,
               new String[]
               {
               "Set", "Abort"
               }
               , "Abort");
         }
      }
      else
      {
         if (globalEditor)
         {
            result = JOptionPane.showOptionDialog (this,
               "This will overwrite all existing values\nin this Filter", "Set values",
               JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE,
               null, new String[]
               {
               "Set anyway", "Cancel", "Create Copy"
               }
               , "Create Copy");
         }
         else
         {
            result = JOptionPane.showOptionDialog (this,
               "This will overwrite all existing values\nin this Filter", "Set values",
               JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE,
               null, new String[]
               {
               "Ok", "Cancel"
               }
               , "Ok");
         }
      }
      if (result == JOptionPane.NO_OPTION)
      {
         return false;
      }
      else if (result == JOptionPane.CANCEL_OPTION)
      {
         return createNewAction();
      }
      else
      {
         editor.setValues();
         Iterator defIter = filter.iteratorOfViewDefinitions();
         while (defIter.hasNext())
         {
            ViewDefinition def = (ViewDefinition) defIter.next();
            if (def.isUpdate())
            {
               def.update();
            }
         }
         return true;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private boolean revertAction()
   {
      Filter filter = getCurrentFilter();
      FilterEditor editor = getEditor();
      if (filter == null || ! (filter instanceof ConfigurableFilter) || editor == null)
      {
         return false;
      }

      int result = JOptionPane.showConfirmDialog (this,
         "Really discard changes?", "Discard changes",
         JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

      if (result == JOptionPane.NO_OPTION)
      {
         return false;
      }

      editor.getValues();
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private boolean createNewAction()
   {
      Filter filter = getCurrentFilter();
      FilterEditor editor = getEditor();
      if (filter == null || ! (filter instanceof ConfigurableFilter) || editor == null)
      {
         return false;
      }

      Filter newFilter = editor.getNewFilter();
      if (newFilter == null)
      {
         JOptionPane.showMessageDialog (this,
            "Creation of new Filter with\nthese Settings failed", "New Filter",
            JOptionPane.ERROR_MESSAGE);

         return false;
      }
      else if (isGlobalEditor())
      {
         if (FilterManager.get().getFilter (newFilter.getName()) != null)
         {
            String name = newFilter.getName();
            int counter = 1;

            while (FilterManager.get().getFilter (name + " <" + counter + ">") != null)
            {
               counter++;
            }
            name = name + "<" + counter + ">";
         }
         FilterManager.get().addToFilters (newFilter);
      }
      setCurrentFilter (newFilter);
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private boolean saveAction()
   {
      Filter filter = getCurrentFilter();
      if (filter == null || ! (filter instanceof ConfigurableFilter))
      {
         return false;
      }

      if (!setAction())
      {
         return false;
      }

      File saveFile = FilterManager.get().getFromSettingsFiles ((ConfigurableFilter) filter);
      do
      {
         JFileChooser fileChooser = new JFileChooser (GeneralPreferences.get().getViewFilterFolder());
         fileChooser.setFileFilter (
            new FileFilter()
            {
               public boolean accept (File file)
               {
                  return  (file.isDirectory() ||
                     file.getName().toLowerCase().endsWith ("." + FilterManager.SETTINGS_FILE_EXT));
               }


               public String getDescription()
               {
                  return "Filter Settings (*." + FilterManager.SETTINGS_FILE_EXT + ")";
               }
            }
            );
         int ret = fileChooser.showSaveDialog (this);
         if (ret == JFileChooser.CANCEL_OPTION || fileChooser.getSelectedFile() == null ||
            fileChooser.getSelectedFile().isDirectory())
         {
            return false;
         }

         saveFile = fileChooser.getSelectedFile();
         if (saveFile.isDirectory())
         {
            saveFile = null;
         }
         else if (!saveFile.exists())
         {
            try
            {
               saveFile.createNewFile();
            }
            catch (IOException ioe)
            {
               ioe.printStackTrace();
               saveFile = null;
            }
         }
      } while (saveFile == null);

      try
      {
         FilterManager.saveFilter ((ConfigurableFilter) filter, saveFile);
         return true;
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void filterComboBoxAction()
   {
      Filter filter = (Filter) filterComboBox.getSelectedItem();

      setCurrentFilter (filter);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FilterEditor editor = null;


   /**
    * Sets the editor attribute of the FilterEditorPanel object
    *
    * @param editor  The new editor value
    */
   public void setEditor (FilterEditor editor)
   {
      if (editor == this.editor)
      {
         return;
      }

      this.editor = editor;

      if (editor == null)
      {
         editorScrollPane.setViewportView (null);
         return;
      }

      editor.getValues();
      JPanel panel = editor.getEditorPanel();
      if (panel != null)
      {
         editorScrollPane.setViewportView (panel);
      }
      else
      {
         editorScrollPane.setViewportView (null);
      }
   }


   /**
    * Get the editor attribute of the FilterEditorPanel object
    *
    * @return   The editor value
    */
   public FilterEditor getEditor()
   {
      return editor;
   }
}

/*
 * $Log: FilterEditorPanel.java,v $
 * Revision 1.11  2004/10/20 17:50:33  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
