/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.*;

import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.fsa.unparse.UnparseManager;
import de.uni_paderborn.fujaba.uml.UMLIncrement;
import de.uni_paderborn.fujaba.views.cf.*;
import de.uni_paderborn.fujaba.views.gui.SmallFilterEditorDialog;


/**
 * implementation of the FilterEditor interface for the CompositeFilter <p>
 *
 * uses the new FSA-Structures to show a diagram for the CompositeFilter
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.18 $
 */
public class CompositeFilterEditor extends AbstractFilterEditor implements ActionListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JLayeredPane noDiagPanel = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JScrollPane scrollPane = null;


   /**
    * Constructor for class CompositeFilterEditor
    *
    * @param filter  No description provided
    */
   public CompositeFilterEditor (ConfigurableFilter filter)
   {
      super (filter);
   }


   /**
    * create the Editor GUI
    */
   protected void buildEditor()
   {
      this.setLayout (new BorderLayout());

      JToolBar buttonBar = createButtonBar();
      scrollPane = createScrollPane();

      JEditorPane textPane = new JEditorPane();
      textPane.setContentType ("text/html");
      textPane.setText ("<div align=\"Center\"><i>This editor might not work 100% correctly <br>" +
         "because of ongoing changes to the <br>" +
         "FUJABA display model</i></div>");
      textPane.setEditable (false);

      JPanel textPanel = new JPanel();
      textPanel.setLayout (new GridLayout (1, 1, 0, 0));
      textPanel.setBackground (FSAObject.COLOR_BACKGROUND);
      textPanel.add (textPane);

      this.add (scrollPane, BorderLayout.CENTER);
      this.add (buttonBar, BorderLayout.NORTH);
      this.add (textPanel, BorderLayout.SOUTH);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JToolBar createButtonBar()
   {
      JToolBar bar = new JToolBar();

      JButton button = new JButton();
      button.setActionCommand ("addAndNode");
      button.addActionListener (this);
      button.setText (" & ");
      bar.add (button);

      button = new JButton();
      button.setActionCommand ("addOrNode");
      button.addActionListener (this);
      button.setText (" v ");
      bar.add (button);

      button = new JButton();
      button.setActionCommand ("addNotNode");
      button.addActionListener (this);
      button.setText (" ! ");
      bar.add (button);

      button = new JButton();
      button.setActionCommand ("addFilterNode");
      button.addActionListener (this);
      button.setText ("Filter");
      bar.add (button);

      button = new JButton();
      button.setActionCommand ("addTransition");
      button.addActionListener (this);
      button.setText ("->");
      bar.add (button);

      bar.addSeparator();

      button = new JButton();
      button.setActionCommand ("deleteNode");
      button.addActionListener (this);
      button.setText ("Delete");
      bar.add (button);
      bar.addSeparator();

      button = new JButton();
      button.setActionCommand ("editFilter");
      button.addActionListener (this);
      button.setText ("Edit Filter");
      bar.add (button);

      return bar;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private JScrollPane createScrollPane()
   {
      JScrollPane scrollPane = new JScrollPane();
      scrollPane.setBackground (FSAObject.COLOR_BACKGROUND);

      JLabel noDiagLabel = new JLabel ("No diagram is selected in the project tree.");
      noDiagLabel.setBackground (FSAObject.COLOR_BACKGROUND);
      noDiagPanel = new JLayeredPane();
      noDiagPanel.setLayout (new FlowLayout());
      noDiagPanel.setBackground (FSAObject.COLOR_BACKGROUND);
      noDiagPanel.add (noDiagLabel, 0);

      scrollPane.getViewport().setDoubleBuffered (true);
      scrollPane.getViewport().setBackground (FSAObject.COLOR_BACKGROUND);
      scrollPane.getViewport().add (noDiagPanel);

      // new swing 1.1.1 property to speed up drawing
      scrollPane.getViewport().setScrollMode (JViewport.BLIT_SCROLL_MODE);

      return scrollPane;
   }


   /**
    * Sets the values attribute of the CompositeFilterEditor object
    *
    * @param filter  The new values value
    */
   protected void setValues (ConfigurableFilter filter)
   {
      if (filter == null || ! (filter instanceof CompositeFilter))
      {
         return;
      }

       ((CompositeFilter) filter).setCFDiagram (getCurrentDiagram());
   }


   /**
    * Get the currentDiagram attribute of the CompositeFilterEditor object
    *
    * @return   The currentDiagram value
    */
   private CFDiagram getCurrentDiagram()
   {
      if (currentDiagram == null)
      {
         Component comp = scrollPane.getViewport().getComponent (0);
         if (comp instanceof JComponent)
         {
            FSAObject obj = FSAObject.getFSAObjectFromJComponent ((JComponent) comp);

            if (obj != null)
            {
               LogicUnparseInterface incr = obj.getLogic();

               if (incr != null && incr instanceof CFDiagram)
               {
                  currentDiagram = (CFDiagram) incr;
               }
            }
         }
      }
      return currentDiagram;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private CFDiagram currentDiagram = null;


   /**
    * Get the values attribute of the CompositeFilterEditor object
    *
    * @param filter  No description provided
    */
   protected void getValues (ConfigurableFilter filter)
   {
      if (scrollPane == null)
      {
         buildEditor();
      }

      scrollPane.getViewport().removeAll();
      SelectionManager.get().clear();

      CFDiagram diag = null;
      if (filter != null && filter instanceof CompositeFilter)
      {
         diag =  ((CompositeFilter) filter).getCFDiagram();

         if (diag != null)
         {
            try
            {
               diag = (CFDiagram) diag.clone();
            }
            catch (Exception e)
            {
               e.printStackTrace();
               throw new RuntimeException ("Unable to clone diagram.\nFailed to set editor up. Cause: " + e.getMessage());
            }
         }
         else
         {
            diag = new CFDiagram();
         }
         initDiagram (diag);
         UnparseManager.get().unparse (diag);

         scrollPane.getViewport().removeAll();
         SelectionManager.get().clear();
         Iterator fsaIter = diag.iteratorOfFsaObjects();
         FSALayeredPane pane = (FSALayeredPane) fsaIter.next();
         scrollPane.getViewport().add (pane.getJComponent());
         currentDiagram = diag;
      }
      else
      {
         scrollPane.getViewport().add (noDiagPanel);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param diag  No description provided
    */
   private void initDiagram (CFDiagram diag)
   {
      if (diag.getStartNode() != null)
      {
         if (!diag.hasInElements (diag.getStartNode()))
         {
            diag.addToElements (diag.getStartNode());
         }
      }
      else
      {
         diag.addToElements (new CFStartNode());
      }
      if (diag.getEndNode() != null)
      {
         if (!diag.hasInElements (diag.getEndNode()))
         {
            diag.addToElements (diag.getEndNode());
         }
      }
      else
      {
         diag.addToElements (new CFEndNode());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      String command = e.getActionCommand();
      CFDiagram diag = getCurrentDiagram();

      if (diag == null)
      {
         return;
      }

      if ("addOrNode".equals (command))
      {
         diag.addToElements (new CFOrNode());
      }
      else if ("addNotNode".equals (command))
      {
         diag.addToElements (new CFNotNode());
      }
      else if ("addAndNode".equals (command))
      {
         diag.addToElements (new CFAndNode());
      }
      else if ("addFilterNode".equals (command))
      {
         CFFilterNode node = new CFFilterNode();
         node.setFilter (new AssocFilter());
         node.setContext (-1);
         diag.addToElements (node);
      }
      else if ("addTransition".equals (command))
      {
         addTransitionAction();
      }
      else if ("deleteNode".equals (command))
      {
         deleteAction();
      }
      else if ("editFilter".equals (command))
      {
         editFilterAction();
      }
   }


   /**
    * Access method for an one to n association.
    */
   private void addTransitionAction()
   {
      FSAObject[] sel = SelectionManager.get().getLastSelections (null, 2);
      if (sel.length < 2 ||
         sel[sel.length - 1] == null || sel[sel.length - 2] == null)
      {
         JOptionPane.showMessageDialog (this,
            "You have to select two items first", "Error", JOptionPane.ERROR_MESSAGE);
      }
      else
      {
         LogicUnparseInterface sourceIncr = sel[sel.length - 2].getLogic();
         LogicUnparseInterface targetIncr = sel[sel.length - 1].getLogic();

         if (! (sourceIncr instanceof CFNode &&
            targetIncr instanceof CFNode))
         {
            JOptionPane.showMessageDialog (this,
               "The last two selections\n" +
               "have to be Nodes!", "Error", JOptionPane.ERROR_MESSAGE);
         }
         else
         {
            CFNode source = (CFNode) sourceIncr;
            CFNode target = (CFNode) targetIncr;

            if (source instanceof CFEndNode)
            {
               JOptionPane.showMessageDialog (this,
                  "An end node cannot have an\n" +
                  "outgoing transition", "Error", JOptionPane.ERROR_MESSAGE);
            }
            else if (target instanceof CFStartNode)
            {
               JOptionPane.showMessageDialog (this,
                  "A start node cannot have an\n" +
                  "incoming transition", "Error", JOptionPane.ERROR_MESSAGE);
            }
            else
            {
               CFTransition trans = new CFTransition();
               trans.setSource (source);
               trans.setTarget (target);
               getCurrentDiagram().addToElements (trans);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void deleteAction()
   {
      Iterator iter = SelectionManager.get().iteratorOfSelection();

      if (!iter.hasNext())
      {
         JOptionPane.showMessageDialog (this,
            "Nothing selected for deletion!", "Error", JOptionPane.ERROR_MESSAGE);
         return;
      }

      Vector remove = new Vector();
      while (iter.hasNext())
      {
         FSAObject obj = (FSAObject) iter.next();
         LogicUnparseInterface incr = obj.getLogic();

         if (incr instanceof CFStartNode)
         {
            JOptionPane.showMessageDialog (this,
               "Cannot delete start node!", "Error", JOptionPane.ERROR_MESSAGE);
         }
         else if (incr instanceof CFEndNode)
         {
            JOptionPane.showMessageDialog (this,
               "Cannot delete end node!", "Error", JOptionPane.ERROR_MESSAGE);
         }
         else
         {
            remove.add (obj);
         }
      }

      iter = remove.iterator();
      while (iter.hasNext())
      {
         FSAObject obj = (FSAObject) iter.next();
         JComponent parent = (JComponent) obj.getJComponent().getParent();
          ((UMLIncrement) obj.getLogic()).removeYou();
         parent.revalidate();
         parent.repaint();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void editFilterAction()
   {
      FSAObject obj = SelectionManager.get().getFocusedObject();

      if (obj != null && obj.getLogic() instanceof CFFilterNode)
      {
         Filter filter =  ((CFFilterNode) obj.getLogic()).getFilter();

         if (filter != null && filter instanceof ConfigurableFilter)
         {
            SmallFilterEditorDialog editor = new SmallFilterEditorDialog (null);
            editor.setFilter (filter);
            editor.showCentered();
         }
         else
         {
            JOptionPane.showMessageDialog (this,
               "No ConfigurableFilter\nin this FilterNode", "Error", JOptionPane.ERROR_MESSAGE);
         }
      }
      else
      {
         JOptionPane.showMessageDialog (this,
            "Please select a FilterNode first!", "Error", JOptionPane.ERROR_MESSAGE);
      }
   }
}

/*
 * $Log: CompositeFilterEditor.java,v $
 * Revision 1.18  2004/10/22 16:41:44  lowende
 * Deprecated warnings removed. Other compile warnings removed.
 *
 */
