/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.update;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Iterator;

import javax.swing.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.fsa.update.VisibilityUpdater;
import de.uni_paderborn.fujaba.metamodel.FDeclaration;
import de.uni_paderborn.fujaba.uml.*;


/**
 * update visibility of Class compartment panels
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.9.2.1 $
 */
public class ClassCompartmentVisibilityUpdater
    extends VisibilityUpdater implements PropertyChangeListener
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ClassCompartmentVisibilityUpdater.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int COMPARTMENT_STEREOTYPES = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int COMPARTMENT_ATTRIBUTES = 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int COMPARTMENT_METHODS = 2;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final static int COMPARTMENT_SIGNALS = 3;


   /**
    * Default Constructor
    */
   public ClassCompartmentVisibilityUpdater()
   {
      super();
      super.setFsaAttrName ("visible");
   }


   /**
    * Constructor for class ClassCompartmentVisibilityUpdater
    *
    * @param incr  No description provided
    * @param attr  No description provided
    */
   public ClassCompartmentVisibilityUpdater (LogicUnparseInterface incr, String attr)
   {
      this();
      setLogicObject (incr);
      setLogicAttrName (attr);
   }


   /**
    * Sets the logicObject attribute of the ClassCompartmentVisibilityUpdater object
    *
    * @param object  The new logicObject value
    * @return        No description provided
    */
   public boolean setLogicObject (Object object)
   {
      if (object == null || object instanceof UMLClass)
      {
         return super.setLogicObject (object);
      }
      throw new IllegalArgumentException ("Object must be of instance UMLClass");
   }


   /**
    * Sets the fsaAttrName attribute of the ClassCompartmentVisibilityUpdater object
    *
    * @param name  The new fsaAttrName value
    * @return      No description provided
    */
   public boolean setFsaAttrName (String name)
   {
      if ("visible".equals (name))
      {
         return false;
      }

      throw new UnsupportedOperationException ("Cannot change VisibilityUpdater.fsaAttrName from \"visible\"");
   }


   /**
    * Sets the logicAttrName attribute of the ClassCompartmentVisibilityUpdater object
    *
    * @param name  The new logicAttrName value
    * @return      No description provided
    */
   public boolean setLogicAttrName (String name)
   {
      if ("methods".equals (name) || "attrs".equals (name) || "signals".equals (name) || "stereotypes".equals (name))
      {
         return super.setLogicAttrName (name);
      }

      throw new UnsupportedOperationException ("Cannot change MethodsVisibilityUpdater.umlAttrName from \"methods\"");
   }


   /**
    * expects an array of 3 int numbers:</p> 0: Number of Stereotypes 1: Number of Attrs</p>
    * 2: Number of Methods</p> 3: Number of Signals</p> translates that array to a boolean
    * value according to UML class compartment visibility rules.
    *
    * @param data  No description provided
    * @return      Boolean.TRUE if compartment is to be shown, Boolean.FALSE else
    */
   public Object translateLogicToFsa (Object data)
   {
      String attrName = getLogicAttrName();
      Boolean result;
      try
      {
         int[] elements = (int[]) data;

         if ("stereotypes".equals (attrName))
         {
            result = elements[COMPARTMENT_STEREOTYPES] > 0
               ? Boolean.TRUE
               : Boolean.FALSE;
         }
         else if ("attrs".equals (attrName))
         {
            result = elements[COMPARTMENT_ATTRIBUTES] > 0
               ? Boolean.TRUE
               : Boolean.FALSE;
         }
         else if ("methods".equals (attrName))
         {
            result = elements[COMPARTMENT_METHODS] > 0
               ? Boolean.TRUE
               : Boolean.FALSE;
         }
         else if ("signals".equals (attrName))
         {
            result = elements[COMPARTMENT_SIGNALS] > 0
               ? Boolean.TRUE
               : Boolean.FALSE;
         }
         else
         {
            result = Boolean.FALSE;
         }

      }
      catch (Exception e)
      {
         log.error ("ClassCompartmentVisibilityUpdater.translateLogicToFsa failed for " + data + " because of " + e.getMessage());
         result = Boolean.FALSE;
      }

      return result;
   }


   /**
    * Access method for an one to n association.
    */
   protected void addListener()
   {
      if (! (getLogicObject() == null || getLogicAttrName() == null ||
         getFsaObject() == null || getFsaAttrName() == null))
      {
         if (getLogicListener() == null)
         {
            setLogicListener (this);
         }
         if ("signals".equals (getLogicAttrName()))
         {
             ((UMLClass) getLogicObject()).addPropertyChangeListener ("signals", (PropertyChangeListener) getLogicListener());
         }
         if ("methods".equals (getLogicAttrName()))
         {
             ((UMLClass) getLogicObject()).addPropertyChangeListener ("methods", (PropertyChangeListener) getLogicListener());
         }
         if ("attrs".equals (getLogicAttrName()))
         {
             ((UMLClass) getLogicObject()).addPropertyChangeListener ("attrs", (PropertyChangeListener) getLogicListener());
         }
         if ("stereotypes".equals (getLogicAttrName()))
         {
             ((UMLClass) getLogicObject()).addPropertyChangeListener ("stereotypes", (PropertyChangeListener) getLogicListener());
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeListener()
   {
      if (! (getLogicObject() == null || getLogicAttrName() == null ||
         getFsaObject() == null || getFsaAttrName() == null))
      {
         if (getLogicListener() != null)
         {
            if ("signals".equals (getLogicAttrName()))
            {
                ((UMLClass) getLogicObject()).removePropertyChangeListener ("signals", (PropertyChangeListener) getLogicListener());
            }
            if ("methods".equals (getLogicAttrName()))
            {
                ((UMLClass) getLogicObject()).removePropertyChangeListener ("methods", (PropertyChangeListener) getLogicListener());
            }
            if ("attrs".equals (getLogicAttrName()))
            {
                ((UMLClass) getLogicObject()).removePropertyChangeListener ("attrs", (PropertyChangeListener) getLogicListener());
            }
            if ("stereotypes".equals (getLogicAttrName()))
            {
                ((UMLClass) getLogicObject()).removePropertyChangeListener ("stereotypes", (PropertyChangeListener) getLogicListener());
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initialize()
   {
      if (! (getLogicObject() == null || getLogicAttrName() == null ||
         getFsaObject() == null || getFsaAttrName() == null))
      {

         attrs = 0;
         methods = 0;
         signals = 0;
         stereotypes = 0;

         UMLClass umlClass = (UMLClass) getLogicObject();

         Iterator iter = umlClass.iteratorOfStereotypes();
         while (iter.hasNext())
         {
            iter.next();
            stereotypes++;
         }

         iter = umlClass.iteratorOfAttrs();
         while (iter.hasNext())
         {
            UMLAttr attr = (UMLAttr) iter.next();

            if (attr.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               attrs++;
            }
         }
         iter = umlClass.iteratorOfMethods();
         while (iter.hasNext())
         {
            UMLMethod method = (UMLMethod) iter.next();

            if (method.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               if (method.isSignal())
               {
                  signals++;
               }
               else
               {
                  methods++;
               }
            }
         }

         Boolean value = (Boolean) getTranslator().translateLogicToFsa (new int[]
            {stereotypes, attrs, methods, signals});
         boolean isVisible = value.booleanValue();

         getFsaObject().setVisible (isVisible);
      }
   }

   // counters for elements
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int stereotypes = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int attrs = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int methods = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int signals = 0;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void propertyChange (PropertyChangeEvent e)
   {
      Object oldValue = e.getOldValue();
      Object newValue = e.getNewValue();

      boolean changed = false;

      if ("attrs".equals (e.getPropertyName()))
      {
         if (oldValue != null && newValue == null)
         {
            UMLAttr attr = (UMLAttr) oldValue;

            if (attr.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               attrs--;
               changed = true;
            }
         }
         else if (oldValue == null && newValue != null)
         {
            UMLAttr attr = (UMLAttr) newValue;

            if (attr.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               attrs++;
               changed = true;
            }
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info ("Strange PCEvent in CompartmentUpdater: " + e);
            }
         }
      }
      else if ("methods".equals (e.getPropertyName()))
      {
         if (oldValue != null && newValue == null)
         {
            UMLMethod method = (UMLMethod) oldValue;

            if (method.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               if (method.isSignal())
               {
                  signals--;
               }
               else
               {
                  methods--;
               }

               changed = true;
            }
         }
         else if (oldValue == null && newValue != null)
         {
            UMLMethod method = (UMLMethod) newValue;

            if (method.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               if (method.isSignal())
               {
                  signals++;
               }
               else
               {
                  methods++;
               }

               changed = true;
            }
         }
         else
         {
            log.error ("Strange PCEvent in CompartmentUpdater: " + e);
         }
      }
      else if ("stereotypes".equals (e.getPropertyName()))
      {
         if (oldValue != null && newValue == null)
         {
            stereotypes--;
            changed = true;
         }
         else if (oldValue == null && newValue != null)
         {
            stereotypes++;
            changed = true;
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info ("Strange PCEvent in CompartmentUpdater: " + e);
            }
         }
      }
      else if ("signals".equals (e.getPropertyName()))
      {
         if (oldValue != null && newValue == null)
         {
            UMLMethod method = (UMLMethod) oldValue;

            if (method.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               signals--;
               methods++;
               changed = true;
            }
         }
         else if (oldValue == null && newValue != null)
         {
            UMLMethod method = (UMLMethod) newValue;

            if (method.getDisplayLevel() > FDeclaration.CODE_DISPLAY_LEVEL)
            {
               signals++;
               methods--;
               changed = true;
            }
         }
         else
         {
            log.error ("Strange PCEvent in CompartmentUpdater: " + e);
         }
      }

      else
      {
         log.error ("Strange PropertyName in CompartmentUpdater: " + e.getPropertyName());
      }

      if (changed)
      {
         Boolean value = (Boolean) getTranslator().translateLogicToFsa (new int[]
            {stereotypes, attrs, methods, signals});
         getFsaObject().setVisible (value.booleanValue());
      }

      JComponent component = getFsaObject().getJComponent();
      JComponent parent = (JComponent) component.getParent();

      if (parent != null)
      {
         while (parent != null && parent.getParent() instanceof JComponent)
         {
            parent.setPreferredSize (null);
            parent = (JComponent) parent.getParent();
         } // end of while ()
         parent.setPreferredSize (null);
         parent.revalidate();
         parent.repaint();
      } // end of if ()
   }

}

/*
 * $Log: ClassCompartmentVisibilityUpdater.java,v $
 * Revision 1.9.2.1  2005/09/30 18:57:02  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
