/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml.actions;

import java.awt.event.ActionEvent;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import javax.swing.*;
import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.uml.*;


/**
 * Invokes the Import Editor
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.15.2.1 $
 */
public class ImportClassReferences extends AbstractAction
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (ImportClassReferences.class);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param event  No description provided
    */
   public void actionPerformed (ActionEvent event)
   {
      FrameMain frame = UMLProject.get().getGui();

      JFileChooser fileChooser = frame.getFileChooser();

      fileChooser.setFileFilter (frame.getFilterJarFiles());
      fileChooser.setFileSelectionMode (JFileChooser.FILES_AND_DIRECTORIES);

      if (fileChooser.showOpenDialog (frame) == JFileChooser.APPROVE_OPTION)
      {
         File file = fileChooser.getSelectedFile();

         if (file.isDirectory())
         {
            importDirectory (file.getAbsoluteFile().toString().lastIndexOf (File.separatorChar) + 1, file);
         }
         else
         {
            importFile (file);
         }
      }
   }


   /**
    * This method converts a name to a normal name. It replaces all / with . and kills .class.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   private String convertToNormalName (String name)
   {
      // we must convert also the other separators because we must understand an archive which have been built on other
      // operating systems.
      name = name.replace (File.separatorChar, '.');
      name = name.replace ('/', '.');
      name = name.replace ('\\', '.');

      if (name.endsWith (".class"))
      {
         name = name.substring (0, name.length() - 6);
      }

      if (name.endsWith ("."))
      {
         name = name.substring (0, name.length() - 1);
      }

      return name;
   }


   /**
    * findPlace finds the place in the tree rooted on root for name. If it must create new
    * packages it will create them in the tree and in the UMLPackage.
    *
    * @param name         No description provided
    * @param rootPackage  No description provided
    * @return             No description provided
    */
   public UMLPackage findPlace (UMLPackage rootPackage, String name)
   {
      int index = name.indexOf ('.');

      if (index != -1)
      {
         // divide the name into first Package and rest
         String first = name.substring (0, index);
         String rest = name.substring (index + 1, name.length());
         //            if (log.isInfoEnabled()) log.info ("First: " + first);
         //            if (log.isInfoEnabled()) log.info ("Rest:  " + rest);

         // find subPackages.toString() == first
         UMLPackage childPackage = rootPackage.getFromPackages (first);
         if (childPackage != null)
         {
            return findPlace (childPackage, rest);
         }
         else
         {
            // no subPackage found, create new one
            UMLPackage newPackage = new UMLPackage();
            newPackage.setName (first);

            rootPackage.addToPackages (newPackage);
            return findPlace (newPackage, rest);
         }
      }
      else
      {
         return rootPackage;
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param name     The object added.
    * @param isClass  The object added.
    */
   public void addToTreeAndPackages (String name, boolean isClass)
   {
      if (name.indexOf ("$") == -1)
      {
         name = convertToNormalName (name);

         if (log.isDebugEnabled())
         {
            log.debug ("adding: " + name);
         }

         UMLPackage place = findPlace (UMLProject.get().getRootPackage(), name);
         name = name.substring (name.lastIndexOf ('.') + 1, name.length());
         if (isClass)
         {
            if (!place.hasKeyInDeclares (name))
            {
               UMLClass newClass = new UMLClass (name);
               UMLStereotype referenceType = UMLStereotypeManager.get().
                  getFromStereotypes (UMLStereotypeManager.REFERENCE);

               newClass.setStereotype (referenceType, true);
               place.addToDeclares (newClass);

               // TODO: here, attributes and methods and assocs should be analysed. AZ
            }
         }
         else
         {
            if (!place.hasKeyInPackages (name))
            {
               UMLPackage newPackage = new UMLPackage();
               newPackage.setName (name);
               place.addToPackages (newPackage);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param file  No description provided
    */
   private void importFile (File file)
   {
      String fileName = file.getName();
      String extension = "";
      if (fileName.lastIndexOf ('.') != -1)
      {
         extension = fileName.substring (fileName.lastIndexOf ('.') + 1, fileName.length());
      }

      if (extension.equals ("jar"))
      {
         try
         {
            JarFile jar = new JarFile (file);
            Enumeration entries = jar.entries();

            while (entries.hasMoreElements())
            {
               JarEntry entry = (JarEntry) entries.nextElement();

               String name = entry.getName();

               if (!name.startsWith ("META") &&  (name.endsWith (".class") || entry.isDirectory()))
               {

                  if (entry.isDirectory())
                  {
                     addToTreeAndPackages (entry.getName(), false);
                  }
                  else
                  {
                     addToTreeAndPackages (entry.getName(), true);
                  }
               }
            }
         }
         catch (IOException e)
         {
            if (log.isInfoEnabled())
            {
               log.info ("Some problem occured while accessing the jar file");
            }
         }

      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param index  No description provided
    * @param file   No description provided
    */
   private void importDirectory (int index, File file)
   {
      int i;
      File[] files = file.listFiles();

      for (i = 0; i < files.length; i++)
      {
         File tmpFile = files[i];
         String fileName = tmpFile.getAbsoluteFile().toString();
         fileName = fileName.substring (index, fileName.length());

         if (!tmpFile.isDirectory())
         {
            if (fileName.endsWith (".class"))
            {
               addToTreeAndPackages (fileName, true);
            }
         }
         else
         {
            if (!fileName.startsWith (".") && !tmpFile.getName().equals ("CVS"))
            {
               addToTreeAndPackages (fileName, false);
               importDirectory (index, tmpFile);
            }
         }
      }
   }

}

/*
 * $Log: ImportClassReferences.java,v $
 * Revision 1.15.2.1  2005/09/30 18:56:52  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
