/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.beans.PropertyChangeListener;
import java.util.*;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.*;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.unparse.UMLUnparseGetter;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FPropHashMap;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *             +------+ 1                 1
 * UMLTypeList | name +--------------------- UMLType
 *             +------+ revTypes      types
 * </pre>
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.66.2.4 $
 */
public class UMLTypeList extends UMLIncrement implements FTypeList
{

   /**
    * Constructor for class UMLTypeList
    */
   public UMLTypeList()
   {
      super();
   }


   /**
    * Constructor for class UMLTypeList
    *
    * @param project  No description provided
    */
   public UMLTypeList (UMLProject project)
   {
      super();
      setProject (project);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected String createUnparseModuleName()
   {
      return UMLUnparseGetter.getUnparseModuleName (this);
   }


   /**
    * Searches the ASG tree for a given id
    *
    * @param id  The id to search for.
    * @return    The Element with the given id, null if not found.
    */
   public ASGElement searchID (String id)
   {
      ASGElement item = super.searchID (id);
      Iterator iter = iteratorOfTypes();
      while ( (item == null) &&  (iter.hasNext()))
      {
         item =  ((ASGElement) iter.next()).searchID (id);
      }
      return item;
   }


   /**
    * returns a new empty list. All PropertyChangeListeners are added to the new list.
    *
    * @return   No description provided
    */
   public UMLTypeList createNewInstance()
   {
      // some temporary variables.
      String key = null;
      Iterator childIter = null;
      Iterator iter = null;
      PropertyChangeListener listener = null;

      // the new list.
      UMLTypeList newList = new UMLTypeList();
      FujabaPropertyChangeSupport support = (FujabaPropertyChangeSupport) getPropertyChangeSupport();

      // all property change listeners must be added to the listeners of the new list.
      iter = support.iteratorOfListeners();
      while (iter.hasNext())
      {
         listener = (PropertyChangeListener) iter.next();
         newList.addPropertyChangeListener (listener);
      } // end of while ()

      iter = support.keysOfChildren();
      while (iter.hasNext())
      {
         key = (String) iter.next();
         childIter = support.iteratorOfListeners (key);
         while (childIter.hasNext())
         {
            listener = (PropertyChangeListener) childIter.next();
            newList.addPropertyChangeListener (key, listener);
         } // end of while ()
      } // end of while ()
      return newList;
   }


   /**
    * Get the sortedTypeList attribute of the UMLTypeList object
    *
    * @return   The sortedTypeList value
    */
   public List getSortedTypeList()
   {
      LinkedList list = new LinkedList();
      if (types != null)
      {
         list.addAll (types.values());
      }
      Collections.sort (list, FujabaComparator.getLessType());
      return list;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator sortedIteratorOfTypes()
   {
      return getSortedTypeList().iterator();
   }


   /**
    * <pre>
    *             +---------------+ 1                 1
    * UMLTypeList | getTypeName() +--------------------- UMLType
    *             +---------------+ revTypes      types
    * </pre>
    */
   private Map types;


   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToTypes (FType obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getName() != null))
      {
         if (this.types == null)
         {
            this.types = new FPropHashMap (this, FTypeList.TYPES_PROPERTY);
         }
         UMLType oldValue = (UMLType) this.types.put (getKeyForTypes (obj), obj);

         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setRevTypes (null);
            }
             ((UMLType) obj).setRevTypes (this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * Get the keyForTypes attribute of the UMLTypeList object
    *
    * @param type  No description provided
    * @return      The keyForTypes value
    */
   protected Object getKeyForTypes (FType type)
   {
      if (type instanceof FClass)
      {
         return  ((FClass) type).getFullClassName();
      }
      return type.getName();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInTypes (FType obj)
   {
      return  ( (this.types != null) &&
          (obj != null) &&  (getKeyForTypes (obj) != null) &&
          (this.types.get (getKeyForTypes (obj)) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInTypes (String key)
   {
      return  ( (this.types != null) &&
          (key != null) &&
         this.types.containsKey (key));
   }


   /**
    * Use this function to inform UMLTypeList, that
    * the key for a FType has changed.
    *
    *
    * @param oldKey  The old key used for the FType
    * @param value   The FType whose key has changed
    */
   public void keyChangedInTypes (String oldKey, FType value)
   {
      Object currentKey = this.getKeyForTypes (value);
      if ( (this.types != null) &&
          (!oldKey.equals (currentKey)))
      {
         UMLType oldValue = (UMLType) this.types.get (oldKey);
         if (oldValue == value)
         {
            this.types.remove (oldKey);
            this.types.put (currentKey, value);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfTypes()
   {
      return  ( (this.types == null)
         ? FEmptyIterator.get()
         : this.types.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfTypes()
   {
      return  ( (this.types == null)
         ? FEmptyIterator.get()
         : this.types.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfTypes()
   {
      return  ( (this.types == null)
         ? FEmptyIterator.get()
         : this.types.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfTypes()
   {
      return  ( (this.types == null)
         ? 0
         : this.types.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromTypes attribute of the UMLTypeList object
    *
    * @param key  No description provided
    * @return     The fromTypes value
    */
   public UMLType getFromTypes (String key)
   {
      return  ( ( (this.types == null) ||  (key == null))
         ? null
         : (UMLType) this.types.get (key));
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FTypeList#getFromFTypes(java.lang.String)
    */
   /**
    * Get the fromFTypes attribute of the UMLTypeList object
    *
    * @param key  No description provided
    * @return     The fromFTypes value
    */
   public FType getFromFTypes (String key)
   {
      return getFromTypes (key);
   }
   // TODO-END


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromTypes (FType obj)
   {
      boolean changed = false;

      if ( (this.types != null) &&  (obj != null) &&  (getKeyForTypes (obj) != null))
      {
         UMLType oldValue = (UMLType) this.types.get (getKeyForTypes (obj));
         if (oldValue == obj)
         {
            this.types.remove (getKeyForTypes (obj));
             ((UMLType) obj).setRevTypes (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromTypes (String key)
   {
      boolean changed = false;

      if ( (this.types != null) &&  (key != null))
      {
         UMLType tmpObj = (UMLType) this.types.get (key);
         if (tmpObj != null)
         {
            this.types.remove (key);
            tmpObj.setRevTypes (null);

            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromTypes()
   {
      UMLType tmpObj;
      Iterator iter = this.iteratorOfTypes();

      while (iter.hasNext())
      {
         tmpObj = (UMLType) iter.next();
         this.removeFromTypes (tmpObj);
      }
   }

   // ######################################
   // additional support
   // ######################################

   /**
    * Access method for type<br>
    * this method is needed only for loading do not use it in other cases
    *
    * @param pair  The object added.
    */
   public void addToTypes (KeyValuePair pair)
   {
      if (pair == null)
      {
         return;
      }
      addToTypes ((String) pair.getKey(), (UMLType) pair.getValue());
   }


   /**
    * add a key-value-pair to types<br>
    * this method is needed only for loading do not use it in other cases
    *
    * @param key    key for
    * @param value  new value
    */
   public void addToTypes (String key, FType value)
   {
      if ( (value != null) &&  (value.getName() == null) &&  (key != null))
      {
         value.setName (key);
      }
      addToTypes (value);
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public UMLType provideType (String key)
   {
      if (!hasKeyInTypes (key))
      {
         new UMLClass (key, null, this, null);
      }
      return getFromTypes (key);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FTypeList#provideFType(java.lang.String)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public FType provideFType (String key)
   {
      return provideType (key);
   }
   // TODO-END

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLProject project; // reverse UMLTypeList typeList


   /**
    * Get the project attribute of the UMLTypeList object
    *
    * @return   The project value
    */
   public UMLProject getProject()
   {
      return project;
   }


   /**
    * Sets the project attribute of the UMLTypeList object
    *
    * @param project  The new project value
    */
   public void setProject (UMLProject project)
   {
      if (this.project != project)
      { // new partner

         UMLProject oldProject = this.project;
         if (this.project != null)
         { // inform old partner

            this.project = null;
            oldProject.setTypeList (null);
         }
         this.project = project;
         if (project != null)
         { // inform new partner

            project.setTypeList (this);
         }
         firePropertyChange ("project", oldProject, project);
      }
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      // first remove array-types, then 'normal' types
      // because UMLArray objects can't be removed, if
      // their 'arrayType' is already destroyed
      Iterator iter = iteratorOfTypes();
      UMLIncrement incr;
      while (iter.hasNext())
      {
         incr = (UMLIncrement) iter.next();

         if (incr instanceof UMLArray)
         {
            removeFromTypes ((UMLType) incr);
            // call removeYou() AFTER UMLArray has been removed from list
            incr.removeYou();
         }
      }

      iter = iteratorOfTypes();
      while (iter.hasNext())
      {
         incr = (UMLIncrement) iter.next();
         incr.removeYou();
         removeFromTypes ((UMLType) incr);
      }

      setProject (null);

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element, may not return null unless this is the top level node (project)
    *         or is not contained in any parent yet
    */
   public FElement getParentElement()
   {
      return getProject();
   }

}

/*
 * $Log: UMLTypeList.java,v $
 * Revision 1.66.2.4  2005/11/23 16:27:24  fklar
 * added method 'keyChangedInTypes(String FType)' that handles a key-change of a type (UMLClass or UMLArray)
 *
 */
