/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Iterator;

import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;
import de.uni_paderborn.fujaba.metamodel.FDiagram;
import de.uni_paderborn.fujaba.views.ViewDiagram;
import de.upb.tools.sdm.FComplexState;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.70.2.2 $
 */
public class UMLStartActivity extends UMLActivity
{
   /**
    * @param coobraPersistent
    */
   public UMLStartActivity (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLStartActivity
    */
   public UMLStartActivity()
   {
      super();
   } // UMLStartActivity


   /**
    * Constructor for class UMLStartActivity
    *
    * @param spec  No description provided
    */
   public UMLStartActivity (UMLMethod spec)
   {
      super();
      setSpec (spec);
   } // UMLStartActivity


   /**
    * Access method for an one to n association.
    *
    * @param diagram  The object added.
    */
   public void addToDiagrams (FDiagram diagram)
   {
      // ensure that the start activity is only in one activity diagram
      if (hasInDiagrams (diagram))
      {
         return;
      }

      if (diagram != null && ! (diagram instanceof ViewDiagram))
      {
         UMLDiagram oldDiagram;
         Iterator diagIter = iteratorOfDiagrams();
         while (diagIter.hasNext())
         {
            oldDiagram = (UMLDiagram) diagIter.next();
            if ( (oldDiagram != null) && ! (oldDiagram instanceof ViewDiagram) &&  (oldDiagram != diagram))
            {
               // the item is contained in more than one diagram
               throw new RuntimeExceptionWithContext ("Error: An start activity can only be contained in one activity diagram", this);
            }
         }
      }
      super.addToDiagrams (diagram);

      // as a side effect connect the diagram and the corresponding method
      UMLMethod myMethod = this.getSpec();
      if (myMethod != null)
      {
         // yes, a story diagram
         myMethod.setStoryDiag ((UMLActivityDiagram) diagram);
      }
   } // addToDiagrams


   /**
    * Sets the entry attribute of the UMLStartActivity object
    *
    * @param elem  The new entry value
    */
   public void setEntry (UMLTransition elem)
   {
      throw new RuntimeExceptionWithContext ("A start activity must not have any entry transitions", this);
   } // setEntry


   /**
    * reverse UMLActivityStart revSpec
    */
   private UMLMethod spec;


   /**
    * Get the spec attribute of the UMLStartActivity object
    *
    * @return   The spec value
    */
   public UMLMethod getSpec()
   {
      return spec;
   } // getSpec


   /**
    * Sets the spec attribute of the UMLStartActivity object
    *
    * @param spec  The new spec value
    */
   public void setSpec (UMLMethod spec)
   {
      if ( (this.spec == null && spec != null) ||
          (this.spec != null && !this.spec.equals (spec)))
      {
         // new partner
         UMLMethod oldSpec = this.spec;
         if (this.spec != null)
         {
            // inform old partner
            this.spec = null;
            oldSpec.setRevSpec (null);
         }
         this.spec = spec;
         if (spec != null)
         {
            // inform new partner
            spec.setRevSpec (this);
         }
         firePropertyChange ("spec", oldSpec, spec);
      }
   } // setSpec


   /**
    * Has one of the following values: FComplexState.HISTORY_NONE, FComplexState.HISTORY_SHALLOW,
    * FComplexState.HISTORY_DEEP
    */
   private int historyKind;


   /**
    * @param historyKind  The new historyKind value
    */
   public void setHistoryKind (int historyKind)
   {
      if ( ( (historyKind == FComplexState.HISTORY_NONE) ||
          (historyKind == FComplexState.HISTORY_DEEP) ||
          (historyKind == FComplexState.HISTORY_SHALLOW)) &&
         this.historyKind != historyKind)
      {
         int oldValue = this.historyKind;
         this.historyKind = historyKind;
         firePropertyChange ("historyKind", oldValue, historyKind);
      }
   } // setHistoryKind


   /**
    * @return   One of the following values: FComplexState.HISTORY_NONE, FComplexState.HISTORY_SHALLOW,
    *      FComplexState.HISTORY_DEEP
    */
   public int getHistoryKind()
   {
      return historyKind;
   } // getHistoryKind


   /**
    * @return   True, if kind of history is FComplexState.HISTORY_SHALLOW or FComplexState.HISTORY_DEEP
    */
   public boolean isHistory()
   {
      return  ( (historyKind == FComplexState.HISTORY_DEEP) ||
          (historyKind == FComplexState.HISTORY_SHALLOW));
   } // isHistory


   /**
    * Get the specClass attribute of the UMLStartActivity object
    *
    * @return   The specClass value
    */
   public UMLClass getSpecClass()
   {
      UMLClass result = this.getRevStartOfStateChart();

      if (result == null)
      {
         UMLMethod myMethod = this.getSpec();

         if (myMethod == null)
         {
            return null;
         }

         result = myMethod.getParent();
      }

      return result;
   } // getSpecClass


   /**
    * Get the startText attribute of the UMLStartActivity object
    *
    * @return   The startText value
    */
   public String getStartText()
   {
      UMLMethod myMethod = this.getSpec();
      if (myMethod != null)
      {
         return myMethod.getQualifiedMethodDecl();
      }

      UMLClass myClass = this.getRevStartOfStateChart();
      if (myClass != null)
      {
         return "Statechart for " + myClass.getName();
      }
      else
      {
         return storyName;
      }
   } // getStartText


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String storyName = "";


   /**
    * Get the storyName attribute of the UMLStartActivity object
    *
    * @return   The storyName value
    */
   public String getStoryName()
   {
      return this.storyName;
   }


   /**
    * Sets the storyName attribute of the UMLStartActivity object
    *
    * @param newStoryName  The new storyName value
    */
   public void setStoryName (String newStoryName)
   {
      if ( (newStoryName != null) && !newStoryName.equals (this.storyName))
      {
         // actually a new story name
         String oldValue = this.storyName;
         this.storyName = newStoryName;
         firePropertyChange ("storyName", oldValue, newStoryName);
      }
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      UMLCommentary comment =  (getSpec() != null) ? getSpec().getComment() : null;

      if (comment != null)
      {
         comment.removeYou();
      }

      setSpec (null);
      setRevStartOfStateChart (null);

      super.removeYou();
   } // removeYou


   /**
    * Get the text attribute of the UMLStartActivity object
    *
    * @return   The text value
    */
   public String getText()
   {
      return isHistory() ? "History" : "Start";
   } // getText


   /**
    * <pre>
    *                   0..1   startOfStateChart   0..1
    * UMLStartActivity --------------------------------- UMLClass
    *                   +                            +
    * </pre>
    */
   private UMLClass revStartOfStateChart;


   /**
    * Get the revStartOfStateChart attribute of the UMLStartActivity object
    *
    * @return   The revStartOfStateChart value
    */
   public UMLClass getRevStartOfStateChart()
   {
      return revStartOfStateChart;
   } // getRevStartOfStateChart


   /**
    * Sets the revStartOfStateChart attribute of the UMLStartActivity object
    *
    * @param revStartOfStateChart  The new revStartOfStateChart value
    */
   public void setRevStartOfStateChart (UMLClass revStartOfStateChart)
   {
      if ( (this.revStartOfStateChart == null && revStartOfStateChart != null) ||
          (this.revStartOfStateChart != null && !this.revStartOfStateChart.equals (revStartOfStateChart)))
      {
         // newPartner
         UMLClass oldRevStartOfStateChart = this.revStartOfStateChart;
         if (this.revStartOfStateChart != null)
         {
            // inform old partner
            this.revStartOfStateChart = null;
            oldRevStartOfStateChart.setStartOfStateChart (null);
         }

         this.revStartOfStateChart = revStartOfStateChart;
         if (revStartOfStateChart != null)
         {
            // inform new partner
            revStartOfStateChart.setStartOfStateChart (this);
         }
         firePropertyChange ("revStartOfStateChart", oldRevStartOfStateChart, revStartOfStateChart);
      }
   } // setRevStartOfStateChart

}

/*
 * $Log: UMLStartActivity.java,v $
 * Revision 1.70.2.2  2006/03/30 15:51:33  lowende
 * Removed compile warnings.
 *
 * Revision 1.70.2.1  2006/03/15 18:47:28  creckord
 * addToDiagrams() had wrong parameter type and didn't override ASGElement.addToDiagrams ()
 *
 */
