/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.beans.PropertyChangeEvent;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.swing.JOptionPane;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.app.FujabaApp;
import de.uni_paderborn.fujaba.app.Version;
import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.asg.ASGProject;
import de.uni_paderborn.fujaba.basic.BasicIncrement;
import de.uni_paderborn.fujaba.basic.ClassMap;
import de.uni_paderborn.fujaba.basic.FD;
import de.uni_paderborn.fujaba.basic.FileHistory;
import de.uni_paderborn.fujaba.basic.FujabaComparator;
import de.uni_paderborn.fujaba.basic.FujabaDebug;
import de.uni_paderborn.fujaba.basic.JavaFactory;
import de.uni_paderborn.fujaba.basic.KeyValuePair;
import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;
import de.uni_paderborn.fujaba.basic.SourceCodeFactory;
import de.uni_paderborn.fujaba.basic.TemplateManager;
import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.fsa.FSAObject;
import de.uni_paderborn.fujaba.metamodel.FClass;
import de.uni_paderborn.fujaba.metamodel.FPackage;
import de.uni_paderborn.fujaba.metamodel.FProject;
import de.uni_paderborn.fujaba.metamodel.FType;
import de.uni_paderborn.fujaba.metamodel.FTypeList;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.uml.unparse.UMLUnparseGetter;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.upb.lib.plugins.PluginProperty;
import de.upb.tools.fca.FDuplicatedTreeMap;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashMap;
import de.upb.tools.fca.FHashSet;
import de.upb.tools.fca.FPropHashMap;
import de.upb.tools.fca.FPropTreeSet;
import de.upb.tools.fca.FTreeSet;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *            +---------+ 1                                   1
 * UMLProject + name    +--------------------------------------- UMLClass
 *            +---------+ declaredInReferences       references
 *
 *            ------ 0..1         Assoc         0..1
 * UMLProject | ID |--------------------------------- BasicIncrement
 *            ------ refProject      objectHashTable
 * </pre>
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.413.2.17 $
 */
public class UMLProject extends ASGProject implements FProject
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLProject.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String PROPERTY_TYPE_LIST = "typeList";
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String PROPERTY_FILES = "files";

   /**
    * The current file version for Fujaba project files.
    */
   public final static transient int FILE_VERSION = 5;

   /**
    * The singleton instance.
    */
   private static UMLProject theUMLProject;

   /**
    * file name for immediate store
    */
   private File backupFile;
   /**
    * output stream for immediate store
    */
   private OutputStream backupStream;


   /**
    * @return   current backup file
    */
   public File getBackupFile()
   {
      return backupFile;
   }


   /**
    * The private constructor.
    *
    * @see   #get
    */
   private UMLProject()
   {
      super();

      //use the immediate store feature
      if (GeneralPreferences.get().isRecoverActivated() && FujabaChangeManager.getVMRepository() != null)
      {
         try
         {
            Date now = new Date();
            String time = new SimpleDateFormat ("yyyy-MM-dd-HH-mm-ss").format (now);
            int i = 0;
            do
            {
               backupFile = new File ("~tmp " + time +  (i > 0 ? "_" + i++ : "") + ".cxri");
            } while (backupFile.exists());
            backupStream = new BufferedOutputStream (new FileOutputStream (backupFile));
            FujabaChangeManager.getVMRepository().store (backupStream, true);
         }
         catch (FileNotFoundException e)
         {
            e.printStackTrace();
         }
         catch (IOException e)
         {
            e.printStackTrace();
         }
      }
   }


   /**
    * @return   true when a UMLProject was already created
    */
   public static boolean isInitialized()
   {
      return theUMLProject != null;
   }


   /**
    * Use this method to get the instance of UMLProject.
    *
    * @return   The singleton instance
    */
   public static UMLProject get()
   {
      if (theUMLProject == null)
      {
         setNewProject (FujabaChangeManager.getUMLProject());
         if (theUMLProject == null && !FujabaChangeManager.isInUndoRedo())
         {
            if (log.isInfoEnabled())
            {
               log.info ("creating a UMLProject in UMLProject.get() - should not happen!");
            }
            setNewProject (createPlainProject());
         }
      }

      return theUMLProject;
   }


   /**
    * Removes the old project and sets a new one.
    *
    * @param newProject  The new newProject value
    */
   public final static void setNewProject (UMLProject newProject)
   {
      if (newProject != theUMLProject)
      {
         // remove the old project
         if (theUMLProject != null)
         {
            if (newProject != null)
            {
               newProject.stereotypeManager = theUMLProject.getStereotypeManager();
            }
            theUMLProject.removeYou();
            theUMLProject = null;
         }
         theUMLProject = newProject;

         // run garbage collector
         System.gc();
      }
   }


   /**
    * The current project will be removed and a new one is returned with an initialized base
    * type list.
    *
    * @return   A new project with an initialized base type list
    */
   public final static UMLProject createPlainProject()
   {
      // build the the project
      UMLProject project = new UMLProject();
      setNewProject (project);

      // init Typelist
      UMLTypeList typeList = new UMLTypeList();
      project.setTypeList (typeList);

      new UMLBaseTypes (UMLBaseTypes.BOOLEAN, "boolean", typeList);
      new UMLBaseTypes (UMLBaseTypes.CHARACTER, "char", typeList);
      new UMLBaseTypes (UMLBaseTypes.STRING, "String", typeList);
      new UMLBaseTypes (UMLBaseTypes.INTEGER, "int", typeList);
      new UMLBaseTypes (UMLBaseTypes.BYTE, "byte", typeList);
      new UMLBaseTypes (UMLBaseTypes.SHORT_INTEGER, "short", typeList);
      new UMLBaseTypes (UMLBaseTypes.LONG_INTEGER, "long", typeList);
      new UMLBaseTypes (UMLBaseTypes.FLOAT, "float", typeList);
      new UMLBaseTypes (UMLBaseTypes.DOUBLE, "double", typeList);

      // add arrays to basetypes
      Iterator baseTypesIter = project.getTypeList().iteratorOfTypes();
      while (baseTypesIter.hasNext())
      {
         UMLBaseTypes baseType = (UMLBaseTypes) baseTypesIter.next();
         String name = baseType.getName();
         if (!name.endsWith ("Array"))
         {
            new UMLBaseTypes (name + "Array", baseType.getProgLangType() + "[]", typeList);
         }
      }

      new UMLBaseTypes (UMLBaseTypes.VOID, "void", typeList);
      new UMLBaseTypes (UMLBaseTypes.CONSTRUCTOR, "", typeList);
      new UMLBaseTypes (UMLBaseTypes.INITIALIZER, "", typeList);

      project.setSaved (true);

      return project;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected String createUnparseModuleName()
   {
      return UMLUnparseGetter.getUnparseModuleName (this);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param id  No description provided
    * @return    No description provided
    */
   public ASGElement searchID (String id)
   {
      ASGElement item = super.searchID (id);
      if (item != null)
      {
         return item;
      }
      item = getRootPackage().searchID (id);
      if (item != null)
      {
         return item;
      }
      if (getTypeList() != null)
      {
         item = getTypeList().searchID (id);
         if (item != null)
         {
            return item;
         }
      }
      item = UMLStereotypeManager.get().searchID (id);
      return item;
   }



   /**
    * The used java-packages are saved in a package-tree. The root of this tree is the variable
    * this.rootPackage. The parameter fullName is the full name of the package, with dots as
    * separators. Attention: The name of a package is the simple name, the full name of a package
    * is build by searching the package-tree. So if you want to add a package, e.g. newPackge,
    * with a full-name, e.g. fullName, you have to do two things: first, leafe the name of
    * the package empty second, call the function this.addToPackages( newPackge,fullName ))
    * <pre>
    *            -/-                     1
    * UMLProject <-----------------------> UMLPackage
    *            -/-           rootPackage
    * </pre>
    *
    * @see   #getRootPackage
    * @see   #getNewFromPackages(String)
    * @see   #getDefaultPackage
    */
   private UMLPackage rootPackage = null;


   /**
    * There should be no need to set the rootPackage outside of this class. But it must be
    * for the load and save mechanism, until now. Hopefully this will be corrected by inro
    * as soon as possible.
    *
    * @param rootPackage  The new rootPackage value
    */
   public void setRootPackage (UMLPackage rootPackage)
   {
      if (this.rootPackage != rootPackage)
      {
         // FujabaDebug.printStackTrace (44);
         UMLPackage oldValue = this.rootPackage;
         if (oldValue != null && oldValue.sizeOfPackages() > 0)
         {
            log.error ("Removed RootPackage with subpackages from project!");
         }
         this.rootPackage = rootPackage;
         firePropertyChange ("rootPackage", oldValue, rootPackage);
      }
   }


   /**
    * Get the rootPackage attribute of the UMLProject object
    *
    * @return   The rootPackage value
    */
   public UMLPackage getRootPackage()
   {
      if (this.rootPackage == null)
      {
         setRootPackage (new UMLPackage ("RootPackage"));
      }
      return this.rootPackage;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  Parameter description
    * @return     Return Value description
    */
   public boolean hasInPackages (String key)
   {
      return key == null ? false :  (this.getFromPackages (key) != null);
   }


   /**
    * Get the fromPackages attribute of the UMLProject object
    *
    * @param key  Parameter description
    * @return     The fromPackages value
    */
   public UMLPackage getFromPackages (String key)
   {
      StringTokenizer tokenizer = new StringTokenizer (key, ".");
      UMLPackage currentPackage = this.getRootPackage();
      String currentKey;

      while (tokenizer.hasMoreElements() && currentPackage != null)
      {
         currentKey = tokenizer.nextToken();
         currentPackage = currentPackage.getFromPackages (currentKey);
      }
      return currentPackage;
   }


   /**
    * Searches for a package with the full-name fullname and returns this package. If there
    * is no package, it will be created.
    *
    * @param fullName  Parameter description
    * @return          The newFromPackages value
    */
   public UMLPackage getNewFromPackages (String fullName)
   {
      return getNewFromPackages (fullName, !ASGElement.isInTransientMode());
   }


   /**
    * Get the newFromPackages attribute of the UMLProject object
    *
    * @param fullName          No description provided
    * @param coobraPersistent  No description provided
    * @return                  The newFromPackages value
    */
   public UMLPackage getNewFromPackages (String fullName, boolean coobraPersistent)
   {
      return findPackage (fullName, true, coobraPersistent);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fullName  No description provided
    * @return          No description provided
    */
   public UMLPackage findPackage (String fullName)
   {
      return findPackage (fullName, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fullName  No description provided
    * @param create    No description provided
    * @return          No description provided
    */
   public UMLPackage findPackage (String fullName, boolean create)
   {
      return findPackage (fullName, create, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fullName          No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public UMLPackage findPackage (String fullName, boolean create, boolean coobraPersistent)
   {
      // dont't return if fullName is empty, because
      // this means the default-package
      if (fullName == null)
      { //if (fullName == null || fullName.length() == 0)

         return null;
      }
      UMLPackage tmpPackage = this.getFromPackages (fullName);
      if (tmpPackage == null && create)
      {
         tmpPackage = new UMLPackage (coobraPersistent);
         if (log.isDebugEnabled())
         {
            log.debug ("Creating new Package: " + fullName);
         }
         this.addToPackages (tmpPackage, fullName);
      }
      return tmpPackage;
   }


   /**
    * Searches for the default project and returns it. Creates a default project if there was
    * no default project.
    *
    * @return   The defaultPackage value
    */
   public UMLPackage getDefaultPackage()
   {
      return this.getRootPackage();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public Iterator iteratorOfPackages()
   {
      return this.getRootPackage().iteratorOfPackages();
   }


   /**
    * gives back the package name of a full qualified class or type. for example: <pre>
    *    type = java.lang.String
    *    returns java.lang
    *
    *    type = java.util.Map$Entry
    *    returns java.util
    *
    *    type = java.util.Map.Entry
    *    returns java.util.Map
    * </pre>
    *
    * @param type  Parameter description
    * @return      The packageOfFullQualifiedType value
    */
   public static String getPackageOfFullQualifiedType (String type)
   {
      String result = "";
      if (type != null &&
         !type.equals (""))
      {
         // cut off any innerclasses
         String outerClass = UMLProject.getOutestClassOfFullQualifiedType (type);

         // if it was a regular type definition use it
         if ("".equals (outerClass))
         {
            outerClass = type;
         }

         int theDot = type.lastIndexOf ('.');
         if (theDot > 0)
         {
            // we have found a full qualified class
            // of style: x.y.z.Classname
            result = type.substring (0, theDot);
         }
      }
      return result;
   }


   /**
    * Works like method 'getOuterClassOfFullQualifiedType' but returns
    * the 'outest' class if an inner-class-definition has been passed.
    * <pre>
    * i.e. type = x.y.z.Outer$Inner1$...$InnerN-1$InnerN
    *      returns x.y.z.Outer
    * </pre>
    *
    * @param type  No description provided
    * @return      see method 'getOuterClassOfFullQualifiedType'
    * @see         UMLProject#getOuterClassOfFullQualifiedType(String)
    */
   public static String getOutestClassOfFullQualifiedType (String type)
   {
      String result = UMLProject.getOuterClassOfFullQualifiedType (type);

      // if we get the empty-string it wasn't an inner class-definition
      if (result == null || "".equals (result))
      {
         return result;
      }
      // if we get a non-empty-string, it is potentialy an inner-class-definition
      // of format 'x.y.z.RootOuterClass$Inner1$...$InnerN-1'
      else
      {
         int firstDollarIndex = result.indexOf ("$");

         // if a '$' exists, we return the part before the dollar
         if (firstDollarIndex > -1)
         {
            return result.substring (0, firstDollarIndex);
         }
         // otherwise it was a non-inner-class-definition and we simply return it
         else
         {
            return result;
         }
      }
   }


   /**
    * gives back the outer class name of an inner class. recognizes the
    * dollar format. for example: <pre>
    *    type = java.lang.String
    *    returns ""
    *
    *    type = java.util.Map$Entry
    *    returns java.util.Map
    *
    *    type = java.util.Map.Entry
    *    returns ""
    *    (because the specified type is not recognized as an innerclass definition)
    *
    *    type = Map$Entry
    *    returns Map
    *
    *    type = Map.Entry
    *    returns Map
    *    (because the first component is not a valid package name
    *    and so represents an outerclass
    *    (by Java convention the first component should start with a lowercase letter))
    *
    *    ------
    *    in general:
    *
    *    type = x.y.z.Outer$Inner1$...$InnerN-1$InnerN
    *    returns x.y.z.Outer$Inner1$...$InnerN-1
    *
    *    type = name1stCharLowerCase.AdditionalSegments
    *    returns ""
    *
    *    type = Name1stCharUpperCase.AdditionalSegments
    *    returns "Name1stCharUpperCase"
    * </pre>
    *
    * Note that java package names may have components, that start
    * with an uppercase letter (like class names do), so
    * one should separate innerclasses from outerclasses by '$'
    * instead of '.' to be sure the outerclass is detected.
    *
    * @param type  A full qualified class name
    * @return      If 'type' is an innerclass definition the full qualified
    * 	outerclass is returned, otherwise if 'type' is not an innerclass,
    *   the empty string is returned.
    */
   public static String getOuterClassOfFullQualifiedType (String type)
   {
      String result = "";
      if (type != null &&
         !type.equals (""))
      {
         int dollar = type.lastIndexOf ('$');

         if (dollar > -1)
         {
            result = type.substring (0, dollar);
         }
         else
         {
            // no explizit innerclass definition found:
            // check if the first component starts with an
            // uppercase letter (i.e., is a class and not a package definition)
            int theDot = type.indexOf ('.');
            if (theDot > -1)
            {
               // wrong package definition '.x.y.z.Class'
               if (theDot == 0)
               {
                  // print an error-message?
               }
               else
               {
                  String firstSegment = type.substring (0, theDot);
                  String firstChar = firstSegment.substring (0, 1);
                  if (firstChar.equals (firstChar.toUpperCase()))
                  {
                     // first segment represents a class, so
                     // we assume this is the outerclass
                     result = firstSegment;
                  }
                  else
                  {
                     // first segment is a package name.
                     // so the type MUST be a non-inner-class definition!
                     // for that we return the empty-string
                     //result = "";	// is already done at method start
                  }
               }
            }
         }
      }
      return result;
   }


   /**
    * gives back the class or type of a full qualified class or type. for example: <pre>
    *    type = java.lang.String
    *    returns String
    *
    *    type = java.util.Map.Entry
    *    returns Entry
    * </pre>
    *
    * @param type  Parameter description
    * @return      The typeOfFullQualifiedType value
    */
   public static String getTypeOfFullQualifiedType (String type)
   {
      // get the outer class of the specified type, if any.
      String result = getOuterClassOfFullQualifiedType (type);

      if (result.length() > 0)
      {
         // the result is either an inner-class-definition or a class-definition
         // with preceeding package name (and class-name)
         // we simply cut off this 'prefix' and the one character separator
         result = type.substring (result.length() + 1);
      }
      else
      {
         // specified type wasn't an inner class definition,
         // so we simply remove its preceeding package name
         result = removePackageFromFullQualifiedType (type);
      }

      return result;
   }


   /**
    * gives back the class name of a full qualified class with the package removed. for example:<pre>
    *    type = java.lang.String
    *    returns String
    *
    *    type = java.util.Map.Entry
    *    returns Map.Entry
    *    </pre>
    *
    * @param type  No description provided
    * @return      No description provided
    */
   public static String removePackageFromFullQualifiedType (String type)
   {
      String result = getPackageOfFullQualifiedType (type);

      if (result.length() > 0)
      {
         result = type.substring (result.length() + 1);
      }
      else
      {
         result = type;
      }

      return result;
   }


   /**
    * Adds a feature to the ToPackages attribute of the UMLProject object
    *
    * @param elem      Access method for an one to n association.
    * @param fullName  Access method for an one to n association.
    */
   public void addToPackages (UMLPackage elem, String fullName)
   {
      if (elem == null || fullName == null)
      {
         throw new RuntimeExceptionWithContext ("Cannot add an Element with key = null to an OrderedMap", elem);
      }

      if (fullName.length() != 0 && !this.hasInPackages (fullName))
      {
         StringTokenizer tokenizer = new StringTokenizer (fullName, ".");
         UMLPackage currentPackage = this.getRootPackage();
         UMLPackage parentOfCurrent = null;
         String currentKey = fullName;

         while (tokenizer.hasMoreElements() && currentPackage != null)
         {
            currentKey = tokenizer.nextToken();
            parentOfCurrent = currentPackage;
            currentPackage = parentOfCurrent.getFromPackages (currentKey);
            if (currentPackage == null && tokenizer.hasMoreElements())
            {
               currentPackage = new UMLPackage (currentKey);
               currentPackage.setParent (parentOfCurrent);
            }
         }
         if (currentPackage != null)
         {
            // replace the last package with elem;
            parentOfCurrent.removeFromPackages (currentPackage);
         }
         // adjust the name of elem
         elem.setName (currentKey);
         parentOfCurrent.addToPackages (elem);
         firePropertyChange (CollectionChangeEvent.get (this, "packages", parentOfCurrent, currentPackage,
            elem, fullName, CollectionChangeEvent.CHANGED));
      }
   }


   /**
    * for loading (versioning): add a package<br>
    * effect: addToPackages( pkg, key );
    *
    * @param fullName  full package name
    * @param pkg       the package to be added
    */
   public void addToPackages (String fullName, UMLPackage pkg)
   {
      addToPackages (pkg, fullName);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllPackages()
   {
      if (this.rootPackage != null)
      {
         rootPackage.removeYou();
         rootPackage = null;
      }
   }


   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient SourceCodeFactory sourceCodeFactory = null;


   /**
    * Get the sourceCodeFactory attribute of the UMLProject class
    *
    * @return   The sourceCodeFactory value
    */
   public static SourceCodeFactory getSourceCodeFactory()
   {
      if (sourceCodeFactory == null)
      {
         sourceCodeFactory = new JavaFactory();
      }
      return sourceCodeFactory;
   }


   // ######################################################################

   // Only used for internal handling, don't save it.
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient File file = null;


   /**
    * Get the file attribute of the UMLProject object
    *
    * @return   The file value
    */
   public File getFile()
   {
      return file;
   }


   /**
    * Sets the file attribute of the UMLProject object
    *
    * @param file  The new file value
    */
   public void setFile (File file)
   {
      if ( (this.file == null && file != null) ||
          (this.file != null && !this.file.equals (file)))
      {
         File oldValue = this.file;
         this.file = file;
         firePropertyChange ("file", oldValue, file);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String rootDir = "./generated"; //System.getProperty ("user.dir");


   /**
    * Get the rootDir attribute of the UMLProject object
    *
    * @return   The rootDir value
    */
   public String getRootDir()
   {
      return rootDir;
   }


   /**
    * Sets the rootDir attribute of the UMLProject object
    *
    * @param rootDir  The new rootDir value
    */
   public void setRootDir (String rootDir)
   {
      if ( (this.rootDir == null && rootDir != null) ||
          (this.rootDir != null && !this.rootDir.equals (rootDir)))
      {
         String oldValue = this.rootDir;
         this.rootDir = rootDir;
         firePropertyChange ("rootDir", oldValue, rootDir);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String additionalClassPath = "";


   /**
    * Get the rootDir attribute of the UMLProject object
    *
    * @return   The additionalClassPath value
    */
   public String getAdditionalClassPath()
   {
      return additionalClassPath;
   }


   /**
    * Sets the additionalClassPath attribute of the UMLProject object
    *
    * @param additionalClassPath  The new additionalClassPath value
    */
   public void setAdditionalClassPath (String additionalClassPath)
   {
      if ( (this.additionalClassPath == null && additionalClassPath != null) ||
          (this.additionalClassPath != null && !this.additionalClassPath.equals (additionalClassPath)))
      {
         String oldValue = this.additionalClassPath;
         this.additionalClassPath = additionalClassPath;
         firePropertyChange ("additionalClassPath", oldValue, additionalClassPath);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = "Unnamed Project";


   /**
    * Get the name attribute of the UMLProject object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Sets the name attribute of the UMLProject object
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if ( (this.name == null && name != null) ||
          (this.name != null && !this.name.equals (name)))
      {
         String oldValue = this.name;
         this.name = name;
         firePropertyChange ("name", oldValue, name);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient UMLIncrement clipboard = null;


   /**
    * Get the clipboard attribute of the UMLProject object
    *
    * @return   The clipboard value
    */
   public UMLIncrement getClipboard()
   {
      return clipboard;
   }


   /**
    * Sets the clipboard attribute of the UMLProject object
    *
    * @param clipboard  The new clipboard value
    */
   public void setClipboard (UMLIncrement clipboard)
   {
      if ( (this.clipboard == null && clipboard != null) ||
          (this.clipboard != null && !this.clipboard.equals (clipboard)))
      {
         UMLIncrement oldValue = this.clipboard;
         this.clipboard = clipboard;
         firePropertyChange ("clipboard", oldValue, clipboard);
      }
   }

   // ######################################################################

   /**
    * use this to get all the files of the project
    */
   private FPropTreeSet files;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  Parameter description
    * @return       Return Value description
    */
   public boolean hasInFiles (UMLFile value)
   {
      return  ( (this.files != null) &&
          (value != null) &&
         this.files.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  Parameter description
    * @return      Return Value description
    */
   public boolean hasInFiles (String name)
   {
      return  (this.getFromFiles (name) != null);
   }


   /**
    * Get the fromFiles attribute of the UMLProject object
    *
    * @param name  Parameter description
    * @return      The fromFiles value
    */
   public UMLFile getFromFiles (String name)
   {
      if (name != null)
      {
         UMLFile file = null;
         Iterator iter = this.iteratorOfFiles();
         while (iter.hasNext())
         {
            file = (UMLFile) iter.next();
            if (name.equals (file.getName()))
            {
               return file;
            }
         }
      }
      return null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param searchName  No description provided
    * @return            No description provided
    */
   public Iterator findFiles (String searchName)
   {
      if (searchName == null)
      {
         return FEmptyIterator.get();
      }

      FHashSet resultSet = new FHashSet();

      Iterator iter = iteratorOfFiles();
      while (iter.hasNext())
      {
         UMLFile aFile = (UMLFile) iter.next();

         if (searchName.equals (aFile.getName()))
         {
            resultSet.add (aFile);
         }

      }

      return resultSet.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void repairFiles()
   {
      // iterate through all files and remove duplicates and unnessary entries

      Iterator patternIter = UMLProject.get().findFiles ("Pattern");
      while (patternIter.hasNext())
      {
         UMLFile element = (UMLFile) patternIter.next();
         element.removeYouAndMyClasses();
      }

      FHashMap seenFiles = new FHashMap();
      Iterator iter = iteratorOfFiles();
      while (iter.hasNext())
      {
         UMLFile aFile = (UMLFile) iter.next();

         if (aFile.sizeOfContains() <= 0)
         {
            aFile.removeYou();
         }
         else
         {

            UMLClass aClass =  ((UMLClass)  (aFile.iteratorOfContains().next()));

            if (!aFile.getName().equals (aClass.getName()))
            {
               if (log.isInfoEnabled())
               {
                  log.info ("class and file name differ " + aFile.getName());
               }

            }

            // is this class know to the UMLProject?
            if (aFile.getName().startsWith ("_@"))
            {
               aFile.removeYouAndMyClasses();
               continue;
            }

            UMLFile oldFile = (UMLFile) seenFiles.get (aFile.getName());
            if (oldFile != null)
            {
               // double file
               if (log.isInfoEnabled())
               {
                  log.info ("I remove double file for reference class " + aFile.getName());
               }
               aFile.removeYou();
               continue;
            }
            else
            {
               seenFiles.put (aFile.getName(), aFile);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public Iterator iteratorOfFiles()
   {
      return  ( (this.files == null)
         ? FEmptyIterator.get()
         : this.files.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public int sizeOfFiles()
   {
      return  ( (this.files == null)
         ? 0
         : this.files.size());
   }


   /**
    * Adds a feature to the ToFiles attribute of the UMLProject object
    *
    * @param value  Access method for an one to n association.
    * @return       Return Value description
    */
   public boolean addToFiles (UMLFile value)
   {
      boolean changed = false;
      if (value != null &&
         !hasInFiles (value))
      {
         if (this.files == null)
         {
            this.files = new FPropTreeSet (this, PROPERTY_FILES);
         }
         changed = this.files.add (value);
         if (changed)
         {
            value.setProject (this);
         }
      }
      return changed;
   }


   /**
    * Adds a feature to the ToFiles attribute of the UMLProject object
    *
    * @param pair  Access method for an one to n association.
    */
   public void addToFiles (KeyValuePair pair)
   {
      // this method is needed only for loading OrderedMaps
      // do not use it in other cases
      if (pair == null)
      {
         return;
      }

      UMLFile elem = (UMLFile) pair.getValue();
      String key = (String) pair.getKey();

      if ( (elem != null) &&
          (key != null))
      {
         elem.setName (key);
      }
      this.addToFiles (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  Parameter description
    * @return       Return Value description
    */
   public boolean removeFromFiles (UMLFile value)
   {
      boolean changed = false;
      if ( (this.files != null) &&  (value != null))
      {
         changed = this.files.remove (value);
         if (changed)
         {
            value.setProject (null);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromFiles()
   {
      UMLFile tmpValue;
      Iterator iter = this.iteratorOfFiles();
      while (iter.hasNext())
      {
         tmpValue = (UMLFile) iter.next();
         this.removeFromFiles (tmpValue);
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  Parameter description
    * @return     Return Value description
    */
   public boolean hasInBaseTypes (String key)
   {
      return  (key == null)
         ? false
         :  (this.getTypeList().getFromTypes (key) instanceof UMLBaseTypes);
   }


   /**
    * Get the fromBaseTypes attribute of the UMLProject object
    *
    * @param key  Parameter description
    * @return     The fromBaseTypes value
    */
   public UMLBaseTypes getFromBaseTypes (String key)
   {
      UMLType aType = this.getTypeList().getFromTypes (key);
      if (aType instanceof UMLBaseTypes)
      {
         return (UMLBaseTypes) aType;
      }
      return null;
   }


   /**
    * use this to get all the classes of the project
    *
    * @return   Return Value description
    */
   public Enumeration elementsOfClasses()
   {
      return new UMLTreeEnumeration (this.getRootPackage(), UMLClass.class, true);
   }


   /**
    * Query if a class exists
    *
    * @param fullClassName  full qualified class name
    * @return               <code>getFromClasses (fullClassName) != null</code>
    */
   public boolean hasInClasses (String fullClassName)
   {
      return this.getFromClasses (fullClassName) != null;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  Parameter description
    * @return      Return Value description
    */
   public boolean hasInClasses (UMLClass elem)
   {
      return  (elem == null) ? false : hasInClasses (elem.getFullClassName());
   }


   /**
    * Retrieve a class with the given full qualified (with package) class name.
    *
    * @param fullClassName  full qualified class name
    * @return               the class with the given package and name
    */
   public UMLClass getFromClasses (String fullClassName)
   {

      if (fullClassName != null)
      {
         Enumeration enumeration = this.elementsOfClasses();
         UMLClass current = null;
         while (enumeration.hasMoreElements())
         {
            current = (UMLClass) enumeration.nextElement();

            if (fullClassName.equals (current.getFullClassName()))
            {
               return current;
            }
         }
      }
      return null;
   }

   // ######################################################################

   /**
    * <pre>
    *            +------+ 1                                   1
    * UMLProject + name +--------------------------------------- UMLClass
    *            +------+ declaredInReferences       references
    * </pre>
    */
   private FPropHashMap references = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  Parameter description
    * @return     Return Value description
    */
   public boolean hasInReferences (UMLClass obj)
   {
      return  ( (this.references != null) &&
          (obj != null) &&  (obj.getFullClassName() != null) &&
          (this.references.get (obj.getFullClassName()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  full qualified class name
    * @return     Return Value description
    */
   public boolean hasKeyInReferences (String key)
   {
      return  ( (this.references != null) &&
          (key != null) &&
         this.references.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public Iterator iteratorOfReferences()
   {
      return  ( (this.references == null)
         ? FEmptyIterator.get()
         : this.references.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public Iterator keysOfReferences()
   {
      return  ( (this.references == null)
         ? FEmptyIterator.get()
         : this.references.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public Iterator entriesOfReferences()
   {
      return  ( (this.references == null)
         ? FEmptyIterator.get()
         : this.references.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public int sizeOfReferences()
   {
      return  ( (this.references == null)
         ? 0
         : this.references.size());
   }


   /**
    * Get the fromReferences attribute of the UMLProject object
    *
    * @param key  full qualified class name
    * @return     The fromReferences value
    */
   public UMLClass getFromReferences (String key)
   {
      return  ( ( (this.references == null) ||  (key == null))
         ? null
         : (UMLClass) this.references.get (key));
   }


   /**
    * Adds a feature to the ToReferences attribute of the UMLProject object
    *
    * @param obj  Access method for an one to n association.
    * @return     Return Value description
    */
   public boolean addToReferences (UMLClass obj)
   {
      boolean changed = false;
      if (!hasInReferences (obj))
      {
         if (this.references == null)
         {
            this.references = new FPropHashMap (this, "references");
         }
         UMLClass oldValue = (UMLClass) this.references.put (obj.getFullClassName(), obj);
         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setDeclaredInReferences (null);
            }
            obj.setDeclaredInReferences (this);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  Parameter description
    * @return     Return Value description
    */
   public boolean removeFromReferences (UMLClass obj)
   {
      boolean changed = false;

      if (hasInReferences (obj))
      {
         UMLClass oldValue = (UMLClass) this.references.get (obj.getFullClassName());
         if (oldValue == obj)
         {
            this.references.remove (obj.getFullClassName());
            obj.setDeclaredInReferences (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * for loading
    *
    * @param key  (ignored)
    * @param obj  Class to remove from references
    * @return     true when something was changed
    */
   public boolean removeFromReferences (String key, UMLClass obj)
   {
      //don't care about the key
      return removeFromReferences (obj);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  full qualified class name
    * @return     Return Value description
    */
   public boolean removeKeyFromReferences (String key)
   {
      boolean changed = false;

      if (hasKeyInReferences (key))
      {
         UMLClass tmpObj = (UMLClass) this.references.get (key);
         if (tmpObj != null)
         {
            this.references.remove (key);
            tmpObj.setDeclaredInReferences (null);
            changed = true;
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromReferences()
   {
      UMLClass tmpObj;
      Iterator iter = this.iteratorOfReferences();

      while (iter.hasNext())
      {
         tmpObj = (UMLClass) iter.next();
         this.removeFromReferences (tmpObj);
         tmpObj.removeYou();
      }
   }


   /**
    * Adds a feature to the ToReferences attribute of the UMLProject object<br>
    * this method is needed only for loading OrderedMaps do not use it in other cases
    *
    * @param pair  key-value-pair
    */
   public void addToReferences (KeyValuePair pair)
   {
      if (pair == null)
      {
         return;
      }

      UMLClass elem = (UMLClass) pair.getValue();
      String key = (String) pair.getKey();

      addToReferences (key, elem);
   }


   /**
    * Adds a feature to the ToReferences attribute of the UMLProject object<br>
    * this method is needed only for loading OrderedMaps do not use it in other cases
    *
    * @param key    key for
    * @param value  new value
    */
   public void addToReferences (String key, UMLClass value)
   {
      if ( (value != null) &&
          (key != null) && !FujabaChangeManager.isInUndoRedo())
      {
         value.setName (key);
      }
      this.addToReferences (value);
   }

   //###############################################################
   // Some additional functionality to retrieve decls from names
   /**
    * Get the orNewType attribute of the UMLProject object
    *
    * @param context  Parameter description
    * @param name     Parameter description
    * @return         The orNewType value
    */
   public UMLType getOrNewType (UMLClass context, String name)
   {
      return getOrNewType (context, name, true);
   }


   /**
    * Get the orNewType attribute of the UMLProject object
    *
    * @param context           No description provided
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @return                  The orNewType value
    */
   public UMLType getOrNewType (UMLClass context, String name, boolean coobraPersistent)
   {
      return findType (context, name, true, coobraPersistent);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context  No description provided
    * @param name     No description provided
    * @return         No description provided
    */
   public UMLType findType (UMLClass context, String name)
   {
      return findType (context, name, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context  No description provided
    * @param name     No description provided
    * @param create   No description provided
    * @return         No description provided
    */
   public UMLType findType (UMLClass context, String name, boolean create)
   {
      return findType (context, name, create, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context           No description provided
    * @param name              No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public UMLType findType (UMLClass context, String name, boolean create, boolean coobraPersistent)
   {
      String className = getTypeOfFullQualifiedType (name);
      if (className.equals (name))
      {
         UMLType result = this.getFromBaseTypes (className);
         if (result != null)
         {
            return result;
         }
      }
      return findClass (context, name, create, coobraPersistent);
   }


   /**
    * Get the orNewClass attribute of the UMLProject object
    *
    * @param name  Parameter description
    * @return      The orNewClass value
    */
   public UMLClass getOrNewClass (String name)
   {
      return getOrNewClass (name, true);
   }


   /**
    * Get the orNewClass attribute of the UMLProject object
    *
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @return                  The orNewClass value
    */
   public UMLClass getOrNewClass (String name, boolean coobraPersistent)
   {
      UMLClass theClass = this.getOrNewClass (null, name, coobraPersistent);

      theClass.removeKeyFromStereotypes ("reference");
      theClass.getFile();

      return theClass;
   }


   /**
    * Get the orNewClass attribute of the UMLProject object
    *
    * @param context  Parameter description
    * @param name     Parameter description
    * @return         The orNewClass value
    */
   public UMLClass getOrNewClass (UMLClass context, String name)
   {
      return getOrNewClass (context, name, true);
   }


   /**
    * Get the orNewClass attribute of the UMLProject object
    *
    * @param context           No description provided
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @return                  The orNewClass value
    */
   public UMLClass getOrNewClass (UMLClass context, String name, boolean coobraPersistent)
   {
      return findClass (context, name, true, coobraPersistent);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context  No description provided
    * @param name     No description provided
    * @return         No description provided
    */
   public UMLClass findClass (UMLClass context, String name)
   {
      return findClass (context, name, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context  No description provided
    * @param name     No description provided
    * @param create   No description provided
    * @return         No description provided
    */
   public UMLClass findClass (UMLClass context, String name, boolean create)
   {
      return findClass (context, name, create, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context           No description provided
    * @param name              No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public UMLClass findClass (UMLClass context, String name, boolean create, boolean coobraPersistent)
   {
      UMLClass umlClass = null;

      String tmpName = getTypeOfFullQualifiedType (name);
      String outerClassName = getOuterClassOfFullQualifiedType (name);
      String packageName = getPackageOfFullQualifiedType (name);

      if (!outerClassName.equals (""))
      {
         // fully qualified name used.
         // just look up the outer class
         UMLClass outerClass = findClass (null, outerClassName, create, coobraPersistent);
         if (outerClass != null)
         {
            umlClass = outerClass.getFromDeclares (tmpName);
         }

         if (umlClass == null)
         {
            umlClass = findReference (name, create, coobraPersistent);
         }
      }
      else if (!packageName.equals (""))
      {
         // fully qualified name used.
         // just look up the package
         UMLPackage tmpPkg = findPackage (packageName, create, coobraPersistent);
         if (tmpPkg != null)
         {
            umlClass = tmpPkg.getFromDeclares (tmpName);
         }
         if (umlClass == null)
         {
            //FIXME: Let's not change the package of an existing class. Does
            //       this break something?
            umlClass = findReference (name, create, coobraPersistent);
            //umlClass = findReference (tmpName, create, coobraPersistent);
            //umlClass.setDeclaredInPackage (tmpPkg);
         }
      }

      if ( (umlClass == null) &&  (context != null))
      {
         umlClass = this.getClassFromImports (context, name);

         if (umlClass == null && packageName.equals ("") && outerClassName.equals (""))
         {
            UMLPackage tmpPkg = context.getDeclaredInPackage();
            // check if context is in java.lang Package
            if (name.equals ("String") ||
               name.equals ("Integer") ||
               name.equals ("Object") ||
               name.equals ("Thread") ||
               name.equals ("Class") ||
               name.equals ("Double") ||
               name.equals ("Boolean") ||
               name.equals ("Float"))
            { // to be continued

               tmpPkg = UMLProject.get().getNewFromPackages ("java.lang");
            }

            if (tmpPkg != null)
            {
               String pkgName = tmpPkg.getFullPackageName();
               tmpName =  (pkgName.length() > 0 ? pkgName + "." + tmpName : tmpName);

               umlClass = findReference (tmpName, create, coobraPersistent);
               //FIXME: Let's not change the package of an existing class. Does
               //       this break something?
               //umlClass.setDeclaredInPackage (tmpPkg);
            }
         }
      }

      if (umlClass == null)
      {
         umlClass = this.getFromClasses (name);
      }

      if (umlClass == null)
      {
         umlClass = this.findReference (name, false, coobraPersistent);
      }

      if (umlClass == null && create)
      {
         umlClass = createClass (name, coobraPersistent, false);

      }
      return umlClass;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   Return Value description
    */
   public Iterator iteratorOfAllUndefinedMethods()
   {
      TreeMap methodMap = new TreeMap();
      UMLMethod method = null;

      Iterator methodIter = null;
      String key = null;

      Iterator iter = iteratorOfReferences();
      while (iter.hasNext())
      {
         UMLClass theClass = (UMLClass) iter.next();
         methodIter = theClass.iteratorOfUndefinedMethods();
         while (methodIter.hasNext())
         {
            method = (UMLMethod) methodIter.next();

            key = theClass.getName() + ":" + method.getName();
            methodMap.put (key, method);
         } // end of while ()

      } // end of while ()

      Enumeration enumeration = elementsOfClasses();
      while (enumeration.hasMoreElements())
      {
         UMLClass theClass = (UMLClass) enumeration.nextElement();
         methodIter = theClass.iteratorOfUndefinedMethods();
         while (methodIter.hasNext())
         {
            method = (UMLMethod) methodIter.next();

            key = theClass.getName() + ":" + method.getName();
            methodMap.put (key, method);
         } // end of while ()

      } // end of while ()

      return methodMap.values().iterator();
   }


   /**
    * Searches the UMLClass with the name in the imports of the current UMLFile.
    *
    * @param context  Parameter description
    * @param name     Parameter description
    * @return         The classFromImports value
    */
   public UMLClass getClassFromImports (UMLClass context, String name)
   {
      if (context == null)
      {
         if (log.isDebugEnabled())
         {
            log.debug (
               "ERROR: getClassFromImports called with null context");
         }
         FujabaDebug.printStackTrace (333);

         return null;
      }

      UMLClass umlClass = null;

      if (log.isDebugEnabled())
      {
         log.debug ("Searching for:" + name);
      }

      // search the current package for the class
      UMLPackage contextPackage = context.getDeclaredInPackage();
      if (contextPackage != null)
      {
         umlClass = contextPackage.getFromDeclares (name);
      }
      else
      {
         if (log.isDebugEnabled())
         {
            log.debug ("ERROR: tried to analyse class without package: "
               + context.getName()
               + "\nThat should not happen");
         }
      }

      if (umlClass == null)
      {
         // search the imports
         UMLFile contextFile = context.getFile();
         if (contextFile != null)
         {
            umlClass = contextFile.getClassFromImports (name);
         }
      }
      if (umlClass != null)
      {
         String fullName = umlClass.getFullClassName();
         String outerClassName = getOuterClassOfFullQualifiedType (name);
         String packageName = getPackageOfFullQualifiedType (name);

         if ( (!outerClassName.equals ("") &&
            !outerClassName.equals (getOuterClassOfFullQualifiedType (fullName))) ||
             (!packageName.equals ("") &&
            !packageName.equals (getPackageOfFullQualifiedType (fullName))))
         {
            umlClass = null;
         }
      }
      return umlClass;
   }


   /**
    * Get the orNewFromReferences attribute of the UMLProject object
    *
    * @param name  full qualified class name
    * @return      The orNewFromReferences value
    */
   public UMLClass getOrNewFromReferences (String name)
   {
      return getOrNewFromReferences (name, true);
   }


   /**
    * Get the orNewFromReferences attribute of the UMLProject object
    *
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @return                  The orNewFromReferences value
    */
   public UMLClass getOrNewFromReferences (String name, boolean coobraPersistent)
   {
      return findReference (name, true, coobraPersistent);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   public UMLClass findReference (String name)
   {
      return findReference (name, false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name    No description provided
    * @param create  No description provided
    * @return        No description provided
    */
   public UMLClass findReference (String name, boolean create)
   {
      return findReference (name, create, true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name              No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public UMLClass findReference (String name, boolean create, boolean coobraPersistent)
   {
      UMLClass umlClass = this.getFromReferences (name);
      if (umlClass == null)
      {
         umlClass = this.getFromClasses (name);
      }
      if (umlClass == null && create)
      {
         umlClass = createClass (name, coobraPersistent, true);
      }

      return umlClass;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @param reference         No description provided
    * @return                  No description provided
    */
   private UMLClass createClass (String name, boolean coobraPersistent, boolean reference)
   {
      UMLClass umlClass = new UMLClass (coobraPersistent);
      String parentClassName = getOuterClassOfFullQualifiedType (name);
      String packageName = getPackageOfFullQualifiedType (name);
      name = getTypeOfFullQualifiedType (name);

      umlClass.setName (name);
      umlClass.setRevTypes (getTypeList());
      if (reference)
      {
         umlClass.addToStereotypes (UMLStereotypeManager.get().getFromStereotypes (UMLStereotypeManager.REFERENCE));
      }

      if (!parentClassName.equals (""))
      {
         UMLClass tmpClz = getOrNewFromReferences (parentClassName, coobraPersistent);
         umlClass.setDeclaredInClass (tmpClz);
      }
      else if (!packageName.equals (""))
      {
         UMLPackage tmpPkg = getNewFromPackages (packageName, coobraPersistent);
         umlClass.setDeclaredInPackage (tmpPkg);
      }
      else
      {
         if (reference)
         {
            addToReferences (umlClass);
         }
         else
         {
            umlClass.setDeclaredInPackage (getDefaultPackage());
         }
      }
      return umlClass;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDiags()
   {
      ASGDiagram diagram;
      Iterator diagIter = iteratorOfDiags();
      while (diagIter.hasNext())
      {
         diagram = (ASGDiagram) diagIter.next();
         if (diagram instanceof UMLDiagram)
         {
            UMLDiagram umlDiagram = (UMLDiagram) diagram;
            Iterator itemIter = umlDiagram.iteratorOfElements();
            while (itemIter.hasNext())
            {
               ASGElement element = (ASGElement) itemIter.next();
               element.removeYou();
            }
         }
         removeFromDiags (diagram);
         diagram.removeYou();
      }
   }


   /**
    * Get the fromDiags attribute of the UMLProject object
    *
    * @param name  Parameter description
    * @return      The fromDiags value
    */
   public UMLDiagram getFromDiags (String name)
   {
      ASGDiagram diag = retrieveDiagramByName (name);
      if (diag instanceof UMLDiagram)
      {
         return (UMLDiagram) diag;
      }
      return null;
   }


   /**
    * Retrieve the diagram by the specified name.
    *
    * @param name  The name of the diagram to search for.
    * @return      Return the diagram or null, if no such diagram exists.
    */
   public static ASGDiagram retrieveDiagramByName (String name)
   {
      ASGDiagram foundDiagram = null;

      Iterator iter = UMLProject.get().iteratorOfDiags();
      while (iter.hasNext() &&  (foundDiagram == null))
      {
         ASGDiagram diagram = (ASGDiagram) iter.next();
         if ( (diagram.getName() != null) &&
             (diagram.getName().equals (name)))
         {
            foundDiagram = diagram;
         }
      }

      return foundDiagram;
   } // retrieveDiagramByName


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLTypeList typeList; // reverse UMLProject project


   /**
    * Get the typeList attribute of the UMLProject object
    *
    * @return   The typeList value
    */
   public UMLTypeList getTypeList()
   {
      return typeList;
   }


   /**
    * Sets the typeList attribute of the UMLProject object
    *
    * @param typeList  The new typeList value
    */
   public void setTypeList (UMLTypeList typeList)
   {
      if ( (this.typeList == null && typeList != null) ||
          (this.typeList != null && !this.typeList.equals (typeList)))
      { // new partner

         UMLTypeList oldTypeList = this.typeList;
         if (this.typeList != null)
         { // inform old partner

            this.typeList = null;
            oldTypeList.setProject (null);
         }
         this.typeList = typeList;
         if (typeList != null)
         { // inform new partner

            typeList.setProject (this);
         }
         firePropertyChange (PROPERTY_TYPE_LIST, oldTypeList, typeList);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FrameMain gui = null; // reverse UMLProject umlProject


   /**
    * Get the gui attribute of the UMLProject object
    *
    * @return   The gui value
    */
   public FrameMain getGui()
   {
      if (gui == null)
      {
         return FrameMain.get();
      }
      return gui;
   }


   /**
    * Sets the gui attribute of the UMLProject object
    *
    * @param gui  The new gui value
    */
   public void setGui (FrameMain gui)
   {
      if ( (this.gui == null && gui != null) ||
          (this.gui != null && !this.gui.equals (gui)))
      { // new partner

         FrameMain oldGui = this.gui;
         if (this.gui != null)
         { // inform old partner

            this.gui = null;
            oldGui.setUMLProject (null);
         }
         this.gui = gui;
         if (gui != null)
         { // inform new partner

            gui.setUMLProject (this);
         }
         firePropertyChange ("gui", oldGui, gui);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean saved = true;


   /**
    * Get the saved attribute of the UMLProject object
    *
    * @return   The saved value
    */
   public boolean isSaved()
   {
      return saved;
   }


   /**
    * Sets the saved attribute of the UMLProject object
    *
    * @param saved  The new saved value
    */
   public void setSaved (boolean saved)
   {
      if (this.saved != saved)
      {
         boolean oldValue = this.saved;
         this.saved = saved;
         firePropertyChange ("saved", oldValue, saved);
      }
   }


   /**
    * Creates a *.bak.fpr project backup if the given outputFile quite exists.
    *
    * @param outputFile  No description provided
    * @return            true if backup was created successfully
    */
   private boolean createProjectBackup (File outputFile)
   {
      //create a *.bak.fpr backup of the file
      try
      {
         if (outputFile.exists())
         {
            File newBackupFile = new File (outputFile.getCanonicalPath());
            String backupName = outputFile.getCanonicalPath();
            backupName = backupName.substring (0, backupName.lastIndexOf (".fpr")) +
               ".bak.fpr";

            File oldBackupFile = new File (backupName);
            if (oldBackupFile.exists())
            {
               if (! (oldBackupFile.canWrite() && oldBackupFile.delete()))
               {
                  if (JOptionPane.showConfirmDialog (UMLProject.get().getGui().getFrame(),
                     "Not able to create a new project backup. \n" +
                     "Backup of project " + outputFile.getName() +
                     "\n" + "exists and is write protected! \n" +
                     "Continue anyway?",
                     "Save project", JOptionPane.YES_NO_OPTION, JOptionPane.ERROR_MESSAGE) == JOptionPane.YES_NO_OPTION)
                  {
                     return true;
                  }
                  else
                  {
                     return false;
                  }
               }
            }

            newBackupFile.renameTo (new File (backupName));
            return true;
         }
         return true;
      }
      catch (Exception e)
      {
         if (JOptionPane.showConfirmDialog (UMLProject.get().getGui().getFrame(),
            "Not able to create a new project backup. \n" +
            e.getMessage() + "\n\n" +
            "Continue anyway?",
            "Save project", JOptionPane.YES_NO_OPTION, JOptionPane.ERROR_MESSAGE) == JOptionPane.YES_NO_OPTION)
         {
            return true;
         }
         else
         {
            return false;
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param outputFile  Parameter description
    * @return            Return Value description
    */
   public boolean saveProject (File outputFile)
   {
      boolean result = true;
      BufferedWriter fw = null;
      int objectTablePos = 0;
      int pluginTablePos = 0;

      StringBuffer outputData = new StringBuffer (131072);
      FTreeSet savedIncrements = new FTreeSet (FujabaComparator.getLessBasicIncr());

      if (GeneralPreferences.get().isCreateProjectBackup())
      {
         //if creating a project backup fails, user can abort saving
         if (!createProjectBackup (outputFile))
         {
            return false;
         }
      }

      try
      {
         PluginProperty plugin = null;

         // write the header at the beginning of the file
         outputData.append ("# Fujaba-Project-File (do not alter this file!!!)\n");
         outputData.append ("# Filename: ").append (outputFile.getName()).append ("\n");
         outputData.append ("# Date    : ").append (new Date (System.currentTimeMillis()).toString()).append ("\n");
         outputData.append ("-;FileVersion;").append (Integer.toString (FILE_VERSION)).append ("\n");

         objectTablePos = outputData.length();
         pluginTablePos = outputData.length();

         outputData.append ("# Object references\n");

         // write every class and their attributes to the StringBuffer
         writeClassToStringBuffer (outputData, savedIncrements);

         // If everything is ok, insert the table [ID, objectType] at the
         // beginning of the StringBuffer
         StringBuffer objectTable = new StringBuffer (16384);
         Iterator iter = savedIncrements.iterator();
         BasicIncrement incr;

         objectTable.append ("# HashTable of this File\n");
         objectTable.append ("-;HashTableLength;");
         objectTable.append (Integer.toString (savedIncrements.size())).append ("\n");

         StringBuffer pluginBuffer = new StringBuffer();
         pluginBuffer.append ("# Used Fujaba core\n");
         // the fujaba core entry has the form: $;name;id;major;minor;revision
         pluginBuffer.append ("$;The Fujaba kernel;" + UPBClassLoader.DEFAULT_CLASSLOADER + ";" + Version.get().getMajor() + ";" + Version.get().getMinor() + ";" + Version.get().getRevision() + "\n");

         pluginBuffer.append ("# used plug-ins\n");
         // the inserted plug-ins have the structure
         // $;plugin-name;pluginID;MajorVersion;MinorVersion;buildnumber
         // $;My Plugin;de.upb.myplugin.MyPlugin;1;10;1020

         Hashtable pluginHashtable = new Hashtable();

         ClassLoader theLoader;

         //String id = null;
         while (iter.hasNext())
         {
            incr = (BasicIncrement) iter.next();
            theLoader = incr.getClass().getClassLoader();

            if (FujabaApp.getPersistencySupport().isPluginClassLoader (theLoader))
            {
               // a plug-in is responsible for this object
               String loaderID = FujabaApp.getPersistencySupport().getClassLoaderKey (theLoader);
               plugin = FujabaApp.getPersistencySupport().getPluginProperty (loaderID);
               if (plugin == null)
               {
                  log.error ("***\nCan not extract plug-in: " + loaderID + " from plug-in list.\n***");
                  continue;
               }
               if (pluginHashtable.containsKey (plugin.getPluginID()))
               {
               }
               else
               {
                  pluginHashtable.put (plugin.getPluginID(), plugin);
                  pluginBuffer.append ("$;" + plugin.getName() + ";" + plugin.getPluginID() + ";" + plugin.getMajor() + ";" + plugin.getMinor() + ";" + plugin.getBuildNumber() + "\n");
               }
               objectTable.append ("+;").append (incr.getID()).append (";");
               objectTable.append (incr.getClass().getName()).append (";");
               objectTable.append (plugin.getPluginID() + "\n"); // counterID for numbering the needed plug-ins to load this project file
            }
            else
            {
               objectTable.append ("+;").append (incr.getID()).append (";");
               objectTable.append (incr.getClass().getName()).append (";");
               objectTable.append (UPBClassLoader.DEFAULT_CLASSLOADER + "\n");
            }
         }

         outputData.insert (pluginTablePos, pluginBuffer);

         outputData.insert (objectTablePos + pluginBuffer.length(), objectTable);

         // now write the StringBuffer to the file
         fw = new BufferedWriter (new OutputStreamWriter (new GZIPOutputStream (new FileOutputStream (outputFile))));
         fw.write (outputData.toString(), 0, outputData.length());

         setSaved (true);
         FileHistory.get().addToHistory (outputFile);
         FileHistory.get().updateActions();
      }
      catch (Exception e)
      {
         result = false;
         if (log.isInfoEnabled())
         {
            log.info (e.getMessage());
         }
         e.printStackTrace();
         throw new RuntimeException ("An error occured while writing the file " +
            outputFile.getName() + "\nMessage : " + e.getMessage());
      }

      finally
      {
         savedIncrements = null;
         BasicIncrement.resetClassInfos();
         System.gc();
         try
         {
            if (fw != null)
            {
               fw.flush();
               fw.close();
            }
            boolean wasSaved = isSaved();
            setFile (outputFile);
            setSaved (wasSaved);
         }
         catch (Exception e)
         {
            result = false;
            if (log.isInfoEnabled())
            {
               log.info (e.getMessage() + "\n" + outputData);
            }
            e.printStackTrace();

            throw new RuntimeException (e.getMessage());
         }
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  Parameter description
    * @return      Return Value description
    */
   private final static StringTokenizer createLineTokenizer (String line)
   {
      StringTokenizer st = new StringTokenizer (line, ";");

      st.nextToken();

      return st;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static transient boolean loading = false;


   /**
    * Get the loading attribute of the UMLProject class
    *
    * @return   The loading value
    */
   public static boolean isLoading()
   {
      return loading;
   }


   /**
    * Sets the loading attribute of the UMLProject class
    *
    * @param flag  The new loading value
    */
   public static void setLoading (boolean flag)
   {
      loading = flag;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param inputFile  No description provided
    * @return           No description provided
    */
   public final static UMLProject loadProject (File inputFile)
   {
      BufferedReader reader;
      try
      {
         try
         {
            reader = new BufferedReader (new InputStreamReader (new GZIPInputStream (new FileInputStream (inputFile))), 32768);
         }
         catch (IOException e)
         {
            reader = new BufferedReader (new FileReader (inputFile), 32768);
         }
         loadProject (reader, inputFile.getName());
         UMLProject.get().setFile (inputFile);
      }
      catch (FileNotFoundException ex)
      {
         JOptionPane.showMessageDialog (null, "Could not open file: " + inputFile.getName(), "Load", JOptionPane.ERROR_MESSAGE);
      }

      return UMLProject.get();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static Hashtable objectHashTable = null;


   /**
    * this method can be used on load time to find a restored object
    *
    * @param id
    * @return    object with specified id
    */
   public static Object getFromObjectHashTable (String id)
   {
      if (isLoading() && objectHashTable != null)
      {
         return objectHashTable.get (id);
      }
      else
      {
         return null;
      }
   }


   /**
    * Original loadProject method.
    *
    * @param reader    No description provided
    * @param fileName  No description provided
    * @return          Return Value description
    */
   public final static UMLProject loadProject (BufferedReader reader, String fileName)
   {
      setLoading (true);
      try
      {
         // get the instance of UMLProject associated with projectKey
         UMLProject project = new UMLProject();
         setNewProject (project);

         BasicIncrement currentIncr = null;
         String line = null;
         String className = null;
         int fileVersion;
         int objectHashTableLength = 0;
         int lineNum = 0;
         int objCount = 0;

         Hashtable plugins = new Hashtable();

         // Changes to the project after loading it should only be applied
         // to the user project, use this switch for this.
         StringTokenizer lineTok;
         FDuplicatedTreeMap attributeHashTable = new FDuplicatedTreeMap();
         boolean notYetWarned = true;

         String maxUniqueId = "0";

         if (log.isDebugEnabled())
         {
            log.debug ("loading project from file " + fileName);
         }

         try
         {
            // check, if it is the correct FileFormat
            line = reader.readLine();
            lineNum++;
            if (!line.startsWith ("# Fujaba-Project-File"))
            {
               throw new Exception ("This is not a Fujaba project file");
            }

            // Now parse the rest of the file
            while (reader.ready())
            {
               lineNum++;
               line = reader.readLine();
               if (line == null)
               {
                  break;
               }

               lineTok = createLineTokenizer (line);

               //  for temporarly use only
               // the plug-in entries with old file format (line=$pluginName?pluginID#Major%Minor)
               // the plug-in entries with new file format (line=$;counterID;pluginName;pluginID;Major;Minor)

               switch (line.charAt (0))
               {
                  case '#': // documentation
                     break;
                  case '$': // plug-ins
                     if (line.charAt (1) != ';')
                     { // old file format (line=$pluginName?pluginID#Major%Minor)
                        line = "$;" + line.substring (1, line.length());
                        line = line.replace ('?', ';');
                        line = line.replace ('#', ';');
                        line = line.replace ('%', ';');
                        line = line.replace ('\u00a7', ';');
                     }

                     StringTokenizer token = createLineTokenizer (line);
                     String name = token.nextToken();
                     String pluginID = token.nextToken();
                     String majorV = token.nextToken();
                     String minorV = token.nextToken();
                     String revision = "0";

                     if (!UPBClassLoader.DEFAULT_CLASSLOADER.equals (pluginID))
                     {
                        if (token.hasMoreTokens())
                        {
                           revision = token.nextToken();
                        }
                        PluginData plugin = new PluginData (pluginID, name, new Integer (majorV).intValue(), new Integer (minorV).intValue(), new Integer (revision).intValue());
                        plugins.put (pluginID, plugin);
                     }
                     else
                     {
                        // which fujaba core...
                        Version version = Version.get();
                        int neededMajor = new Integer (majorV).intValue();
                        int neededMinor = new Integer (minorV).intValue();
                        int neededRev = 0;
                        if (token.hasMoreTokens())
                        {
                           neededRev = new Integer (token.nextToken()).intValue();
                        }

                        if (neededMajor > version.getMajor())
                        {
                           throw new RuntimeException ("\nThe project file \"" + fileName + "\" needs Fujaba core version: " + neededMajor + "." + neededMinor + "." + neededRev + "\n" +
                              "Available Fujaba core version: " + version.getVersion() + ".\n");
                        }
                        else if (neededMajor == version.getMajor() && neededMinor > version.getMinor())
                        {
                           throw new RuntimeException ("\nThe project file \"" + fileName + "\" needs Fujaba core version: " + neededMajor + "." + neededMinor + "." + neededRev + "\n" +
                              "Available Fujaba core version: " + version.getVersion() + ".\n");
                        }
                     }
                     break;
                  case '-': // file specific variables
                     String variable = lineTok.nextToken();
                     if (variable.compareTo ("FileVersion") == 0)
                     {
                        fileVersion = new Integer (lineTok.nextToken()).intValue();
                     }
                     if (variable.compareTo ("HashTableLength") == 0)
                     {
                        objectHashTableLength = new Integer (lineTok.nextToken()).intValue();
                        objectHashTable = new Hashtable (objectHashTableLength * 2);
                        if (log.isDebugEnabled())
                        {
                           log.debug ("Hashtable created with length = " + String.valueOf (objectHashTableLength * 2));
                        }
                     }
                     break;
                  case '+': // hashtable entry
                     if (objectHashTable == null)
                     {
                        throw new Exception ("Hashtable entry found,\n" + "but there is no hashtable created yet!");
                     }

                     // get id and compare to maxUniqueId
                     String strg = lineTok.nextToken();
                     if (!strg.endsWith ("]") && lessUniqueId (maxUniqueId, strg))
                     {
                        maxUniqueId = strg;
                     }

                     // get class name
                     className = lineTok.nextToken();
                     Object o = null;
                     if (className.endsWith ("UMLProject"))
                     {
                        o = UMLProject.get();
                     }
                     else
                     {
                        try
                        {
                           // if true ==> we have the new file format
                           if (lineTok.hasMoreTokens())
                           {
                              String loaderID = lineTok.nextToken();
                              if (UPBClassLoader.DEFAULT_CLASSLOADER.equals (loaderID))
                              {
                                 o = ClassMap.get().forName (className).newInstance();
                              }
                              else
                              {
                                 PluginData pluginData = (PluginData) plugins.get (loaderID);

                                 pluginID = pluginData.getId();

                                 PluginProperty plugin = FujabaApp.getPersistencySupport().getPluginProperty (pluginID);

                                 if (plugin != null)
                                 {
                                    if (plugin.getMajor() >= pluginData.getMajor())
                                    {
                                       if ( (plugin.getMajor() == pluginData.getMajor() &&
                                          plugin.getMinor() >= pluginData.getMinor()) ||
                                          plugin.getMajor() > pluginData.getMajor())
                                       {
                                          ClassLoader pluginClassLoader = FujabaApp.getPersistencySupport().getClassLoader (pluginID);
                                          o = pluginClassLoader.loadClass (className).newInstance();
                                       }
                                       else
                                       {
                                          // the available minor version is less than the needed minor version of plug-in
                                          throw new RuntimeException ("Error: \"" + className + "\" could not be instantiated.\n" +
                                             "Available minor version is less than needed minor version\n\n" +
                                             "\tNeeded Plug-In: \t \"" +
                                             pluginData.getName() + " version " + pluginData.getMajor() + "." + pluginData.getMinor() + "." + pluginData.getBuildNumber() + "\"\n\t" +
                                             "Available Plug-In: \t \"" + plugin.getName() + " version " + plugin.getMajor() + "." + plugin.getMinor() + "." + plugin.getBuildNumber() + "\"\n");
                                       }
                                    }
                                    else
                                    {
                                       // not the same major version
                                       throw new RuntimeException ("Error: \"" + className + "\" could not be instantiated.\n" +
                                          "Available major version is less than needed major version\n\n" +
                                          "\tNeeded Plug-In: \t \"" +
                                          pluginData.getName() + " version " + pluginData.getMajor() + "." + pluginData.getMinor() + "." + pluginData.getBuildNumber() + "\"\n\t" +
                                          "Available Plug-In: \t \"" + plugin.getName() + " version " + plugin.getMajor() + "." + plugin.getMinor() + "." + plugin.getBuildNumber() + "\"\n");
                                    }
                                 }
                                 else
                                 {
                                    PluginData neededPlugin = (PluginData) plugins.get (pluginID);
                                    //the plug-in with plug-in id does not exists
                                    throw new RuntimeException ("Error: \"" + className + "\" could not be instantiated.\n" +
                                       "Plug-in \"" + neededPlugin.getName() + "\" version " + neededPlugin.getMajor() +
                                       "." + neededPlugin.getMinor() + "." + neededPlugin.getBuildNumber() + " is missing.\n" +
                                       "Please download and install the plug-in to load the project file: \"" + fileName + "\"");
                                 }
                              }
                           }
                           else
                           { // we have the old file format

                              o = loadPluginClass (className, plugins);
                              if (o == null)
                              {
                                 o = ClassMap.get().forName (className).newInstance();
                              }
                           }
                        }
                        finally
                        {
                           if (o == null)
                           {
                              if (log.isInfoEnabled())
                              {
                                 log.info ("ERROR in line " + lineNum + ": Could not execute\n   "
                                    + "Class.forName (" + className + ").newInstance (); \n"
                                    + "Perhaps class " + className
                                    + " has no standard (parameterless) constructor?");
                              }
                           }
                        }
                     }
                     if (o != null)
                     {
                        objectHashTable.put (strg, o);
                        // we want FUJABA to keep the old IDs
                         ((BasicIncrement) o).setID (strg);
                     }
                     break;
                  case '*': // object reference
                     // is this the very first time?

                     if (notYetWarned)
                     {
                        // check HashTable size.
                        if (objectHashTable.size() != objectHashTableLength)
                        {
                           if (log.isInfoEnabled())
                           {
                              log.info ("WARNING:Object reference found,");
                           }
                           if (log.isInfoEnabled())
                           {
                              log.info ("\nbut the hashtable is not complete!");
                           }
                        }

                        if (log.isDebugEnabled())
                        {
                           log.debug ("Objects created, start filling: ... ");
                        }

                        // Adjust the uniqueId counter to avoid that already used
                        // numbers are not used even more.
                        setUniqueId (new StringBuffer (maxUniqueId));

                        notYetWarned = false;
                     }

                     if (currentIncr != null)
                     {
                        currentIncr.readAttributes (objectHashTable, attributeHashTable);
                        attributeHashTable.clear();

                        objCount++;
                        if ( (log.isDebugEnabled()) &&
                            ( (objCount % 1000) == 0))
                        {
                           log.debug ("objCount: " + objCount);
                        }
                     }
                     currentIncr = (BasicIncrement) objectHashTable.get (lineTok.nextToken());
                     break;
                  case '~': // attribute reference

                     if (currentIncr == null)
                     {
                        throw new Exception ("Attribute reference found in " + line + ",\nbut no current increment is set!");
                     }
                     attributeHashTable.put (lineTok.nextToken(), lineTok);

                     break;
               } // switch

            } // while fpr file has more lines

            if (currentIncr != null)
            {
               currentIncr.readAttributes (objectHashTable, attributeHashTable);
            }

            // mark and refresh all increments
            if (log.isDebugEnabled())
            {
               log.debug ("mark and refresh all increments");
            }
            Enumeration objects = objectHashTable.elements();

            inconsistencyErrors = 0;
            while (objects.hasMoreElements())
            {
               Object tmpObject = objects.nextElement();
               if (tmpObject instanceof ASGElement)
               {
                  ASGElement tmpItem = (ASGElement) tmpObject;

                  if ( (tmpItem instanceof UMLFile))
                  {
                     UMLFile tmpFile = (UMLFile) tmpItem;
                     if (tmpFile.sizeOfContains() == 0)
                     {
                        inconsistencyRemoveObject (tmpFile);
                     }
                  }
                  else if (tmpItem instanceof UMLParam)
                  {
                     UMLParam tmpParam = (UMLParam) tmpItem;
                     if ( (tmpParam.getRevParam() == null))
                     {
                        inconsistencyRemoveObject (tmpParam);
                     }
                  }
                  /*
                   *  Here check if there are irregular diagram elements
                   *  and delete them.  This should be replaced by an
                   *  real consistency checker mechanism.
                   */
                  if ( (tmpItem instanceof UMLObject) ||
                      (tmpItem instanceof UMLLink) ||
                      (tmpItem instanceof UMLTransition))
                  {
                     if (tmpItem.sizeOfDiagrams() == 0)
                     {
                        inconsistencyRemoveObject (tmpItem);
                     }

                     if (tmpItem instanceof UMLLink)
                     {
                        UMLLink tmpLink = (UMLLink) tmpItem;
                        if (tmpLink.getSource() == null ||
                           tmpLink.getTarget() == null)
                        {
                           inconsistencyRemoveObject (tmpLink);
                        }
                     }
                  }
                  else if (tmpItem instanceof UMLGeneralization)
                  {
                     UMLGeneralization tmpGeneralisation = (UMLGeneralization) tmpItem;
                     if ( (tmpGeneralisation.getSuperclass() == null) ||
                         (tmpGeneralisation.getSubclass() == null))
                     {
                        inconsistencyRemoveObject (tmpGeneralisation);
                     }
                  }
                  else if (tmpItem instanceof UMLAttr)
                  {
                     UMLAttr tmpAttr = (UMLAttr) tmpItem;
                     if (tmpAttr.getParent() == null)
                     {
                        inconsistencyRemoveObject (tmpAttr);
                     }
                  }
                  else if (tmpItem instanceof UMLMethod)
                  {
                     UMLMethod tmpMethod = (UMLMethod) tmpItem;
                     if ( (tmpMethod.getParent() == null))
                     {
                        inconsistencyRemoveObject (tmpMethod);
                     }
                  }
                  else if (tmpItem instanceof UMLClassDiagram)
                  {
                     UMLDiagram tmpDiagram = (UMLDiagram) tmpItem;
                     if (tmpDiagram.getProject() == null)
                     {
                        inconsistencyRemoveObject (tmpDiagram);
                     }
                  }
                  else if (tmpItem instanceof UMLStoryPattern)
                  {
                     UMLStoryPattern tmpPattern = (UMLStoryPattern) tmpItem;
                     if (tmpPattern.getRevStoryPattern() == null)
                     {
                        inconsistencyRemoveObject (tmpPattern);
                     }
                  }
                  else if (tmpItem instanceof UMLClass)
                  {
                     UMLClass tmpClass = (UMLClass) tmpItem;
                     if (tmpClass.getName() == null ||  (tmpClass.getName() != null && tmpClass.getID().equals (tmpClass.getName())))
                     {
                        inconsistencyRemoveObject (tmpClass);
                     }
                  }
               } // end of if ()
            }

            if (inconsistencyErrors > 0)
            {
               if (log.isDebugEnabled())
               {
                  log.debug (inconsistencyErrors + " inconsistency errors found");
               }
            }

            // The project is just loaded
            UMLProject.get().setSaved (true);

            if (log.isDebugEnabled())
            {
               log.debug ("Loading done.");
            }
         }
         catch (Exception e)
         {
            e.printStackTrace();
            throw new RuntimeException ("An error occured in line " +
               Integer.toString (lineNum) + "=" +
               line +
               "\nwhile reading the file " +
               fileName +
               "\nMessage: " + e);
         }
         finally
         {
            try
            {
               reader.close();
            }
            catch (Exception e)
            {
               if (log.isInfoEnabled())
               {
                  log.info
                      ("ERROR in line " + lineNum + ": Perhaps I could not execute\n   "
                     + "Class.forName (" + className + ").newInstance (); \n"
                     + "Perhaps class " + className
                     + " has no standard (parameterless) constructor?");
               }
               e.printStackTrace();
               throw new RuntimeException (e.getMessage());
            }
         }

         // remove the old project
         if (project != UMLProject.get())
         {
            project.removeYou();
         }

         reader = null;

         System.gc();

         return UMLProject.get();
      }
      finally
      {
         setLoading (false);
         objectHashTable = null;
      }
   } // loadProject


   /**
    * This method is only for old project file format. It can be removed, if we do not want
    * load the project files less than 4.0.1
    *
    * @param className   No description provided
    * @param plugins     No description provided
    * @return            No description provided
    * @throws Exception  Exception description not provided
    */
   private static Object loadPluginClass (String className, Hashtable plugins) throws Exception
   {
      Object o = null;

      boolean loaded = false;
      boolean wrongVersion = false;
      PluginProperty plugin = null;
      PluginData pluginData = null;
      Iterator pluginIter = plugins.values().iterator();

      while (pluginIter.hasNext() && !loaded)
      {
         try
         {
            pluginData = (PluginData) pluginIter.next();
            plugin = FujabaApp.getPersistencySupport().getPluginProperty (pluginData.getId());
            if (plugin != null)
            {
               ClassLoader pluginClassLoader = FujabaApp.getPersistencySupport().getClassLoader (plugin.getPluginID());
               o = pluginClassLoader.loadClass (className).newInstance();
               if (plugin.getMajor() == pluginData.getMajor() && plugin.getMinor() >= pluginData.getMinor() && plugin.getBuildNumber() >= pluginData.getBuildNumber())
               {
                  loaded = true;
               }
               else
               {
                  wrongVersion = true;
                  throw new RuntimeException ("Error: \"" + className + "\" could not be instantiated.\n" +
                     "Wrong Plug-in version is available\n\n" +
                     "\tNeeded Plug-In: \t \"" +
                     pluginData.getName() + " version " + pluginData.getMajor() + "." + pluginData.getMinor() + "." + pluginData.getBuildNumber() + "\"\n\t" +
                     "Available Plug-In: \t \"" + plugin.getName() + " version " + plugin.getMajor() + "." + plugin.getMinor() + "." + plugin.getBuildNumber() + "\"\n");
               }
            }
         }
         catch (Exception ex)
         {
            loaded = false;
            if (wrongVersion)
            {
               throw ex;
            }
         }
      }
      return o;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static int inconsistencyErrors = 0;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param incr  Parameter description
    */
   private static void inconsistencyRemoveObject (BasicIncrement incr)
   {
      if (log.isDebugEnabled())
      {
         if (log.isInfoEnabled())
         {
            log.info ("Inconsistency found! remove object " + FD.toString (incr));
         }
      }

      incr.removeYou();
      inconsistencyErrors++;
   } // inconsistencyRemoveObject


   /**
    * Get the currentUMLDiagram attribute of the UMLProject object
    *
    * @return   The currentUMLDiagram value
    */
   public UMLDiagram getCurrentUMLDiagram()
   {
      ASGDiagram diag = getCurrentDiagram();
      if (diag instanceof UMLDiagram)
      {
         return (UMLDiagram) diag;
      }

      return null;
   }


   /**
    * Refreshs the visual representation by explicit synchronization.
    */
   public synchronized void refreshDisplay()
   {
      ASGDiagram currentDiagram = getCurrentDiagram();
      if (currentDiagram != null)
      {
         Iterator iter = currentDiagram.iteratorOfFsaObjects();
         while (iter.hasNext())
         {
            FSAObject fsa = (FSAObject) iter.next();
            fsa.getJComponent().invalidate();
            fsa.getJComponent().repaint();
         }

      }
   } // refreshDisplay


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      //kill display first to save time
      Iterator diagIt = iteratorOfDiags();
      while (diagIt.hasNext())
      {
         ASGDiagram asgDiagram = (ASGDiagram) diagIt.next();

         Iterator fsaIt = asgDiagram.iteratorOfFsaObjects();

         while (fsaIt.hasNext())
         {
            FSAObject fsa = (FSAObject) fsaIt.next();

            fsa.removeYou();
         }
      }

      // remove all elements from all diagrams,
      // before package-structure is destroyed
      removeAllFromDiags();

      // remove type-list,
      // before package-structure is destroyed
      UMLTypeList typeList = getTypeList();
      if (typeList != null)
      {
         setTypeList (null);
         typeList.removeYou();
      }

      this.removeAllFromReferences();
      this.removeAllPackages();
      this.setClipboard (null);

      this.removeAllFromFiles();

      this.setGui (null);

      BasicIncrement.resetClassInfos();
      BasicIncrement.resetFieldWriteMethods();

      TemplateManager.get().resetTemplates();

      super.removeYou();

      //delete backup file _after_ complete remove
      deleteBackupFile();

   } // removeYou


   /**
    * delete backup file used for immediate store
    */
   public void deleteBackupFile()
   {
      if (backupStream != null)
      {
         try
         {
            backupStream.close();
         }
         catch (IOException e)
         {
            e.printStackTrace();
         }
         backupStream = null;
      }
      if (backupFile != null)
      {
         if (!backupFile.delete())
         {
            log.error ("failed to delete temporary backup file!");
         }
      }
   }


   /**
    * @return   the UMLStereotypeManager for this Project
    */
   public UMLStereotypeManager getStereotypeManager()
   {
      if (stereotypeManager == null)
      {
         stereotypeManager = new UMLStereotypeManager();
         if (!isLoading() && !FujabaChangeManager.isInUndoRedo())
         {
            stereotypeManager.initDefaultStereotypes();
         }
      }
      return stereotypeManager;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private UMLStereotypeManager stereotypeManager;


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#setFRootPackage(de.uni_paderborn.fujaba.metamodel.FPackage)
    */
   /**
    * Sets the rootPackage attribute of the UMLProject object
    *
    * @param rootPackage  The new rootPackage value
    */
   public void setRootPackage (FPackage rootPackage)
   {
      setRootPackage ((UMLPackage) rootPackage);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getFRootPackage()
    */
   /**
    * Get the fRootPackage attribute of the UMLProject object
    *
    * @return   The fRootPackage value
    */
   public FPackage getFRootPackage()
   {
      return getRootPackage();
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findFPackage(java.lang.String, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fullName  No description provided
    * @param create    No description provided
    * @return          No description provided
    */
   public FPackage findFPackage (String fullName, boolean create)
   {
      return findPackage (fullName, create);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findFPackage(java.lang.String, boolean, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fullName          No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public FPackage findFPackage (String fullName, boolean create, boolean coobraPersistent)
   {
      return findPackage (fullName, create, coobraPersistent);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getNewFromFPackages(java.lang.String)
    */
   /**
    * Get the newFromFPackages attribute of the UMLProject object
    *
    * @param fullName  No description provided
    * @return          The newFromFPackages value
    */
   public FPackage getNewFromFPackages (String fullName)
   {
      return getNewFromPackages (fullName);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getNewFromFPackages(java.lang.String, boolean)
    */
   /**
    * Get the newFromFPackages attribute of the UMLProject object
    *
    * @param fullName          No description provided
    * @param coobraPersistent  No description provided
    * @return                  The newFromFPackages value
    */
   public FPackage getNewFromFPackages (String fullName, boolean coobraPersistent)
   {
      return getNewFromPackages (fullName, coobraPersistent);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findFPackage(java.lang.String)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fullName  No description provided
    * @return          No description provided
    */
   public FPackage findFPackage (String fullName)
   {
      return findPackage (fullName);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getFromFBaseTypes(java.lang.String)
    */
   /**
    * Get the fromFBaseTypes attribute of the UMLProject object
    *
    * @param key  No description provided
    * @return     The fromFBaseTypes value
    */
   public FType getFromFBaseTypes (String key)
   {
      return getFromBaseTypes (key);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getFromFReferences(java.lang.String)
    */
   /**
    * Get the fromFReferences attribute of the UMLProject object
    *
    * @param key  No description provided
    * @return     The fromFReferences value
    */
   public FClass getFromFReferences (String key)
   {
      return getFromReferences (key);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getOrNewFType(de.uni_paderborn.fujaba.metamodel.FClass, java.lang.String)
    */
   /**
    * Get the orNewFType attribute of the UMLProject object
    *
    * @param context  No description provided
    * @param name     No description provided
    * @return         The orNewFType value
    */
   public FType getOrNewFType (FClass context, String name)
   {
      return getOrNewType ((UMLClass) context, name);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getOrNewFType(de.uni_paderborn.fujaba.metamodel.FClass, java.lang.String, boolean)
    */
   /**
    * Get the orNewFType attribute of the UMLProject object
    *
    * @param context           No description provided
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @return                  The orNewFType value
    */
   public FType getOrNewFType (FClass context, String name, boolean coobraPersistent)
   {
      return getOrNewType ((UMLClass) context, name, coobraPersistent);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findType(de.uni_paderborn.fujaba.metamodel.FClass, java.lang.String)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context  No description provided
    * @param name     No description provided
    * @return         No description provided
    */
   public FType findFType (FClass context, String name)
   {
      return findType ((UMLClass) context, name);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findType(de.uni_paderborn.fujaba.metamodel.FClass, java.lang.String, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context  No description provided
    * @param name     No description provided
    * @param create   No description provided
    * @return         No description provided
    */
   public FType findFType (FClass context, String name, boolean create)
   {
      return findType ((UMLClass) context, name, create);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findType(de.uni_paderborn.fujaba.metamodel.FClass, java.lang.String, boolean, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param context           No description provided
    * @param name              No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public FType findFType (FClass context, String name, boolean create, boolean coobraPersistent)
   {
      return findType ((UMLClass) context, name, create, coobraPersistent);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getOrNewFromFReferences(java.lang.String)
    */
   /**
    * Get the orNewFromFReferences attribute of the UMLProject object
    *
    * @param name  No description provided
    * @return      The orNewFromFReferences value
    */
   public FClass getOrNewFromFReferences (String name)
   {
      return getOrNewFromReferences (name);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getOrNewFromFReferences(java.lang.String, boolean)
    */
   /**
    * Get the orNewFromFReferences attribute of the UMLProject object
    *
    * @param name              No description provided
    * @param coobraPersistent  No description provided
    * @return                  The orNewFromFReferences value
    */
   public FClass getOrNewFromFReferences (String name, boolean coobraPersistent)
   {
      return getOrNewFromReferences (name, coobraPersistent);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findFReference(java.lang.String)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   public FClass findFReference (String name)
   {
      return findReference (name);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findFReference(java.lang.String, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name    No description provided
    * @param create  No description provided
    * @return        No description provided
    */
   public FClass findFReference (String name, boolean create)
   {
      return findReference (name, create);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#findFReference(java.lang.String, boolean, boolean)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name              No description provided
    * @param create            No description provided
    * @param coobraPersistent  No description provided
    * @return                  No description provided
    */
   public FClass findFReference (String name, boolean create, boolean coobraPersistent)
   {
      return findReference (name, create, coobraPersistent);
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getFTypeList()
    */
   /**
    * Get the fTypeList attribute of the UMLProject object
    *
    * @return   The fTypeList value
    */
   public FTypeList getFTypeList()
   {
      return getTypeList();
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FProject#getFClassFromImports(de.uni_paderborn.fujaba.metamodel.FClass, java.lang.String)
    */
   /**
    * Get the fClassFromImports attribute of the UMLProject object
    *
    * @param context  No description provided
    * @param name     No description provided
    * @return         The fClassFromImports value
    */
   public FClass getFClassFromImports (FClass context, String name)
   {
      return getClassFromImports ((UMLClass) context, name);
   }


   /**
    * Get the persistencyChange attribute of the UMLProject object
    *
    * @param e  No description provided
    * @return   The persistencyChange value
    */
   protected boolean isPersistencyChange (PropertyChangeEvent e)
   {
      return super.isPersistencyChange (e) &&
         ! ("saved".equals (e.getPropertyName()) || "gui".equals (e.getPropertyName()));
   }
}


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.413.2.17 $
 */
class PluginData
{


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String id;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int major;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int minor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int buildNumber;


   /**
    * Constructor for class PluginData
    *
    * @param id           No description provided
    * @param name         No description provided
    * @param major        No description provided
    * @param minor        No description provided
    * @param buildNumber  No description provided
    */
   public PluginData (String id, String name, int major, int minor, int buildNumber)
   {
      this.id = id;
      this.name = name;
      this.major = major;
      this.minor = minor;
      this.buildNumber = buildNumber;
   }


   /**
    * Get the id attribute of the PluginData object
    *
    * @return   The id value
    */
   public String getId()
   {
      return id;
   }


   /**
    * Get the major number of the project
    *
    * @return   The major value
    */
   public int getMajor()
   {
      return major;
   }


   /**
    * Get the minor number of the project
    *
    * @return   The minor value
    */
   public int getMinor()
   {
      return minor;
   }


   /**
    * Get the name of the project
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Get the build number of the project
    *
    * @return   The buildNumber value
    */
   public int getBuildNumber()
   {
      return buildNumber;
   }


   /**
    * @param id  The new id value
    */
   public void setId (String id)
   {
      this.id = id;
   }


   /**
    * @param major  The new major value
    */
   public void setMajor (int major)
   {
      this.major = major;
   }


   /**
    * @param minor  The new minor value
    */
   public void setMinor (int minor)
   {
      this.minor = minor;
   }


   /**
    * @param name  The new name value
    */
   public void setName (String name)
   {
      this.name = name;
   }


   /**
    * @param buildNumber  The new buildNumber value
    */
   public void setBuildNumber (int buildNumber)
   {
      this.buildNumber = buildNumber;
   }

}

/*
 * $Log: UMLProject.java,v $
 * Revision 1.413.2.17  2006/07/10 08:05:35  cschneid
 * sysouts removed, exceptions printed (instead of ignoring completely), caption changed to 'Fujaba 4'
 *
 */
