/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.io.File;
import java.util.*;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.*;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.unparse.UMLUnparseGetter;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;
import de.upb.tools.sdm.Path;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *          +-----------+ 1                          1
 * UMLClass | getName() +------------------------------ UMLPackage
 *          +-----------+ declares   declaredInPackage
 *
 *            +-----------+ 1                1
 * UMLPackage | getName() +-------------------- UMLPackage
 *            +-----------+ parent    packages
 * </pre>
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.125.2.2 $
 */
public class UMLPackage extends UMLIncrement implements FPackage
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (UMLPackage.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static transient String DEFAULT_PACKAGE_NAME = "default";


   /**
    * Constructor for class UMLPackage
    */
   public UMLPackage()
   {
      super();
   }


   /**
    * Constructor for class UMLPackage
    *
    * @param coobraPersistent  No description provided
    */
   public UMLPackage (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLPackage
    *
    * @param name  No description provided
    */
   public UMLPackage (String name)
   {
      super();
      this.setName (name);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected String createUnparseModuleName()
   {
      return UMLUnparseGetter.getUnparseModuleName (this);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   public UMLClass findClass (String name)
   {
      UMLClass result = getFromDeclares (name);
      if (result == null)
      {
         Iterator packages = iteratorOfPackages();
         while (packages.hasNext() && result == null)
         {
            UMLPackage pack = (UMLPackage) packages.next();
            result = pack.findClass (name);
         }
      }
      return result;
   }


   /**
    * Searches the ASG tree for a given id
    *
    * @param id  The id to search for.
    * @return    The Element with the given id, null if not found.
    */
   public ASGElement searchID (String id)
   {
      ASGElement item = super.searchID (id);
      Iterator iter = iteratorOfPackages();
      while ( (item == null) &&  (iter.hasNext()))
      {
         item =  ((ASGElement) iter.next()).searchID (id);
      }
      return item;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = null;


   /**
    * Get the name attribute of the UMLPackage object
    *
    * @return   The name value
    */
   public String getName()
   {
      // @@@FIX ME: there should be no nee to do this
      if (this.name == null)
      {
         this.name = "_@" + getID().toString();
         if (log.isDebugEnabled())
         {
            log.debug ("create a temp name for a package");
         }
      }
      return this.name;
   }


   /**
    * Sets the name attribute of the UMLPackage object
    *
    * @param name  The new name value
    */
   public void setName (String name)
   {
      if (name == null || name.indexOf ('.') != -1)
      {
         throw new RuntimeExceptionWithContext ("Cannot set '" + name + "' with this function", this);
      }

      if (this.getName() == null ||
         this.getName().equals (name) == false)
      {
         UMLPackage myParent = getParent();
         if (this.name != null && myParent != null)
         {
            setParent (null);
         }
         String oldValue = this.name;
         this.name = name;

         //----- update all qualified assoccs
         if (this.name != null && myParent != null)
         {
            setParent (parent);
         }
         firePropertyChange (NAME_PROPERTY, oldValue, name);
      }
   }


   /**
    * Get the text attribute of the UMLPackage object
    *
    * @return   The text value
    */
   public String getText()
   {
      return this.getName();
   }


   /**
    * @return   the full packagename of the this package, returns an empty string if this package
    *      is the default- or the rootpackage
    */
   public String getFullPackageName()
   {
      UMLPackage rootPackage = UMLProject.get().getRootPackage();
      if (this == rootPackage ||
         this == UMLProject.get().getDefaultPackage())
      {
         return "";
      }
      else
      {
         StringBuffer fullName = new StringBuffer();
         UMLPackage currentPack = this;

         while (currentPack != rootPackage)
         {
            if (fullName.length() > 0)
            {
               fullName.insert (0, ".");
            }
            fullName.insert (0, currentPack.getName());
            if (currentPack.getParent() == null)
            {
               Logger.getLogger (UMLPackage.class).warn ("potential corrupt package found: " + fullName + " is not attached to another package and is not the root-package!");

               // BUGFIX: there are fpr-Files which are corrupted
               currentPack.setParent (rootPackage);
               break;
            }

            if (currentPack.getParent() != null && currentPack.getParent() != rootPackage
               && currentPack.getParent().getName().equals (rootPackage.getName()))
            {
               currentPack.setParent (rootPackage);
               Logger.getLogger (UMLPackage.class).warn (fullName + " was contained in a RootPackage that is not part of the project!");
            }
            currentPack = currentPack.getParent();
         }
         return fullName.toString();
      }
   }


   /**
    * Get the packagePath attribute of the UMLPackage object
    *
    * @return   The packagePath value
    */
   public String getPackagePath()
   {
      // check if this is the default package
      if (this == UMLProject.get().getDefaultPackage())
      {
         // the default
         return ".";
      }
      else
      {
         StringBuffer fullName = new StringBuffer (this.getName());
         UMLPackage currentPackage = this.getParent();
         UMLPackage rootPackage = UMLProject.get().getRootPackage();

         while (currentPackage != rootPackage && currentPackage != null)
         {
            fullName.insert (0, File.separator);
            fullName.insert (0, currentPackage.getName());
            currentPackage = currentPackage.getParent();
         }
         return fullName.toString();
      }
   }


   /**
    * <pre>
    *          +-----------+ 1                          1
    * UMLClass | getName() +------------------------------ UMLPackage
    *          +-----------+ declares   declaredInPackage
    * </pre>
    */
   private transient FPropTreeMap declares;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInDeclares (FClass obj)
   {
      return  ( (this.declares != null) &&
          (obj != null) &&  (obj.getName() != null) &&
          (this.declares.get (obj.getName()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInDeclares (String key)
   {
      return  ( (this.declares != null) &&
          (key != null) &&
         this.declares.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfDeclares()
   {
      return  ( (this.declares == null)
         ? FEmptyIterator.get()
         : this.declares.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfDeclares()
   {
      return  ( (this.declares == null)
         ? 0
         : this.declares.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromDeclares attribute of the UMLPackage object
    *
    * @param key  No description provided
    * @return     The fromDeclares value
    */
   public UMLClass getFromDeclares (String key)
   {
      return  ( ( (this.declares == null) ||  (key == null))
         ? null
         : (UMLClass) this.declares.get (key));
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FPackage#getFromFDeclares(java.lang.String)
    */
   /**
    * Get the fromFDeclares attribute of the UMLPackage object
    *
    * @param key  No description provided
    * @return     The fromFDeclares value
    */
   public FClass getFromFDeclares (String key)
   {
      return getFromDeclares (key);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToDeclares (FClass obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getName() != null))
      {
         if (this.declares == null)
         {
            this.declares = new FPropTreeMap (FujabaComparator.getLessString(),
               this, DECLARES_PROPERTY);
         }

         UMLClass oldValue = (UMLClass) this.declares.put (obj.getName(), obj);
         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setDeclaredInPackage (null);
            }
            obj.setDeclaredInPackage (this);
            changed = true;

            // side effects

         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromDeclares (FClass obj)
   {
      boolean changed = false;

      if ( (this.declares != null) &&  (obj != null) &&  (obj.getName() != null))
      {
         UMLClass oldValue = (UMLClass) this.declares.get (obj.getName());
         if (oldValue == obj)
         {
            this.declares.remove (obj.getName());
            obj.setDeclaredInPackage (null);
            changed = true;

            // side effects

         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromDeclares (String key)
   {
      boolean changed = false;

      if ( (this.declares != null) &&  (key != null))
      {
         UMLClass tmpObj = (UMLClass) this.declares.get (key);
         if (tmpObj != null)
         {
            this.declares.remove (key);
            tmpObj.setDeclaredInPackage (null);
            changed = true;

            // side effects
         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDeclares()
   {
      UMLClass tmpObj;
      Iterator iter = this.iteratorOfDeclares();

      while (iter.hasNext())
      {
         tmpObj = (UMLClass) iter.next();
         this.removeFromDeclares (tmpObj);
      }
   }


   /**
    * This method is needed only for loading FTreeMaps, don't use it in other cases.
    *
    * @param pair  The object added.
    */
   protected void addToDeclares (KeyValuePair pair)
   {
      if (pair != null)
      {
         UMLClass elem = (UMLClass) pair.getValue();
         String key = (String) pair.getKey();

         if ( (elem != null) &&  (key != null) && ! (key.equals (elem.getName())))
         {
            elem.setName (key);
         }

         addToDeclares (elem);
      }
   }

   // ######################################################################

   /**
    * all files which import this package
    *
    * <pre>
    *             n                                  n
    * UMLPackage -------------------------------------- UMLFile
    *             importedPackage   revImportedPackage
    * </pre>
    */
   private FTreeSet revImportedPackages = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInRevImportedPackages (UMLFile elem)
   {
      return this.revImportedPackages.contains (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfRevImportedPackages()
   {
      return new EnumerationForAnIterator (iteratorOfRevImportedPackages());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfRevImportedPackages()
   {
      return revImportedPackages.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToRevImportedPackages (UMLFile elem)
   {
      if (elem != null && !this.hasInRevImportedPackages (elem))
      {
         this.revImportedPackages.add (elem);
         elem.addToImportedPackages (this);

         firePropertyChange (CollectionChangeEvent.get (this, "revImportedPackages", revImportedPackages,
            null, elem, CollectionChangeEvent.ADDED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromRevImportedPackages (UMLFile elem)
   {
      if (this.hasInRevImportedPackages (elem))
      {
         this.revImportedPackages.remove (elem);
         elem.removeFromImportedPackages (this);

         firePropertyChange (CollectionChangeEvent.get (this, "revImportedPackages", revImportedPackages,
            elem, null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromRevImportedPackages()
   {
      UMLFile item;
      Iterator iter = iteratorOfRevImportedPackages();

      while (iter.hasNext())
      {
         item = (UMLFile) iter.next();
         item.removeFromImportedPackages (this);
         firePropertyChange (CollectionChangeEvent.get (this, "revImportedPackages", revImportedPackages,
            item, null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * To represent the java-package structure it is necessary to put every depth of the package-tree
    * in one UMLPackage.
    *
    * <pre>
    *            +-----------+ 1                1
    * UMLPackage | getName() +-------------------- UMLPackage
    *            +-----------+ parent    packages
    * </pre>
    *
    * The key is the short package name (without any extension), e.g. the package de.uni_paderborn.fujaba
    * consists of three packages: the package 'de' contains the package 'uni_paderborn' and
    * the package 'uni_paderborn' contains the package 'fujaba'.
    */
   private FPropTreeMap packages;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInPackages (FPackage obj)
   {
      return  ( (this.packages != null) &&
          (obj != null) &&  (obj.getName() != null) &&
          (this.packages.get (obj.getName()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInPackages (String key)
   {
      return  ( (this.packages != null) &&
          (key != null) &&
         this.packages.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfPackages()
   {
      return  ( (this.packages == null)
         ? FEmptyIterator.get()
         : this.packages.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfPackages()
   {
      return  ( (this.packages == null)
         ? FEmptyIterator.get()
         : this.packages.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfPackages()
   {
      return  ( (this.packages == null)
         ? FEmptyIterator.get()
         : this.packages.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfPackages()
   {
      return  ( (this.packages == null)
         ? 0
         : this.packages.size());
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the fromPackages attribute of the UMLPackage object
    *
    * @param key  No description provided
    * @return     The fromPackages value
    */
   public UMLPackage getFromPackages (String key)
   {
      return  ( ( (this.packages == null) ||  (key == null))
         ? null
         : (UMLPackage) this.packages.get (key));
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FPackage#getFromFPackages(java.lang.String)
    */
   /**
    * Get the fromFPackages attribute of the UMLPackage object
    *
    * @param key  No description provided
    * @return     The fromFPackages value
    */
   public FPackage getFromFPackages (String key)
   {
      return getFromPackages (key);
   }
   // TODO-END

   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToPackages (FPackage obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getName() != null))
      {
         if (this.packages == null)
         {
            this.packages = new FPropTreeMap (this, PACKAGES_PROPERTY);
         }
         UMLPackage oldValue = (UMLPackage) this.packages.put (obj.getName(), obj);
         if (oldValue != obj)
         {
            if (oldValue != null)
            {
               oldValue.setParent (null);
            }
            obj.setParent (this);
            changed = true;

            // side effects

         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromPackages (FPackage obj)
   {
      boolean changed = false;

      if ( (this.packages != null) &&  (obj != null) &&  (obj.getName() != null))
      {
         UMLPackage oldValue = (UMLPackage) this.packages.get (obj.getName());
         if (oldValue == obj)
         {
            this.packages.remove (obj.getName());
            obj.setParent (null);
            changed = true;

            // side effects

         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromPackages (String key)
   {
      boolean changed = false;

      if ( (this.packages != null) &&  (key != null))
      {
         UMLPackage tmpObj = (UMLPackage) this.packages.get (key);
         if (tmpObj != null)
         {
            this.packages.remove (key);
            tmpObj.setParent (null);
            changed = true;

            // side effects

         }
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromPackages()
   {
      UMLPackage tmpObj;
      Iterator iter = this.iteratorOfPackages();

      while (iter.hasNext())
      {
         tmpObj = (UMLPackage) iter.next();
         this.removeFromPackages (tmpObj);
      }
   }


   /**
    * This method is needed only for loading FTreeMaps, Don't use it in other cases.
    *
    * @param pair  The object added.
    */
   protected void addToPackages (KeyValuePair pair)
   {
      if (pair != null)
      {
         UMLPackage elem = (UMLPackage) pair.getValue();
         String key = (String) pair.getKey();

         if ( (elem != null) &&  (key != null) && ! (key.equals (elem.getName())))
         {
            elem.setName (key);
         }

         addToPackages (elem);
      }
   }

   // ######################################################################

   /**
    * <pre>
    *            +-----------+ 1                1
    * UMLPackage | getName() +-------------------- UMLPackage
    *            +-----------+ parent    packages
    * </pre>
    */
   private transient UMLPackage parent;


   /**
    * Sets the parent attribute of the UMLPackage object
    *
    * @param obj  The new parent value
    * @return     No description provided
    */
   public boolean setParent (FPackage obj)
   {
      boolean changed = false;

      if (this.parent != obj)
      {
         UMLPackage oldValue = this.parent;
         if (this.parent != null)
         {
            this.parent = null;
            oldValue.removeFromPackages (this);
         }
         this.parent = (UMLPackage) obj;
         if (obj != null)
         {
            obj.addToPackages (this);
         }
         changed = true;

         // side effects

         firePropertyChange (PARENT_PROPERTY, oldValue, obj);
      }

      return changed;
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the parent attribute of the UMLPackage object
    *
    * @return   The parent value
    */
   public UMLPackage getParent()
   {
      return this.parent;
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FPackage#getFParent()
    */
   /**
    * Get the fParent attribute of the UMLPackage object
    *
    * @return   The fParent value
    */
   public FPackage getFParent()
   {
      return getParent();
   }
   // TODO-END

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean hasParent()
   {
      return  ( (this.parent != null) &&
          (this.parent != UMLProject.get().getRootPackage()));
   }

   // ######################################################################

   /**
    * Use this function to get all children of a package (all classes and all packages).
    *
    * @return   No description provided
    */
   public Enumeration elementsOfAllChildren()
   {
      return new EnumerationForAnIterator (Path.iterUnion (iteratorOfDeclares(), iteratorOfPackages()));
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param objects     No description provided
    * @param attributes  No description provided
    */
   public void readAttributes (Hashtable objects, FDuplicatedTreeMap attributes)
   {
      super.readAttributes (objects, attributes);

      // BUGFIX: there are old fpr-files which have saved the packages
      //         in the wriong order
      Object obj = readFromStringTokenizer ("parent", null, objects, attributes);
      if (obj != null)
      {
         setParent ((UMLPackage) obj);
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return getName();
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.removeAllFromRevImportedPackages();
      this.removeAllFromPackages();
      this.removeAllFromDeclares();
      this.setParent (null);

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element, may not return null unless this is the top level node (project)
    */
   public FElement getParentElement()
   {
      if (getParent() != null)
      {
         return getParent();
      }
      else
      {
         return UMLProject.get();
      }
   }

}

/*
 * $Log: UMLPackage.java,v $
 * Revision 1.125.2.2  2006/06/22 15:41:24  fklar
 * print warning to log, if corrupt package has been detected
 *
 */
