/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Enumeration;
import java.util.Iterator;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.basic.KeyValuePair;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.metamodel.*;
import de.upb.tools.fca.*;


/**
 * <h2>Associations</h2>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.321.2.1 $
 */
public abstract class UMLIncrement extends ASGElement implements LogicUnparseInterface, FIncrement
{
   /**
    * Constructor for class UMLIncrement
    */
   public UMLIncrement()
   {
      super();
   }


   /**
    * Constructor for class UMLIncrement
    *
    * @param coobraPersistent  No description provided
    */
   protected UMLIncrement (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfAllChildren()
   {
      return new FEmptyEnumeration();
   } // elementsOfAllChildren


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FPropTreeSet constraints;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInConstraints (FConstraint value)
   {
      return  ( (this.constraints == null)
         ? false
         : this.constraints.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfConstraints()
   {
      return  ( (this.constraints == null)
         ? FEmptyIterator.get()
         : this.constraints.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfConstraint()
   {
      return  ( (this.constraints == null)
         ? 0
         : this.constraints.size());
   }


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToConstraints (FConstraint value)
   {
      boolean changed = false;
      if (value != null)
      {
         if (this.constraints == null)
         {
            this.constraints = new FPropTreeSet (this, CONSTRAINTS_PROPERTY);
         }
         changed = this.constraints.add (value);
         if (changed)
         {
            value.addToIncrements (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromConstraints (FConstraint value)
   {
      boolean changed = false;
      if ( (this.constraints != null) &&  (value != null))
      {
         changed = this.constraints.remove (value);
         if (changed)
         {
            value.removeFromIncrements (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromConstraints()
   {
      UMLConstraint tmpValue;
      Iterator iter = this.iteratorOfConstraints();
      while (iter.hasNext())
      {
         tmpValue = (UMLConstraint) iter.next();
         this.removeFromConstraints (tmpValue);
      }
   }


   /**
    * UMLAssociation: 'comment'
    */
   private UMLCommentary comment = null; // reverse attribute revComment


   /**
    * Get the comment attribute of the UMLIncrement object
    *
    * @return   The comment value
    */
   public UMLCommentary getComment()
   {
      return comment;
   } // getComment


   /**
    * Sets the comment attribute of the UMLIncrement object
    *
    * @param comment  The new comment value
    */
   public void setComment (FCommentary comment)
   {
      if (this.comment != comment)
      {
         UMLCommentary oldComment = this.comment;
         // newPartner
         if (this.comment != null)
         {
            // inform old partner
            this.comment = null;

            oldComment.setRevComment (null);
         } // if

         this.comment = (UMLCommentary) comment;
         if (comment != null)
         {
            // inform new partner
            comment.setRevComment (this);
         } // if

         firePropertyChange (COMMENT_PROPERTY, oldComment, comment);

      } // if

   } // setComment


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean assertInUnitTest = false;


   /**
    * Sets the assertInUnitTest attribute of the UMLIncrement object
    *
    * @param value  The new assertInUnitTest value
    */
   public void setAssertInUnitTest (boolean value)
   {
      if (assertInUnitTest != value)
      {
         boolean oldValue = assertInUnitTest;
         assertInUnitTest = value;
         firePropertyChange (ASSERT_IN_UNIT_TEST_PROPERTY, oldValue, value);
      }
   }


   /**
    * Get the assertInUnitTest attribute of the UMLIncrement object
    *
    * @return   The assertInUnitTest value
    */
   public boolean isAssertInUnitTest()
   {
      return assertInUnitTest;
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.removeAllFromConstraints();
      this.removeAllFromStereotypes();
      UMLCommentary comment = this.getComment();
      if (comment != null)
      {
         comment.removeYou();
      }

      super.removeYou();
   }


   /**
    * Get the fComment attribute of the UMLIncrement object
    *
    * @return   The fComment value
    */
   public FCommentary getFComment()
   {
      return getComment();
   }


   /**
    * <pre>
    *          +-----------+ 1                    1
    * UMLClass | getText() +------------------------ FStereotype
    *          +-----------+ umlItem    stereotypes
    * </pre>
    */
   private FPropHashMap stereotypes;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean hasInStereotypes (FStereotype obj)
   {
      return  ( (this.stereotypes != null) &&
          (obj != null) &&  (obj.getText() != null) &&
          (this.stereotypes.get (obj.getText()) == obj));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean hasKeyInStereotypes (String key)
   {
      return  ( (this.stereotypes != null) &&
          (key != null) &&
         this.stereotypes.containsKey (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfStereotypes()
   {
      return  ( (this.stereotypes == null)
         ? FEmptyIterator.get()
         : this.stereotypes.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator keysOfStereotypes()
   {
      return  ( (this.stereotypes == null)
         ? FEmptyIterator.get()
         : this.stereotypes.keySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator entriesOfStereotypes()
   {
      return  ( (this.stereotypes == null)
         ? FEmptyIterator.get()
         : this.stereotypes.entrySet().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfStereotypes()
   {
      return  ( (this.stereotypes == null)
         ? 0
         : this.stereotypes.size());
   }


   /**
    * Get the fromStereotypes attribute of the UMLClass object
    *
    * @param key  No description provided
    * @return     The fromStereotypes value
    */
   public FStereotype getFromStereotypes (String key)
   {
      return  ( ( (this.stereotypes == null) ||  (key == null))
         ? null
         : (FStereotype) this.stereotypes.get (key));
   }


   /**
    * Get the fromStereotypes attribute of the UMLClass object
    *
    * @param value  No description provided
    * @return       The fromStereotypes value
    */
   public FStereotype getFromStereotypes (FStereotype value)
   {
      return  ( ( (this.stereotypes == null) ||  (value == null))
         ? null
         : (FStereotype) this.stereotypes.get (value.getText()));
   }


   /**
    * Access method for an one to n association.
    *
    * @param obj  The object added.
    * @return     No description provided
    */
   public boolean addToStereotypes (FStereotype obj)
   {
      boolean changed = false;

      if ( (obj != null) &&  (obj.getText() != null) &&
          (!hasInStereotypes (obj)))
      {
         if (this.stereotypes == null)
         {
            this.stereotypes = new FPropHashMap (this, "stereotypes");
         }

         if (UMLStereotypeManager.get() != null)
         {
            obj = UMLStereotypeManager.get().getFromStereotypes (obj.getText());
         }

         this.stereotypes.put (obj.getText(), obj);

         obj.addToIncrements (this);
         changed = true;
      }
      return changed;
   }


   /**
    * Access method for an one to n association.
    *
    * @param kind  The object added.
    * @return      No description provided
    */
   private boolean addKeyToStereotypes (String kind)
   {
      if (kind != null)
      {
         FStereotype stereotype = UMLStereotypeManager.get().getFromStereotypes (kind);

         if (!hasInStereotypes (stereotype))
         {
            return addToStereotypes (stereotype);
         }
      }

      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param obj  No description provided
    * @return     No description provided
    */
   public boolean removeFromStereotypes (FStereotype obj)
   {
      boolean changed = false;
      if (obj != null)
      {
         changed = removeKeyFromStereotypes (obj.getText());
      }

      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   public boolean removeKeyFromStereotypes (String key)
   {
      boolean changed = false;

      if (hasKeyInStereotypes (key))
      {
         FStereotype tmpObj = getFromStereotypes (key);
         this.stereotypes.remove (key);
         tmpObj.removeFromIncrements (this);
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromStereotypes()
   {
      FStereotype tmpObj;
      Iterator iter = this.iteratorOfStereotypes();
      while (iter.hasNext())
      {
         tmpObj = (FStereotype) iter.next();
         this.removeFromStereotypes (tmpObj);
      }
   }


   /**
    * This method is needed only for loading FTreeMaps, don't use it in other cases.
    *
    * @param pair  The object added.
    */
   protected void addToStereotypes (KeyValuePair pair)
   {
      if (pair != null)
      {
         FStereotype elem = (FStereotype) pair.getValue();
         String key = (String) pair.getKey();
         elem.setText (key);
         addToStereotypes (elem);
      }
   } // addToStereotypes


   /**
    * This method sets the stereotype if the value is true and deletes all of the specified
    * kind if the value is false.
    *
    * @param stereotype  The new stereotype value
    * @param value       The new stereotype value
    */
   public void setStereotype (FStereotype stereotype, boolean value)
   {
      if (value)
      {
         addToStereotypes (stereotype);
      }
      else
      {
         removeFromStereotypes (stereotype);
      }
   } // setStereotype


   /**
    * Get the fromStereotypes attribute of the FClass object
    *
    * @param key  No description provided
    * @return     The fromStereotypes value
    */
   public FStereotype getFromFStereotypes (String key)
   {
      return getFromStereotypes (key);
   }
}

/*
 * $Log: UMLIncrement.java,v $
 * Revision 1.321.2.1  2005/12/15 15:21:48  creckord
 * - FujabaApp can determine the Fujaba install location now
 * - Template dir and default plugin dir are determined based on install location instead of current dir
 *
 * - UMLCollabStat has an int number for its sequence number now instead of noText
 * - hopefully fixed UMLCollabStats losing/messing up their order and number between save & load
 * - threadId is honored by renumbering methods and GUI
 *
 * - FTreeEnumeration can enumerate elements nested in other elements of target type now
 * - iterator/elementsOfClasses in FProject returns inner classes now, too
 *
 * - UMLClass doesn't lose inner classes anymore (hopefully)
 *
 */
