/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Enumeration;
import java.util.Iterator;

import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.unparse.UMLUnparseGetter;
import de.upb.tools.fca.*;
import de.upb.tools.pcs.CollectionChangeEvent;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *         +-----------+ 1                   1
 * UMLFile | getName() +----------------------- UMLClass
 *         +-----------+ file         contains
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.164.2.1 $
 */
public class UMLFile extends UMLIncrement implements FFile
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static String DEFAULT_FILE_NAME = "default";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean newFile;


   /**
    * Constructor for class UMLFile
    */
   public UMLFile()
   {
      super();
      setNewFile (false);
   }


   /**
    * Constructor for class UMLFile
    *
    * @param name  No description provided
    */
   public UMLFile (String name)
   {
      this();
      this.setProject (UMLProject.get());
      this.setName (name);
      setNewFile (false);
   }


   /**
    * Constructor for class UMLFile
    *
    * @param clazz  No description provided
    */
   public UMLFile (UMLClass clazz)
   {
      this();
      this.addToContains (clazz);
      this.setProject (UMLProject.get());
      this.setName (clazz.getName());
      setNewFile (false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   protected String createUnparseModuleName()
   {
      return UMLUnparseGetter.getUnparseModuleName (this);
   }


   /**
    * The name of the the file specified by the object.
    */
   private String name;


   /**
    * Get the name attribute of the UMLFile object
    *
    * @return   The name value
    */
   public String getName()
   {
      //----- @@FIX ME: what about, if there are more than one class in a file
      //      check if the filename is the same as the first class-name

      //----- search for a class with the current filename
      String myName = null;
      if (this.name != null)
      {
         Iterator iter = this.iteratorOfContains();
         while (iter.hasNext())
         {
            myName =  ((UMLClass) iter.next()).getName();
            if (this.name.equals (myName))
            {
               break;
            }
            myName = null;
         }
      }

      if (myName == null)
      {
         //----- there is no class with the current filename or
         //      the class has not a name yet
         Iterator iter = this.iteratorOfContains();
         if (iter.hasNext())
         {
            this.setName ( ((UMLClass) iter.next()).getName());
         }
         else
         {
            this.setName ("_@" + getID().toString());
         }
      }
      return this.name;
   }


   /**
    * Sets the name attribute of the UMLFile object
    *
    * @param newName  The new name value
    */
   public void setName (String newName)
   {
      if (newName != null && !newName.equals (this.name))
      {
         String oldValue = this.name;

         String rootDir = UMLProject.get() != null ? UMLProject.get().getRootDir() : "";
         if (newName.startsWith (rootDir) == true)
         {
            // adjust the file name. if it begins with the inputDir, which is
            // specified in OptionParser
            newName = newName.substring (rootDir.length());
         }
         this.name = newName;
         firePropertyChange (NAME_PROPERTY, oldValue, newName);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean tokensDirty;


   /**
    * Get the tokensDirty attribute of the UMLFile object
    *
    * @return       The tokensDirty value
    * @deprecated
    */
   public boolean getTokensDirty()
   {
      return tokensDirty;
   }


   /**
    * Sets the tokensDirty attribute of the UMLFile object
    *
    * @param tokensDirty  The new tokensDirty value
    * @deprecated
    */
   public void setTokensDirty (boolean tokensDirty)
   {
      boolean oldValue = this.tokensDirty;
      this.tokensDirty = tokensDirty;

      firePropertyChange ("tokensDirty", oldValue, tokensDirty);
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLFile[");
      result.append (name);
      result.append ("]");

      return result.toString();
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * the package to which this file belongs <pre>
    *          0,1          n            n          0,1
    * UMLFile <--------------> UMLClass <--------------> UMLPackage
    * </pre>
    *
    * @return   The package value
    */
   public UMLPackage getPackage()
   {
      UMLPackage pack = null;
      Iterator iter = this.iteratorOfContains();
      UMLClass clazz = null;

      while (pack == null && iter.hasNext())
      {
         clazz = (UMLClass) iter.next();
         pack = clazz.getDeclaredInPackage();
      }
      return pack;
   }


   /**
    * Get the fPackage attribute of the UMLFile object
    *
    * @return   The fPackage value
    * @see      de.uni_paderborn.fujaba.metamodel.FFile#getFPackage()
    */
   public FPackage getFPackage()
   {
      return getPackage();
   }
   // TODO-END

   // ####################################################################

   /**
    * UMLFile <-----------------------> UMLProject
    */
   private transient UMLProject project = null;


   /**
    * Get the project attribute of the UMLFile object
    *
    * @return   The project value
    */
   public UMLProject getProject()
   {
      return this.project;
   }


   /**
    * Sets the project attribute of the UMLFile object
    *
    * @param proj  The new project value
    */
   public void setProject (UMLProject proj)
   {
      if (this.project != proj)
      {
         UMLProject oldProject = this.project;
         if (this.project != null)
         {
            this.project = null;
            oldProject.removeFromFiles (this);
         }
         this.project = proj;
         if (this.project != null)
         {
            this.project.addToFiles (this);
         }
         firePropertyChange ("project", oldProject, proj);
      }
   }


   /**
    * if (codeGenDenied) UMLProject is not allowed to create the UMLFile
    *
    * @return   true , if code generation is denied for this UMLFile
    */
   public boolean isCodeGenDenied()
   {
      boolean codeGenDenied = false;

      Iterator iter = iteratorOfContains();
      UMLClass tmpUMLClass = null;

      while (iter.hasNext() && !codeGenDenied)
      {
         tmpUMLClass = (UMLClass) iter.next();

         codeGenDenied |= tmpUMLClass.isCodeGenDenied();
      }

      return codeGenDenied;
   }


   /**
    * <pre>
    *           n      hasClasses       0..1
    * UMLClass ------------------------------ UMLFile
    *           contains   {ordered,}   file
    * </pre>
    */
   private FLinkedList contains;


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #contains
    */
   public boolean hasInContains (FClass value)
   {
      return  ( (this.contains != null) &&
          (value != null) &&
         this.contains.contains (value));
   }


   /**
    * @return   No description provided
    * @see      #contains
    */
   public Iterator iteratorOfContains()
   {
      return  ( (this.contains == null)
         ? FEmptyIterator.get()
         : this.contains.iterator());
   }


   /**
    * @return   No description provided
    * @see      #contains
    */
   public int sizeOfContains()
   {
      return  ( (this.contains == null)
         ? 0
         : this.contains.size());
   }


   /**
    * @param value  The object added.
    * @return       No description provided
    * @see          #contains
    */
   public boolean addToContains (FClass value)
   {
      boolean changed = false;
      if (value != null && !hasInContains (value))
      {
         if (this.contains == null)
         {
            this.contains = new FPropLinkedList (this, CONTAINS_PROPERTY);
         }
         changed = this.contains.add (value);
         if (changed)
         {
            value.setFile (this);
         }
      }
      return changed;
   }


   /**
    * @param value  No description provided
    * @return       No description provided
    * @see          #contains
    */
   public boolean removeFromContains (FClass value)
   {
      boolean changed = false;
      if ( (this.contains != null) &&  (value != null))
      {
         changed = this.contains.remove (value);
         if (changed)
         {
            value.setFile (null);
         }
      }
      return changed;
   }


   /**
    * @see   #contains
    */
   public void removeAllFromContains()
   {
      UMLClass tmpValue;
      Iterator iter = this.iteratorOfContains();
      while (iter.hasNext())
      {
         tmpValue = (UMLClass) iter.next();
         this.removeFromContains (tmpValue);
      }
   }

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @param index  No description provided
    * @return       The containsAt value
    * @see          #contains
    */
   public UMLClass getContainsAt (int index)
   {
      if (index >= 0 && index < sizeOfContains())
      {
         return (UMLClass) this.contains.get (index);
      }
      else
      {
         throw new IllegalArgumentException ("getContainsAt(" + index + ")");
      }
   }


   /**
    * Get the fContainsAt attribute of the UMLFile object
    *
    * @param index  No description provided
    * @return       The fContainsAt value
    * @see          de.uni_paderborn.fujaba.metamodel.FFile#getFContainsAt(int)
    */
   public FClass getFContainsAt (int index)
   {
      return getContainsAt (index);
   }
   // TODO-END


   /**
    * @param elem  No description provided
    * @return      No description provided
    * @see         #contains
    */
   public int indexOfContains (FClass elem)
   {
      return  ( (this.contains == null)
         ? -1
         : this.contains.indexOf (elem));
   }


   /**
    * @param elem  No description provided
    * @return      No description provided
    * @see         #contains
    */
   public int lastIndexOfContains (FClass elem)
   {
      return  ( (this.contains == null)
         ? -1
         : this.contains.lastIndexOf (elem));
   }


   /**
    * todo: remove this method to make edobs and coobra2 work
    *
    * @param lowerBound  No description provided
    * @return            No description provided
    * @see               #contains
    * @deprecated        useless and lets the 'contains' assoc look like a qualified one
    */
   public Iterator iteratorOfContains (FClass lowerBound)
   {
      Iterator result = FEmptyIterator.get();
      if (contains == null)
      {
         result = FEmptyIterator.get();
      }
      else if (contains != null && lowerBound != null)
      {
         int index = contains.indexOf (lowerBound) + 1;
         result = contains.listIterator (index);
      }
      else if (contains != null && lowerBound == null)
      {
         result = contains.listIterator (0);
      }
      return result;
   }


   /**
    * @param leftObject   No description provided
    * @param rightObject  No description provided
    * @return             The beforeOfContains value
    * @see                #contains
    */
   public boolean isBeforeOfContains (FClass leftObject, FClass rightObject)
   {
      if (contains == null)
      {
         return false;
      }
      else
      {
         return contains.isBefore (leftObject, rightObject);
      }
   }


   /**
    * @param leftObject   No description provided
    * @param rightObject  No description provided
    * @return             The afterOfContains value
    * @see                #contains
    */
   public boolean isAfterOfContains (FClass leftObject, FClass rightObject)
   {
      if (contains == null)
      {
         return false;
      }
      else
      {
         return contains.isAfter (leftObject, rightObject);
      }
   }

   // TODO-BEGIN: Merge JDK 1.5
   /**
    * @return   The firstOfContains value
    * @see      #contains
    */
   public UMLClass getFirstOfContains()
   {
      if (contains == null)
      {
         return null;
      }
      else
      {
         return (UMLClass) contains.getFirst();
      }
   }


   /**
    * Get the firstOfFContains attribute of the UMLFile object
    *
    * @return   The firstOfFContains value
    * @see      de.uni_paderborn.fujaba.metamodel.FFile#getFirstOfFContains()
    */
   public FClass getFirstOfFContains()
   {
      return getFirstOfContains();
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @return   The lastOfContains value
    * @see      #contains
    */
   public UMLClass getLastOfContains()
   {
      if (contains == null)
      {
         return null;
      }
      else
      {
         return (UMLClass) contains.getLast();
      }
   }


   /**
    * Get the lastOfFContains attribute of the UMLFile object
    *
    * @return   The lastOfFContains value
    * @see      de.uni_paderborn.fujaba.metamodel.FFile#getLastOfFContains()
    */
   public FClass getLastOfFContains()
   {
      return getLastOfContains();
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @param object  No description provided
    * @return        The nextOfContains value
    * @see           #contains
    */
   public UMLClass getNextOfContains (FClass object)
   {
      if (contains == null)
      {
         return null;
      }
      else
      {
         return (UMLClass) contains.getNextOf (object);
      }
   }


   /**
    * Get the nextOfFContains attribute of the UMLFile object
    *
    * @param object  No description provided
    * @return        The nextOfFContains value
    * @see           de.uni_paderborn.fujaba.metamodel.FFile#getNextOfFContains(de.uni_paderborn.fujaba.metamodel.FClass)
    */
   public FClass getNextOfFContains (FClass object)
   {
      return getNextOfContains (object);
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @param object  No description provided
    * @param index   No description provided
    * @return        The nextIndexOfContains value
    * @see           #contains
    */
   public UMLClass getNextIndexOfContains (FClass object, int index)
   {
      if (contains == null)
      {
         return null;
      }
      else
      {
         return (UMLClass) contains.getNextOf (object, index);
      }
   }


   /**
    * Get the nextIndexOfFContains attribute of the UMLFile object
    *
    * @param object  No description provided
    * @param index   No description provided
    * @return        The nextIndexOfFContains value
    * @see           de.uni_paderborn.fujaba.metamodel.FFile#getNextIndexOfFContains(de.uni_paderborn.fujaba.metamodel.FClass,
    *      int)
    */
   public FClass getNextIndexOfFContains (FClass object, int index)
   {
      return getNextIndexOfContains (object, index);
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @param object  No description provided
    * @return        The previousOfContains value
    * @see           #contains
    */
   public UMLClass getPreviousOfContains (FClass object)
   {
      if (contains == null)
      {
         return null;
      }
      else
      {
         return (UMLClass) contains.getPreviousOf (object);
      }
   }


   /**
    * Get the previousOfFContains attribute of the UMLFile object
    *
    * @param object  No description provided
    * @return        The previousOfFContains value
    * @see           de.uni_paderborn.fujaba.metamodel.FFile#getPreviousOfFContains(de.uni_paderborn.fujaba.metamodel.FClass)
    */
   public FClass getPreviousOfFContains (FClass object)
   {
      return getPreviousOfContains (object);
   }
   // TODO-END

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * @param object  No description provided
    * @param index   No description provided
    * @return        The previousIndexOfContains value
    * @see           #contains
    */
   public UMLClass getPreviousIndexOfContains (FClass object, int index)
   {
      if (contains == null)
      {
         return null;
      }
      else
      {
         return (UMLClass) contains.getPreviousOf (object, index);
      }
   }


   /**
    * Get the previousIndexOfFContains attribute of the UMLFile object
    *
    * @param object  No description provided
    * @param index   No description provided
    * @return        The previousIndexOfFContains value
    * @see           de.uni_paderborn.fujaba.metamodel.FFile#getPreviousIndexOfFContains(de.uni_paderborn.fujaba.metamodel.FClass,
    *      int)
    */
   public FClass getPreviousIndexOfFContains (FClass object, int index)
   {
      return getPreviousIndexOfContains (object, index);
   }
   // TODO-END


   /**
    * UMLAttribute : 'footer : StringBuffer '
    */
   private StringBuffer footer;


   /**
    * Get the value of footer.
    *
    * @return   Value of footer.
    */
   public StringBuffer getFooter()
   {
      return this.footer;
   }


   /**
    * Set the value of footer.
    *
    * @param footer  Value to assign to footer.
    */
   public void setFooter (StringBuffer footer)
   {
      if (this.footer != footer)
      {
         this.footer = footer;
      }
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Enumeration elementsOfAllChildren()
   {
      return new EnumerationForAnIterator (this.iteratorOfContains());
   }

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FTreeSet importedClasses = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInImportedClasses (FClass elem)
   {
      return this.importedClasses.contains (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfImportedClasses()
   {
      return  ( (this.importedClasses == null)
         ? 0
         : this.importedClasses.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return       No description provided
    * @deprecated
    */
   public Enumeration elementsOfImportedClasses()
   {
      return new EnumerationForAnIterator (iteratorOfImportedClasses());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfImportedClasses()
   {
      return importedClasses.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToImportedClasses (FClass elem)
   {
      if (elem != null &&
         !this.hasInImportedClasses (elem))
      {
         this.importedClasses.add (elem);
         elem.addToRevImportedClasses (this);
         firePropertyChange (CollectionChangeEvent.get (this, "importedClasses", this.importedClasses, null,
            elem, CollectionChangeEvent.ADDED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromImportedClasses (FClass elem)
   {
      if (this.hasInImportedClasses (elem))
      {
         this.importedClasses.remove (elem);
         elem.removeFromRevImportedClasses (this);
         firePropertyChange (CollectionChangeEvent.get (this, "importedClasses", this.importedClasses, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeAllFromImportedClasses()
   {
      UMLClass item;
      Iterator iter = iteratorOfImportedClasses();

      while (iter.hasNext())
      {
         item = (UMLClass) iter.next();
         item.removeFromRevImportedClasses (this);
         firePropertyChange (CollectionChangeEvent.get (this, "importedClasses", this.importedClasses, item,
            null, CollectionChangeEvent.REMOVED));
      }
   }

   // ######################################################################

   /**
    * UMLFile <----------------------------------------> UMLPackage
    */
   private FTreeSet importedPackages = new FTreeSet();


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInImportedPackages (FPackage elem)
   {
      return this.importedPackages.contains (elem);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfImportedPackages()
   {
      return  ( (this.importedPackages == null)
         ? 0
         : this.importedPackages.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return       No description provided
    * @deprecated
    */
   public Enumeration elementsOfImportedPackages()
   {
      return new EnumerationForAnIterator (iteratorOfImportedPackages());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfImportedPackages()
   {
      return this.importedPackages.iterator();
   }


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToImportedPackages (FPackage elem)
   {
      if (elem != null && !this.hasInImportedPackages (elem))
      {
         this.importedPackages.add (elem);
          ((UMLPackage) elem).addToRevImportedPackages (this);

         firePropertyChange (CollectionChangeEvent.get (this, "importedPackages", this.importedPackages, null,
            elem, CollectionChangeEvent.ADDED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromImportedPackages (FPackage elem)
   {
      if (elem != null &&
         this.hasInImportedPackages (elem))
      {
         this.importedPackages.remove (elem);
          ((UMLPackage) elem).removeFromRevImportedPackages (this);

         firePropertyChange (CollectionChangeEvent.get (this, "importedPackages", this.importedPackages, elem,
            null, CollectionChangeEvent.REMOVED));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final void removeAllFromImportedPackages()
   {
      UMLPackage item;
      Iterator iter = iteratorOfImportedPackages();

      while (iter.hasNext())
      {
         item = (UMLPackage) iter.next();
         item.removeFromRevImportedPackages (this);

         firePropertyChange (CollectionChangeEvent.get (this, "importedPackages", this.importedPackages, item,
            null, CollectionChangeEvent.REMOVED));
      }
   }

   // ######################################################################

   /**
    * Get the classFromImports attribute of the UMLFile object
    *
    * @param fullName  No description provided
    * @return          The classFromImports value
    */
   public UMLClass getClassFromImports (String fullName)
   {
      UMLClass umlClass = null;
      String className = fullName;

      //----- split a qualified classname
      int lastDot = fullName.lastIndexOf ('.');
      if (lastDot != -1)
      {
         className = fullName.substring (lastDot + 1);
      }

      //----- search current package
      if (this.getPackage() != null)
      {
         umlClass = this.getPackage().getFromDeclares (className);
      }

      //----- search imported classes
      Iterator iter = this.iteratorOfImportedClasses();
      while (umlClass == null && iter.hasNext())
      {
         umlClass = (UMLClass) iter.next();

         if (!umlClass.getName().equals (className))
         {
            umlClass = null;
         }
      }

      //----- search imported packages
      iter = this.iteratorOfImportedPackages();
      while (umlClass == null && iter.hasNext())
      {
         umlClass =  ((UMLPackage) iter.next()).getFromDeclares (className);
      }

      //----- check if the found class is the correct one
      if (lastDot != -1 && umlClass != null &&
         !umlClass.getFullClassName().equals (fullName))
      {
         umlClass = null;
      }

      return umlClass;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean necessaryToCreateFile()
   {
      boolean create = false;
      Iterator iter = iteratorOfContains();
      while (!create && iter.hasNext())
      {
         UMLClass tmpClass = (UMLClass) iter.next();
         create = tmpClass.necessaryToGenerateSourceCode();
      }
      return create;
   }


   /**
    * Isolates the UMLFile and all its classes in order to prepare a new load.
    */
   public void removeYouAndMyClasses()
   {
      // first remove all my contained classes
      Iterator iter = iteratorOfContains();
      while (iter.hasNext())
      {
          ((UMLClass) iter.next()).removeYou();
      }

      this.removeYou();
   }


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      this.removeAllFromContains();
      this.removeAllFromImportedClasses();
      this.removeAllFromImportedPackages();
      this.setProject (null);

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getPackage();
   }


   /**
    * Get the newFile attribute of the UMLFile object
    *
    * @return       The newFile value
    * @deprecated
    */
   public boolean isNewFile()
   {
      return newFile;
   }


   /**
    * Sets the newFile attribute of the UMLFile object
    *
    * @param isNew  The new newFile value
    * @deprecated
    */
   public void setNewFile (boolean isNew)
   {
      boolean oldValue = this.newFile;
      newFile = isNew;
      firePropertyChange ("newFile", oldValue, isNew);
   }

}

/*
 * $Log: UMLFile.java,v $
 * Revision 1.164.2.1  2005/10/10 00:54:49  lowende
 * Two enumerationOf methods set to deprecated.
 * Two missing sizeOf methods added.
 *
 */
