/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.StringTokenizer;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.metamodel.*;
import de.uni_paderborn.fujaba.uml.unparse.UMLUnparseGetter;


/**
 * Provides commentary nodes for the diags.
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *                1                       1
 * UMLCommentary --------------------------- UMLIncrement
 *                comment        revComment
 * </pre>
 *
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.53.2.1 $
 */
public class UMLCommentary extends UMLIncrement implements FCommentary
{

   /**
    * Constructor for class UMLCommentary
    */
   public UMLCommentary()
   {
      super();
   }


   /**
    * Constructor for class UMLCommentary
    *
    *
    * @param coobraPersistent  No description provided
    */
   public UMLCommentary (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLCommentary
    *
    *
    * @param text  No description provided
    */
   public UMLCommentary (String text)
   {
      super();
      setText (text);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    *
    * @return   No description provided
    */
   protected String createUnparseModuleName()
   {
      return UMLUnparseGetter.getUnparseModuleName (this);
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    *
    * @param value  No description provided
    *
    * @deprecated   Use <code>UMLCommentary#setVisible(boolean)</code> instead.
    */
   public void swapVisibilityTo (boolean value)
   {
   }


   /**
    * Get the visibilityOfFirstIncr attribute of the UMLCommentary object
    *
    *
    * @return       The visibilityOfFirstIncr value
    *
    * @deprecated   Use <code>UMLCommentary#isVisible()</code> instead.
    */
   public boolean getVisibilityOfFirstIncr()
   {
      return false;
   }


   /**
    * Get visibility-state of this commentary in the current diagram.
    *
    * If no diagram is currently active, false is returned.
    *
    *
    * @return   The current visibility-state of this commentary.
    */
   public boolean isVisible()
   {
      ASGDiagram diag = UMLProject.get().getCurrentDiagram();
      return this.hasInDiagrams (diag);
   }


   /**
    * Set visibility-state of this commentary in the current diagram.
    *
    * Nothing will happen, if no diagram is currently active.
    *
    * Note to refresh the display after calling this method.
    *
    *
    * @param value  The new visibility-state of this commentary.
    */
   public void setVisible (boolean value)
   {
      if (value == true)
      {
         // add comment to current diagram
         ASGDiagram diag = UMLProject.get().getCurrentDiagram();
         this.addToDiagrams (diag);
      }
      else
      {
         // remove comment from current diagram
         ASGDiagram diag = UMLProject.get().getCurrentDiagram();
         this.removeFromDiagrams (diag);
      }
   }

   // ######################################################################

   /**
    * <pre>
    *                1                       1
    * UMLCommentary --------------------------- UMLIncrement
    *                comment        revComment
    * </pre>
    */
   private transient UMLIncrement revComment; // reverse attribute comment

   // TODO-BEGIN: Merge with JDK 1.5
   /**
    * Get the revComment attribute of the UMLCommentary object
    *
    *
    * @return   The revComment value
    */
   public UMLIncrement getRevComment()
   {
      return revComment;
   } // getRevComment


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.metamodel.FCommentary#getFRevComment()
    */
   /**
    * Get the fRevComment attribute of the UMLCommentary object
    *
    *
    * @return   The fRevComment value
    */
   public FIncrement getFRevComment()
   {
      return getRevComment();
   }

   // TODO-END

   /**
    * Sets the revComment attribute of the UMLCommentary object
    *
    *
    * @param revComment  The new revComment value
    */
   public void setRevComment (FIncrement revComment)
   {
      if (this.revComment != revComment)
      {
         // newPartner
         UMLIncrement oldRevComment = this.revComment;
         if (this.revComment != null)
         {
            // inform old partner
            this.revComment = null;

            oldRevComment.setComment (null);
         } // if

         this.revComment = (UMLIncrement) revComment;
         if (revComment != null)
         {
            // inform new partner
            revComment.setComment (this);
         } // if

         firePropertyChange (REV_COMMENT_PROPERTY, oldRevComment, revComment);
      } // if

   } // setRevComment

   // ######################################################################

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    *
    * @param text   No description provided
    * @return       No description provided
    *
    * @deprecated   If no one insists on using this method (e.g.
    *             plugin-developers), it may be deleted, otherwise remove this
    *             deprecated-flag.
    */
   public String deleteBlancLines (String text)
   {
      StringTokenizer st = new StringTokenizer (text, "\n", true);
      boolean blancLines = true;
      String str;
      StringBuffer initialBlancs = new StringBuffer();

      // get the blancs in front of the first non-blanc line
      while (blancLines && st.hasMoreTokens())
      {
         str = st.nextToken();
         if (str.trim().length() > 0)
         {
            blancLines = false;
            StringBuffer strBuf = new StringBuffer (str);
            boolean ready = false;
            int i = 0;
            while (!ready && i < strBuf.length())
            {
               char chr = strBuf.charAt (i);
               if (chr == ' ' || chr == '\t')
               {
                  initialBlancs.append (chr);
               }
               else
               {
                  ready = true;
               }
               i++;
            }
         }
      }
      return  (text.trim().length() > 0) ? initialBlancs + text.trim() + "\n" : "";
   }


   /**
    * Get the incrText attribute of the UMLCommentary object
    *
    *
    * @return   The incrText value
    */
   public String getIncrText()
   {
      return getText();
   }


   /**
    * UMLAttribute: '+ text : String'
    */
   private String text = "";


   /**
    * Get the text attribute of the UMLCommentary object
    *
    *
    * @return   The text value
    */
   public String getText()
   {
      return text;
   } // getText


   /**
    * Sets the text attribute of the UMLCommentary object
    *
    *
    * @param text  The new text value
    * @return      No description provided
    *
    * @see         UMLCommentary#getUncommentedText(String)
    */
   public String setText (String text)
   {
      if (text != null && !this.text.equals (text))
      {
         String oldValue = this.text;
         this.text = text;
         firePropertyChange ("text", oldValue, this.text);
      } // if

      return this.text;
   } // setText


   /**
    * Get the text of this UMLCommentary in a Java-like multi-line-comment or
    * single-line representation.
    *
    *
    * @return   The commented text.
    */
   public String getCommentedText()
   {
      String trimmedText = this.text.trim();

      // test if text already is a multi-line-comment
      if (trimmedText.startsWith ("/*"))
      {
         // multi-line-comment is corrupt, so repair it
         if (!trimmedText.endsWith ("*/"))
         {
            return  (this.text + "*/\n");
         }
         // multi-line-comment is valid
         else
         {
            return this.text;
         }
      }
      // test if text already is a single-line-comment
      else if (trimmedText.startsWith ("//"))
      {
         // if the text has additional lines
         // we have to make shure these start with "//" as well
         StringTokenizer tokenizer = new StringTokenizer (text, "\n", true);
         String token;
         String trimmedToken;
         StringBuffer textBuffer = new StringBuffer();

         while (tokenizer.hasMoreTokens())
         {
            token = tokenizer.nextToken();

            // is token a delimiter
            if (token.length() == 1 && token.startsWith ("\n"))
            {
               textBuffer.append (token);
            }
            else
            {
               trimmedToken = token.trim();

               if (trimmedToken.startsWith ("//"))
               {
                  textBuffer.append (token);
               }
               else
               {
                  textBuffer.append ("// ");
                  textBuffer.append (token);
               }
            }
         }
         return textBuffer.toString();
      }
      // text is of unkown comment-format or not commented, so comment it
      else
      {
         return  ("/*\n" + this.text + "*/\n");
      }
   }


   /**
    * Use this method to get an uncommented version of Java-style-commented
    * text.
    *
    * <pre>
    * Use this function, e.g. if you want to adjust the text of a UMLCommentary
    * but only have commented text, or don't know if the text you have
    * contains comment-symbols and you want them to be removed.
    *
    * Example code:
    *   String uncommentedText = UMLCommentary.getUncommentedText(myCommentedText);
    *   UMLCommentary theCommentary = ...;
    *   theCommentary.setText(uncommentedText);
    * </pre>
    *
    *
    * @param text  A String containing commented Java-style text.
    *
    * @return      A String without comment-symbols.
    */
   public static String getUncommentedText (String text)
   {
      if (text == null || text.length() == 0)
      {
         return "";
      }

      // get text without leading and trailing (white)spaces
      String trimmedText = text.trim();

      // comment-types:
      // -1 unknown
      //  0 multi-line
      //  1 javadoc-style (multi-line)
      //  2 single-line
      int commentType = -1;

      // check what type of comment
      if (trimmedText.startsWith ("/**"))
      {
         commentType = 1;
      }
      else if (trimmedText.startsWith ("/*"))
      {
         commentType = 0;
      }
      else if (trimmedText.startsWith ("//"))
      {
         commentType = 2;
      }
      else
      {
         commentType = -1;
      } // maybe uncommented text has been passed to this
      // method?

      switch (commentType)
      {
         case -1: // unknown comment-type
            break;
         case 0: // multi-line-comment
         case 1:
         { // java-doc-style-comment

            // text between start and end is the commented text
            int firstMultiIndex = -1;
            if (commentType == 0)
            {
               firstMultiIndex = text.indexOf ("/*");
            }
            else
            {
               firstMultiIndex = text.indexOf ("/**");
            }

            int textEndIndex = text.lastIndexOf ("*/");

            String pureText;
            // correct multi-line-comment
            if (textEndIndex != -1)
            {
               pureText = text.substring (firstMultiIndex + 2, textEndIndex);
            }
            // corrupted multi-line-comment
            else
            {
               pureText = text.substring (firstMultiIndex + 2);
            }

            // trim each line, so (white)spaces will be removed
            if (commentType == 0)
            {
               return UMLCommentary.deleteInitialBlancs (pureText, false);
            }
            // additionally remove preceeding "*"s
            else
            {
               return UMLCommentary.deleteInitialBlancs (pureText, true);
            }
         }
         case 2:
         { // single-line-comment

            StringTokenizer tokenizer = new StringTokenizer (text, "\n", true);
            String token;
            String trimmedToken;
            StringBuffer textBuffer = new StringBuffer();

            // remove leading and trailing whitespace and the single-line-symbol
            while (tokenizer.hasMoreTokens())
            {
               token = tokenizer.nextToken();

               // is token a delimiter
               if (token.length() == 1 && token.startsWith ("\n"))
               {
                  textBuffer.append (token);
               }
               else
               {
                  trimmedToken = token.trim();

                  if (trimmedToken.startsWith ("//"))
                  {
                     textBuffer.append (trimmedToken.substring (2).trim());
                  }
                  else
                  {
                     textBuffer.append (trimmedToken);
                  }
               }
            }
            return textBuffer.toString();
         }
      }

      return text;
   }


   /**
    * Deletes leading blancs, from each line of the text and if specified
    * preceeding asterisks as well.
    *
    *
    * @param text            A String.
    * @param removeAsterisk  Should preceeding asteriks be removed as well?
    *
    * @return                The String without leading blancs.
    */
   public static String deleteInitialBlancs (String text, boolean removeAsterisk)
   {
      StringTokenizer tokenizer = new StringTokenizer (text, "\n", true);
      String token;
      String trimmedToken;
      StringBuffer textBuffer = new StringBuffer();

      // remove leading and trailing whitespace
      while (tokenizer.hasMoreTokens())
      {
         token = tokenizer.nextToken();

         // is token a delimiter
         if (token.length() == 1 && token.startsWith ("\n"))
         {
            textBuffer.append (token);
         }
         else
         {
            trimmedToken = token.trim();

            // remove asterisk if specified
            if (removeAsterisk && trimmedToken.startsWith ("*"))
            {
               textBuffer.append (trimmedToken.substring (1).trim());
            }
            else
            {
               textBuffer.append (trimmedToken);
            }
         }
      }
      return textBuffer.toString();
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public void removeYou()
   {
      setRevComment (null);

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class,
    * diagram of an object).
    *
    *
    * @return   the logical parent of this element, may not return null unless
    *         this is the top level node (project)
    */
   public FElement getParentElement()
   {
      return getRevComment();
   }
}

/*
 * $Log: UMLCommentary.java,v $
 * Revision 1.53.2.1  2005/07/04 16:41:34  fklar
 * refactored API of UMLCommentary:
 * + added method that converts a commented to an uncommented text
 * + added methods that change visibility of a commentary in the current diagram
 * + marked some methods as "deprecated"
 *
 */
