/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import de.uni_paderborn.fujaba.basic.RuntimeExceptionWithContext;

import de.uni_paderborn.fujaba.coobra.FujabaChangeManager;
import de.uni_paderborn.fujaba.metamodel.FElement;


/**
 * <h2>Associations</h2> <pre>
 *                  0..n    instanceOf    0..1
 * UMLAttrExprPair ---------------------------- UMLAttr
 *                  instances       instanceOf
 *
 *                  0..n    attrs    0..1
 * UMLAttrExprPair ----------------------- UMLObject
 *                  attrs        revAttrs
 * </pre>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.61.2.1 $
 */
public class UMLAttrExprPair extends UMLDiagramItem
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NONE = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int PRE = NONE + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int POST = PRE + 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int EQUAL = 0;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NOTEQUAL = EQUAL + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int LESS = NOTEQUAL + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int GREATER = LESS + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int LESSEQUAL = GREATER + 1;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int GREATEREQUAL = LESSEQUAL + 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int REG_EXPRESSION = GREATEREQUAL + 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int IN = REG_EXPRESSION + 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int NOT_IN = IN + 1;


   /**
    * Constructor for class UMLAttrExprPair
    */
   public UMLAttrExprPair()
   {
      super();
   }


   /**
    * Constructor for class UMLAttrExprPair
    *
    * @param name        No description provided
    * @param qualifier   No description provided
    * @param operation   No description provided
    * @param expression  No description provided
    * @param instanceOf  No description provided
    * @param revAttrs    No description provided
    */
   public UMLAttrExprPair (String name, int qualifier,
                           int operation, String expression, UMLAttr instanceOf,
                           UMLObject revAttrs)
   {
      super();
      setName (name);
      setQualifier (qualifier);
      setOperation (operation);
      setExpression (expression);
      setRevAttrs (revAttrs);
      setInstanceOf (instanceOf);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = "";


   /**
    * Get the name attribute of the UMLAttrExprPair object
    *
    * @return   The name value
    */
   public String getName()
   {
      return name;
   }


   /**
    * Set the value of name.
    *
    * @param name  Value to assign to name.
    */
   public void setName (String name)
   {
      if ( (this.name == null && name != null) ||
          (this.name != null && !this.name.equals (name)))
      {
         String oldName = this.name;

         this.name = name;

         firePropertyChange ("name", oldName, this.name);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int qualifier = NONE;


   /**
    * Get the qualifier attribute of the UMLAttrExprPair object
    *
    * @return   The qualifier value
    */
   public int getQualifier()
   {
      return qualifier;
   }


   /**
    * Sets the qualifier attribute of the UMLAttrExprPair object
    *
    * @param qualifier  The new qualifier value
    */
   public void setQualifier (int qualifier)
   {
      if (this.qualifier != qualifier)
      {
         int oldQualifier = this.qualifier;
         String oldAttrOperationAsText = getAttrOperationAsText();

         this.qualifier = qualifier;

         firePropertyChange ("qualifier", oldQualifier, this.qualifier);

         firePropertyChange ("attrOperationAsText", oldAttrOperationAsText, getAttrOperationAsText());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int operation = EQUAL;


   /**
    * Get the operation attribute of the UMLAttrExprPair object
    *
    * @return   The operation value
    */
   public int getOperation()
   {
      return operation;
   }


   /**
    * Sets the operation attribute of the UMLAttrExprPair object
    *
    * @param operation  The new operation value
    */
   public void setOperation (int operation)
   {
      if (this.operation != operation)
      {
         int oldOperation = this.operation;
         String oldAttrOperationAsText = getAttrOperationAsText();

         this.operation = operation;

         firePropertyChange ("operation", oldOperation, this.operation);

         firePropertyChange ("attrOperationAsText", oldAttrOperationAsText, getAttrOperationAsText());
      }
   }


   /**
    * Get the operationText attribute of the UMLAttrExprPair object
    *
    * @return   The operationText value
    */
   public String getOperationText()
   {
      switch (getOperation())
      {
         case EQUAL:
            return  ("=");
         case NOTEQUAL:
            return  ("!=");
         case LESS:
            return  ("<");
         case GREATER:
            return  (">");
         case LESSEQUAL:
            return  ("<=");
         case GREATEREQUAL:
            return  (">=");
         case REG_EXPRESSION:
            return  ("== RegExp:");
         case IN:
            return  (" in ");
         case NOT_IN:
            return  (" not in ");
         default:
            // should not happen, otherwise model is incorrect
            return  ("");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String expression = "";


   /**
    * Get the expression attribute of the UMLAttrExprPair object
    *
    * @return   The expression value
    */
   public String getExpression()
   {
      return expression;
   }


   /**
    * Set the value of expression.
    *
    * @param expression  Value to assign to expression.
    */
   public void setExpression (String expression)
   {
      if ( (this.expression == null && expression != null) ||
          (this.expression != null && !this.expression.equals (expression)))
      {
         String oldExpression = this.expression;
         this.expression = expression;
         firePropertyChange ("expression", oldExpression, this.expression);
      }
   }


   /**
    * <pre>
    *                  0..n    instanceOf    0..1
    * UMLAttrExprPair ---------------------------- UMLAttr
    *                  instances       instanceOf
    * </pre>
    */
   private UMLAttr instanceOf;


   /**
    * Get the instanceOf attribute of the UMLAttrExprPair object
    *
    * @return   The instanceOf value
    */
   public UMLAttr getInstanceOf()
   {
      return instanceOf;
   }


   /**
    * Sets the instanceOf attribute of the UMLAttrExprPair object
    *
    * @param instanceOf  The new instanceOf value
    */
   public void setInstanceOf (UMLAttr instanceOf)
   {
      if (this.instanceOf != instanceOf)
      {
         // new partner
         UMLAttr oldInstanceOf = this.instanceOf;
         if (this.instanceOf != null)
         {
            // inform old partner
            this.instanceOf = null;
            oldInstanceOf.removeFromInstances (this);
         }
         this.instanceOf = instanceOf;
         firePropertyChange ("instanceOf", oldInstanceOf, instanceOf);
         if (instanceOf != null)
         {
            // inform new partner
            instanceOf.addToInstances (this);
         }
      }
   }


   /**
    * <pre>
    *                  0..n    attrs    0..1
    * UMLAttrExprPair ----------------------- UMLObject
    *                  attrs        revAttrs
    * </pre>
    */
   private UMLObject revAttrs;


   /**
    * Get the revAttrs attribute of the UMLAttrExprPair object
    *
    * @return   The revAttrs value
    */
   public UMLObject getRevAttrs()
   {
      return revAttrs;
   }


   /**
    * Sets the revAttrs attribute of the UMLAttrExprPair object
    *
    * @param revAttrs  The new revAttrs value
    */
   public void setRevAttrs (UMLObject revAttrs)
   {
      if (this.revAttrs != revAttrs)
      { // new partner

         UMLObject oldRevAttrs = this.revAttrs;
         if (this.revAttrs != null)
         { // inform old partner

            this.revAttrs = null;
            oldRevAttrs.removeFromAttrs (this);
         }
         this.revAttrs = revAttrs;
         firePropertyChange ("revAttrs", oldRevAttrs, revAttrs);
         if (revAttrs != null)
         { // inform new partner

            revAttrs.addToAttrs (this);
         }
      }
   }


   /**
    * Get the text attribute of the UMLAttrExprPair object
    *
    * @return   The text value
    */
   public String getText()
   {
      StringBuffer newText = new StringBuffer();

      newText.append (getName());

      switch (getQualifier())
      {
         case PRE:
            if (getOperation() == EQUAL)
            {
               newText.append ("=");
            }
            break;
         case POST:
            newText.append (":");
            break;
         default:
            newText.append (" ");
            break;
      }
      newText.append (getOperationText()).append (" ").append (getExpression());

      return newText.toString();
   }


   /**
    * Get the attrOperationAsText attribute of the UMLAttrExprPair object
    *
    * @return   The attrOperationAsText value
    */
   public String getAttrOperationAsText()
   {
      StringBuffer newText = new StringBuffer();

      switch (getQualifier())
      {
         case PRE:
            if (getOperation() == EQUAL)
            {
               newText.append ("=");
            }
            break;
         case POST:
            if (getOperation() == EQUAL)
            {
               newText.append (":");
            }
            break;
         default:
            break;
      }
      newText.append (getOperationText());

      return newText.toString();
   }


   /**
    * Sets the attrOperationAsText attribute of the UMLAttrExprPair object
    *
    * @param newText  The new attrOperationAsText value
    */
   public void setAttrOperationAsText (String newText)
   {
      if (newText.equals (":="))
      {
         setOperation (EQUAL);
         setQualifier (POST);
      }
      else if (newText.equals ("=="))
      {
         setOperation (EQUAL);
         setQualifier (PRE);
      }
      else if (newText.equals ("!="))
      {
         setOperation (NOTEQUAL);
         setQualifier (NONE);
      }
      else if (newText.equals ("<"))
      {
         setOperation (LESS);
         setQualifier (NONE);
      }
      else if (newText.equals (">"))
      {
         setOperation (GREATER);
         setQualifier (NONE);
      }
      else if (newText.equals ("<="))
      {
         setOperation (LESSEQUAL);
         setQualifier (NONE);
      }
      else if (newText.equals (">="))
      {
         setOperation (GREATEREQUAL);
         setQualifier (NONE);
      }
      else if (newText.equals ("== RegExp:"))
      {
         setOperation (REG_EXPRESSION);
         setQualifier (NONE);
      }
      else if (newText.trim().equals ("in"))
      {
         setOperation (IN);
         setQualifier (NONE);
      }
      else if (newText.trim().equals ("not in"))
      {
         setOperation (NOT_IN);
         setQualifier (NONE);
      }
      else
      {
         // this can happen while undoing some changes while the qualifier is not set
         if (!FujabaChangeManager.isInUndoRedo())
         {
            throw new RuntimeExceptionWithContext ("newText \"" + newText + "\" should not happen !", this);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static String[] allAttrOperationsAsText = null;


   /**
    * Get the allAttrOperationsAsText attribute of the UMLAttrExprPair class
    *
    * @return   The allAttrOperationsAsText value
    */
   public static String[] getAllAttrOperationsAsText()
   {
      if (allAttrOperationsAsText == null)
      {
         allAttrOperationsAsText = new String[]
            {
            ":=",
            "==",
            "!=",
            "<",
            ">",
            "<=",
            ">=",
            "== RegExp:",
            " in ",
            " not in "
            };
      }

      return allAttrOperationsAsText;
   } // getAllAttrOperationsAsText


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      setRevAttrs (null);
      setInstanceOf (null);

      super.removeYou();
   }


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      return getRevAttrs();
   }


   /**
    * @return   short string representation of current object
    */
   public String toString()
   {
      StringBuffer result = new StringBuffer();

      result.append ("UMLAttrExprPair[name=");
      result.append (getName());
      result.append (",operation=");
      result.append (getOperationText());
      result.append (",expression=");
      result.append (getExpression());
      result.append (",instanceOf=");
      result.append (getInstanceOf());
      result.append ("]");

      return result.toString();
   }

}

/*
 * $Log: UMLAttrExprPair.java,v $
 * Revision 1.61.2.1  2005/05/02 11:51:06  creckord
 * Fix for Multilinks
 * Small change in CoObRA change filtering
 *
 */
