/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.uml;

import java.util.Iterator;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.sequencer.FlowActivity;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashMap;
import de.upb.tools.fca.FLinkedList;
import de.upb.tools.sdm.Path;


/**
 * <h2>Associations</h2>
 *
 * <pre>
 *                  0..1      contains      n
 * UMLComplexState --------------------------- UMLActivityDiagram
 *                  revContains      contains
 * </pre>
 *
 * @author    $Author: l3_g5 $
 * @version   $Revision: 1.224.2.4 $
 */
public class UMLActivityDiagram extends UMLDiagram
{
   /**
    * @param coobraPersistent
    */
   public UMLActivityDiagram (boolean coobraPersistent)
   {
      super (coobraPersistent);
   }


   /**
    * Constructor for class UMLActivityDiagram
    */
   public UMLActivityDiagram()
   {
      super();
   } // UMLActivityDiagram


   /**
    * Constructor for class UMLActivityDiagram
    *
    * @param name     No description provided
    * @param project  No description provided
    */
   public UMLActivityDiagram (String name, UMLProject project)
   {
      super (name, project);
   } // UMLActivityDiagram


   /**
    * Get the name attribute of the UMLDiagram object
    *
    * @return   The name of the associated method if any, super.getName() else
    */
   public String getName()
   {
      if (getStartActivity() != null && getStartActivity().getSpec() != null)
      {
         return getStartActivity().getSpec().getFullMethodName();
      }
      else
      {
         return super.getName();
      }
   }


   /**
    * Get the statechart attribute of the UMLActivityDiagram object
    *
    * @return   The statechart value
    */
   public boolean isStatechart()
   {
      UMLStartActivity start = getStartActivity();
      if (start != null)
      {
         return  (start.getSpecClass() != null && start.getSpec() == null);
      }
      return false;
   }


   /**
    * reverse UMLActivityStart revSpec
    */
   private UMLMethod storyMethod;


   /**
    * Get the spec attribute of the UMLStartActivity object
    *
    * @return   The spec value
    */
   public UMLMethod getStoryMethod()
   {
      return storyMethod;
   } // getSpec


   /**
    * Sets the spec attribute of the UMLStartActivity object
    *
    * @param storyMethod  The new storyMethod value
    */
   public void setStoryMethod (UMLMethod storyMethod)
   {
      if ( (this.storyMethod == null && storyMethod != null) ||
          (this.storyMethod != null && !this.storyMethod.equals (storyMethod)))
      {
         // new partner
         UMLMethod oldStoryMethod = this.storyMethod;
         if (this.storyMethod != null)
         {
            // inform old partner
            this.storyMethod = null;
            oldStoryMethod.setStoryDiag (null);
         }
         this.storyMethod = storyMethod;
         if (storyMethod != null)
         {
            // inform new partner
            storyMethod.setStoryDiag (this);
         }
         firePropertyChange ("storyMethod", oldStoryMethod, storyMethod);
      }
   } // setSpec


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean createTest = true;


   /**
    * Get the createTest attribute of the UMLActivityDiagram object
    *
    * @return   The createTest value
    */
   public boolean isCreateTest()
   {
      return this.createTest;
   }


   /**
    * Sets the createTest attribute of the UMLActivityDiagram object
    *
    * @param value  The new createTest value
    */
   public void setCreateTest (boolean value)
   {
      if (this.createTest != value)
      {
         boolean oldValue = this.createTest;
         this.createTest = value;
         firePropertyChange ("createTest", new Boolean (oldValue), new Boolean (value));
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean isStoryBoard = false;


   /**
    * Get the isStoryBoard attribute of the UMLActivityDiagram object
    *
    * @return   The isStoryBoard value
    */
   public boolean getIsStoryBoard()
   {
      return this.isStoryBoard;
   }


   /**
    * Sets the isStoryBoard attribute of the UMLActivityDiagram object
    *
    * @param flag  The new isStoryBoard value
    */
   public void setIsStoryBoard (boolean flag)
   {
      this.isStoryBoard = flag;
   }


   /**
    * Returns an iterator containing all objects in the diagram. If there are no UMLActivityStory's
    * in the diagram an empty iterator is returned.
    *
    * @return   iterator of all objects
    */
   public Iterator iteratorOfObjects()
   {
      Iterator tmpIter = FEmptyIterator.get();
      Iterator itemsIter = iteratorOfElements();
      while (itemsIter.hasNext())
      {
         ASGElement tmpItem = (ASGElement) itemsIter.next();

         if (tmpItem instanceof UMLStoryActivity)
         {
            UMLStoryPattern pattern =  ((UMLStoryActivity) tmpItem).getStoryPattern();
            if (pattern != null)
            {
               tmpIter = Path.iterUnion (tmpIter, pattern.iteratorOfObjects());
            }
         }

         if (tmpItem instanceof UMLComplexState)
         {
            UMLComplexState complexState = (UMLComplexState) tmpItem;
            UMLStoryActivity storyActivity = complexState.getStory();
            if (storyActivity != null)
            {
               UMLStoryPattern storyPattern = storyActivity.getStoryPattern();

               if (storyPattern != null)
               {
                  tmpIter = Path.iterUnion (tmpIter, storyPattern.iteratorOfObjects());
               }
            }
            else
            {
               UMLActivityDiagram activityDiagram = complexState.getFirstFromContains();
               if (activityDiagram != null)
               {
                  tmpIter = Path.iterUnion (tmpIter, activityDiagram.iteratorOfObjects());
               }
            }
         }
      }

      return tmpIter;
   } // iteratorOfObjects


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient boolean hasFlowAssociations = false;


   /**
    * Creates a correspondent FlowActivity for each activity in this diagram. Necessary for
    * code generation of the related method. Use removeFlowAssociations() afterwards syncronization
    * is realized by the semaphore hasFlowAssociations
    */
   public synchronized void createFlowAssociations()
   {
      while (hasFlowAssociations)
      {
         try
         {
            wait (200);
         }
         catch (InterruptedException e)
         {
         }
      }

      hasFlowAssociations = true;

      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         ASGElement tmpItem = (ASGElement) iter.next();
         if (tmpItem instanceof UMLActivity)
         {
            FlowActivity tmpFlowActivity = new FlowActivity();
            tmpFlowActivity.setUMLActivity ((UMLActivity) tmpItem);
         }
      }
   } // createFlowAssociations


   /**
    * Cuts associatons to correspondent FlowActivities generated by createFlowAssociation().
    * Use this method always in a finally part to prevent deadlocks.
    */
   public void removeFlowAssociations()
   {
      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         ASGElement tmpItem = (ASGElement) iter.next();
         if (tmpItem instanceof UMLActivity)
         {
            FlowActivity tmpFlowActivity =  ((UMLActivity) tmpItem).getFlowActivity();
            tmpFlowActivity.removeYou();
         }
      }
      hasFlowAssociations = false;
   } // removeFlowAssociations


   /**
    * Returns the start activity of this diagram
    *
    * @return   The startActivity value
    */
   public UMLStartActivity getStartActivity()
   {
      UMLStartActivity theStartActivity = null;

      Iterator iter = iteratorOfElements();
      while (iter.hasNext() && theStartActivity == null)
      {
         ASGElement tmpItem = (ASGElement) iter.next();
         if (tmpItem instanceof UMLStartActivity)
         {
            theStartActivity = (UMLStartActivity) tmpItem;
         }
      }

      return theStartActivity;
   } // getStartActivity


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FLinkedList transitions = null;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void removeAllFromTransitions()
   {
      if (transitions != null)
      {
         transitions.clear();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean belongsToState()
   {
      return this.belongsToMasterState() || this.belongsToSubState();
   } // belongsToState


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean belongsToMasterState()
   {
      // is this the master-state?
      UMLStartActivity startActivity = getStartActivity();
      if (startActivity != null)
      {
         if (startActivity.getRevStartOfStateChart() != null)
         {
            return true;
         }
      }

      return false;
   } // belongsToMasterState


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean belongsToSubState()
   {
      // is there a corresponding UMLComplexState?
      if (this.getRevContains() != null)
      {
         return true;
      }

      return false;
   } // belongsToSubState


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean belongsToSimpleState()
   {
      if (!this.belongsToState())
      {
         return false;
      }

      // the state is a simple one if the activity is empty
      return  (sizeOfElements() == 0);
   } // belongsToSimpleState


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public boolean belongsToStateWithSubStates()
   {
      if (!this.belongsToState())
      {
         return false;
      }

      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         if (iter.next() instanceof UMLComplexState)
         {
            return true;
         }
      }

      return false;
   } // belongsToStateWithSubStates


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public UMLActivityDiagram findActivityDiagramOfMasterState()
   {
      UMLStartActivity startActivity = this.getStartActivity();
      if (startActivity != null)
      {
         if (this.belongsToMasterState())
         {
            // this one is the master
            return this;
         }
      }

      UMLActivityDiagram activityDiag = null;
      UMLActivityDiagram tmpDiag = null;

      UMLComplexState state = getRevContains();
      UMLDiagram diag = null;

      if (state != null)
      {
         diag = state.getFirstFromDiagrams();
      }

      if (diag != null &&
         diag instanceof UMLActivityDiagram)
      {
         tmpDiag = (UMLActivityDiagram) diag;
         activityDiag = tmpDiag.findActivityDiagramOfMasterState();
      }

      return activityDiag;
   } // findActivityDiagramOfMasterState


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private transient FHashMap newStoryObjects = null;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    */
   protected void addToNewStoryObjects (UMLObject value)
   {
      if (value != null && value.getText() != null &&
         !hasInNewStoryObjects (value))
      {
         if (newStoryObjects == null)
         {
            this.newStoryObjects = new FHashMap();
         }
         this.newStoryObjects.put (value.getText(), value);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   protected boolean hasInNewStoryObjects (UMLObject value)
   {
      return  ( (this.newStoryObjects != null) &&
          (value != null) &&  (value.getText() != null) &&
          (this.newStoryObjects.get (value.getText()) == value));
   }


   /**
    * Get the fromNewStoryObjects attribute of the UMLActivityDiagram object
    *
    * @param key  No description provided
    * @return     The fromNewStoryObjects value
    */
   protected UMLObject getFromNewStoryObjects (String key)
   {
      return  ( ( (this.newStoryObjects == null) ||  (key == null))
         ? null
         : (UMLObject) this.newStoryObjects.get (key));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeAllFromNewStoryObjects()
   {
      if (this.newStoryObjects != null && !this.newStoryObjects.isEmpty())
      {
         this.newStoryObjects.clear();
      }
   }


   /**
    * Returns the full name like class::method
    *
    * @return   The fullName value
    */
   public String getFullName()
   {
      String text = null;
      UMLClass parent = null;
      UMLMethod method = null;
      UMLStartActivity activity = getStartActivity();

      if (activity != null)
      {
         method = activity.getSpec();
      }

      if (method != null)
      {
         parent = method.getParent();
      }

      if (parent != null &&
         parent.getName() != null &&
         parent.getName().length() > 0)
      {
         text = parent.getName();
      }
      else
      {
         text = "?";
      }
      return text + "::" + this;
   } // getFullName


   /**
    * Returns a correct name for the tree.
    *
    * @return   No description provided
    */
   public String toString()
   {
      String text = null;
      UMLMethod method = null;
      UMLStartActivity activity = getStartActivity();

      if (activity != null)
      {
         method = activity.getSpec();
      }

      if (method != null &&
         method.getName() != null &&
         method.getName().length() > 0)
      {
         text = method.getName();
      }
      else
      {
         text = getName();
      }

      return text;
   } // toString


   /**
    * <pre>
    *                  0..1      Contains      n
    * UMLComplexState --------------------------- UMLActivityDiagram
    *                  revContains      contains
    * </pre>
    */
   private transient UMLComplexState revContains = null;


   /**
    * Sets the revContains attribute of the UMLActivityDiagram object
    *
    * @param elem  The new revContains value
    */
   public void setRevContains (UMLComplexState elem)
   {
      if ( (this.revContains == null && elem != null) ||
          (this.revContains != null && !this.revContains.equals (elem)))
      {
         UMLComplexState oldRevContains = this.revContains;
         // newPartner
         if (this.revContains != null)
         {
            // inform old partner
            this.revContains = null;
            oldRevContains.removeFromContains (this);
         }

         this.revContains = elem;
         if (elem != null)
         {
            // inform new partner
            elem.addToContains (this);
         }

         firePropertyChange ("revContains", oldRevContains, elem);
      }
   }


   /**
    * Get the revContains attribute of the UMLActivityDiagram object
    *
    * @return   The revContains value
    */
   public UMLComplexState getRevContains()
   {
      return this.revContains;
   } // getRevContains



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int priority;


   /**
    * Get the value of priority.
    *
    * @return   Value of priority.
    */
   public int getPriority()
   {
      return priority;
   } // getPriority


   /**
    * Set the value of priority.
    *
    * @param v  Value to assign to priority.
    */
   public void setPriority (int v)
   {
      if (this.priority != v)
      {
         int oldValue = this.priority;
         // ugly hack to let the UMLActivityDiagramPriorityComparator work
         UMLComplexState revContains = this.getRevContains();
         if (revContains != null)
         {
            revContains.removeFromContains (this);
            this.priority = v;
            revContains.addToContains (this);
         }
         this.priority = v;
         firePropertyChange ("priority", oldValue, v);
      }
   } // setPriority


   /**
    * Isolates the object so the garbage collector can remove it.
    */
   public void removeYou()
   {
      // A ActivityDiagram should aggregate all it's items, so
      // if the diagram is removed, remove also the items.
      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         // but not nodes of the Java-ASG, they are removed by the method
         ASGElement item = (ASGElement) iter.next();
         item.removeYou();
      }

      removeAllFromNewStoryObjects();
      setRevContains (null);

      removeAllFromTransitions();
      setStoryMethod (null);

      super.removeYou();
   } // removeYou


   /**
    * Query the logical parent of this element (e.g. package of a class, diagram of an object).
    *
    * @return   the logical parent of this element;
    */
   public FElement getParentElement()
   {
      UMLStartActivity startActivity = getStartActivity();
      if (getStoryMethod() != null)
      {
         //is method spec
         return getStoryMethod();
      }
      else if (startActivity != null)
      {
         if (startActivity.getSpec() != null)
         {
            //is method spec
            return startActivity.getSpec();
         }
         else if (startActivity.getRevStartOfStateChart() != null)
         {
            //is statechart
            return startActivity.getRevStartOfStateChart();
         }
         else
         {
            return super.getParentElement();
         }
      }
      else
      {
         return super.getParentElement();
      }
   }

}

/*
 * $Log: UMLActivityDiagram.java,v $
 * Revision 1.224.2.4  2006/06/19 12:35:00  l3_g5
 * remove method link in removeYou
 *
 */
