/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences.gui;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import de.uni_paderborn.fujaba.logging.LoggerInfo;
import de.uni_paderborn.fujaba.preferences.LoggingPreferences;


/**
 * A panel to configure a particular log4j appender
 *
 * @author    $Author: koenigs $
 * @version   $Revision: 1.7.2.1 $
 */
public class AppenderPanel extends JPanel
{
   /**
    * Name used to refer to appender in GUI
    */
   private String displayName;

   /**
    * Name used to refer to appender in log4j config file
    */
   private String configName;


   /**
    * getter for field configName
    *
    * @return   current value of field configName
    */
   public String getConfigName()
   {
      return this.configName;
   }


   /**
    * Determines whether appender is enabled or not
    */
   JCheckBox enabledBox;

   /**
    * Selectable thresholds
    */
   private final String[] THRESHOLDS = new String[]
      {LoggerInfo.DEBUG,
      LoggerInfo.INFO,
      LoggerInfo.WARN,
      LoggerInfo.ERROR,
      LoggerInfo.FATAL};
   /**
    * Label for threshold combo box
    */
   private JLabel thresholdLabel = new JLabel ("Threshold");

   /**
    * Threshold of appender
    */
   private JComboBox thresholdCombo = new JComboBox (THRESHOLDS);

   /**
    * Layout of panel
    */
   private GridBagLayout gridBag = new GridBagLayout();

   /**
    * Layout constraints for panel
    */
   private GridBagConstraints constraints = new GridBagConstraints();

   /**
    * Components in panel which are additional to the default mapped to corresponding parameter
    * names
    */
   private Hashtable componentMap = new Hashtable();

   /**
    * Constant to show that text field contains info on a pattern
    */
   private final static String CONVERSION_PATTERN = "Conversion Pattern";

   private final static String PORT = "Port";

   /**
    * Constant for JLabels to map to
    */
   private final static String JLABEL = "";


   /**
    * Creates a new appender panel
    *
    * @param theDisplayName  name used to refer to appender in GUI
    * @param theConfigName   name used to refer to appender in log4j config file
    */
   public AppenderPanel (String theDisplayName, String theConfigName)
   {
      displayName = theDisplayName;
      configName = theConfigName;

      setLayout (gridBag);

      // set up the check box
      enabledBox = new JCheckBox (displayName);

      enabledBox.addChangeListener (
         new ChangeListener()
         {
            public void stateChanged (ChangeEvent e)
            {
               setFieldsEnabled (enabledBox.isSelected());
            }
         });

      // add check box to panel
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      constraints.insets = new Insets (0, 4, 0, 0);
      gridBag.setConstraints (enabledBox, constraints);

      add (enabledBox);

      // add the threshold box too
      addComponent (thresholdLabel, thresholdCombo);
   }


   /**
    * Add a labelled check box
    *
    * @param name   the label for the check box in the GUI
    * @param param  the parameter which the check box corresponds to
    */
   public void addCheckBox (String name, String param)
   {
      JLabel label = new JLabel (name);
      JCheckBox checkBox = new JCheckBox();
      addComponent (label, checkBox);

      // add to list of extra components
      componentMap.put (label, JLABEL);
      componentMap.put (checkBox, param);
   }


   /**
    * Add a labelled text field
    *
    * @param name   the label for the field in the GUI
    * @param param  the parameter which the field corresponds to
    */
   public void addTextField (String name, String param)
   {
      JLabel label = new JLabel (name);
      JTextField textField = new JTextField (25);
      addComponent (label, textField);

      // add to list of extra components
      componentMap.put (label, JLABEL);
      componentMap.put (textField, param);
   }


   /**
    * Add a labelled file selector
    *
    * @param name   the label for the file selector in the GUI
    * @param param  the parameter which the file selector corresponds to
    */
   public void addFileSelector (String name, String param)
   {
      JLabel label = new JLabel (name);
      FileSelector selector = new FileSelector();
      addComponent (label, selector);

      // add to list of extra components
      componentMap.put (label, JLABEL);
      componentMap.put (selector, param);
   }


   /**
    * Add a text field to configure the conversion pattern used
    */
   public void addPatternField()
   {
      addTextField ("Conversion Pattern", CONVERSION_PATTERN);
   }


   public void addPortField()
   {
      addTextField ("Port", PORT);
   }


   /**
    * Add a component to panel with a label.
    *
    * @param label      the label for the component
    * @param component  the component
    */
   private void addComponent (JLabel label, JComponent component)
   {
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      constraints.gridwidth = GridBagConstraints.RELATIVE;
      constraints.insets = new Insets (0, 30, 0, 0);
      gridBag.setConstraints (label, constraints);

      label.setEnabled (false);
      add (label);

      constraints.weightx = 1.0;
      constraints.weighty = 1.0;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.insets = new Insets (0, 4, 0, 0);
      gridBag.setConstraints (component, constraints);

      component.setEnabled (false);
      add (component);
   }


   /**
    * Set the options in the panel according to information in log4j config file
    */
   public void setOptions()
   {
      // get a handle on OptionsLog4J
      LoggingPreferences options = LoggingPreferences.get();

      // get threshold deals with the two components we know we have
      getThreshold();

      // iterate through any extra components to set
      // according to relevant parameters
      for (Enumeration components = componentMap.keys();
         components.hasMoreElements(); )
      {
         JComponent component = (JComponent) components.nextElement();
         String param = (String)  (componentMap.get (component));

         // determine action to take based on class of component
         if (component instanceof JCheckBox)
         {
             ((JCheckBox) component).setSelected (Boolean.valueOf (options.getAppenderParam (param, configName)).booleanValue());
         }
         else if (component instanceof JTextField)
         {
            if (CONVERSION_PATTERN.equals (param))
            {
                ((JTextField) component).setText (options.getAppenderPattern (configName));
            }
            else
            {
                ((JTextField) component).setText (options.getAppenderParam (param, configName));
            }
         }
         else if (component instanceof FileSelector)
         {
             ((FileSelector) component).setFilename (options.getAppenderParam (param, configName));
         }

         // if it's a label then we're ignoring it
         // if it's anything else you need to write the
         // appropriate code to deal with it here!
      }
   }


   /**
    * Update the log4j config file according to the selections in the panel
    */
   public void updateOptions()
   {
      LoggingPreferences options = LoggingPreferences.get();

      // set threshold deals with the two components we know we have
      setThreshold();
      // iterate through any extra components to set according to
      // relevant parameters
      for (Enumeration components = componentMap.keys();
         components.hasMoreElements(); )
      {
         JComponent component = (JComponent) components.nextElement();
         String param = (String)  (componentMap.get (component));

         // determine action to take based on class of component
         if (component instanceof JCheckBox)
         {
            options.setAppenderParam (param, String.valueOf ( ((JCheckBox) component).isSelected()), configName);
         }
         else if (component instanceof JTextField)
         {
            if (CONVERSION_PATTERN.equals (param))
            {
               options.setAppenderPattern ( ((JTextField) component).getText(), configName);
            }
            else
            {
               options.setAppenderParam (param,  ((JTextField) component).getText(), configName);
            }
         }
         else if (component instanceof FileSelector)
         {
            options.setAppenderParam (param,  ((FileSelector) component).getFilename(), configName);
         }

         // if it's a label then we're ignoring it
         // if it's anything else you need to write the
         // appropriate code to deal with it here!
      }
   }


   /**
    * Enables or disables the panel contents. Different to <CODE>setEnabled</CODE> as it affects
    * all the components in the panel, rather than the panel itself.
    *
    * @param enabled  <CODE>true</CODE> to enable, <CODE>false</CODE> to disable
    */
   public void setContentsEnabled (boolean enabled)
   {
      enabledBox.setEnabled (enabled);
      setFieldsEnabled (enabled && enabledBox.isSelected());
   }


   /**
    * Enables or disables the contents of the panel, except for the checkbox.
    *
    * @param enabled  <CODE>true</CODE> to enable, <CODE>false</CODE> to disable
    */
   void setFieldsEnabled (boolean enabled)
   {
      Enumeration components = componentMap.keys();

      if (enabled)
      {
         thresholdLabel.setEnabled (true);
         thresholdCombo.setEnabled (true);
         while (components.hasMoreElements())
         {
            JComponent component = (JComponent) components.nextElement();
            component.setEnabled (true);
         }
      }
      else
      {
         thresholdLabel.setEnabled (false);
         thresholdCombo.setEnabled (false);
         while (components.hasMoreElements())
         {
            JComponent component = (JComponent) components.nextElement();
            component.setEnabled (false);
         }
      }
   }


   /**
    * Determine whether an appender is on from the node which defines it. If it's on, determine
    * which threshold it is running at.
    */
   private void getThreshold()
   {
      LoggingPreferences options = LoggingPreferences.get();
      String threshold = options.getAppenderParam ("Threshold", configName);
      if (threshold.equalsIgnoreCase (LoggerInfo.OFF))
      {
         enabledBox.setSelected (false);
      }
      else
      {
         enabledBox.setSelected (true);
         boolean found = false;
         for (int i = 0; i < THRESHOLDS.length; i++)
         {
            if (threshold.equalsIgnoreCase (THRESHOLDS[i]))
            {
               thresholdCombo.setSelectedItem (THRESHOLDS[i]);
               found = true;
               break;
            }
         }

         // following behaviour corresponds to Log4J behaviour
         // at time of coding
         if (!found)
         {
            thresholdCombo.setSelectedItem (LoggerInfo.DEBUG);
         }
      }
   }


   /**
    * Set the threshold for an appender according to current options selected in the GUI
    */
   private void setThreshold()
   {
      LoggingPreferences options = LoggingPreferences.get();

      if (!enabledBox.isSelected())
      {
         options.setAppenderParam ("Threshold", LoggerInfo.OFF.toLowerCase(), configName);
      }
      else
      {
         options.setAppenderParam ("Threshold",  ((String) thresholdCombo.getSelectedItem()).toLowerCase(), configName);
      }
   }


   /**
    * A text field to show the selected file and a button to bring the file selector up
    *
    * @author    $Author: koenigs $
    * @version   $Revision: 1.7.2.1 $
    */
   private class FileSelector extends JPanel
   {
      /**
       * Click to browse
       */
      private JButton browseButton;

      /**
       * Displays the currently selected file
       */
      private JTextField fileField;


      /**
       * Creates a new FileSelector
       */
      public FileSelector()
      {
         browseButton = new JButton ("...");

         browseButton.addActionListener (
            new ActionListener()
            {
               public void actionPerformed (ActionEvent e)
               {
                  JFileChooser fileChooser = new JFileChooser();

                  File currentFile = new File (getFilename());
                  fileChooser.setSelectedFile (currentFile);

                  fileChooser.setDialogType (JFileChooser.SAVE_DIALOG);
                  fileChooser.setDialogTitle ("Select file");

                  fileChooser.setApproveButtonText ("Select");
                  fileChooser.setApproveButtonMnemonic (KeyStroke.getKeyStroke (KeyEvent.VK_S, ActionEvent.ALT_MASK).getKeyCode());
                  fileChooser.setApproveButtonToolTipText ("Select the named file");

                  if (fileChooser.showDialog (FujabaPreferencesDialog.get(), null) == JFileChooser.APPROVE_OPTION)
                  {
                     File file = fileChooser.getSelectedFile();
                     String filePath = file.getAbsolutePath();
                     setFilename (filePath);
                  }
               }
            });

         fileField = new JTextField (20);
         // Font fileFont = fileField.getFont();
         // log.info("File font size was " + fileFont.getSize());
         fileField.setEditable (false);
         // fileField.setBorder(BorderFactory.createEtchedBorder());

         GridBagLayout selBag = new GridBagLayout();
         GridBagConstraints selConstraints = new GridBagConstraints();
         super.setLayout (selBag);
         // super.setBorder(BorderFactory.createEmptyBorder());

         selConstraints.gridwidth = GridBagConstraints.RELATIVE;
         selConstraints.fill = GridBagConstraints.BOTH;
         selConstraints.weightx = 1.0;
         selConstraints.weighty = 1.0;
         selConstraints.insets = new Insets (1, 0, 1, 1);
         selBag.setConstraints (fileField, selConstraints);
         super.add (fileField);

         selConstraints.gridwidth = GridBagConstraints.RELATIVE;
         selConstraints.fill = GridBagConstraints.VERTICAL;
         selConstraints.weightx = 0.0;
         selConstraints.weighty = 0.0;
         selConstraints.insets = new Insets (1, 0, 0, 1);
         selBag.setConstraints (browseButton, selConstraints);
         super.add (browseButton);
      }


      /**
       * Enables or disables the FileSelector and the components it contains
       *
       * @param enabled  <CODE>true</CODE> to enable, <CODE>false</CODE> to disable
       */
      public void setEnabled (boolean enabled)
      {
         browseButton.setEnabled (enabled);
         fileField.setEnabled (enabled);
         super.setEnabled (enabled);
      }


      /**
       * Sets the filename shown in the text box
       *
       * @param filename  the file name to show
       */
      public void setFilename (String filename)
      {
         fileField.setText (filename.replace (File.separatorChar, '/'));
      }


      /**
       * Gets the filename shown in the text box
       *
       * @return   the file name in the text box
       */
      public String getFilename()
      {
         return fileField.getText().replace (File.separatorChar, '/');
      }
   }
}

/*
 * $Log: AppenderPanel.java,v $
 * Revision 1.7.2.1  2006/06/22 09:04:05  koenigs
 * Added SocketHubAppender support. [ak]
 *
 */
