/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.preferences;


import java.awt.Color;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Properties;
import java.util.Vector;

import de.uni_paderborn.fujaba.app.FujabaApp;
import de.uni_paderborn.lib.util.StringManipulation;


/**
 * Singleton Class 'PreferencesProperties' handles the preferences for the Fujaba application.
 * 
 * @author $Author: lowende $
 * @version $Revision: 1.10.2.4 $
 */
public class PreferencesProperties
{
   /**
    * Fujaba core properties file name
    */
   private final static String CORE_PROPERTIES = "core/Fujaba.properties";

   /**
    * Key for the 'propertyDir' attribute that will be used by FujabaApp to store the propertyDir in
    * a (property) file.
    * 
    * <pre>
    * Note: The key contains the fujaba base-directory
    *   (installation-directory) as keyword, so
    *   <b>every fujaba installation</b>
    *   may define its own propertyDir.
    * </pre>
    */
   public final static String PROPERTY_DIR_PROPERTY = "propertyDir" + "_"
         + FujabaApp.FUJABA_BASE.toString();

   /**
    * Directory for all properties. Version number of fujaba is NOT included, because of backwards
    * compatibility to existing Fujaba 4 installations. In Fujaba 5 the version number is included.
    */
   private final static String PROPERTY_DIR = "fujaba";

   /**
    * The file for the properties.
    */
   private File settingsFile;

   /**
    * The properties hash table.
    */
   private Properties properties;

   private PropertyChangeSupport propertyChangeSupport;

   /**
    * The directory where the options are stored.
    */
   private static String baseDir = System.getProperty("user.home");

   /**
    * The table where all propertiesOptions are stored.
    * 
    * @see #get()
    * @see #get(String)
    */
   private static HashMap propertyOptions = new HashMap();


   /**
    * Private constructor, use get () or get(key) to get an instance.
    * 
    * @see #get()
    * @see #get(String)
    */
   private PreferencesProperties()
   {
      properties = new Properties();
      propertyChangeSupport = new PropertyChangeSupport(this);
   }


   /**
    * Wrapper for the old singleton get() method.
    * 
    * @return The Fujaba core properties.
    */
   public static PreferencesProperties get()
   {
      PreferencesProperties coreOptions = (PreferencesProperties) propertyOptions
            .get(CORE_PROPERTIES);

      if (coreOptions == null)
      {
         coreOptions = new PreferencesProperties();
         propertyOptions.put(CORE_PROPERTIES, coreOptions);
         coreOptions.load(CORE_PROPERTIES);
      }

      return coreOptions;
   }


   /**
    * @return The PropertyOptions for a plugin, for example.
    */
   public static PreferencesProperties get(String fileName)
   {
      PreferencesProperties currentOptions = (PreferencesProperties) propertyOptions
            .get(fileName);

      if (currentOptions == null)
      {
         currentOptions = new PreferencesProperties();
         propertyOptions.put(fileName, currentOptions);
         currentOptions.load(fileName);
      }

      return currentOptions;
   }


   /**
    * Saves all PropertiesOptions including core and plugin options.
    */
   public static void saveAll()
   {
      Iterator iter = propertyOptions.values().iterator();
      while (iter.hasNext())
      {
         PreferencesProperties currentOptions = (PreferencesProperties) iter
               .next();
         currentOptions.save();
      }
   } // saveAll


   /**
    * Saves the options in separate file.
    */
   public void save()
   {
      try
      {
         properties.store(new FileOutputStream(settingsFile),
               "Fujaba Properties, Do NOT edit!");
      }
      catch (FileNotFoundException e)
      {
         System.err.println("File " + settingsFile + " could not be saved!");
         System.err.println(e.getMessage());
      }
      catch (IOException e)
      {
         System.err.println("File " + settingsFile + " could not be saved!");
         System.err.println(e.getMessage());
      }
   } // save


   /**
    * Loads the options.
    */
   private void load(String fileName)
   {
      // fix file separators '/' to '\' and vice versa as needed by file system
      String fixedFileName = StringManipulation.fixFileNameSeparators(fileName);

      // create directories if they don't exist
      String dirName = getPropertyDir();
      if (fixedFileName.lastIndexOf(File.separatorChar) != -1)
      {
         dirName += fixedFileName.substring(0, fixedFileName
               .lastIndexOf(File.separatorChar));
      }
      File dir = new File(dirName);
      dir.mkdirs();

      // load file
      settingsFile = new File(getPropertyDir() + fixedFileName);

      try
      {
         if (settingsFile.exists())
         {
            properties.load(new FileInputStream(settingsFile));
         }
         else
         {
            boolean fileCreated = settingsFile.createNewFile();
            if (fileCreated)
            {
               System.out.println("created preference file '" + settingsFile
                     + "'");
            }

            if (CORE_PROPERTIES.equals(fileName))
            {
               generateDefaultSettings();
               save();
            }
         }
      }
      catch (Exception e)
      {
         System.err.println("The preferences " + settingsFile
               + " could not be loaded!");
         e.printStackTrace();
      }
   } // load


   /**
    * Cache for property directory.
    * 
    * <pre>
    * Note: changing property directory during runtime
    * is not recommended until Fujaba code has been reviewed.
    * </pre>
    */
   private static String propertyDir = null;


   /**
    * Get the directory in which Fujaba properties are stored.
    * 
    * <pre>
    * Note: the propertyDir ends with a file separator.
    * Note: if the propertyDir has not been set, a call to
    *       this function will initialize the propertyDir with
    *       the proposedPropertyDir.
    * </pre>
    * 
    * @return The directory in which properties are stored.
    * @see PreferencesProperties#getProposedPropertyDir()
    */
   public static String getPropertyDir()
   {
      // use cached property directory, because property dir
      // should not change during runtime for now
      if (propertyDir == null)
      {
         setPropertyDir(getProposedPropertyDir());
      }

      return propertyDir;
   }


   /**
    * Set the directory in which Fujaba properties are stored.
    * 
    * <pre>
    *
    * Note: do not change propertyDir during runtime, for now!
    * Properties won't be adjusted, if the propertyDir changes.
    * Use startup-parameter in class 'FujabaApp' to change the
    * propertyDir during startup.
    * </pre>
    * 
    * @param value
    * @see de.uni_paderborn.fujaba.app.FujabaApp#main(String[])
    */
   public static void setPropertyDir(String value)
   {
      // make sure propertyDir ends with separator
      if (!value.endsWith(File.separator))
      {
         value += File.separator;
      }

      if (value != propertyDir)
      {
         // TODO: activate property change support for this property,
         // after 'changing of propertyDir during runtime' works
         // String oldValue = propertyDir;
         propertyDir = value;
         // firePropertyChange(PROPERTY_DIR_PROPERTY, oldValue, value);
      }
   }


   /**
    * Get the default directory in which Fujaba properties should be stored. The directory-name is
    * constructed dependent on the version number of the fujaba application and is basically located
    * in the users home directory. Note: the propertyDir ends with a file separator.
    * 
    * @return The directory in which properties should be stored.
    */
   public static String getProposedPropertyDir()
   {
      StringBuffer propertyDirBuffer = new StringBuffer();

      propertyDirBuffer.append(baseDir);
      propertyDirBuffer.append(File.separatorChar);

      if ("Linux".equals(System.getProperty("os.name"))
            || "Unix".equals(System.getProperty("os.name")))
      {
         propertyDirBuffer.append(".");
      }

      propertyDirBuffer.append(PROPERTY_DIR);
      propertyDirBuffer.append(File.separatorChar);

      return propertyDirBuffer.toString();
   }


   /**
    * Generate Default settings for Fujaba, needs to be modified for new settings.
    */
   private void generateDefaultSettings()
   {
      GeneralPreferences.get().setDefaults();
      JavaPreferences.get().setDefaults();
      IconsPreferences.get().setDefaults();
      ColorsPreferences.get().setDefaults();
      DebugPreferences.get().setDefaults();
      LoggingPreferences.get().setDefaults();
      PlugInsPreferences.get().setDefaults();
   } // generateDefaultSettings


   // ----------------------------------------------------------------------------
   // property change support
   // ----------------------------------------------------------------------------

   /**
    * Access method for an one to n association.
    * 
    * @param listener The object added.
    */
   public void addPropertyChangeListener(PropertyChangeListener listener)
   {
      propertyChangeSupport.addPropertyChangeListener(listener);
   }


   /**
    * Access method for an one to n association.
    * 
    * @param propertyName The object added.
    * @param listener The object added.
    */
   public void addPropertyChangeListener(String propertyName,
         PropertyChangeListener listener)
   {
      propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param listener No description provided
    */
   public void removePropertyChangeListener(PropertyChangeListener listener)
   {
      propertyChangeSupport.removePropertyChangeListener(listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param propertyName No description provided
    * @param listener No description provided
    */
   public void removePropertyChangeListener(String propertyName,
         PropertyChangeListener listener)
   {
      propertyChangeSupport
            .removePropertyChangeListener(propertyName, listener);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param event No description provided
    */
   public void firePropertyChange(PropertyChangeEvent event)
   {
      propertyChangeSupport.firePropertyChange(event);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param propertyName No description provided
    * @param oldValue No description provided
    * @param newValue No description provided
    */
   public void firePropertyChange(String propertyName, boolean oldValue,
         boolean newValue)
   {
      propertyChangeSupport
            .firePropertyChange(propertyName, oldValue, newValue);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param propertyName No description provided
    * @param oldValue No description provided
    * @param newValue No description provided
    */
   public void firePropertyChange(String propertyName, int oldValue,
         int newValue)
   {
      propertyChangeSupport
            .firePropertyChange(propertyName, oldValue, newValue);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param propertyName No description provided
    * @param oldValue No description provided
    * @param newValue No description provided
    */
   public void firePropertyChange(String propertyName, Object oldValue,
         Object newValue)
   {
      propertyChangeSupport
            .firePropertyChange(propertyName, oldValue, newValue);
   }


   /**
    * Get the propertyChangeListeners attribute of the PreferencesProperties object
    * 
    * @return The propertyChangeListeners value
    */
   public PropertyChangeListener[] getPropertyChangeListeners()
   {
      return propertyChangeSupport.getPropertyChangeListeners();
   }


   /**
    * Get the propertyChangeListeners attribute of the PreferencesProperties object
    * 
    * @param arg0 No description provided
    * @return The propertyChangeListeners value
    */
   public PropertyChangeListener[] getPropertyChangeListeners(String arg0)
   {
      return propertyChangeSupport.getPropertyChangeListeners(arg0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param arg0 No description provided
    * @return No description provided
    */
   public boolean hasListeners(String arg0)
   {
      return propertyChangeSupport.hasListeners(arg0);
   }


   // ----------------------------------------------------------------------------
   // getter and setter for the properties
   // ----------------------------------------------------------------------------

   /**
    * get a specified setting from the Properties
    * 
    * @param key The settings key
    * @return The value of the setting specified by key
    */
   public String getSetting(String key)
   {
      return properties.getProperty(key);
   } // getSetting (String)


   /**
    * get a specified setting from the Properties
    * 
    * @param key The settings key
    * @param defaultValue No description provided
    * @return The value of the setting specified by key
    */
   public String getSetting(String key, String defaultValue)
   {
      String result = getSetting(key);
      if (result == null)
      {
         putSetting(key, defaultValue);
         result = defaultValue;
      }

      return result;
   } // getSetting (String, String)


   /**
    * Get the boolSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @return The boolSetting value
    */
   public boolean getBoolSetting(String key)
   {
      return getBoolSetting(key, false);
   } // getBoolSetting (String)


   /**
    * Get the boolSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @param defaultValue No description provided
    * @return The boolSetting value
    */
   public boolean getBoolSetting(String key, boolean defaultValue)
   {
      String result = getSetting(key, String.valueOf(defaultValue));

      return (Boolean.valueOf(result)).booleanValue();
   } // getBoolSetting (String, boolean)


   /**
    * Get the intSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @return The intSetting value
    * @throws NumberFormatException Exception description not provided
    */
   public int getIntSetting(String key) throws NumberFormatException
   {
      return getIntSetting(key, 0);
   } // getIntSetting (String)


   /**
    * Get the intSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @param defaultValue No description provided
    * @return The intSetting value
    * @throws NumberFormatException Exception description not provided
    */
   public int getIntSetting(String key, int defaultValue)
         throws NumberFormatException
   {
      String result = getSetting(key, String.valueOf(defaultValue));

      return (Integer.valueOf(result)).intValue();
   } // getIntSetting (String, int)


   /**
    * Get the floatSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @return The floatSetting value
    * @throws NumberFormatException Exception description not provided
    */
   public float getFloatSetting(String key) throws NumberFormatException
   {
      return getFloatSetting(key, 0.0f);
   } // getFloatSetting (String)


   /**
    * Get the floatSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @param defaultValue No description provided
    * @return The floatSetting value
    * @throws NumberFormatException Exception description not provided
    */
   public float getFloatSetting(String key, float defaultValue)
         throws NumberFormatException
   {
      String result = getSetting(key, String.valueOf(defaultValue));

      return (Float.valueOf(result)).floatValue();
   } // getFloatSetting (String, float)


   /**
    * Get the doubleSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @return The doubleSetting value
    * @throws NumberFormatException Exception description not provided
    */
   public double getDoubleSetting(String key) throws NumberFormatException
   {
      return getDoubleSetting(key, 0.0);
   } // getDoubleSetting (String)


   /**
    * Get the doubleSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @param defaultValue No description provided
    * @return The doubleSetting value
    * @throws NumberFormatException Exception description not provided
    */
   public double getDoubleSetting(String key, double defaultValue)
         throws NumberFormatException
   {
      String result = getSetting(key, String.valueOf(defaultValue));

      return (Double.valueOf(result)).doubleValue();
   } // getDoubleSetting (String, double)


   /**
    * Get a preferences vector.
    * 
    * @param key The property key
    * @return The preferences vector, if the key can be found, otherwise an empty vector.
    */
   public Vector getVectorSetting(String key)
   {
      return getVectorSetting(key, new Vector());
   }


   /**
    * Get a preferences vector.
    * 
    * @param key The property key
    * @param defVector The default vector
    * @return The preferences vector, if the key can be found, otherwise the default vector.
    */
   public Vector getVectorSetting(String key, Vector defVector)
   {
      Vector vector = new Vector();

      int i = 0;
      String value = getSetting(key + i);

      if (value == null)
      {
         return defVector;
      }

      while (value != null)
      {
         vector.addElement(value);
         i++;
         value = getSetting(key + i);
      }

      return vector;
   } // getVectorSetting


   /**
    * Get the hashtableSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @param name No description provided
    * @return The hashtableSetting value
    */
   public Hashtable getHashtableSetting(String key, String name)
   {
      Hashtable hashtable = new Hashtable();

      int i = 0;
      String secondaryKey = getSetting(key + name + i);
      String value = null;
      while (secondaryKey != null)
      {
         value = getSetting(key + name + secondaryKey);
         if (value != null)
         {
            hashtable.put(secondaryKey, value);
         }

         i++;
         secondaryKey = getSetting(key + name + i);
      }

      return hashtable;
   } // getHashtableSetting


   /**
    * Get the colorSetting attribute of the PropertyOptions object
    * 
    * @param key No description provided
    * @param defaultValue No description provided
    * @return The colorSetting value
    */
   public Color getColorSetting(String key, Color defaultValue)
   {
      int red = getIntSetting(key + "_R", defaultValue.getRed());
      int green = getIntSetting(key + "_G", defaultValue.getGreen());
      int blue = getIntSetting(key + "_B", defaultValue.getBlue());

      return (new Color(red, green, blue));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param value No description provided
    */
   public void putSetting(String key, String value)
   {
      Object oldValue = properties.get(key);
      properties.put(key, value);

      // fire property change event
      propertyChangeSupport.firePropertyChange(key, oldValue, value);
   } // putSetting (String, String)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param value No description provided
    */
   public void putSetting(String key, boolean value)
   {
      putSetting(key, String.valueOf(value));
   } // putSetting (String, boolean)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param value No description provided
    */
   public void putSetting(String key, int value)
   {
      putSetting(key, String.valueOf(value));
   } // putSetting (String, int)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param value No description provided
    */
   public void putSetting(String key, float value)
   {
      putSetting(key, String.valueOf(value));
   } // putSetting (String, float)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param value No description provided
    */
   public void putSetting(String key, double value)
   {
      putSetting(key, String.valueOf(value));
   } // putSetting (String, double)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param settings No description provided
    */
   public void putSetting(String key, List settings)
   {
      int i = 0;

      // remove old settings
      while (getSetting(key + i) != null)
      {
         properties.remove(key + i);
         i++;
      }

      // save new settings
      ListIterator listIter = settings.listIterator();
      while (listIter.hasNext())
      {
         i = listIter.nextIndex();
         String value = listIter.next().toString();
         putSetting(key + i, value);
      }
   } // putSetting (String, Vector)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param name No description provided
    * @param settings No description provided
    */
   public void putSetting(String key, String name, Hashtable settings)
   {
      int i = 0;
      String secondaryKey;

      // remove old settings
      secondaryKey = getSetting(key + name + i);
      while (secondaryKey != null)
      {
         properties.remove(key + name + i);
         properties.remove(key + name + secondaryKey);
         i++;
         secondaryKey = getSetting(key + name + i);
      }

      i = 0;
      // save new settings
      Enumeration keyEnum = settings.keys();
      while (keyEnum.hasMoreElements())
      {
         secondaryKey = keyEnum.nextElement().toString();
         String value = settings.get(secondaryKey).toString();
         putSetting(key + name + i, secondaryKey);
         putSetting(key + name + secondaryKey, value);
         i++;
      }
   } // putSetting (String, Hashtable)


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    * 
    * @param key No description provided
    * @param col No description provided
    */
   public void putSetting(String key, Color col)
   {
      // remove old settings
      properties.remove(key + "_R");
      properties.remove(key + "_G");
      properties.remove(key + "_B");
      // save new settings
      putSetting(key + "_R", col.getRed());
      putSetting(key + "_G", col.getGreen());
      putSetting(key + "_B", col.getBlue());
   }
}

/*
 * $Log: PreferencesProperties.java,v $
 * Revision 1.10.2.4  2006/04/03 14:37:33  lowende
 * Patch from HEAD: property dir will be saved dependent from Fujaba's installation dir.
 * Revision 1.10.2.3 2006/03/30 15:51:32 lowende Removed
 * compile warnings.
 * 
 * Revision 1.10.2.2 2006/03/13 11:43:36 fklar changed default property-directory back to "fujaba"
 * (without version-number) for backwards comaptibility with existing installations
 * 
 */
