/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.parser;

import java.util.HashSet;

import de.uni_paderborn.fujaba.metamodel.FFile;
import de.uni_paderborn.fujaba.metamodel.FMethod;
import de.uni_paderborn.fujaba.uml.ASTRootNode;


/**
 * Abstract super class for parsers connected to Fujaba
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *         0..1      currentParser      0..1
 * Parser <---------------------------------- ParserManager
 *         currentParser      parserManager
 *
 *         0..n     parsers      0..1
 * Parser ---------------------------- ParserManager
 *         parsers      parserManager
 * </pre>
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.13 $
 */
public abstract class Parser
{
   /**
    * Parses the given files.
    *
    * @param files  The files to be parsed
    */
   public abstract void parseFiles (HashSet files);


   /**
    * Parses the compilation unit and returns the abstract syntax graph (ASG)
    * with an FFile as root.
    *
    * @param filename  The name of the file to be parsed
    * @param reader    An input reader
    * @return          An FFile with the ASG of the parsed compilation unit
    */
   public abstract FFile parseCompilationUnit (String filename, java.io.Reader reader);


   /**
    * Parses the given method body and returns the root node
    * of the abstract syntax tree.
    *
    * @param fMethod  The method/constructor to be parsed
    * @return         The ASTRootNode
    */
   public abstract ASTRootNode parseMethodBody (FMethod fMethod);


   /**
    * Access method for a To N-association.
    *
    * @param parserPostProcessor  The object added.
    */
   public abstract void addParserPostProcessor (ParserPostProcessor parserPostProcessor);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param parserPostProcessor  No description provided
    */
   public abstract void removeParserPostProcessor (ParserPostProcessor parserPostProcessor);


   /**
    * <pre>
    *                0..1     parsers      0..n
    * ParserManager ---------------------------- Parser
    *                parserManager      parsers
    * </pre>
    */
   private ParserManager parserManager;


   /**
    * Get the parserManager attribute of the Parser object
    *
    * @return   The parserManager value
    */
   public ParserManager getParserManager()
   {
      return this.parserManager;
   }


   /**
    * Sets the parserManager attribute of the Parser object
    *
    * @param value  The new parserManager value
    * @return       No description provided
    */
   public boolean setParserManager (ParserManager value)
   {
      boolean changed = false;
      if (this.parserManager != value)
      {
         if (this.parserManager != null)
         {
            ParserManager oldValue = this.parserManager;
            this.parserManager = null;
            oldValue.removeFromParsers (this);
         }
         this.parserManager = value;
         if (value != null)
         {
            value.addToParsers (this);
         }
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      ParserManager tmpParserManager = getParserManager();
      if (tmpParserManager != null)
      {
         setParserManager (null);
      }
   }

}

/*
 * $Log: Parser.java,v $
 * Revision 1.13  2005/03/14 08:39:52  lowende
 * Enhanced the parser interface. The JavaParser 3.0 version won't work.
 * A new JavaParser version 3.1 will be released soon.
 *
 */
