/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.packagediagrams;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.asg.ASGDiagram;
import de.uni_paderborn.fujaba.fsa.swing.DiagramKindComboBoxModel;
import de.uni_paderborn.fujaba.metamodel.FDiagram;
import de.uni_paderborn.fujaba.uml.UMLDiagramItem;


/**
 * UMLClass: 'DiagramUsage' A diagram usage is a placeholder for a diagram within a packagediagram.
 *
 * @author    $Author: mksoft $
 * @version   $Revision: 1.33.2.3 $
 */
public class DiagramUsage extends UMLDiagramItem
{
   /**
    * log4j logging
    */
   private final static transient Logger log = Logger.getLogger (DiagramUsage.class);


   /**
    * Constructor for class DiagramUsage
    */
   public DiagramUsage()
   {
      super();
   }


   /**
    * Constructor for class DiagramUsage
    *
    * @param name  No description provided
    */
   public DiagramUsage (String name)
   {
      super();
      this.setName (name);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String name = "";


   /**
    * Get the name attribute of the DiagramUsage object
    *
    * @return   The name value
    */
   public String getName()
   {
      return this.name;
   }


   /**
    * Sets the name attribute of the DiagramUsage object
    *
    * @param strg  The new name value
    */
   public void setName (String strg)
   {
      if (strg != null && !this.name.equals (strg))
      {
         String oldName = this.name;

         ASGDiagram diagram = this.getDiagram();

         if (diagram != null && !diagram.getName().equals (strg))
         {
            diagram.setName (strg);
         }

         this.name = strg;

         firePropertyChange ("name", oldName, this.name);
      }
   }



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int x = -1;


   /**
    * Get the x attribute of the DiagramUsage object
    *
    * @return   The x value
    */
   public int getX()
   {
      return x;
   }


   /**
    * Sets the x attribute of the DiagramUsage object
    *
    * @param tempX  The new x value
    */
   public void setX (int tempX)
   {
      this.x = tempX;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int y = -1;


   /**
    * Get the y attribute of the DiagramUsage object
    *
    * @return   The y value
    */
   public int getY()
   {
      return y;
   }


   /**
    * Sets the y attribute of the DiagramUsage object
    *
    * @param tempY  The new y value
    */
   public void setY (int tempY)
   {
      this.y = tempY;
   }


   /**
    * <pre>
    *             0..1    usages    n
    * UMLDiagram --------------------- DiagramUsage
    *             diagram      usages
    * </pre>
    */
   private ASGDiagram diagram;


   /**
    * UMLMethod: '+ setDiagram (value: UMLDiagram): Boolean'
    *
    * @param value  The new diagram value
    * @return       No description provided
    */
   public boolean setDiagram (ASGDiagram value)
   {
      if (log.isInfoEnabled())
      {
         log.info ("Setting " + this + "'s diagram to " + value);
      }
      boolean changed = false;
      if (this.diagram != value)
      {
         ASGDiagram oldValue = this.diagram;
         if (this.diagram != null)
         {
            this.diagram = null;
            oldValue.removeFromUsages (this);
         }
         this.diagram = value;
         if (value != null)
         {
            value.addToUsages (this);
         }
         changed = true;

         if (value != null)
         {
            this.setName (this.diagram.getName());
            if (log.isInfoEnabled())
            {
               log.info ("value's class is " + value);
            }
            int kind = DiagramKindComboBoxModel.getDiagramKindIndex (value.getClass());
            if (kind >= 0)
            {
               this.setDiagramKind (kind);
            }
         }
         firePropertyChange ("diagram", oldValue, value);
      }
      return changed;
   }


   /**
    * UMLMethod: '+ getDiagram (): UMLDiagram'
    *
    * @return   The diagram value
    */
   public ASGDiagram getDiagram()
   {
      return this.diagram;
   }


   /**
    * UMLMethod: '+ removeYou (): Void'
    */
   public void removeYou()
   {
      FDiagram tmpDiagram = getDiagram();
      if (tmpDiagram != null)
      {
         setDiagram (null);
      } // if

      super.removeYou();
   }


   /**
    * UMLAttribute: 'whichDiagram : String'
    */
   private String whichDiagram = "package";


   /**
    * UMLMethod: 'Read access method for attribute whichDiagram : String'
    *
    * @return   The whichDiagram value
    */
   public String getWhichDiagram()
   {
      return whichDiagram;
   }


   /**
    * UMLAttribute: 'diagramKind : Integer = 1'
    */
   private int diagramKind = 0;


   /**
    * UMLMethod: 'Read access method for attribute diagramKind : Integer = 1'
    *
    * @return   The diagramKind value
    */
   public int getDiagramKind()
   {
      return this.diagramKind;
   }


   /**
    * UMLMethod: 'Write access method for attribute diagramKind : Integer = 1'
    * FIX ME: should use diagram kind name here! Indices may change!
    *
    * @param diagramKind  The new diagramKind value
    * @return             No description provided
    */
   public int setDiagramKind (int diagramKind)
   {
      if (log.isInfoEnabled())
      {
         log.info ("new diagram kind: " + diagramKind);
      }

      int oldKind = this.diagramKind;

      FDiagram diagram = this.getDiagram();

      if (diagram == null)
      {
         this.diagramKind = diagramKind;
         firePropertyChange ("diagramKind", oldKind, diagramKind);
      }
      else
      {
         int kind = DiagramKindComboBoxModel.getDiagramKindIndex (diagram.getClass());

         if (kind >= 0)
         {
            int currentDiagramKind = kind;
            if (currentDiagramKind != this.diagramKind)
            {
               this.diagramKind = currentDiagramKind;
               firePropertyChange ("diagramKind", oldKind, currentDiagramKind);
            }
         }
         else
         {
            if (log.isInfoEnabled())
            {
               log.info ("This should never happen. Its Mike's fault. \n"
                  + "DiagramKindComboBoxModel.diagClassToKindNo does not contain " + diagram.getClass());
            }
         }

         if (this.diagramKind != diagramKind)
         {
            firePropertyChange ("diagramKind", diagramKind, this.diagramKind);
            if (log.isInfoEnabled())
            {
               log.info ("overridden the setdiagramkind");
            }
         }

      }

      return this.diagramKind;
   }
}

/*
 * $Log: DiagramUsage.java,v $
 * Revision 1.33.2.3  2005/09/30 18:57:31  mksoft
 * replacing many System.out.println with if (log.isInfoEnabled()) log.info ()
 *
 */
