/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.mpEdit;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.util.Hashtable;

import javax.swing.*;

import org.apache.log4j.Logger;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.22.2.3 $
 */
final class TextCanvas
    extends JComponent
    implements MouseMotionListener,
   MouseListener,
   ComponentListener,
   AdjustmentListener,
   KeyListener,
   FocusListener
{
   /**
    * log4j logging
    */
   final static transient Logger log = Logger.getLogger (TextCanvas.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected final int EDGE = 10;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected JScrollBar horiz; // control

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected JScrollBar vert; // control

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean eactive,
      hactive; // is there currently a range?

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean mouseDown;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean autoIndent = true; // auto indentation after ENTER?

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean doSeparator = false; // print separator bar?

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int sx,
      sy,
      ny; // current scroll positions

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int line,
      column,
      pix; // the current column

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int eline,
      ecolumn,
      epix; // the current end column

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int oldlines; // repaint to old line count (if longer)

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int fontHeight; // save time

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int fontDescent; // save time

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int tabSize; // preference

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Font font; // preference

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected FontMetrics fontMetrics; // save time

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Ruler ruler; // measure strings in pixels

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected TextFrame textFrame; // friendly object

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected TextCursor textCursor; // friendly object

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected TextMenu textMenu; // friendly object

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Dimension dimension; // save time

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected DocMan docMan; // the document

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected mpEDIT mpEdit; // the big cheese

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color textColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color textXColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color commentColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color commentXColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color keywordColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color keywordXColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color quoteColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color quoteXColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color hideColor;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Color hideXColor;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int linesEmpty = 0; // how many succesive lines was empty at the moment

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int lastVerticalSize = 10; // vertical size of view in text lines

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected TextScroller textScroller = null;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected Hashtable actionDictionary;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   final int LINE_MAX = 2000;


   /**
    * Constructor for class TextCanvas
    *
    * @param mpe  No description provided
    * @param doc  No description provided
    * @param tf   No description provided
    * @param h    No description provided
    * @param v    No description provided
    * @param r    No description provided
    */
   public TextCanvas (mpEDIT mpe,
                      DocMan doc,
                      TextFrame tf,
                      JScrollBar h,
                      JScrollBar v,
                      Ruler r)
   {
      super();

      mpEdit = mpe;
      docMan = doc;
      ruler = r;

      createActionDictionary();

      setBackground (Color.white);
      setForeground (Color.black);

      textFrame = tf;

      horiz = h;
      horiz.addAdjustmentListener (this);
      horiz.addKeyListener (this);

      vert = v;
      vert.addAdjustmentListener (this);
      vert.addKeyListener (this);

      addComponentListener (this);
      addMouseListener (this);
      addKeyListener (this);
      addFocusListener (this);

      applyProperties();

      clear();
      validate();
      requestFocus();

      textCursor = new TextCursor (this);
      textCursor.start();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected boolean gotFocus = false;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void focusGained (FocusEvent e)
   {
      gotFocus = true;
      release_cursor();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void focusLost (FocusEvent e)
   {
      pause_cursor();
      gotFocus = false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void release_cursor()
   {
      if ( (textCursor != null) && !eactive && gotFocus)
      {
         textCursor.release_cursor (true);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void pause_cursor()
   {
      if ( (textCursor != null) && !eactive && gotFocus)
      {
         textCursor.pause_cursor (true);
      }
   }


   /**
    * Sets the textMenu attribute of the TextCanvas object
    *
    * @param tm  The new textMenu value
    */
   public void setTextMenu (TextMenu tm)
   {
      textMenu = tm;
   }


   /**
    * Get the highest attribute of the TextCanvas object
    *
    * @return   The highest value
    */
   public int getHighest()
   { // highest visible line

      return sy + ny - 1;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void clear()
   {
      clear_area (null);
      hactive = eactive = false;
      updateCopyActions (false);
      oldlines = line = column = sx = sy = 0;
      pix = EDGE;
      horiz.setValue (0);
      vert.setValue (0);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void applyProperties()
   {

      font = docMan.getFont();
      tabSize = docMan.getTabSize();

      Graphics g = getGraphics();

      if (g != null)
      {
         clear_area (g);
         updateFonts (g);
         ruler.setTabSize (tabSize);
         resizeLines();
         redoCanvas();
         repaint();
         g.dispose();
      }
      else
      {
         fontMetrics = null;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   protected void clear_area (Graphics g)
   {
      boolean foobar = false;

      pause_cursor();

      if (g == null)
      {
         g = getGraphics();
         foobar = true;
      }

      if (g != null)
      {
         g.clearRect (0, 0, dimension.width, dimension.height);
      }

      if (foobar &&  (g != null))
      {
         g.dispose();
      }
   }


   /**
    * Sets the pos attribute of the TextCanvas object
    *
    * @param tp  The new pos value
    */
   public void setPos (TextPosition tp)
   {
      hactive = eactive = false;
      line = eline = tp.line;
      column = ecolumn = tp.column;
      pix = epix = ruler.length (line, column) + EDGE;
      shiftVert (line);
      shiftHoriz (pix);
      repaint();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param y  No description provided
    */
   public void Goto (int y)
   {
      y--; // line is zero based

      if ( (y >= 0) &&  (y < docMan.getLineCount()))
      {
         boolean paint = hactive;
         hactive = eactive = false;
         line = eline = y;
         column = ecolumn = 0;
         pix = epix = EDGE;

         if (shiftVert (line))
         {
            paint = true;
         }

         if (shiftHoriz (pix))
         {
            paint = true;
         }

         if (paint)
         {
            repaint();
         }
      }
   }


   /**
    * Get the line attribute of the TextCanvas object
    *
    * @return   The line value
    */
   public int getLine()
   {
      return line + 1; // line is zero based

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void legalizeCursor()
   {
      String whole;
      boolean illegal = false;
      int max_line;
      int max_column;

      max_line = docMan.getLineCount();

      if (line >= max_line)
      {
         line = max_line - 1;
         illegal = true;
      }

      whole = docMan.getLine (line);
      max_column = whole.length();

      if (column > max_column)
      {
         column = max_column;
         illegal = true;
      }

      if (eactive &&  (eline >= max_line))
      {
         illegal = true;
      }

      if (eactive && !illegal)
      {
         whole = detabbed (docMan.getLine (eline));
         max_column = whole.length();

         if (ecolumn > max_column)
         {
            illegal = true;
         }
      }

      if (line < 0)
      {
         line = 0;
      }

      if (column < 0)
      {
         column = 0;
      }

      if (illegal)
      {
         hactive = eactive = false;
         pix = epix = ruler.length (line, column) + EDGE;
         shiftVert (line);
         shiftHoriz (pix);
         repaint();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   protected void updateFonts (Graphics g)
   {
      boolean foobar = false;

      if (g == null)
      {
         g = getGraphics();
         foobar = true;
      }

      if (g != null)
      {
         g.setFont (font);
         fontMetrics = g.getFontMetrics (font);
         fontHeight = fontMetrics.getHeight();
         fontDescent = fontMetrics.getDescent();

         ruler.setFontMetrics (fontMetrics);

         if (foobar)
         {
            g.dispose();
         }
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param patt  The object added.
    */
   public void addSearchPattern (String patt)
   {
      mpEdit.addSearchPattern (patt);
   }


   /**
    * Get the searchPatterns attribute of the TextCanvas object
    *
    * @return   The searchPatterns value
    */
   public String[] getSearchPatterns()
   {
      return mpEdit.getSearchPatterns();
   }


   /**
    * Get the latestSearchPattern attribute of the TextCanvas object
    *
    * @return   The latestSearchPattern value
    */
   public String getLatestSearchPattern()
   {
      return mpEdit.getLatestSearchPattern();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param direction  No description provided
    */
   public void findAgain (int direction)
   {
      String lastFind = getLatestSearchPattern();

      if (lastFind == null)
      {
         release_cursor();
      }
      else
      {
         find (lastFind, direction);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pattern  No description provided
    * @return         No description provided
    */
   public boolean find (String pattern)
   {
      return find (pattern, 1);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pattern    No description provided
    * @param direction  No description provided
    * @return           No description provided
    */
   public boolean find (String pattern, int direction)
   {
      int i;
      int
         max;
      int
         ct;
      int
         start;
      int
         from;
      String whole;

      pause_cursor();

      max = docMan.getLineCount();
      i = line;
      ct = 0;

      while (ct++ <= max)
      {
         whole = docMan.getLine (i);

         if ( (i == line) &&  (ct == 1))
         {
            if (eactive)
            {
               from = column + direction;
            }
            else
            {
               from = column;
            }
         }
         else
         {
            from =  (direction > 0) ? 0 : whole.length();
         }

         if (direction > 0)
         {
            start = whole.indexOf (pattern, from);
         }
         else
         {
            start = whole.lastIndexOf (pattern, from);
         }

         if (start >= 0)
         {
            line = eline = i;
            column = start;
            pix = ruler.length (line, column) + EDGE;
            ecolumn = start + pattern.length();
            epix = ruler.length (eline, ecolumn) + EDGE;
            eactive = true;
            setup_h();
            save_h();
            updateCopyActions (eactive);
            shiftVert (line);
            shiftHoriz (pix);
            shiftHoriz (epix);
            repaint();
            return true;
         }

         if (direction > 0)
         {
            i++;
         }
         else
         {
            i--;
         }

         if (i >= max)
         {
            i = 0;
            showStatus ("Search wrapped");
         }
         else if (i < 0)
         {
            i = max - 1;
            showStatus ("Search wrapped");
         }
      }

      release_cursor();
      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public synchronized void mousePressed (MouseEvent e)
   {
      Graphics g;

      requestFocus();

      if (mouseDown)
      {
         return;
      }

      mouseDown = true;

      pause_cursor();
      g = getGraphics();
      updateFonts (g);

      clickPosition (e.getX(), e.getY());

      if ( (cline < 0) ||  (cline >= docMan.getLineCount()))
      {
         return;
      }

      eline = cline;
      ecolumn = ccolumn;
      epix = cpix;

      int clickCount = e.getClickCount();

      if (clickCount == 3)
      {
         eline = line = cline;
         column = 0;
         pix = EDGE;
         String whole = docMan.getLine (cline);
         ecolumn = whole.length();
         epix = ruler.length (cline, 10000000) + EDGE;
         eactive = true;
         setup_h();
         save_h();
         updateCopyActions (eactive);
         repaint();
      }
      else if (clickCount == 2)
      {
         eline = line = cline;
         String whole = docMan.getLine (cline);
         column = ecolumn = ccolumn;
         while (true)
         {
            if (column == 0)
            {
               break;
            }

            if (!Character.isLetterOrDigit (whole.charAt (column - 1)))
            {
               break;
            }

            column--;
         }
         pix = ruler.length (cline, column) + EDGE;
         int max = whole.length();
         while (true)
         {
            if (ecolumn >= max)
            {
               break;
            }

            if (!Character.isLetterOrDigit (whole.charAt (ecolumn)))
            {
               break;
            }

            ecolumn++;
         }
         epix = ruler.length (cline, ecolumn) + EDGE;
         eactive = true;
         setup_h();
         save_h();
         updateCopyActions (eactive);
         repaint();
      }
      else if (e.isShiftDown())
      {
         eline = cline;
         ecolumn = ccolumn;
         epix = cpix;
         eactive = true;
         setup_h();
         save_h();
         updateCopyActions (eactive);
         repaint();
      }
      else
      {
         if (hactive)
         {
            flip_h (g, oline, opix, oeline, oepix);
         }
         eline = line = cline;
         ecolumn = column = ccolumn;
         epix = pix = cpix;
         hactive = false;
         eactive = true;
         updateCopyActions (eactive);
         addMouseMotionListener (this);
      }
      repaint(); //Fixes obsolete cursor bug (hope so) //TRI
      g.dispose();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int opix,
      oepix,
      oline,
      oeline;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int hpix,
      hepix,
      hline,
      heline,
      hcolumn,
      hecolumn;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int lastx,
      lasty;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public synchronized void mouseDragged (MouseEvent e)
   {
      Graphics g = getGraphics();
      if (g != null)
      {
         updateFonts (g);

         if (e != null)
         {
            lastx = e.getX();
            lasty = e.getY();
         }

         clickPosition (lastx, lasty);

         eline = cline;
         ecolumn = ccolumn;
         epix = cpix;

         eactive =  ( (eline != line) ||  (ecolumn != column));
         if (eactive || hactive)
         {
            setup_h();

            if ( (eline < sy) ||
                (eline >= sy + ny) ||
                (epix < sx) ||
                (epix >= dimension.width))
            {
               if (textScroller == null)
               {
                  textScroller = new TextScroller (this);
                  textScroller.start();
               }
               shiftVert (eline);
               shiftHoriz (epix);
               setup_h();
               save_h();
               docMan.extendHilite (sy + ny - 1);
               for (int i = 0; i < ny; i++)
               {
                  drawLine (g, sy + i);
               }
            }
            else
            {
               if (textScroller != null)
               {
                  textScroller.setStopThread (true);
                  textScroller = null;
               }
               if (hactive)
               {
                  if ( (hline < oline) ||  ( (hline == oline) &&  (hpix < opix)))
                  {
                     flip_h (g, hline, hpix, oline, opix);
                  }
                  if ( (hline > oline) ||  ( (hline == oline) &&  (hpix > opix)))
                  {
                     flip_h (g, oline, opix, hline, hpix);
                  }
                  if ( (heline < oeline) ||  ( (heline == oeline) &&  (hepix < oepix)))
                  {
                     flip_h (g, heline, hepix, oeline, oepix);
                  }
                  if ( (heline > oeline) ||  ( (heline == oeline) &&  (hepix > oepix)))
                  {
                     flip_h (g, oeline, oepix, heline, hepix);
                  }
               }
               else
               {
                  flip_h (g, hline, hpix, heline, hepix);
               }

               hactive = eactive;
               save_h();
            }
         }
         g.dispose();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected int cline,
      ccolumn,
      cpix;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param x  No description provided
    * @param y  No description provided
    */
   void clickPosition (int x, int y)
   {
      cline =  (y / fontHeight) + sy;

      if (cline < 0)
      {
         cline = ccolumn = cpix = 0;
         return;
      }

      if (cline >= docMan.getLineCount())
      {
         cline = docMan.getLineCount() - 1;
         String whole = docMan.getLine (cline);
         ccolumn = whole.length();
         cpix = ruler.length (cline, 10000000) + EDGE;
         return;
      }

      x = x + sx - EDGE;

      ccolumn = cpix = 0;

      if (x > 0)
      {
         TextPosition tp = ruler.position (cline, x);
         ccolumn = tp.column;
         cpix = tp.pix + EDGE;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g      No description provided
    * @param sline  No description provided
    * @param spix   No description provided
    * @param eline  No description provided
    * @param epix   No description provided
    */
   protected void flip_h (Graphics g, int sline, int spix, int eline, int epix)
   {
      int i;
      int bx;
      int
         ex;
      int
         by;

      g.setColor (Color.black);
      g.setXORMode (Color.white);

      if (eline >= sy + ny)
      {
         eline = sy + ny - 1;
         epix = 5000;
      }

      for (i = sline; i <= eline; i++)
      {
         by =  (i - sy) * fontHeight;
         bx = 0;
         ex = 5000;
         if (i == sline)
         {
            bx = spix - sx;
         }
         if (i == eline)
         {
            ex = epix - sx;
            if (sline == eline)
            {
               ex -= bx;
            }
         }
         g.fillRect (bx, by, ex, fontHeight);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void setup_h()
   {
      if ( (line < eline) ||  ( (line == eline) &&  (column <= ecolumn)))
      {
         hline = line;
         hcolumn = column;
         hpix = pix;
         heline = eline;
         hecolumn = ecolumn;
         hepix = epix;
      }
      else
      {
         hline = eline;
         hcolumn = ecolumn;
         hpix = epix;
         heline = line;
         hecolumn = column;
         hepix = pix;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void save_h()
   {
      opix = hpix;
      oepix = hepix;
      oline = hline;
      oeline = heline;
      hactive = eactive;
   }

   // it is still fooled by comments and literals because LineInfo
   // works with display columns and here we have logical columns
   // I can detab every line, but it is not worth it - I'll wait until
   // everything will move to virtual spaces
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param direction  No description provided
    */
   public void findMatchingBrace (int direction)
   {
      String tmplineStr = docMan.getLine (line);
      int tmpcolumn = column;
      int tmpline = line;
      boolean bracesearch = true;

      if (column >= tmplineStr.length())
      {
         return;
      }

      char atchar = tmplineStr.charAt (column);
      char nestchar = atchar;
      char opposite;
      switch (atchar)
      {
         case '[':
            opposite = ']';
            break;
         case ']':
            opposite = '[';
            break;
         case '(':
            opposite = ')';
            break;
         case ')':
            opposite = '(';
            break;
         case '{':
            opposite = '}';
            break;
         case '}':
            opposite = '{';
            break;
         default:
            opposite = atchar;
            nestchar = 0;
            bracesearch = false;
            break;
      }

      // seek braces in correct direction. not in provided one
      if (bracesearch)
      {
         if ( (nestchar == '(') ||  (nestchar == '{') ||  (nestchar == '['))
         {
            direction = 1;
         }
         else
         {
            direction = -1;
         }
      }

      int nest = 1;

      if (direction < 0)
      { // left

         while (true)
         {
            tmpcolumn--;
            if (tmpcolumn < 0)
            {
               tmpline--;
               if (tmpline < 0)
               {
                  return;
               }

               tmplineStr = docMan.getLine (tmpline);
               tmpcolumn = tmplineStr.length() - 1;
               if (tmpcolumn < 0)
               {
                  continue;
               }
            }

            atchar = tmplineStr.charAt (tmpcolumn);
            if (atchar == opposite &&
                (!bracesearch || docMan.getLineInfo (tmpline).isFreeStanding (tmpcolumn)))
            {
               nest--;
               if (nest == 0)
               {
                  line = tmpline;
                  column = tmpcolumn;
                  cursorAdjust();
                  return;
               }
            }
            else if (atchar == nestchar &&
                (docMan.getLineInfo (tmpline).isFreeStanding (tmpcolumn)))
            {
               nest++;
            }
         }
      }
      else
      { // right

         while (true)
         {
            tmpcolumn++;
            if (tmpcolumn >= tmplineStr.length())
            {
               tmpline++;
               if (tmpline >= docMan.getLineCount())
               {
                  return;
               }

               tmplineStr = docMan.getLine (tmpline);
               if (tmplineStr.length() == 0)
               {
                  continue;
               }
               tmpcolumn = 0;
            }

            atchar = tmplineStr.charAt (tmpcolumn);
            if (atchar == opposite &&
                (!bracesearch || docMan.getLineInfo (tmpline).isFreeStanding (tmpcolumn)))
            {
               nest--;
               if (nest == 0)
               {
                  line = tmpline;
                  column = tmpcolumn;
                  cursorAdjust();
                  return;
               }
            }
            else if (atchar == nestchar &&
                (docMan.getLineInfo (tmpline).isFreeStanding (tmpcolumn)))
            {
               nest++;
            }
         }
      }

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public synchronized void mouseReleased (MouseEvent e)
   {
      if (!mouseDown)
      {
         return;
      }

      mouseDown = false;

      if (textScroller != null)
      {
         textScroller.setStopThread (true);
         textScroller = null;
      }
      removeMouseMotionListener (this);
      eactive =  ( (eline != line) ||  (ecolumn != column));
      release_cursor();
      updateCopyActions (eactive);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void mouseEntered (MouseEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void mouseExited (MouseEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void mouseClicked (MouseEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void mouseMoved (MouseEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void keyTyped (KeyEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void keyReleased (KeyEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void keyPressed (KeyEvent e)
   {
      int keyModifiers = e.getModifiers();
      int keyCode = e.getKeyCode();
      char keyChar = e.getKeyChar();
      String s;

      if (!gotFocus)
      {
         return;
      }

      s = mpEdit.getKeyAction (keyModifiers, keyCode);

      if (s != null)
      {
         MpAction act = (MpAction) actionDictionary.get (s);

         if (act != null)
         {
            e.consume();
            if (!act.isEnabled())
            {
               return;
            }
            pause_cursor();

            if (s.indexOf ("dialog") >= 0)
            { // use a thread for dialogs

               ButtonPusher bp = new ButtonPusher (act);
               bp.start();
            }
            else
            {
               act.actionPerformed (null);
            }
            release_cursor();
            return;
         }
      }

      if (e.isMetaDown() || e.isControlDown() || e.isAltDown())
      {
         return;
      }

      if (docMan.isReadOnly())
      {
         return;
      }

      pause_cursor();

      // default character insert
      if (keyChar != KeyEvent.CHAR_UNDEFINED)
      {
         if (keyCode != KeyEvent.VK_ESCAPE &&
            keyChar != '\n' &&
            keyChar != '\b')
         {
            e.consume();
            if (hactive)
            {
               copy (true, false);
            }
            column++;
            docMan.insert_char (line, column - 1, keyChar);
            // JJ 1998.01.03  - "pix" must be updated with "column"
            pix = ruler.length (line, column) + EDGE;
            shiftHoriz (pix);
            return;
         }
      }
      release_cursor();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param s  No description provided
    */
   public void paste (String s)
   {
      int oldline;
      TextPosition tp;
      boolean paint;

      pause_cursor();

      if (hactive)
      {
         copy (true, false);
      }

      oldline = line;

      tp = docMan.insert_section (line, column, s, false);

      line = tp.line;
      column = tp.column;
      ruler.invalidate (oldline, line);
      pix = ruler.length (line, column) + EDGE;

      docMan.updateFrames (oldline, line);

      paint = oldline != line;

      paint = false;

      if (shiftVert (line))
      {
         paint = true;
      }

      if (shiftHoriz (pix))
      {
         paint = true;
      }

      if (paint)
      {
         repaint();
      }

   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param cut      No description provided
    * @param visible  No description provided
    * @return         No description provided
    */
   public String copy (boolean cut, boolean visible)
   {
      String s;
      boolean paint = false;

      if (!hactive)
      {
         return null;
      }

      if (hline != heline)
      {
         paint = true;
      }

      s = docMan.delete_section (hline, hcolumn, heline, hecolumn, cut);

      if (visible)
      {
         docMan.updateFrames (hline, heline);
      }

      if (cut)
      {
         line = hline;
         column = hcolumn;
         hactive = eactive = false;

         if (shiftVert (line))
         {
            paint = true;
         }

         pix = ruler.length (line, column) + EDGE;
         if (shiftHoriz (pix))
         {
            paint = true;
         }

         if (paint)
         {
            repaint();
         }
         else
         {
            if (visible)
            {
               linesChanged (line, line);
            }
         }
      }

      return s;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void cursorAdjust()
   {
      cursorAdjust (false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param force_paint  No description provided
    */
   protected void cursorAdjust (boolean force_paint)
   {
      int max;
      boolean paint = force_paint;

      if (eactive)
      {
         paint = true;
         hactive = eactive = false;
         updateCopyActions (eactive);
      }

      // straighten up lines first

      if (line < 0)
      {
         line = 0;
      }
      else
      {
         max = docMan.getLineCount();

         if (line >= max)
         {
            line = max - 1;
         }
      }

      // straighten up columns

      if (column < 0)
      {
         column = 0;
      }
      else
      {
         max =  (docMan.getLine (line)).length();
         if (column > max)
         {
            column = max;
         }
      }

      // off page?

      if (shiftVert (line))
      {
         paint = true;
      }

      // JJ 1998.01.03  - "pix" must be updated with "column"

      pix = ruler.length (line, column) + EDGE;
      if (shiftHoriz (pix))
      {
         paint = true;
      }

      if (paint)
      {
         repaint();
      }
      else
      {
         release_cursor();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param line  No description provided
    * @return      No description provided
    */
   protected boolean shiftVert (int line)
   {
      if (line < sy)
      {
         sy = line;
         if (sy < 0)
         {
            sy = 0;
         }
         vert.setValue (sy);
         return true;
      }

      if (line >= sy + ny)
      {
         sy = line - ny + 1;
         vert.setValue (sy);
         docMan.extendHilite (sy + ny - 1);
         return true;
      }

      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param x  No description provided
    * @return   No description provided
    */
   protected boolean shiftHoriz (int x)
   {
      if (x <= sx)
      {
         sx = x -  (dimension.width / 5);

         if (sx < 0)
         {
            sx = 0;
         }

         horiz.setValue (sx);
         return true;
      }

      if (x >= sx + dimension.width)
      {
         sx = x - dimension.width +  (dimension.width / 5);
         horiz.setValue (sx);
         return true;
      }

      return false;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void componentHidden (ComponentEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void componentShown (ComponentEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void componentMoved (ComponentEvent e)
   {
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void componentResized (ComponentEvent e)
   {
      redoControls (horiz.getValue(), vert.getValue(), true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void redoCanvas()
   {
      redoControls (horiz.getValue(), vert.getValue(), false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param h      No description provided
    * @param v      No description provided
    * @param clear  No description provided
    */
   protected void redoControls (int h, int v, boolean clear)
   {
      Graphics g = getGraphics();

      if (g != null)
      {
         updateFonts (g);
         dimension = getSize();

         if (clear)
         {
            clear_area (g);
         }

         int maxpix = 6000;
         int bubble = dimension.width;

         if (bubble > maxpix)
         {
            maxpix = bubble;
         }

         horiz.setValues (h, bubble, 0, maxpix);

         int maxline = docMan.getLineCount();

         if (maxline <= 0)
         {
            maxline = 10;
         }

         bubble = ny = dimension.height / fontHeight;

         docMan.extendHilite (sy + ny - 1);

         if (bubble > maxline)
         {
            bubble = maxline;
         }

         if (v > maxline)
         {
            v = maxline;
         }

         vert.setValues (v, bubble, 0, maxline);
         repaint();

         g.dispose();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void resizeLines()
   {
      if ( (hline >= 0) &&  (hline < docMan.getLineCount()))
      {
         opix = hpix = ruler.length (hline, hcolumn) + EDGE;
      }

      if ( (heline >= 0) &&  (heline < docMan.getLineCount()))
      {
         oepix = hepix = ruler.length (heline, hecolumn) + EDGE;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param s  No description provided
    * @return   No description provided
    */
   protected String detabbed (String s)
   {
      if (s.indexOf ('\t') < 0)
      {
         return s;
      }

      char c;
      String t = "";
      int j = 0;
      int tabs;
      int max = s.length();

      for (int i = 0; i < max; i++)
      {
         c = s.charAt (i);
         if (c == '\t')
         {
            tabs = tabSize -  (j % tabSize);
            j += tabs;
            while (tabs-- > 0)
            {
               t = t + ' ';
            }
         }
         else
         {
            t = t + c;
            j++;
         }
      }

      return t;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   char buffer[];


   /**
    * Fills char[] buffer with line li and replaces all tab characters '\t' with blanks Same
    * method as used in Hilite.
    *
    * @param li  No description provided
    * @return    length of buffer
    */
   protected int fillBuffer (LineInfo li)
   {
      char c;
      int i;
      int
         j;
      int
         tabs;
      int
         max;
      char before[];

      before = li.data.toCharArray();

      max = before.length;
      for (j = i = 0; i < max; i++)
      {
         if (before[i] == '\t')
         {
            j++;
         }
      }

      if (j == 0)
      {
         buffer = before;
         j = max;
      }
      else
      {
         buffer = new char[max +  (j *  (tabSize - 1))];

         for (j = i = 0; i < max; i++)
         {
            c = before[i];
            if (c == '\t')
            {
               tabs = tabSize -  (j % tabSize);
               while (tabs-- > 0)
               {
                  buffer[j++] = ' ';
               }
            }
            else
            {
               buffer[j++] = c;
            }
         }
      }

      return j;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param start   No description provided
    * @param length  No description provided
    * @return        No description provided
    */
   protected int charsLength (int start, int length)
   {
      if (fontMetrics == null)
      {
         return -1;
      }
      return fontMetrics.charsWidth (buffer, start, length);
   }


   /**
    * Get the cursorPos attribute of the TextCanvas object
    *
    * @param g  No description provided
    * @param r  No description provided
    */
   public void getCursorPos (Graphics g, Rectangle r)
   {
      updateFonts (g);
      r.width = 2;
      r.height = fontHeight;
      r.y =  (line - sy) * r.height;

      if (column > 0)
      {
         r.x = ruler.length (line, column) + EDGE - sx;
      }
      else
      {
         r.x = EDGE - sx;
      }
   }


   /**
    * Get the selectionOrWordUnderCursor attribute of the TextCanvas object
    *
    * @return   The selectionOrWordUnderCursor value
    */
   public String getSelectionOrWordUnderCursor()
   {
      String str = copy (false, false);
      if (str != null)
      {
         return str;
      }
      int leftBound = column;
      int rightBound = column;
      String currLine = docMan.getLine (line);
      int rightMax = currLine.length();

      leftB :
      while ( (--leftBound >= 0))
      {
         switch (currLine.charAt (leftBound))
         {
            case ' ':
            case '\t':
            case '.':
            case '(':
            case ')':
            case '[':
            case ']':
            case '{':
            case '}':
            case ';':
               leftBound++;
               break leftB;
         }
      }
      if (leftBound < 0)
      {
         leftBound = 0;
      }

      rightB :
      while (rightBound < rightMax)
      {
         switch (currLine.charAt (rightBound))
         {
            case ' ':
            case '\t':
            case '.':
            case '(':
            case ')':
            case '[':
            case ']':
            case '{':
            case '}':
            case ';':
               break rightB;
            default:
               rightBound++;

         }
      }

      if (rightBound == 0)
      {
         return "";
      }
      return currLine.substring (leftBound, rightBound);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void adjustmentValueChanged (AdjustmentEvent e)
   {
      int temp;
      boolean paint = false;

      pause_cursor();
      if (e.getSource() == horiz)
      {
         temp = e.getValue();
         if (temp != sx)
         {
            horiz.setValue (sx = temp);
            paint = true;
         }
      }
      else if (e.getSource() == vert)
      {
         temp = e.getValue();
         if (temp != sy)
         {
            vert.setValue (sy = temp);
            docMan.extendHilite (sy + ny - 1);
            paint = true;
         }
      }

      if (paint)
      {
         repaint();
      }
      else
      {
         release_cursor();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param first  No description provided
    * @param last   No description provided
    */
   public void linesChanged (int first, int last)
   {
      if ( (first > sy + ny) ||  (last < sy))
      {
         return;
      }

      if (first == last)
      {
         Graphics g = getGraphics();
         updateFonts (g);
         drawLine (g, first);
         g.dispose();
         release_cursor();
      }
      else
      {
         repaint();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param un  No description provided
    * @param re  No description provided
    */
   public void updateUndoActions (boolean un, boolean re)
   {
      undoAction.setEnabled (un);
      redoAction.setEnabled (re);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param active  No description provided
    */
   public void updateCopyActions (boolean active)
   {
      cutAction.setEnabled (active);
      copyAction.setEnabled (active);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param txt  No description provided
    */
   public void showStatus (String txt)
   {
      // in future this shoud display info in status line
      // for now just stdout
      if (log.isInfoEnabled())
      {
         log.info (txt);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   public void update (Graphics g)
   {
      paint (g);
   }


   /**
    * Get the colors attribute of the TextCanvas object
    */
   protected void getColors()
   {
      textColor = docMan.getTextColor();
      textXColor = docMan.getTextXColor();
      commentColor = docMan.getCommentColor();
      commentXColor = docMan.getCommentXColor();
      keywordColor = docMan.getKeywordColor();
      keywordXColor = docMan.getKeywordXColor();
      quoteColor = docMan.getQuoteColor();
      quoteXColor = docMan.getQuoteXColor();
      hideColor = docMan.getHideColor();
      hideXColor = docMan.getHideXColor();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    */
   public void paint (Graphics g)
   {
      if (!isShowing())
      { // why java tries to repaint window during file load ?

         return;
      }

      int i;

      int
         j;

      int
         t;
      Rectangle clip;

      pause_cursor();

      getColors();

      g.setPaintMode();
      updateFonts (g);

      i = docMan.getLineCount();
      if (oldlines != i)
      {
         oldlines = i;
         redoControls (horiz.getValue(), vert.getValue(), true);
      }

      if (dimension == null)
      {
         dimension = getSize();
      }

      clip = g.getClipBounds();

      Color oldColor = g.getColor();
      g.setColor (Color.white);
      g.fillRect (clip.x, clip.y, clip.width, clip.height);
      g.setColor (oldColor);

      i = clip.y / fontHeight;
      t = clip.y + clip.height;
      j = t / fontHeight;
      if ( (t % fontHeight) == 0)
      {
         j--;
      }
      if (j >= ny)
      {
         j = ny - 1;
      }

      linesEmpty = 0;
      lastVerticalSize = j;

      for (i = 0; i <= j; i++)
      {
         drawLine (g, sy + i);
      }

      release_cursor();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   static Color separatorInsideColor = new Color (160, 0, 0);

   // This shouldn't be such hardcoded, but without Swing I'm a bit
   // helpless - Artur
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    * @param y  No description provided
    */
   protected void drawLineSeparator (Graphics g, int y)
   {
      int center = y - fontHeight - 2;
      g.setColor (separatorInsideColor);
      g.drawLine (3, center, dimension.width - 12, center);
      g.setColor (Color.gray);
      g.drawLine (3, center, 6, center + 1);
      g.drawLine (3, center, 6, center - 1);
      g.drawLine (6, center + 1, dimension.width - 15, center + 1);
      g.drawLine (6, center - 1, dimension.width - 15, center - 1);
      g.drawLine (dimension.width - 15, center + 1, dimension.width - 12, center);
      g.drawLine (dimension.width - 15, center - 1, dimension.width - 12, center);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param g  No description provided
    * @param i  No description provided
    */
   protected synchronized void drawLine (Graphics g, int i)
   {
      int x;
      int y;
      int m;
      int bx;
      int ex;
      int a;
      int b;
      int c;
      int max;
      int key;
      LineInfo hi;
      boolean xor = false;

      //Calculate the upper left corner of the text area
      x = EDGE - sx;
      y =  (i - sy + 1) * fontHeight;
      m = fontHeight;

      g.setColor (Color.black);
      g.setPaintMode();

      if (i >= docMan.getLineCount())
      { //if this line not exists in the document, return
         return;
      }

      if (hactive &&  (i > oline) &&  (i < oeline))
      { //if this line is selected...

         g.fillRect (0, y - fontHeight, dimension.width, m); //...clear line and set xor flag
         xor = true;
      }
      else
      {
         //Clear line
         Color oldColor = g.getColor();
         g.setColor (Color.white);
         g.fillRect (0, y - fontHeight, dimension.width, m);
         g.setColor (oldColor);

         //g.clearRect (0, y - fontHeight, dimension.width, m);
      }

      hi = docMan.getLineInfo (i); //get LineInfo i from LineMan
      max = fillBuffer (hi); //and fill char[] max with it (tabs are replaced with blanks)
      a = 0;
      key = 0;

      if (hi.tagColor != null)
      { //the line has a special background color

         if (hactive &&  (i > oline) &&  (i < oeline))
         {
            g.setColor (new Color (hi.tagColor.getRGB() ^ 0xffffff));
         }
         else
         {
            g.setColor (hi.tagColor);
         }
         g.fillRect (0, y - fontHeight, dimension.width, m);
      }

      //draw all characters within this line
      while (a < max)
      {
         if (key < hi.keyCt)
         { //if the actual keyword is less than all existing keywords

            b = hi.keyStarts[key]; //set the keyword borders
            c = hi.keyEnds[key];
         }
         else
         { //no more keywords exist so draw the whole rest of the line

            b = max;
            c = max;
         }

         if (b > a)
         { //we haven't reached a keyword yet

            if (xor)
            { //selected?
               g.setColor (textXColor);
            }
            else
            {
               g.setColor (textColor);
            }

            //draw all characters up to the keyword
            g.drawChars (buffer, a, b - a, x, y - fontDescent);

            x += charsLength (a, b - a);
         }

         //check if whe have now a keyword
         if (c > b)
         {
            //Skip hide tags for LIFE

            if (hi.keyTypes[key] >= Hilite.HIDE)
            {
               if (hi.keyTypes[key] > Hilite.HIDE + docMan.getHideLevel())
               {
                  a = max;
               }
               else
               {
                  a = b;
               }
               key++;
               continue;
            }

            //if so set the right color
            switch (hi.keyTypes[key])
            {
               case Hilite.COMMENT:
                  if (xor)
                  {
                     g.setColor (commentXColor);
                  }
                  else
                  {
                     g.setColor (commentColor);
                  }
                  break;
               case Hilite.KEYWORD:
                  if (xor)
                  {
                     g.setColor (keywordXColor);
                  }
                  else
                  {
                     g.setColor (keywordColor);
                  }
                  break;
               case Hilite.QUOTE:
                  if (xor)
                  {
                     g.setColor (quoteXColor);
                  }
                  else
                  {
                     g.setColor (quoteColor);
                  }
                  break;
            }

            //and draw the keyword
            //for(int q = b; q < c; q++) System.out.print(buffer[q]);

            g.drawChars (buffer, b, c - b, x, y - fontDescent);
            x += charsLength (b, c - b);
         }
         a = c; //if we found a keyword, its end is the new starting point, else a=c=max and we can finish
         key++; //prepare for the next key
      }

      //if  doSeperator flag is true, and more than one empty line follows another, draw a seperator
      if (doSeparator)
      {
         if (max == 0)
         {
            linesEmpty++;
         }
         else
         {
            linesEmpty = 0;
         }

         if (linesEmpty == 2)
         {
            drawLineSeparator (g, y);
         }
      }

      if (hactive &&  ( (i == oline) ||  (i == oeline)))
      {
         bx = 0;
         ex = 5000;
         if (i == oline)
         {
            bx = opix - sx;
         }
         if (i == oeline)
         {
            ex = oepix - sx;
            if (oline == oeline)
            {
               ex -= bx;
            }
         }
         g.setColor (Color.black);
         g.setXORMode (Color.white);
         g.fillRect (bx, y - fontHeight, ex, m);
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param mpa  The object added.
    */
   protected void addToDict (MpAction mpa)
   {
      actionDictionary.put (mpa.getIdString(), mpa);
   }


   /**
    * Get the mpAction attribute of the TextCanvas object
    *
    * @param id  No description provided
    * @return    The mpAction value
    */
   public MpAction getMpAction (String id)
   {
      return (MpAction) actionDictionary.get (id);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void createActionDictionary()
   {
      actionDictionary = new Hashtable();

      addToDict (undoAction);
      addToDict (redoAction);
      addToDict (
         new
            AbstractMpAction ("line-swap")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               if (docMan.getLineCount() - 1 == line)
               {
                  return;
               }
               docMan.swap_lines (line, line + 1);
               cursorAdjust (true);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-up")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               line--;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-down")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               line++;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-forward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               column++;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-backward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               column--;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-word-forward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               _wordForward();
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-word-backward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               _wordBackward();
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-line-begin")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               column = 0;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-line-end")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               column = docMan.getLine (line).length();
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-page-begin")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               line = sy;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-page-end")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               line = Math.min (docMan.getLineCount() - 1, sy + ny - 1);
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-document-begin")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               line = 0;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("cursor-document-end")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               line = docMan.getLineCount() - 1;
               cursorAdjust();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("page-up")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               int offset = line - sy;
               line -= lastVerticalSize;
               shiftVert (Math.max (line - offset, 0));
               cursorAdjust (true);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("page-down")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               int offset = line - sy;
               line += lastVerticalSize;
               shiftVert (Math.min (line - offset + ny - 1, docMan.getLineCount() - 1));
               cursorAdjust (true);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("find-next-forward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               findAgain (1);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("find-next-backward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               findAgain (-1);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("brace-match-forward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               findMatchingBrace (1);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("brace-match-backward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               findMatchingBrace (-1);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("character-delete-forward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               _pressedDelete();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("character-delete-backward")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               _pressedBackspace();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("line-break")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               _insertNewline();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("line-clone")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               docMan.insert_line (line, docMan.getLine (line));
               line++;
               cursorAdjust (true);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("line-delete")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               _deleteLine();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("document-save")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               docMan.fileSave (textFrame);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("frame-close")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               docMan.closeFrame (textFrame);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("mode-autoindent-switch")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               autoIndent = !autoIndent;
               showStatus ("Autindentation turned " +  (autoIndent ? "ON" : "OFF"));
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("document-new")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               mpEdit.newDoc();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("document-open-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               mpEdit.openDocDialog (docMan, textFrame);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("document-save-as-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               docMan.fileSave (textFrame);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("document-print-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               docMan.filePrint (textFrame);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("find-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               String search = getSelectionOrWordUnderCursor();
               if (!search.equals (""))
               {
                  addSearchPattern (search);
               }
               FindDialog findDialog;
               findDialog = new FindDialog (textFrame,
                  textMenu,
                  textMenu.getStrings(),
                  textMenu.getStrings().getString ("DialogFind"));
               findDialog.show();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("goto-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               GotoDialog gotoDialog;
               gotoDialog = new GotoDialog (textFrame,
                  textMenu,
                  textMenu.getStrings(),
                  textMenu.getStrings().getString ("DialogGoto"));
               gotoDialog.show();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("replace-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               ReplaceDialog replaceDialog;
               replaceDialog = new ReplaceDialog (textFrame,
                  textMenu,
                  textMenu.getStrings(),
                  textMenu.getStrings().getString ("DialogReplace"));
               replaceDialog.show();
            }
         }
         );

      addToDict (copyAction =
         new
            AbstractMpAction ("selection-copy")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               String srcData = copy (false, false);
               if (srcData != null)
               {
                  StringSelection contents = new StringSelection (srcData);
                  textMenu.getClipboard().setContents (contents, textMenu);
               }
            }
         }
         );

      addToDict (cutAction =
         new
            AbstractMpAction ("selection-cut")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               String srcData = copy (true, true);
               if (srcData != null)
               {
                  StringSelection contents = new StringSelection (srcData);
                  textMenu.getClipboard().setContents (contents, textMenu);
               }
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("buffer-paste")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               String dstData;
               Transferable content = textMenu.getClipboard().getContents (textMenu);
               if (content != null)
               {
                  try
                  {
                     dstData = (String) content.getTransferData (DataFlavor.stringFlavor);
                  }
                  catch (Exception exc)
                  {
                     if (log.isInfoEnabled())
                     {
                        log.info ("Could not read clipboard");
                     }
                     return;
                  }
                  paste (dstData);
               }
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("properties-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               PropDialog pd = new PropDialog (textFrame, textMenu.getStrings(), docMan, textMenu.getStrings().getString ("DialogProps"));
               pd.show();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("frame-clone")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               Dimension size = mpEdit.getWindowSize();
               Point place = mpEdit.getPlace (size);
               TextFrame textFrame = docMan.newFrame (place, size);
               docMan.setTitles();
               textFrame.setVisible (true);
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("help-about-dialog")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               AboutDialog ab = new AboutDialog (textFrame, textMenu.getStrings(), textMenu.getStrings().getString ("DialogAbout"));
               ab.show();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("keytable-save")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               mpEdit.saveKeytable();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("keytable-load")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               mpEdit.loadKeytable();
            }
         }
         );

      addToDict (
         new
            AbstractMpAction ("mode-readonly-switch")
         {
            {
            }

            public void actionPerformed (ActionEvent e)
            {
               docMan.setReadOnly (!docMan.isReadOnly());
            }
         }
         );

   }


   /*
    *  Action template
    *  addToDict( new
    *  AbstractMpAction("") {
    *  {
    *  }
    *  public void actionPerformed( ActionEvent e ) {
    *  }
    *  }
    *  );
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   MpAction undoAction =
      new
         AbstractMpAction ("undo")
      {
         {
         }

         public void actionPerformed (ActionEvent e)
         {
            docMan.undo (textFrame);
         }
      };

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   MpAction redoAction =
      new
         AbstractMpAction ("redo")
      {
         {
         }

         public void actionPerformed (ActionEvent e)
         {
            docMan.redo (textFrame);
         }
      };

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   MpAction copyAction;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   MpAction cutAction;


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void _deleteLine()
   {
      if (docMan.getLineCount() == 1)
      {
         docMan.clear_line (0);
         column = 0;
         pix = EDGE;
      }
      else
         if (docMan.getLineCount() - 1 == line)
      {
         docMan.clear_line (line);
         line--;
      }
      else
      {
         docMan.delete_line (line);
      }
      cursorAdjust (true);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void _wordForward()
   {
      boolean skippedWord = false;
      char[] buf = docMan.getLine (line).toCharArray();

      if (buf.length == 0)
      { // failed on empty lines
         return;
      }

      while (column < buf.length)
      {
         if ( (buf[column] != ' ') &&  (buf[column] != '\t'))
         {
            if (skippedWord)
            {
               break;
            }
         }
         else
         {
            skippedWord = true;
         }
         column++;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void _wordBackward()
   {
      boolean skippedSpace = false;
      char[] buf = docMan.getLine (line).toCharArray();

      if (buf.length == 0)
      { // failed on empty lines
         return;
      }

      column = Math.min (column, buf.length - 1);
      if ( (buf[column] != ' ') &&  (buf[column] != '\t'))
      {
         column--;
      }
      while (column >= 0)
      {
         if ( (buf[column] == ' ') ||  (buf[column] == '\t'))
         {
            if (skippedSpace)
            {
               column++;
               break;
            }
         }
         else
         {
            skippedSpace = true;
         }
         column--;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void _pressedDelete()
   {
      String s;
      int max;

      if (hactive)
      {
         copy (true, true);
         return;
      }
      s = docMan.getLine (line);
      max = s.length();
      if (column < max)
      {
         docMan.delete_char (line, column);
         return;
      }
      else
         if (line + 1 < docMan.getLineCount())
      {
         docMan.join_line (line, column);
         repaint();
         return;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void _pressedBackspace()
   {
      String s;

      if (hactive)
      {
         copy (true, true);
         return;
      }
      if (column > 0)
      {
         column--;
         docMan.delete_char (line, column);
         // JJ 1998.01.03  - "pix" must be updated with "column"
         pix = ruler.length (line, column) + EDGE;
         shiftHoriz (pix);
         return;
      }
      else
         if (line > 0)
      {
         line--;
         s = docMan.getLine (line);
         column = s.length();
         shiftHoriz (ruler.length (line, 10000000) + EDGE);
         docMan.join_line (line, column);
         shiftVert (line);
         repaint();
         return;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   void _insertNewline()
   {
      if (hactive)
      {
         copy (true, false);
      }
      docMan.split_line (line, column);
      shiftVert (++line);
      column = 0;
      if (autoIndent)
      {
         int i = 0;
         String prevline = docMan.getLine (line - 1);
         char ch;
         while ( (i < prevline.length()) &&
             ( ( (ch = prevline.charAt (i)) == ' ') ||
             (ch == '\t')))
         {
            docMan.insert_char (line, column, ch);
            column++;
            i++;
         }
         // JJ 1998.01.03  - "pix" must be updated with "column"
         pix = ruler.length (line, column) + EDGE;
         shiftHoriz (pix);
      }
      else
      {
         pix = EDGE;
         shiftHoriz (0);
      }
      repaint();
      return;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String[] readonlyActions =
      {
      "line-swap",
      "line-delete",
      "line-clone",
      "character-delete-forward",
      "character-delete-backward",
      "replace-dialog",
      "selection-cut",
      "buffer-paste",
      "line-break",
      "document-save",
      "cursor-forward",  //Entry for Hilite Life, cosmetics only
   "cursor-backward" //Entry for Hilite Life, cosmetics only
   };


   /**
    * Disables/enables all actions which modify text This method does not check if readOnly
    * status have changed - you should check this before and do not call this if not needed
    *
    * @param readOnly  The new readOnly value
    */
   public void setReadOnly (boolean readOnly)
   {
      int max = readonlyActions.length;
      int i;
      for (i = 0; i < max; i++)
      {
         getMpAction (readonlyActions[i]).setEnabled (!readOnly);
      }
   }

}

/*
 * $Log: TextCanvas.java,v $
 * Revision 1.22.2.3  2005/10/09 19:01:31  lowende
 * Removed some compile warnings.
 *
 */
