/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.layout.classdiag;

import java.awt.Dimension;
import java.awt.Point;
import java.util.Vector;

import de.uni_paderborn.fujaba.fsa.FSAPanel;
import de.uni_paderborn.fujaba.layout.classdiag.internalmodel.LayoutedNode;


/**
 * This class represents a node in the classdiagram (a class, interface or
 * package). The class has been taken from Argo UML source code.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.1.2.1 $ $Date: 2005/08/23 08:24:46 $
 */
class ClassdiagramNode implements LayoutedNode
{

   /**
    * Operation ClassdiagramNode creates a new ClassdiagramNode.
    *
    * @param f  represents the figure in the diagram, that peers this layout
    *           node.
    */
   public ClassdiagramNode (FSAPanel f)
   {
      setFigure (f);
   }


   /**
    * Operation getSize returns the size of the figure associated with this
    * layout node.
    *
    * @return   The size of the associated figure.
    */
   public Dimension getSize()
   {
      return getFigure().getSize();
   }


   /**
    * Operation getLocation returns the location of the associated figure in the
    * diagram.
    *
    * @return   The location of the associated figure.
    */
   public Point getLocation()
   {
      return getFigure().getLocation();
   }


   /**
    * Operation setLocation set the new location of the associated figure in the
    * diagram.
    *
    * @param newLocation  represents the new location for this figure.
    */
   public void setLocation (Point newLocation)
   {
      getFigure().setLocation (newLocation);
   }


   /**
    * Check if this node is associated with a package.
    *
    * @return   true, if this node is associated with a package, false otherwise.
    */
   boolean isPackage()
   {
      return false; //(getFigure() instanceof FigPackage);
   }


   /**
    * Get the realRank attribute of the ClassdiagramNode object
    *
    * @return   The realRank value
    */
   public int getRealRank()
   {
      return rank;
   }


   /**
    * Compute or just return the rank of this node.
    *
    * @return   The rank for this node.
    */
   public int getRank()
   {

      if (rank == NORANK)
      {
         // If the rank was not computed yet, compute it now.
         if (getUplinks().size() == 0)
         { // If there are no uplinks,
            rank = 0; // place the node in the 1st row.
         }
         else
         { // Otherwise compute the max rank of the uplinks + 1
            for (int i = 0; i < getUplinks().size(); i++)
            {

               // avoid recursing, causing stack overflow error, if the
               // uplinks are cyclic.
               if (getUplink (i) == this)
               {
                  return rank;
               }

               if (getUplink (i).getRank() + 1 > rank)
               {

                  rank = getUplink (i).getRank() + 1;
               }
            }
         }
      }
      return rank;
   }


   /**
    * Operation setRank changes the value of the attribute _rank.
    *
    * @param newRank  represents the new value of _rank.
    */
   public void setRank (int newRank)
   {
      rank = newRank;
   }


   /**
    * Add a constant to the rank of this node.
    *
    * @param n  The value to add.
    */
   public void addRank (int n)
   {
      setRank (n + getRank());
   }


   /**
    * Operation getColumn returns the value of the attribute _column.
    *
    * @return   The value of the attribute _column.
    */
   public int getColumn()
   {
      return column;
   }


   /**
    * Operation setColumn changes the value of the attribute _column.
    *
    * @param newColumn  represents the new value of _column.
    */
   public void setColumn (int newColumn)
   {
      column = newColumn;
   }


   /**
    * Operation getUplinks returns the value of the attribute _uplinks.
    *
    * @return   The value of the attribute _uplinks.
    */
   public Vector getUplinks()
   {
      return uplinks;
   }


   /**
    * Get the uplink for a given index.
    *
    * @param i  The index of this uplink.
    * @return   The ClassdiagramNode for this uplink.
    */
   public ClassdiagramNode getUplink (int i)
   {
      return (ClassdiagramNode)  (uplinks.elementAt (i));
   }


   /**
    * Add an uplink to this node.
    *
    * @param newUplink  represents the new uplinks.
    */
   public void addUplink (ClassdiagramNode newUplink)
   {
      uplinks.add (newUplink);
   }


   /**
    * Operation getDownlinks returns the value of the attribute _downlinks.
    *
    * @return   The value of the attribute _downlinks.
    */
   public Vector getDownlinks()
   {
      return downlinks;
   }


   /**
    * Get the downlink for a given index.
    *
    * @param i  The index of this downlink.
    * @return   The ClassdiagramNode of this downlink.
    */
   public ClassdiagramNode getDownlink (int i)
   {
      return (ClassdiagramNode)  (downlinks.elementAt (i));
   }


   /**
    * Operation setDownlinks changes the value of the attribute _downlinks.
    *
    * @param newDownlink  Represents the new value of _downlinks.
    */
   public void addDownlink (ClassdiagramNode newDownlink)
   {
      downlinks.add (newDownlink);
   }


   /**
    * Operation getFigure returns the value of the attribute _figure.
    *
    * @return   The value of the attribute _figure.
    */
   public FSAPanel getFigure()
   {
      return figure;
   }


   /**
    * Operation setFigure changes the value of the attribute _figure.
    *
    * @param newFigure  represents the new value of _figure.
    */
   public void setFigure (FSAPanel newFigure)
   {
      figure = newFigure;
   }


   /**
    * Return the weight of this node.
    *
    * @return   The weight of this node.
    */
   public float getWeight()
   {
      return weight;
   }


   /**
    * Set a new weight for this node.
    *
    * @param w  The new weight of this node.
    */
   public void setWeight (float w)
   {
      weight = w;
   }


   /**
    * a node is movable when it hasn't got any up- or downlinks.
    *
    * @return   whether the node is movable
    */
   public boolean isMovable()
   {
      return  (downlinks.size() == 0 && uplinks.size() == 0);
   }


   /**
    * A placementhint gives an indication where it might be feasible to place
    * this node. It is used by the layouter, and there is no guarantee that it
    * will be used.
    *
    * @param hint  x coordinate of the desired placement
    */
   public void setPlacementHint (int hint)
   {
      placementHint = hint;
   }


   /**
    * get the current placementhint.
    *
    * @return   The placementHint value
    */
   public int getPlacementHint()
   {
      return placementHint;
   }

   // Attributes

   /**
    * Constant to be used as an initializer when this node has no rank assigned
    * yet.
    */
   public final static int NORANK = -1;

   /**
    * Attribute _rank represents the current rank of this node.
    */
   private int rank = NORANK;

   /**
    * Constant to be used as an initializer when this node is not placed at an
    * column.
    */
   public final static int NOCOLUMN = -1;

   /**
    * Attribute _column represents the current column of this node.
    */
   private int column = NOCOLUMN;

   /**
    * Attribute _uplinks represents the links I consider as an 'uplink'. An
    * uplink is a link going to a superclass or a interface that this class
    * implements. Figures that are usually placed above this figure.
    */
   private Vector uplinks = new Vector();

   /**
    * Attribute _downlinks represents the links I consider as an 'downlink'. The
    * opposite of an uplink. See explanation above.
    */
   private Vector downlinks = new Vector();

   /**
    * Attribute _figure represents the figure, that this ClassdiagramNode
    * represents during the layout process.
    */
   private FSAPanel figure = null;

   /**
    * This attributes stores the 'weight' of this node. This is a computed
    * attribute that is used during the horizontal placement process. It's based
    * on the position of the 'uplinked' objects. The actual purpose is to
    * minimize the number of link crossings in the diagram. Since we don't
    * compute the actual number of link crossings, we look where our uplinked
    * objects are, and then try to place our object in a way, that we can expect
    * to have a minimal number of crossings.
    */
   private float weight = 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int placementHint = -1;

}

/*
 * $Log: ClassdiagramNode.java,v $
 * Revision 1.1.2.1  2005/08/23 08:24:46  lowende
 * Removed compile warnings.
 *
 */
