/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.layout;

import java.awt.*;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.*;

import de.uni_paderborn.fujaba.app.ProgressBarWrapper;
import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.fsa.swing.*;
import de.uni_paderborn.fujaba.fsa.unparse.LogicUnparseInterface;
import de.uni_paderborn.fujaba.layout.options.LayoutPreferences;
import de.uni_paderborn.fujaba.uml.UMLStartActivity;
import de.uni_paderborn.fujaba.uml.UMLStopActivity;


/**
 * Class 'TreeLayout' specified in class diagram 'TreeLayout.java diagram'.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.22 $
 */
public class TreeLayout extends AbstractLayouter
{
   /**
    * The children of the abstract Layouter are singletons, so define singleton instance.
    */
   private static TreeLayout myLayouter = null;


   /**
    * Constructor of class TreeLayout
    */
   private TreeLayout()
   {
      super();
   }


   /**
    * Use this method to get a reference to singleton layouter classes The class var is defined
    * in AbstractLayouter
    *
    * @return   reference to singleton instance of Layout-Classes
    */
   public static TreeLayout get()
   {
      if (myLayouter == null)
      {
         myLayouter = new TreeLayout();
      }
      // simply return reference to myLayouter
      return myLayouter;
   } // get


   /**
    * This function is called recursively to reLayout the displayed classes, the initial values
    * of depth and left have to be depth = 0, left = 0 and pos = 0
    *
    * @param currentClass           No description provided
    * @param depth                  No description provided
    * @param left                   No description provided
    * @param pos                    No description provided
    * @param rowDepths              No description provided
    * @throws InterruptedException  Exception description not provided
    */
   private void treeLayout (FSAObject currentClass, int depth, int left,
                            int pos, Vector rowDepths)
       throws InterruptedException
   {
      // variables needed locally in this function
      Dimension currentSize;
      Iterator enumGrabs;
      FSAGrab itemGrab;
      //FSAObject frameGrabbed;

      // First check, if the class is not empty, if so, break
      if ( (currentClass == null) ||  (!currentClass.isVisible()))
      {
         return;
      }
      // First Layout the inner classes
      if (currentClass instanceof FSAContainer)
      {
         //innerLayout ((FSAContainer) currentClass);
      }

      // Set this class as visited
      currentClass.setGenerated (!this.isLastCondition());

      // Depending on the (depth, left) - Tupel set up the coordinates
      left = getLeft (depth, rowDepths);
      currentClass.setLocation (this.getHorizDist() + left,
         this.getHorizDist() + pos);
      currentSize = currentClass.getPreferredSize();
      left +=  (currentSize.width + 10);
      setLeft (left, depth, rowDepths);

      // Now move to all sons by iteration
      // But first set LEFT to count all sons displayed in this row
      left = getLeft (depth + 1, rowDepths);

      LogicUnparseInterface entry = currentClass.getLogic();
      JComponent comp = currentClass.getJComponent();
      if (entry instanceof UMLStartActivity)
      {
         comp = (JCircle)  ((JPanel) comp.getComponent (1)).getComponent (0);
      }
      else if (entry instanceof UMLStopActivity)
      {
         comp = (JCircle)  ((JPanel) comp.getComponent (0)).getComponent (0);
      }

      // Get a List of Grabs leaving this Class
      GrabManager manager = (GrabManager) comp.getClientProperty (GrabManager.TARGET_PROPERTY);

      if (manager != null)
      {
         enumGrabs = manager.iteratorOfGrabs();

         FSAObject childFrame = null;
         while (enumGrabs.hasNext())
         {
            // Get next Element in Enumeration-List and move one element ahead
            itemGrab = (FSAGrab) FSAObject.getFSAObjectFromJComponent ((JGrab) enumGrabs.next());

            // Move to sons
            childFrame = getChild (itemGrab);
            if ( (childFrame != null) &&  (!childFrame.equals (currentClass))
               &&  (! (childFrame.isGenerated() == !this.isLastCondition())))
            {
               treeLayout (childFrame, depth + 1, left,
                  this.getVertDist() + pos + currentSize.height,
                  rowDepths);
               // next son, next recursion
            }
            // The if prevents the scope and the recursion from running
            // endlessly!
         }
      }
   } // treeLayout


   /**
    * UMLMethod: '+ reLayout (currentCanvas : DisResizeable) : Void'
    *
    * @param currentCanvas          No description provided
    * @throws InterruptedException  Exception description not provided
    */
   public void reLayout (FSAContainer currentCanvas)
       throws InterruptedException
   {
      FSAObject currentClass = null;

      ProgressBarWrapper myProgress = new ProgressBarWrapper (getFrame (currentCanvas), "Tree layout", 2);

      // Set Options before start
      LayoutPreferences options = LayoutPreferences.get();

      setHorizDist (options.getHorizDist());
      setVertDist (options.getVertDist());

      // Next search for starting class
      // take the first selected Class
      FSAObject[] selected = SelectionManager.get().getLastSelectionsOfType (FSAPanel.class, 1);
      if (selected.length > 0)
      {
         currentClass = selected[0];
      }

      // toggle Condition for next reLayout
      FSAObject searchResult = searchStart (currentCanvas, false);
      if (searchResult != null)
      {
         setLastCondition (searchResult.isGenerated());
      } //if

      else
      {
         setLastCondition (false);
      } // else

      // now check, if a class is selected, or if we have to search for one
      // lastCondition needs to be resetted at this time
      if ( (currentClass == null) || !currentClass.isVisible())
      {
         currentClass = searchStart (currentCanvas, true);
      }

      // now start the (recursive) Layout-Algorithm
      // First reset the values
      Vector rowDepths = new Vector (10);

      myProgress.openWindow();
      try
      {
         myProgress.increment ("Layouter working...");
         while (currentClass != null)
         {
            // then start the treeLayout
            treeLayout (currentClass, 0, 0, 0, rowDepths);

            // now look if there are not layouted trees, if yes start again
            currentClass = searchStart (currentCanvas, true);
         }
         myProgress.increment ("Layouter working...");
      }
      finally
      {
         myProgress.closeWindow();
      }

   } // reLayout

}

/*
 * $Log: TreeLayout.java,v $
 * Revision 1.22  2004/10/20 17:49:59  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
