/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.layout;

import java.awt.*;
import java.util.Iterator;

import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.layout.options.LayoutPreferences;
import de.uni_paderborn.fujaba.metamodel.FElement;
import de.uni_paderborn.fujaba.sequencer.*;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FLinkedList;


/**
 * Class 'ControlFlowLayout' specified in class diagram 'ControlFlowLayout.java diagram'.
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.26 $
 */
public class ControlFlowLayout extends AbstractLayouter
{

   /**
    * The children of the abstract Layouter are singletons, so define singleton instance
    */
   private static ControlFlowLayout myLayouter = null;


   /**
    * Constructor of class ControlFlowLayout
    */
   private ControlFlowLayout()
   {
      super();
   }


   /**
    * This method visits recursively the seq and sets the coordinates
    *
    * @param flow  A subclass of Flow e.g. Seq, Sel, Rep, FlowActivity in the CFG
    * @param nw    NorthWest Point
    * @return      se SouthEast Point
    */
   private Point layoutCFG (Flow flow, Point nw)
   {
      Point se = new Point (nw);

      if (flow instanceof Seq)
      {
         FLinkedList items =  ((Seq) flow).getItems();
         Point xy = new Point();
         Point newNW = new Point (nw);

         Flow currentFlow;
         Iterator iter = items.iterator();
         while (iter.hasNext())
         {
            currentFlow = (Flow) iter.next();
            xy = layoutCFG (currentFlow, newNW);

            // next flow under it
            newNW.y = xy.y;

            // adjust SouthEast Point (max x)
            se.y = xy.y;
            se.x =  (xy.x > se.x) ? xy.x : se.x;
         }
      }
      else if (flow instanceof Sel)
      {
         se = layoutCFG ( ((Sel) flow).getFlowActivity(), nw);

         Point newNW = new Point (nw.x, se.y);
         // log.info("1st point: " + newNW);
         se = layoutCFG ( ((Sel) flow).getThenSeq(), newNW);
         Point xy = new Point (se.x, newNW.y);
         // log.info("2nd point: " + xy);
         xy = layoutCFG ( ((Sel) flow).getElseSeq(), xy);

         // adjust SouthEast Point (max y)
         se.x = xy.x;
         se.y =  (se.y > xy.y) ? se.y : xy.y;
      }
      else if (flow instanceof Rep)
      {
         se = layoutCFG ( ((Rep) flow).getBodySeq(), nw);
      }
      else if (flow instanceof FlowActivity)
      {
         FSAObject curDisFrame = getConcerningDisFrame ((FlowActivity) flow);
         curDisFrame.setLocation (nw);
         Dimension curDimension = curDisFrame.getPreferredSize();
         se.x = nw.x + curDimension.width + getHorizDist();
         se.y = nw.y + curDimension.height + getVertDist();

         // in case here starts a for each flow
         if ( ((FlowActivity) flow).isForEach())
         {

            // start left from the activity
            Point xy = new Point (se.x, nw.y);
            xy = layoutCFG ( ((FlowActivity) flow).getForEachSeq(), xy);

            // adjust SouthEast Point (max y)
            se.x = xy.x;
            se.y =  (se.y > xy.y) ? se.y : xy.y;
         }
      }

      return se;
   }


   /**
    * Searches in the IncrementList of the Activity
    *
    * @param curFlowActivity  The FlowActivity for the activity in the diagram
    * @return                 The DisFrame in this diagram concerning the activity
    */
   private FSAObject getConcerningDisFrame (FlowActivity curFlowActivity)
   {
      FSAObject curDisObject;
      curDisObject = curFlowActivity.getUMLActivity().getFSAInterface().getFirstFromFsaObjects();
      return curDisObject;
   }


   /**
    * Use this method to get a reference to singleton layouter classes The class var is defined
    * in AbstractLayouter
    *
    * @return   reference to singleton instance of Layout-Classes
    */
   public static ControlFlowLayout get()
   {
      if (myLayouter == null)
      {
         myLayouter = new ControlFlowLayout();
      }
      // simply return reference to myLayouter
      return myLayouter;
   } // get


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param currentCanvas  No description provided
    */
   public void reLayoutNoThrow (FSAContainer currentCanvas)
   {
      try
      {
         reLayout (currentCanvas);
      }
      catch (Exception e)
      {

      }
   }

   //private FSAContainer currentCanvas = null;
   /**
    * UMLMethod: '+ reLayout (currentCanvas : DisResizeable) : Void'
    *
    * @param currentCanvas          No description provided
    * @throws InterruptedException  Exception description not provided
    */
   public void reLayout (FSAContainer currentCanvas)
       throws InterruptedException
   {
      // Set Options before start
      LayoutPreferences options = LayoutPreferences.get();

      setHorizDist (options.getHorizDist());
      setVertDist (options.getVertDist());

      UMLActivityDiagram curActivityDiagram = null;

      // set the NorthWest corner of the layout
      Point nw = new Point (getHorizDist(), getVertDist());
      if (currentCanvas.getFSAInterface().getLogic() instanceof UMLActivityDiagram)
      {
         try
         {
            //this.currentCanvas = currentCanvas;
            curActivityDiagram = (UMLActivityDiagram) currentCanvas.getFSAInterface().getLogic();

            Sequencer sequencer = Sequencer.get();

            curActivityDiagram.createFlowAssociations();

            // start of mksoft hack
            // layout story activities with TreeLayout

            Iterator itemIter = curActivityDiagram.iteratorOfElements();
            FElement item = null;

            while (itemIter.hasNext())
            {
               item = (FElement) itemIter.next();

               if (item instanceof UMLStoryActivity)
               {
                  UMLStoryPattern pattern =  ((UMLStoryActivity) item).getStoryPattern();
                  Iterator iter2 = pattern.getFSAInterface().iteratorOfFsaObjects();

                  while (iter2.hasNext())
                  {
                     FSAObject ob = (FSAObject) iter2.next();

                     if (ob instanceof FSAPanel)
                     {
                        FSAPanel dr = (FSAPanel) ob;

                        TreeLayout.get().reLayout (dr);
                     }
                  }
               }
            }

            // end of mksoft hack

            FlowActivity startActivity = curActivityDiagram.getStartActivity().getFlowActivity();

            // if exploreCFG fails a runtime exception is thrown
            Seq seq = sequencer.exploreCFG (startActivity);

            // layout recursively through the CFG
            layoutCFG (seq, nw);

         }
         catch (RuntimeException runEx)
         {
            // FujabaDebug.println ("ERROR during code generation simulation for ControlFlowLayouter");
            // runEx.printStackTrace ();

            // if there is a runtime exception in the sequencer, there is a structure error
            // one way is to make the sequencer fault tolerant, the other to ignore and use another layouter
         }
         finally
         {
            if (curActivityDiagram != null)
            {
               curActivityDiagram.removeFlowAssociations();
            }
            currentCanvas = null;
         }
      }
   } // reLayout

}

/*
 * $Log: ControlFlowLayout.java,v $
 * Revision 1.26  2004/10/20 17:49:58  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
