/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.layout;

import java.awt.*;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.*;

import de.uni_paderborn.fujaba.fsa.*;
import de.uni_paderborn.fujaba.fsa.swing.JBend;
import de.uni_paderborn.fujaba.fsa.swing.JBendLine;
import de.uni_paderborn.fujaba.layout.options.LayoutPreferences;
import de.uni_paderborn.fujaba.uml.UMLActivityDiagram;
import de.uni_paderborn.fujaba.uml.UMLStartActivity;


/**
 * Class 'AbstractLayouter' specified in class diagram 'TreeLayout.java diagram'.
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.31 $
 */
public abstract class AbstractLayouter
{
   /**
    * Constructor of Class AbstractLayouter
    */
   protected AbstractLayouter()
   {
      super();
      // some initial settings for the layouter
      lastCondition = false;
   } // AbstractLayouter


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean lastCondition = false;


   /**
    * Get the lastCondition attribute of the AbstractLayouter object
    *
    * @return   The lastCondition value
    */
   protected boolean isLastCondition()
   {
      return lastCondition;
   } // isLastCondition


   /**
    * Sets the lastCondition attribute of the AbstractLayouter object
    *
    * @param lastCondition  The new lastCondition value
    * @return               No description provided
    */
   protected boolean setLastCondition (boolean lastCondition)
   {
      if (this.lastCondition != lastCondition)
      {
         this.lastCondition = lastCondition;
      } // if

      return this.lastCondition;
   } // setLastCondition


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int horizDist;


   /**
    * Get the horizDist attribute of the AbstractLayouter object
    *
    * @return   The horizDist value
    */
   public int getHorizDist()
   {
      return horizDist;
   } // getHorizDist


   /**
    * Sets the horizDist attribute of the AbstractLayouter object
    *
    * @param horizDist  The new horizDist value
    * @return           No description provided
    */
   public int setHorizDist (int horizDist)
   {
      if (horizDist >= 0)
      {
         this.horizDist = horizDist;
      } // if

      return this.horizDist;
   } // setHorizDist


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int vertDist;


   /**
    * Get the vertDist attribute of the AbstractLayouter object
    *
    * @return   The vertDist value
    */
   public int getVertDist()
   {
      return vertDist;
   } // getVertDist


   /**
    * Sets the vertDist attribute of the AbstractLayouter object
    *
    * @param vertDist  The new vertDist value
    * @return          No description provided
    */
   public int setVertDist (int vertDist)
   {
      if (vertDist >= 0)
      {
         this.vertDist = vertDist;
      } // if

      return this.vertDist;
   } // setVertDist


   /**
    * Function refreshes the function settings for the AbstractLayouter
    */
   public void refreshOptions()
   {
      LayoutPreferences myOptions = LayoutPreferences.get();
      // now read the options from myOptions
      horizDist = myOptions.getHorizDist();
      vertDist = myOptions.getVertDist();
   } // refreshOptions


   /**
    * Function searches for Child of a father class, connected by a grab
    *
    * @param itemGrab  The grab which should be searched
    * @return          The grabbed to Frame
    */
   protected FSAObject getChild (FSABend itemGrab)
   {
      FSAObject item = null;
      FSABend targetGrab = null;

      JBend grab = (JBend) itemGrab.getJComponent();
      JBendLine line = grab.getFirstFromLines();
      // Now search for the Target Class which this line is connected to
      if (grab == line.getStartBend())
      {
         targetGrab = (FSABend) FSAObject.getFSAObjectFromJComponent (line.getEndBend());
      }
      else if (grab == line.getEndBend())
      {
         targetGrab = (FSABend) FSAObject.getFSAObjectFromJComponent (line.getStartBend());
      }
      if (targetGrab instanceof FSAGrab)
      {
         item =  ((FSAGrab) targetGrab).getTarget();
      }
      else if (targetGrab != null)
      {
         getChild (targetGrab);
      }

      return item;
   } // getChild


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param currentClass           No description provided
    * @throws InterruptedException  Exception description not provided
    */
   protected void innerLayout (FSAContainer currentClass)
       throws InterruptedException
   {
      // Some variables
      Iterator myIter = currentClass.iteratorOfChildren();
      Object myObject;

      // check if there are inner classes, if so relayout them
      if (myIter.hasNext())
      {
         myObject = myIter.next();
         if (myObject instanceof FSAContainer)
         {
            // The inner Canvas needs to be Layouted
            this.reLayout ((FSAContainer) myObject);
         }
      }
   } // innerLayout


   /**
    * This method returns TRUE if a Object already is set as visited, the behavior of the method
    * depends on the var lastCondition
    *
    * @param tocheckObject  The object to be checked, needs to be a instance of DisFrame
    * @return               No description provided
    */
   protected boolean checkVisited (Object tocheckObject)
   {
      if (tocheckObject instanceof FSAObject)
      {
         return  ((FSAObject) tocheckObject).isGenerated() != lastCondition;
      }
      else
      {
         return false;
      }
   } // checkVisited


   /**
    * searchStart searches for DisFrame on the searchCanvas, from which the reLayouter can
    * start. This method only is invoked, if there is no DisFrame selected on searchCanvas.
    * Not used by all layouters
    *
    * @param searchCanvas  The Canvas to be searched
    * @param withVisited   The Toggle status of the visited condition (for performance reasons
    *      this value is toggled at each invocation of the layouter)
    * @return              No description provided
    */
   protected FSAObject searchStart (FSAContainer searchCanvas,
                                    boolean withVisited)
   {
      FSAObject myObject = null;
      FSAObject myFirstObject = null;
      // search element for MrLayout to start
      Iterator myIter = searchCanvas.iteratorOfChildren();
      boolean found = false;
      boolean isActivityDiag =  (searchCanvas.getLogic() != null && searchCanvas.getLogic() instanceof UMLActivityDiagram);

      while (!found && myIter.hasNext())
      {
         myObject = (FSAObject) myIter.next();
         found =  (myObject instanceof FSAPanel) &&  (!checkVisited (myObject)
            || !withVisited);
         // use only if visible
         if (found)
         {
            found = myObject.isVisible();

            //always use start node for activity diagrams
            if (found && isActivityDiag)
            {
               found =  (myObject.getLogic() != null && myObject.getLogic() instanceof UMLStartActivity);
               if (myFirstObject == null)
               {
                  myFirstObject = myObject;
               }
            }
         }

      }

      if (!found && isActivityDiag)
      {
         myObject = myFirstObject;
         found =  (myObject != null);
      }

      // did we really find a DisFrame ?
      if (found)
      {
         return myObject;
      }
      else
      {
         return null;
      }
   } // searchStart


   /**
    * Function is responsible for the rowDepths vector, it returns the value of Element at
    * position <pos>
    *
    * in the vector, if the size of the array is smaller as the requested element, the function
    * adds one element and returns 0. Only for layouter, which layout in rows
    *
    * @param pos        The position of the leftmost element
    * @param rowDepths  The rowDepths-Vector per reference (needed for recursion reasons)
    * @return           The left value
    */
   protected int getLeft (int pos, Vector rowDepths)
   {
      int left;
      if (pos < rowDepths.size())
      {
         left =  ((Integer) rowDepths.elementAt (pos)).intValue();
      }
      else
      {
         left = 0;
         rowDepths.addElement (new Integer (left));
      }
      return left;
   } // getLeft


   /**
    * Function set value <left> at pos in vector rowDepths
    *
    * @param left       The new left value
    * @param pos        The new left value
    * @param rowDepths  The new left value
    */
   protected void setLeft (int left, int pos, Vector rowDepths)
   {
      rowDepths.setElementAt (new Integer (left), pos);
   } // setLeft


   /**
    * Returns the frame where the wait cursor have to be set.
    *
    * @param canvas  the canvas to be layouted.
    * @return        the highest frame.
    */
   public JFrame getFrame (FSAObject canvas)
   {
      while (canvas != null)
      {
         if (canvas instanceof FSALayeredPane)
         {
            Container p = canvas.getJComponent();
            while (p != null)
            {
               if (p instanceof JFrame)
               {
                  return (JFrame) p;
               }

               p = p.getParent();
            }
            return null;
         }

         canvas = canvas.getParent();
      }

      return null;
   }


   /**
    * This function needs to be implemented in the child-classes.
    *
    * @param currentCanvas          No description provided
    * @throws InterruptedException  Exception description not provided
    */
   public abstract void reLayout (FSAContainer currentCanvas)
       throws InterruptedException;


   /**
    * 1 myLayouter 1 AbstractLayouter -------------------- MrLayout
    */
   private MrLayout revMyLayouter;


   /**
    * Get the revMyLayouter attribute of the AbstractLayouter object
    *
    * @return   The revMyLayouter value
    */
   public MrLayout getRevMyLayouter()
   {
      return revMyLayouter;
   } // getRevMyLayouter


   /**
    * Sets the revMyLayouter attribute of the AbstractLayouter object
    *
    * @param revMyLayouter  The new revMyLayouter value
    */
   public void setRevMyLayouter (MrLayout revMyLayouter)
   {
      if (this.revMyLayouter != revMyLayouter)
      {
         // newPartner
         if (this.revMyLayouter != null)
         {
            // inform old partner
            MrLayout oldRevMyLayouter = this.revMyLayouter;
            this.revMyLayouter = null;

            oldRevMyLayouter.setMyLayouter (null);
         } // if

         this.revMyLayouter = revMyLayouter;
         if (revMyLayouter != null)
         {
            // inform new partner
            revMyLayouter.setMyLayouter (this);
         } // if

      } // if

   } // setRevMyLayouter


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      setRevMyLayouter (null);
   } // removeYou


   /**
    * Function returns the frame at the one side of a Frame depending on the what-Parameter
    *
    * @param myLine  The connecting Line
    * @param what    =1 front of bends, =2 back of bends
    * @return        The fromToFrame value
    */
   protected FSAObject getFromToFrame (FSABendLine myLine, int what)
   {
      FSAObject whatsThere = null;
      if (what == 1)
      {
         whatsThere =  ((FSAGrab) myLine.getStartBend()).getTarget();
      }
      else if (what == 2)
      {
         whatsThere =  ((FSAGrab) myLine.getEndBend()).getTarget();
      }

      if (whatsThere instanceof FSABendLine)
      {
         return this.getFromToFrame ((FSABendLine) whatsThere, what);
      }
      return whatsThere;
   }


   /**
    * Needed for getPreferredLen, value comes from enviroment settings
    */
   private double presetAdjustment = 1.4;


   /**
    * Get the presetAdjustment attribute of the AbstractLayouter object
    *
    * @return   The presetAdjustment value
    */
   public double getPresetAdjustment()
   {
      return presetAdjustment;
   }


   /**
    * Sets the presetAdjustment attribute of the AbstractLayouter object
    *
    * @param presetAdjustment  The new presetAdjustment value
    */
   public void setPresetAdjustment (double presetAdjustment)
   {
      if (presetAdjustment > 0)
      {
         this.presetAdjustment = presetAdjustment;
      }
   }


   /**
    * Function returns the preferred length of a given DisLine
    *
    * @param myLine  The Line
    * @return        The preferredLen value
    */
   protected int getPreferredLen (FSABendLine myLine)
   {
      FSAObject front242Frame;
      FSAObject
         backFrame;
      int height;
      int
         width;
      int
         len;

      front242Frame = getFromToFrame (myLine, 1);
      backFrame = getFromToFrame (myLine, 2);

      // Don`t forget to initialize len, to avoid exceptions
      len = 0;

      if (front242Frame != null)
      {
         height = front242Frame.getPreferredSize().height;
         width = front242Frame.getPreferredSize().width;
         len = (int) Math.sqrt (height * height + width * width);
      }

      if (backFrame != null)
      {
         height = backFrame.getPreferredSize().height;
         width = backFrame.getPreferredSize().width;
         len += (int) Math.sqrt (height * height + width * width);
      }

      // multiply the adjustment from the presettings.
      len *= presetAdjustment;

      return len;
   }
}

/*
 * $Log: AbstractLayouter.java,v $
 * Revision 1.31  2004/11/23 18:20:00  creckord
 * Fixed some bugs in Story Pattern codegen:
 * <a href="http://uther.uni-paderborn.de/tracker/?func=detail&aid=73&group_id=39&atid=197">Bug #73</a>
 * <a href="http://uther.uni-paderborn.de/tracker/?func=detail&aid=74&group_id=39&atid=197">Bug #74</a>
 * <a href="http://uther.uni-paderborn.de/tracker/?func=detail&aid=98&group_id=39&atid=197">Bug #98</a>
 * <a href="http://uther.uni-paderborn.de/tracker/?func=detail&aid=99&group_id=39&atid=197">Bug #99</a>
 *
 */
