/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui.mdi;

import java.awt.*;
import java.beans.PropertyVetoException;

import javax.swing.*;


/**
 * This class provides internal frame positioning methods for use by DesktopMenu.
 *
 * @author    <a href="mailto:tessier@gabinternet.com">Tom Tessier</a>
 * @version   1.0 11-Aug-2001
 */
public class FramePositioning
{
   /**
    * maximum number of internal frames allowed
    */
   public final static int MAX_FRAMES = 20;

   /**
    * default x offset of first frame in cascade mode, relative to desktop
    */
   public final static int X_OFFSET = 30;

   /**
    * default y offset of first frame in cascade mode, relative to desktop
    */
   public final static int Y_OFFSET = 30;

   /**
    * minimum width of frame toolbar buttons
    */
   public final static int MINIMUM_BUTTON_WIDTH = 30;

   /**
    * maximum width of frame toolbar buttons
    */
   public final static int MAXIMUM_BUTTON_WIDTH = 80;

   /**
    * the foreground color of inactive buttons whose associated frame contents have changed
    */
   public final static Color CONTENTS_CHANGED_COLOR = Color.red;

   /**
    * desktop whose frames are positioned
    */
   private JDesktopPane desktop;


   /**
    * creates the FramePositioning object
    *
    * @param desktop  a reference to the JDesktopPane
    */
   public FramePositioning (JDesktopPane desktop)
   {
      this.desktop = desktop;
   }


   /**
    * cycles through and cascades all internal frames
    */
   public void cascadeInternalFrames()
   {
      JInternalFrame[] frames = desktop.getAllFrames();
      JInternalFrame f;

      int frameCounter = 0;

      for (int i = frames.length - 1; i >= 0; i--)
      {
         f = frames[i];

         // don't include iconified frames in the cascade
         if (!f.isIcon())
         {
            try
            {
               f.setMaximum (false);
            }
            catch (PropertyVetoException e)
            {
            }
            f.setLocation (cascadeInternalFrame (f, frameCounter++));
            f.setSize (new Dimension ((int) Math.min (f.getWidth(), desktop.getVisibleRect().getWidth() - f.getX()),
               (int) Math.min (f.getHeight(), desktop.getVisibleRect().getHeight() - f.getY())));
         }
      }
   }


   /**
    * cascades the given internal frame based upon the current number of internal frames
    *
    * @param f  the internal frame to cascade
    * @return   a Point object representing the location assigned to the internal frame upon
    *      the virtual desktop
    */
   public Point cascadeInternalFrame (JInternalFrame f)
   {
      return cascadeInternalFrame (f, desktop.getComponentCount());
   }


   /**
    * cascades the given internal frame based upon supplied count
    *
    * @param f      the internal frame to cascade
    * @param count  the count to use in cascading the internal frame
    * @return       a Point object representing the location assigned to the internal frame
    *      upon the virtual desktop
    */
   private Point cascadeInternalFrame (JInternalFrame f, int count)
   {
      int windowWidth = f.getWidth();
      int windowHeight = f.getHeight();

      Rectangle viewP = desktop.getVisibleRect();

      // get # of windows that fit horizontally
      int numFramesWide =  (viewP.width - windowWidth) / X_OFFSET;

      if (numFramesWide < 1)
      {
         numFramesWide = 1;
      }

      // get # of windows that fit vertically
      int numFramesHigh =  (viewP.height - windowHeight) / Y_OFFSET;

      if (numFramesHigh < 1)
      {
         numFramesHigh = 1;
      }

      // position relative to the current viewport (viewP.x/viewP.y)
      // (so new windows appear onscreen)
      int xLoc = viewP.x +
          (X_OFFSET *  ( (count
      /*
       *  + 1
       */
         ) -
          ( (numFramesWide - 1) *  (count / numFramesWide))));
      int yLoc = viewP.y +
          (Y_OFFSET *  ( (count
      /*
       *  + 1
       */
         ) -
          (numFramesHigh *  (count / numFramesHigh))));

      return new Point (xLoc, yLoc);
   }


   /**
    * tiles internal frames upon the desktop. <BR>
    * <BR>
    * Based upon the following tiling algorithm: <BR>
    * <BR>
    * - take the sqroot of the total frames rounded down, that gives the number of columns.
    * <BR>
    * <BR>
    * - divide the total frames by the # of columns to get the # of rows in each column, and
    * any remainder is distributed amongst the remaining rows from right to left) <BR>
    * <BR>
    * eg) <BR>
    * 1 frame, remainder 0, 1 row<BR>
    * 2 frames, remainder 0, 2 rows<BR>
    * 3 frames, remainder 0, 3 rows<BR>
    * 4 frames, remainder 0, 2 rows x 2 columns <BR>
    * 5 frames, remainder 1, 2 rows in column I, 3 rows in column II<BR>
    * 10 frames, remainder 1, 3 rows in column I, 3 rows in column II, 4 rows in column III
    * <BR>
    * 16 frames, 4 rows x 4 columns <BR>
    * <BR>
    * <BR>
    * Pseudocode: <BR>
    * <BR>
    * <code>
    *     while (frames) { <BR>
    * numCols = (int)sqrt(totalFrames); <BR>
    * numRows = totalFrames / numCols; <BR>
    * remainder = totalFrames % numCols <BR>
    * if ((numCols-curCol) <= remainder) { <BR>
    * numRows++; // add an extra row for this column <BR>
    * } <BR>
    * } </code><BR>
    *
    */
   public void tileInternalFrames()
   {
      Rectangle viewP = desktop.getVisibleRect();

      int totalNonIconFrames = 0;

      JInternalFrame[] frames = desktop.getAllFrames();

      for (int i = 0; i < frames.length; i++)
      {
         if (!frames[i].isIcon())
         { // don't include iconified frames...
            try
            {
               frames[i].setMaximum (false);
            }
            catch (PropertyVetoException e)
            {
            }
            totalNonIconFrames++;
         }
      }

      int curCol;
      int curRow;
      int i = 0;

      if (totalNonIconFrames > 0)
      {
         // compute number of columns and rows then tile the frames
         int numCols = (int) Math.sqrt (totalNonIconFrames);

         int frameWidth = viewP.width / numCols;

         for (curCol = 0; curCol < numCols; curCol++)
         {
            int numRows = totalNonIconFrames / numCols;
            int remainder = totalNonIconFrames % numCols;

            if ( (numCols - curCol) <= remainder)
            {
               numRows++; // add an extra row for this guy
            }

            int frameHeight = viewP.height / numRows;

            for (curRow = 0; curRow < numRows; curRow++)
            {
               while (frames[i].isIcon())
               { // find the next visible frame
                  i++;
               }

               frames[i].setBounds (curCol * frameWidth,
                  curRow * frameHeight, frameWidth, frameHeight);

               i++;
            }
         }
      }
   }
}

/*
 * $Log: FramePositioning.java,v $
 * Revision 1.5  2004/10/22 16:41:38  lowende
 * Deprecated warnings removed. Other compile warnings removed.
 *
 */
