/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui.mdi;

import java.awt.*;
import java.awt.event.*;
import java.beans.PropertyVetoException;
import java.util.Enumeration;

import javax.swing.*;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;


/**
 * This class provides a "Window" menu for the JDesktopPane.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.5.2.1 $ $Date: 2005/08/23 08:24:56 $
 */
public class DesktopMenu extends JMenu implements ActionListener
{
   /**
    * managed desktop
    */
   final JDesktopPane desktop;
   /**
    * strategy for frame positioning
    */
   private final FramePositioning positioning;
   /**
    * last index of non-dynamic menu entries
    */
   private int baseItemsEndIndex;
   /**
    * group for frame radio buttons
    */
   ButtonGroup frameRadioButtonMenuItemGroup;
   /**
    * menuitem to close selected window
    */
   JMenuItem closeMenuItem;
   /**
    * menuitem to close all internal frames
    */
   JMenuItem closeAllMenuItem;


   /**
    * creates the DesktopMenu object
    *
    * @param desktop  a reference to the DesktopMediator object
    */
   public DesktopMenu (JDesktopPane desktop)
   {
      this (desktop, false);
   }


   /**
    * creates the DesktopMenu object with the specified tileMode
    *
    * @param desktop   a reference to the DesktopMediator object
    * @param tileMode  the tile mode to use (<code>true</code> = tile internal frames, <code>false</code>
    *      = cascade internal frames)
    */
   public DesktopMenu (JDesktopPane desktop, boolean tileMode)
   {
      super ("Window");
      setMnemonic (KeyEvent.VK_W);

      this.desktop = desktop;
      this.positioning = new FramePositioning (desktop);

      frameRadioButtonMenuItemGroup = new ButtonGroup();

      constructMenuItems (this);

      // set the default item count (ie: number of items comprising
      // current menu contents)
      baseItemsEndIndex = getItemCount();

      desktop.addContainerListener (
         new ContainerListener()
         {
            public void componentAdded (ContainerEvent e)
            {
               if (e.getChild() instanceof JInternalFrame)
               {
                  addFrame ((JInternalFrame) e.getChild());
               }
            }


            public void componentRemoved (ContainerEvent e)
            {

            }
         });

   }


   /**
    * set mnemonic and shortcut for a menu item, add this as action listener
    *
    * @param item      the item
    * @param mnemonic  the mnemonic used to access the menu
    * @param shortcut  the keyboard shortcut used to access the menu. -1 indicates no shortcut.
    * @return          the initialized menu item
    */
   public JMenuItem initMenuItem (JMenuItem item, int mnemonic, int shortcut)
   {
      item.setMnemonic (mnemonic);

      // set the alt-Shortcut accelerator
      if (shortcut != -1)
      {
         item.setAccelerator (KeyStroke.getKeyStroke (shortcut, ActionEvent.CTRL_MASK));
      }

      item.addActionListener (this);
      return item;
   }


   /**
    * constructs the actual menu items.
    *
    * @param sourceMenu  the source menu to apply the menu items
    */
   private void constructMenuItems (JMenu sourceMenu)
   {
      sourceMenu.add (initMenuItem (new JMenuItem ("Tile"), KeyEvent.VK_T, -1));

      sourceMenu.add (initMenuItem (new JMenuItem ("Cascade"), KeyEvent.VK_C, -1));
      sourceMenu.addSeparator();

//      JMenu autoMenu = new JMenu ("Auto");
//      autoMenu.setMnemonic (KeyEvent.VK_U);
//
//      ButtonGroup autoMenuGroup = new ButtonGroup();
//      JMenuItem radioItem = initMenuItem ( new JRadioButtonMenuItem( "Tile", tileMode )
//            , KeyEvent.VK_T, -1 );
//      autoMenu.add (radioItem);
//      autoMenuGroup.add (radioItem);
//
//      radioItem = initMenuItem( new JRadioButtonMenuItem ( "Cascade", !tileMode), KeyEvent.VK_C,
//            -1);
//      autoMenu.add (radioItem);
//      autoMenuGroup.add (radioItem);
//
//      sourceMenu.add (autoMenu);
//      sourceMenu.addSeparator();

      closeMenuItem = initMenuItem (new JMenuItem ("Close"), KeyEvent.VK_S,
         KeyEvent.VK_F4);
      sourceMenu.add (closeMenuItem);
      closeMenuItem.setEnabled (false);

      closeAllMenuItem = initMenuItem (new JMenuItem ("Close All"), KeyEvent.VK_A, -1);
      sourceMenu.add (closeAllMenuItem);
      closeMenuItem.setEnabled (desktop.getComponentCount() > 0);

      sourceMenu.addSeparator();
   }


   /**
    * adds a radio item to the menu and associates it with an internal frame
    *
    * @param associatedFrame  the internal frame to associate with the menu item
    */
   void addFrame (JInternalFrame associatedFrame)
   {
      //check if we already have it
      final Component[] menuComponents = this.getMenuComponents();
      for (int i = 0; i < menuComponents.length; i++)
      {
         Component menuComponent = menuComponents[i];
         if (menuComponent instanceof JComponent)
         {
            if ( ((JComponent) menuComponent).getClientProperty ("associatedFrame") == associatedFrame)
            {
               return;
            }
         }
      }

      int displayedCount = getItemCount() - baseItemsEndIndex + 1;
      int currentMenuCount = displayedCount;

      // compute the key mnemonic based upon the currentMenuCount
      if (currentMenuCount > 9)
      {
         currentMenuCount /= 10;
      }

      final JMenuItem menuButton = initMenuItem (new JRadioButtonMenuItem (
         displayedCount + " " + associatedFrame.getTitle(), true),
         KeyEvent.VK_0 + currentMenuCount, -1);
      menuButton.putClientProperty ("associatedFrame", associatedFrame);

      associatedFrame.addInternalFrameListener (
         new InternalFrameAdapter()
         {
            /**
             * Invoked when an internal frame is activated.
             *
             * @param e  event
             */
            public void internalFrameActivated (InternalFrameEvent e)
            {
               closeMenuItem.setEnabled (true);
               menuButton.removeActionListener (DesktopMenu.this);
               menuButton.setSelected (true);
               menuButton.addActionListener (DesktopMenu.this);
            }


            /**
             * Invoked when an internal frame is de-activated.
             *
             * @param e  event
             */
            public void internalFrameDeactivated (InternalFrameEvent e)
            {
               closeMenuItem.setEnabled (false);
               menuButton.setSelected (false);
            }


            /**
             * Invoked when an internal frame has been closed.
             *
             * @param e  event
             */
            public void internalFrameClosed (InternalFrameEvent e)
            {
               remove (menuButton);
               frameRadioButtonMenuItemGroup.remove (menuButton);
               closeAllMenuItem.setEnabled (desktop.getComponentCount() > 0);
            }


            /**
             * Invoked when an internal frame has been opened.
             *
             * @param e  event
             */
            public void internalFrameOpened (InternalFrameEvent e)
            {
               add (menuButton);
               frameRadioButtonMenuItemGroup.add (menuButton);
            }
         });

      add (menuButton);
      frameRadioButtonMenuItemGroup.add (menuButton);

      menuButton.setSelected (true);
      closeAllMenuItem.setEnabled (true);
   }


   /**
    * removes the specified radio menu button from the menu
    *
    * @param menuButton  the JRadioButtonMenuItem to remove
    */
   public void remove (JRadioButtonMenuItem menuButton)
   {
      frameRadioButtonMenuItemGroup.remove (menuButton);
      super.remove (menuButton);

      // cannot simply remove the radio menu button, as need to
      // renumber the keyboard shortcut keys as well. Hence, a
      // call to refreshMenu is in order...
      refreshMenu(); // refresh the mnemonics associated
      // with the other items
   }


   /**
    * refreshes a given menu item
    */
   private void refreshMenu()
   {
      // refresh the associated mnemonics, so that the keyboard shortcut
      // keys are properly renumbered...
      // get an enumeration to the elements of the current button group
      Enumeration enumeration = frameRadioButtonMenuItemGroup.getElements();

      int displayedCount = 1;
      int currentMenuCount;

      while (enumeration.hasMoreElements())
      {
         JMenuItem item = (JMenuItem) enumeration.nextElement();

         // compute the key mnemonic based upon the currentMenuCount
         currentMenuCount = displayedCount;

         if (currentMenuCount > 9)
         {
            currentMenuCount /= 10;
         }

         item.setMnemonic (KeyEvent.VK_0 + currentMenuCount);
         final JInternalFrame associatedFrame = (JInternalFrame) item.getClientProperty ("associatedFrame");
         item.setText (displayedCount + associatedFrame.getTitle());
         displayedCount++;
      }
   }


   /**
    * called when a menu item was clicked.
    *
    * @param e  event
    */
   public void actionPerformed (ActionEvent e)
   {
      String actionCmd = e.getActionCommand();

      if (actionCmd.equals ("Tile"))
      {
         positioning.tileInternalFrames();
      }
      else if (actionCmd.equals ("Cascade"))
      {
         positioning.cascadeInternalFrames();
      }
      else if (actionCmd.equals ("Close"))
      {
         final JInternalFrame selectedFrame = desktop.getSelectedFrame();
         if (selectedFrame != null)
         {
            selectedFrame.doDefaultCloseAction();
         }
      }
      else if (actionCmd.equals ("Close All"))
      {
         final Component[] frames = desktop.getComponents();
         for (int i = 0; i < frames.length; i++)
         {
            Component frame = frames[i];
            if (frame instanceof JInternalFrame)
            {
                ((JInternalFrame) frame).dispose();
            }
            else
            {
               remove (frame);
            }
         }
      }
//      else if (actionCmd.equals ("TileRadio"))
//      {
//         setAutoTile (true);
//      }
//      else if (actionCmd.equals ("CascadeRadio"))
//      {
//         setAutoTile (false);
//      }
      else if (e.getSource() instanceof JRadioButtonMenuItem)
      {
         // then select the associated frame (if any)
         JRadioButtonMenuItem item = (JRadioButtonMenuItem) e.getSource();

         if (item.isSelected())
         {
            JInternalFrame associatedFrame = (JInternalFrame)  ((JComponent) e.getSource()).getClientProperty ("associatedFrame");

            if (associatedFrame != null)
            {
               activate (associatedFrame);
            }
         }
      }
   }


   /**
    * make the frame the selected frame.
    *
    * @param frame  what to activate
    */
   private void activate (JInternalFrame frame)
   {
      try
      {
         frame.setSelected (true);
         frame.setVisible (true);
         frame.setIcon (false);
      }
      catch (PropertyVetoException e1)
      {
         //leave it
      }
   }
}

/*
 * $Log: DesktopMenu.java,v $
 * Revision 1.5.2.1  2005/08/23 08:24:56  lowende
 * Removed compile warnings.
 *
 */
