/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.util.Iterator;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.uml.*;



/**
 * The class PETypeSelection derives the class PESelection to initialize the
 * source list with a possible type.
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.30.2.3 $
 */
public class PETypeSelection extends PESelection
{
   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private PETextField typeName;

   /**
    * Shows a more detailed description of the selected type in a textfield.
    * Helpful, if there are types with the same name. E.g. if you add a
    * UMLClass with name 'java.lang.String' to your project, that will be shown
    * as 'String' in the type-list, and the UMLBaseType 'String' also exists
    * you won't know which 'String' is the base-type and which one is the
    * UMLClass.
    */
   private PETextField detailedTypeDescription;
   private String initialSelection;


   /**
    * Constructor for class PETypeSelection
    *
    * @param parent            No description provided
    * @param initialSelection
    */
   public PETypeSelection (BasicPropertyEditor parent, String initialSelection)
   {
      super (parent);
      getSource().setHeader ("Types");
      typeName = new PETextField (this.parent, "Type Name");
      this.initialSelection = initialSelection;
   }


   /**
    * Get the singleton instance of the detailed type-description.
    *
    * @return   The singleton instance.
    */
   private PETextField getDetailedTypeDescription()
   {
      if (detailedTypeDescription == null)
      {
         // don't set a title:
         // what this text-field represents is self-explanatory
         detailedTypeDescription = new PETextField (this.parent, "");
         detailedTypeDescription.setReadOnly (true);
      }

      return detailedTypeDescription;
   }


   /**
    * Sets the typeName attribute of the PETypeSelection object
    *
    * @param type  The new typeName value
    */
   public void setTypeName (String type)
   {
      typeName.setText (type);
   }


   /**
    * The function filter enables the class to preselect the kind of type. For
    * example you can filter all basetypes.
    *
    * @param type  a valid reference to an object of the AST
    * @return      results true if the type is valid
    */
   protected boolean filter (UMLType type)
   {
      // override this method to filter the kind of Type
      // e.g. only basetypes
      return true;
   }


   /**
    * Add all UMLType instances contained in the typelist of the current
    * UMLProject to the type-list.
    */
   protected void fillSourceList()
   {
      clearSource();
      ASGElement incr = getIncrement();
      if (incr instanceof UMLClass)
      {
         UMLTypeList typeList = UMLProject.get().getTypeList();
         if (typeList != null)
         {
            UMLType umlType;
            Iterator types = typeList.sortedIteratorOfTypes();
            while (types.hasNext())
            {
               umlType = (UMLType) types.next();
               if ( (umlType != null) &&  (filter (umlType)))
               {
                  addToSource ((UMLIncrement) umlType);

                  // initially select 'Void'
                  if (umlType.getName().equals (initialSelection))
                  {
                     setSourceSelectedIncr ((UMLIncrement) umlType);
                  }
               }
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   protected void fillDestList()
   {
   }


   /**
    * Get the typeName attribute of the PETypeSelection object
    *
    * @return   The typeName value
    */
   public String getTypeName()
   {
      return typeName.getText();
   }


   /**
    * Called whenever selection in the type-list is changed.
    */
   protected void sourceSelectionChanged()
   {
      ASGElement incr = source.getSelectedIncrement();
      if (incr != null)
      {
         typeName.setText (incr.getText());

         PETextField tmpDescritionTextField = getDetailedTypeDescription();
         // construct a more detailed description
         String detailedDescription = incr.getText();

         if (incr instanceof UMLClass)
         {
            detailedDescription += " - UMLClass '" +  ((UMLClass) incr).getTypeName() + "'";
         }
         else if (incr instanceof UMLBaseTypes)
         {
            detailedDescription += " - UMLBaseType";
         }
         else if (incr instanceof UMLArray)
         {
            detailedDescription += " - UMLArray of type '" +  ((UMLArray) incr).getBaseTypeName() + "'";
         }

         tmpDescritionTextField.setText (detailedDescription);
         tmpDescritionTextField.setToolTipText ("selected type: " + detailedDescription);
      }
   }


   /*
    *  (non-Javadoc)
    *  @see de.uni_paderborn.fujaba.gui.PEBaseComponent#addComponents()
    */
   /**
    * Access method for a To N-association.
    */
   protected void addComponents()
   {
      super.addComponents();

      // additionally add the more detailed description to the bottom
      // of the PESelection
      this.add (getDetailedTypeDescription());
   }
}

/*
 * $Log: PETypeSelection.java,v $
 * Revision 1.30.2.3  2006/08/14 11:02:41  cschneid
 * initial selection in Attribute and Parameter Editor is "String" now
 *
 */
