/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import javax.swing.ButtonGroup;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import org.apache.log4j.Logger;

import de.uni_paderborn.fujaba.app.FrameMain;
import de.uni_paderborn.fujaba.gui.comp.FujabaDialog;
import de.uni_paderborn.fujaba.uml.UMLClass;
import de.uni_paderborn.fujaba.uml.UMLFile;
import de.uni_paderborn.fujaba.uml.UMLIncrement;
import de.uni_paderborn.fujaba.uml.UMLPackage;
import de.uni_paderborn.fujaba.uml.UMLProject;
import de.uni_paderborn.fujaba.uml.UMLStereotype;
import de.uni_paderborn.fujaba.uml.UMLStereotypeManager;


/**
 * This Dialog can be used to change the imports. You can add or remove packages and classes
 * using the package structure on UMLProject's rootPackage. The adding and removing is only
 * simulated until you hit the ok/cancel buttons so we can to an easy cancel operation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.36.2.3 $
 */
public class EditImportDialog extends FujabaDialog
{
   /**
    * log4j logging
    */
   final static transient Logger log = Logger.getLogger (EditImportDialog.class);

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JTree fullTree;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JList fileImportList;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JComboBox fileBox;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private DefaultMutableTreeNode rootNodeFullTree = new DefaultMutableTreeNode (UMLProject.get().getRootPackage());
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   DefaultListModel importListModel = new DefaultListModel();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   DefaultTreeModel fullTreeModel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   HashSet simAddHashSet = new HashSet();
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   HashSet simRemoveHashSet = new HashSet();


   /**
    * Constructor for class EditImportDialog
    *
    * @param frame  No description provided
    */
   public EditImportDialog (JFrame frame)
   {
      super (frame, "Import Editor", true);
      this.guiInit();
      this.unparse();
   }


   /**
    * guiInit creates the base panel and puts the button and the guiWorkPanel on it.
    */
   private final void guiInit()
   {
      JPanel panel = new JPanel();
      panel.setLayout (new BorderLayout());
      panel.add (this.guiWorkPanel(), BorderLayout.CENTER);
      panel.add (this.guiPanelOkCancelHelp(), BorderLayout.SOUTH);
      this.getContentPane().add (panel);
   }


   /**
    * guiWorkPanel creates the Panel which does the real work. Here the tree, the additional
    * buttons and and list are created.
    *
    * @return   No description provided
    */
   private final JPanel guiWorkPanel()
   {
      GridBagLayout gridbag = new GridBagLayout();
      GridBagConstraints c = new GridBagConstraints();
      JPanel mainPanel = new JPanel (gridbag);

      JPanel addRemovePanel = new JPanel (new GridLayout (2, 1));
      JButton add = new JButton ("->");
      JButton remove = new JButton ("<-");
      addRemovePanel.add (add);
      addRemovePanel.add (remove);
      remove.addActionListener (new RemoveFromImportsListener());
      add.addActionListener (new AddToImportsListener());

      JPanel leftPanel = new JPanel (new BorderLayout());
      fullTreeModel = new DefaultTreeModel (rootNodeFullTree);
      fullTree = new JTree (fullTreeModel);
      fullTree.setCellRenderer (new PackageTreeCellRenderer());
      JPanel fullTreePanel = makeLabeledPanel ("full tree", new JScrollPane (fullTree), SwingConstants.CENTER);
      leftPanel.add (fullTreePanel, BorderLayout.CENTER);
      JPanel leftButtonPanel = new JPanel();
      JButton importButton = new JButton ("import");
      JButton newButton = new JButton ("new");
      JButton removeButton = new JButton ("remove");
      newButton.addActionListener (new NewActionListener (this));
      removeButton.addActionListener (new RemoveActionListener (this));
      importButton.addActionListener (new ImportActionListener());
      leftButtonPanel.add (importButton);
      leftButtonPanel.add (newButton);
      leftButtonPanel.add (removeButton);
      leftPanel.add (leftButtonPanel, BorderLayout.SOUTH);

      JPanel rightPanel = new JPanel (new BorderLayout());
      JPanel rightComboPanel = new JPanel();
      fileBox = new JComboBox();
      fileBox.addItemListener (new FileBoxItemListener());
      rightComboPanel.add (fileBox);

      fileImportList = new JList (importListModel);
      fileImportList.setCellRenderer (new ImportListCellRenderer());
      rightPanel.add (rightComboPanel, BorderLayout.SOUTH);
      rightPanel.add (makeLabeledPanel ("imports", new JScrollPane (fileImportList), SwingConstants.CENTER), BorderLayout.CENTER);

      c.weightx = 1;
      c.weighty = 1;
      c.gridwidth = 4;
      c.fill = GridBagConstraints.BOTH;
      gridbag.setConstraints (leftPanel, c);
      mainPanel.add (leftPanel);

      c.weighty = 0;
      c.gridwidth = 1;
      c.fill = GridBagConstraints.HORIZONTAL;
      gridbag.setConstraints (addRemovePanel, c);
      mainPanel.add (addRemovePanel);

      c.weighty = 1;
      c.gridwidth = 4;
      c.fill = GridBagConstraints.BOTH;
      gridbag.setConstraints (rightPanel, c);
      mainPanel.add (rightPanel);

      return mainPanel;
   }


   /**
    * Sets the file attribute of the EditImportDialog object
    *
    * @param file  The new file value
    */
   public void setFile (UMLFile file)
   {
      if (file != null)
      {
         fileBox.setSelectedItem (file);
      }
   }


   /**
    * fillTree fills a JTree with all Packages and class which are in a UMLPackage. It will
    * recurse into subPackages.
    *
    * @param root  No description provided
    * @param pack  No description provided
    */
   private final void fillTree (DefaultMutableTreeNode root, UMLPackage pack)
   {
      Iterator iter = pack.iteratorOfDeclares();
      while (iter.hasNext())
      {
         UMLClass clazz = (UMLClass) iter.next();
         root.add (new DefaultMutableTreeNode (clazz));
      }

      iter = pack.iteratorOfPackages();
      while (iter.hasNext())
      {
         UMLPackage child = (UMLPackage) iter.next();
         DefaultMutableTreeNode newNode = new DefaultMutableTreeNode (child);
         root.add (newNode);
         fillTree (newNode, child);
      }
   }


   /**
    * This method takes the UMLPackage hierarchie and fills the tree and takes all known UMLFiles
    * and put them in the comboBox.
    */
   public void unparse()
   {
      fillTree (rootNodeFullTree, UMLProject.get().getRootPackage());
      fullTree.expandRow (0);

      Iterator iter = UMLProject.get().iteratorOfFiles();
      while (iter.hasNext())
      {
         UMLFile file = (UMLFile) iter.next();
         fileBox.addItem (file);
      }
   }


   /**
    * This method applies all simulated changes to the imports.
    */
   public void parse()
   {
      UMLFile file = (UMLFile) fileBox.getSelectedItem();

      if (file != null)
      {
         Iterator addIter = simAddHashSet.iterator();
         while (addIter.hasNext())
         {
            Object obj = addIter.next();

            if (obj instanceof UMLPackage)
            {
               file.addToImportedPackages ((UMLPackage) obj);
            }
            if (obj instanceof UMLClass)
            {
               file.addToImportedClasses ((UMLClass) obj);
            }
         }

         Iterator removeIter = simRemoveHashSet.iterator();
         while (removeIter.hasNext())
         {
            Object obj = removeIter.next();

            if (obj instanceof UMLPackage)
            {
               file.removeFromImportedPackages ((UMLPackage) obj);
            }
            if (obj instanceof UMLClass)
            {
               file.removeFromImportedClasses ((UMLClass) obj);
            }
         }
      }
   }


   /**
    * findPlace finds the place in the tree rooted on root for name. If it must create new
    * packages it will create them in the tree and in the UMLPackage.
    *
    * @param rootNode  No description provided
    * @param name      No description provided
    * @return          No description provided
    */
   public DefaultMutableTreeNode findPlace (DefaultMutableTreeNode rootNode, String name)
   {
      int index = name.indexOf ('.');

      if (index != -1)
      {
         // divide the name into first Package and rest
         String first = name.substring (0, index);
         String rest = name.substring (index + 1, name.length());
         //            if (log.isInfoEnabled()) log.info ("First: " + first);
         //            if (log.isInfoEnabled()) log.info ("Rest:  " + rest);

         // find subPackages.toString() == first
         Enumeration packages = rootNode.children();

         while (packages.hasMoreElements())
         {
            DefaultMutableTreeNode pack = (DefaultMutableTreeNode) packages.nextElement();

            // we only want packages
            if (pack.getUserObject() instanceof UMLClass)
            {
               continue;
            }

            UMLPackage umlPack = (UMLPackage) pack.getUserObject();

            if (umlPack.toString().equals (first))
            {
               //recurse
               return findPlace (pack, rest);
            }
         }

         // no subPackage found, create new one
         UMLPackage newPackage = new UMLPackage();
         newPackage.setName (first);
         UMLPackage rootPackage = (UMLPackage) rootNode.getUserObject();
         DefaultMutableTreeNode newNode = new DefaultMutableTreeNode (newPackage);

         rootNode.add (newNode);
         rootPackage.addToPackages (newPackage);
         return findPlace (newNode, rest);
      }
      else
      {
         return rootNode;
      }
   }


   /**
    * This method converts a name to a normal name. It replaces all / with . and kills .class.
    *
    * @param name  No description provided
    * @return      No description provided
    */
   private String convertToNormalName (String name)
   {
      // we must convert also the other separators because we must understand an archive which have been built on other
      // operating systems.
      name = name.replace (File.separatorChar, '.');
      name = name.replace ('/', '.');
      name = name.replace ('\\', '.');

      if (name.endsWith (".class"))
      {
         name = name.substring (0, name.length() - 6);
      }

      if (name.endsWith ("."))
      {
         name = name.substring (0, name.length() - 1);
      }

      return name;
   }


   /**
    * Access method for an one to n association.
    *
    * @param name     The object added.
    * @param isClass  The object added.
    */
   public void addToTreeAndPackages (String name, boolean isClass)
   {
      if (name.indexOf ("$") == -1)
      {
         name = convertToNormalName (name);

         if (log.isDebugEnabled())
         {
            log.debug ("adding: " + name);
         }

         DefaultMutableTreeNode placeNode = findPlace (rootNodeFullTree, name);
         UMLPackage place = (UMLPackage) placeNode.getUserObject();
         name = name.substring (name.lastIndexOf ('.') + 1, name.length());
         if (isClass)
         {
            if (!place.hasKeyInDeclares (name))
            {
               UMLClass newClass = new UMLClass (name);
               UMLStereotype referenceType = UMLStereotypeManager.get().
                  getFromStereotypes (UMLStereotypeManager.REFERENCE);

               newClass.setStereotype (referenceType, true);
               place.addToDeclares (newClass);
               placeNode.add (new DefaultMutableTreeNode (newClass));
            }
         }
         else
         {
            if (!place.hasKeyInPackages (name))
            {
               UMLPackage newPackage = new UMLPackage();
               newPackage.setName (name);
               place.addToPackages (newPackage);
               placeNode.add (new DefaultMutableTreeNode (newPackage));
            }
         }
      }
   }


   /**
    * This ItemListener updates the importList according to the selected file in the fileBox.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class FileBoxItemListener implements ItemListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param event  No description provided
       */
      public void itemStateChanged (ItemEvent event)
      {
         if (event.getStateChange() == ItemEvent.SELECTED)
         {
            UMLFile file = (UMLFile) event.getItem();

            // fill import Box
            importListModel.clear();

            // with classes
            Iterator iter = file.iteratorOfImportedClasses();
            while (iter.hasNext())
            {
               UMLClass clazz = (UMLClass) iter.next();
               importListModel.addElement (clazz);
            }

            // with packages
            iter = file.iteratorOfImportedPackages();
            while (iter.hasNext())
            {
               UMLPackage pack = (UMLPackage) iter.next();
               importListModel.addElement (pack);
            }

            // clear simSets
            simAddHashSet.clear();
            simRemoveHashSet.clear();
         }
      }
   }


   /**
    * This class simulates the adding the selected package or class to the imports box. After
    * adding and removing a class/package the state remains the same.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class AddToImportsListener implements ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param event  No description provided
       */
      public void actionPerformed (ActionEvent event)
      {
         TreePath paths[] = fullTree.getSelectionPaths();

         if (paths != null)
         {
            for (int i = 0; i < paths.length; i++)
            {
               Object obj = null;
               TreePath path = paths[i];
               DefaultMutableTreeNode node = (DefaultMutableTreeNode) path.getLastPathComponent();
               if (node != null)
               {
                  obj = node.getUserObject();
               }

               if (obj != null)
               {
                  if (simRemoveHashSet.contains (obj))
                  {
                     simRemoveHashSet.remove (obj);
                  }
                  else
                  {
                     simAddHashSet.add (obj);
                  }
                  if (!importListModel.contains (obj))
                  {
                     importListModel.addElement (obj);
                  }
               }

            }
         }
      }
   }


   /**
    * This class simulates the removing the selected package or class to the imports box. After
    * adding and removing a class/package the state remains the same.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class RemoveFromImportsListener implements ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param event  No description provided
       */
      public void actionPerformed (ActionEvent event)
      {
         Object obj[] = fileImportList.getSelectedValues();

         if (obj != null)
         {
            for (int i = 0; i < obj.length; i++)
            {
               Object tmp = obj[i];
               if (simAddHashSet.contains (tmp))
               {
                  simAddHashSet.remove (tmp);
               }
               else
               {
                  simRemoveHashSet.add (tmp);
               }
               importListModel.removeElement (tmp);
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class RemoveActionListener implements ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private Dialog dialog;


      /**
       * Constructor for class RemoveActionListener
       *
       * @param aDialog  No description provided
       */
      public RemoveActionListener (Dialog aDialog)
      {
         dialog = aDialog;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param remove  No description provided
       */
      public void removeAllSubPackages (UMLPackage remove)
      {
         Enumeration enumeration = remove.elementsOfAllChildren();

         while (enumeration.hasMoreElements())
         {
            UMLIncrement incr = (UMLIncrement) enumeration.nextElement();
            importListModel.removeElement (incr);

            if (incr instanceof UMLPackage)
            {
               removeAllSubPackages ((UMLPackage) incr);
            }
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param event  No description provided
       */
      public void actionPerformed (ActionEvent event)
      {
         DefaultMutableTreeNode node = (DefaultMutableTreeNode) fullTree.getLastSelectedPathComponent();
         if (node != null)
         {
            int rc;
            UMLIncrement incr = (UMLIncrement) node.getUserObject();
            String name = null;

            if (incr instanceof UMLPackage)
            {
               name =  ((UMLPackage) incr).getFullPackageName();
            }
            if (incr instanceof UMLClass)
            {
               name =  ((UMLClass) incr).getFullClassName();
            }

            rc = JOptionPane.showConfirmDialog (dialog, "Deleting this class/package may do harm to your project. Are you sure?",
               "Really delete " + name, JOptionPane.YES_NO_OPTION, JOptionPane.WARNING_MESSAGE);
            if (rc == JOptionPane.YES_OPTION)
            {
               // remove from tree
               node.removeFromParent();
               fullTreeModel.reload();

               // remove from importList
               if (incr instanceof UMLPackage)
               {
                  removeAllSubPackages ((UMLPackage) incr);
               }
               importListModel.removeElement (incr);

               // remove from UMLPackage structure
               incr.removeYou();

            }
         }
      }
   }


   /**
    * This actionListener is responsible for displaying the new Dialog.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class NewActionListener implements ActionListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private Dialog dialog;


      /**
       * Constructor for class NewActionListener
       *
       * @param aDialog  No description provided
       */
      public NewActionListener (Dialog aDialog)
      {
         dialog = aDialog;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void actionPerformed (ActionEvent e)
      {
         NewDialog newDialog = new NewDialog (dialog);
         newDialog.showCentered();
      }
   }


   /**
    * This dialog implements the possibility to create new packages or new classes. It will
    * add the new classes both to the tree and to the package hierarchie in fujaba.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class NewDialog extends FujabaDialog
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JTextField nameField;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JRadioButton packButton;
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private JRadioButton classButton;


      /**
       * Constructor for class NewDialog
       *
       * @param dialog  No description provided
       */
      public NewDialog (Dialog dialog)
      {
         super (dialog, "New", true);
         this.guiInit();
         this.unparse();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      private final void guiInit()
      {
         JPanel panel = new JPanel();
         panel.setLayout (new BorderLayout());
         panel.add (this.guiWorkPanel(), BorderLayout.CENTER);
         panel.add (this.guiPanelOkCancelHelp(), BorderLayout.SOUTH);
         this.getContentPane().add (panel);
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @return   No description provided
       */
      private final JPanel guiWorkPanel()
      {
         JPanel mainPanel = new JPanel (new GridLayout (2, 1));
         JPanel namePanel = new JPanel();
         JPanel groupPanel = new JPanel();

         JLabel nameLabel = new JLabel ("Name:");
         nameField = new JTextField (30);
         namePanel.add (nameLabel);
         namePanel.add (nameField);

         packButton = new JRadioButton ("Package");
         packButton.setSelected (true);
         classButton = new JRadioButton ("Class");
         ButtonGroup group = new ButtonGroup();
         group.add (packButton);
         group.add (classButton);
         groupPanel.add (packButton);
         groupPanel.add (classButton);

         mainPanel.add (namePanel);
         mainPanel.add (groupPanel);

         return mainPanel;
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void unparse()
      {
         // do nothing
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       */
      public void parse()
      {
         addToTreeAndPackages (nameField.getText(), classButton.isSelected());
         fullTreeModel.reload();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: lowende $
    * @version   $Revision: 1.36.2.3 $
    */
   private class ImportActionListener implements ActionListener
   {
      /**
       * Constructor for class ImportActionListener
       */
      public ImportActionListener()
      {
         super();
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param event  No description provided
       */
      public void actionPerformed (ActionEvent event)
      {
         FrameMain frame = UMLProject.get().getGui();

         JFileChooser fileChooser = frame.getFileChooser();

         fileChooser.setFileFilter (frame.getFilterJarFiles());
         fileChooser.setFileSelectionMode (JFileChooser.FILES_AND_DIRECTORIES);

         if (fileChooser.showOpenDialog (frame) == JFileChooser.APPROVE_OPTION)
         {
            File file = fileChooser.getSelectedFile();

            if (file.isDirectory())
            {
               importDirectory (file.getAbsoluteFile().toString().lastIndexOf (File.separatorChar) + 1, file);
            }
            else
            {
               importFile (file);
            }
            fullTreeModel.reload();
         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param file  No description provided
       */
      private void importFile (File file)
      {
         String fileName = file.getName();
         String extension = "";
         if (fileName.lastIndexOf ('.') != -1)
         {
            extension = fileName.substring (fileName.lastIndexOf ('.') + 1, fileName.length());
         }

         if (extension.equals ("jar"))
         {
            try
            {
               JarFile jar = new JarFile (file);
               Enumeration entries = jar.entries();

               while (entries.hasMoreElements())
               {
                  JarEntry entry = (JarEntry) entries.nextElement();

                  String name = entry.getName();

                  if (!name.startsWith ("META") &&  (name.endsWith (".class") || entry.isDirectory()))
                  {

                     if (entry.isDirectory())
                     {
                        EditImportDialog.this.addToTreeAndPackages (entry.getName(), false);
                     }
                     else
                     {
                        EditImportDialog.this.addToTreeAndPackages (entry.getName(), true);
                     }
                  }
               }
            }
            catch (IOException e)
            {
               if (log.isInfoEnabled())
               {
                  log.info ("Some problem occured while accessing the jar file");
               }
            }

         }
      }


      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param index  No description provided
       * @param file   No description provided
       */
      private void importDirectory (int index, File file)
      {
         int i;
         File[] files = file.listFiles();

         for (i = 0; i < files.length; i++)
         {
            File tmpFile = files[i];
            String fileName = tmpFile.getAbsoluteFile().toString();
            fileName = fileName.substring (index, fileName.length());

            if (!tmpFile.isDirectory())
            {
               if (fileName.endsWith (".class"))
               {
                  addToTreeAndPackages (fileName, true);
               }
            }
            else
            {
               if (!fileName.startsWith (".") && !tmpFile.getName().equals ("CVS"))
               {
                  addToTreeAndPackages (fileName, false);
                  importDirectory (index, tmpFile);
               }
            }
         }
      }
   }
}

/*
 * $Log: EditImportDialog.java,v $
 * Revision 1.36.2.3  2005/10/10 00:54:50  lowende
 * Two enumerationOf methods set to deprecated.
 * Two missing sizeOf methods added.
 *
 */
