/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.*;

import de.uni_paderborn.fujaba.gui.comp.FujabaDialog;
import de.uni_paderborn.fujaba.uml.*;


/**
 * EditCommentDialog is an editor for all UMLCommentarys. It is introduced for
 * the Fujaba life version to replace editing of comments with a text editor.
 *
 *
 * @author    $Author: fklar $
 * @version   $Revision: 1.13.2.1 $
 */
public class EditCommentDialog extends FujabaDialog implements TextEditAction.ClosableDialog, ActionListener
{
   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   JTextArea sourceEditor;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JButton switchCommentVisibilityButton;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JButton convertToTextButton;


   /**
    * Constructor for class EditCommentDialog
    *
    *
    * @param frame  No description provided
    */
   public EditCommentDialog (JFrame frame)
   {
      super (frame, "Comment editor", true);
      this.guiInit();
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public void guiInit()
   {
      JPanel mainPanel = new JPanel();

      GridBagLayout gridBag = new GridBagLayout();
      GridBagConstraints constraints = new GridBagConstraints();

      mainPanel.setLayout (gridBag);

      constraints.fill = GridBagConstraints.BOTH;
      constraints.gridwidth = GridBagConstraints.REMAINDER;
      constraints.anchor = GridBagConstraints.NORTH;
      constraints.weightx = 1.0;

      JLabel commentLabel = new JLabel ("Comment");
      constraints.insets = new Insets (0, 3, 0, 3);
      gridBag.setConstraints (commentLabel, constraints);
      mainPanel.add (commentLabel);

      sourceEditor = new JTextArea();
      sourceEditor.setToolTipText ("Edit Comment");
      // TODO: add mechanism that stores these values in preferences!
      sourceEditor.setColumns (55);
      sourceEditor.setRows (25);

      JScrollPane scrollPane = new JScrollPane (sourceEditor);
      constraints.weighty = 1.0;
      constraints.insets = new Insets (0, 3, 3, 3);
      gridBag.setConstraints (scrollPane, constraints);
      mainPanel.add (scrollPane);

      // some buttons...
      JPanel buttonPanel = new JPanel();
      buttonPanel.setLayout (new FlowLayout (FlowLayout.CENTER));
      mainPanel.add (buttonPanel);

      constraints.weightx = 0.5;
      constraints.weighty = 0.0;
      JButton textEditButton = new JButton ("Open in editor");
      textEditButton.addActionListener (new TextEditAction (this, sourceEditor));
      gridBag.setConstraints (textEditButton, constraints);
      buttonPanel.add (textEditButton);

      switchCommentVisibilityButton = new JButton ("switch visibility");
      switchCommentVisibilityButton.setActionCommand ("editor:switchVisibility");
      switchCommentVisibilityButton.addActionListener (this);
      switchCommentVisibilityButton.setToolTipText ("show/hide the UML commentary in the current diagram");
      gridBag.setConstraints (switchCommentVisibilityButton, constraints);
      buttonPanel.add (switchCommentVisibilityButton);
      // disable the button, only enable, if 'this.uMLCommentary' is non-null
      switchCommentVisibilityButton.setEnabled (false);

      convertToTextButton = new JButton ("comment -> text");
      convertToTextButton.setActionCommand ("editor:convertToText");
      convertToTextButton.addActionListener (this);
      convertToTextButton.setToolTipText ("convert java-style-commented text to uncommented text");
      gridBag.setConstraints (convertToTextButton, constraints);
      buttonPanel.add (convertToTextButton);

      Container container = this.getContentPane();
      container.add (mainPanel, BorderLayout.CENTER);
      JPanel panel = this.guiPanelOkCancelHelp();
      container.add (panel, BorderLayout.SOUTH);
      this.pack();
   }


   /**
    * The UMLCommentary that is edited in this EditCommentDialog object.
    *
    * <pre>
    * Note:
    * This attribute may be 'null', so make shure to perform a null-pointer-check,
    * before working on it in this class.
    * </pre>
    */
   private UMLCommentary uMLCommentary;


   /**
    * Get the UMLCommentary that is edited in this EditCommentDialog object.
    *
    *
    * @return   The UMLCommentary edited in this dialog, may be null.
    */
   public UMLCommentary getUMLCommentary()
   {
      return this.uMLCommentary;
   }


   /**
    * Was a new UMLCommentary created by the editor?
    */
   private boolean newUMLCommentaryCreated = false;


   /**
    * Tells whether this editor has created a new UMLCommentary.
    *
    * <pre>
    *
    * If a new UMLCommentary has been created, caller of this dialog
    * has to ashure, that its revComment (its parent-UMLIncrement) will be set!
    * </pre>
    *
    *
    * @return   True if a new UMLCommentary has been created.
    */
   public boolean isNewUMLCommentaryCreated()
   {
      return this.newUMLCommentaryCreated;
   }


   /**
    * Called when the dialog's Ok-Button is pressed.
    */
   public void parse()
   {
      if (this.uMLCommentary == null)
      {
         this.uMLCommentary = new UMLCommentary();
         this.newUMLCommentaryCreated = true;
      }

      this.uMLCommentary.setText (sourceEditor.getText());
      UMLProject.get().refreshDisplay();
   }


   /**
    * Called when the dialog is initialized.
    *
    *
    * @param value  The commentary that should be edited.
    */
   public void unparse (UMLCommentary value)
   {
      if (value == null)
      {
         this.uMLCommentary = new UMLCommentary();
         this.newUMLCommentaryCreated = true;

         // visibility of comment may NOT be switched, because
         // it is not finished... it has no revComment (parent), yet!
         switchCommentVisibilityButton.setEnabled (false);
      }
      else
      {
         this.uMLCommentary = value;

         // visibility of comment may be switched, because
         // it seems to be ready
         switchCommentVisibilityButton.setEnabled (true);
      }

      String commentStr = this.uMLCommentary.getText();

      if (commentStr != null)
      {
         sourceEditor.setText (commentStr);
      }
   }


   /**
    * called when sourceEditor content is shown in the TextEditor
    */
   public void hideWhileEditing()
   {
      hide();
   }


   /**
    * called when the user wants to save the content of the TextEditor
    */
   public void okPressed()
   {
      super.actionOkButton();
   }


   /**
    * this function is called if the cancel-button is pressed
    */
   public void actionCancelButton()
   {
      // if user canceled the dialog, we will
      // remove a created commentary
      if (isNewUMLCommentaryCreated())
      {
         this.uMLCommentary.removeYou();
         this.uMLCommentary = null;
      }

      super.actionCancelButton();
   }


   /*
    *  (non-Javadoc)
    *  @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
    */
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      if (e.getActionCommand().equals ("editor:switchVisibility"))
      {
         if (this.uMLCommentary != null)
         {
            // switch visibility of commentary
            this.uMLCommentary.setVisible (!this.uMLCommentary.isVisible());

            UMLProject.get().refreshDisplay();
         }
      }
      else if (e.getActionCommand().equals ("editor:convertToText"))
      {
         if (this.uMLCommentary != null)
         {
            sourceEditor.setText (UMLCommentary.getUncommentedText (sourceEditor.getText()));
         }
      }
   }
}

/*
 * $Log: EditCommentDialog.java,v $
 * Revision 1.13.2.1  2005/07/04 16:42:45  fklar
 * + adapted to modified UMLCommentary-API
 * + added features to dialog:
 * 1. "show/hide comment"-button
 * 2. "convert java-comment to text"-button
 *
 */
