/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.gui;

import java.awt.*;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.uml.UMLActivityDiagram;
import de.uni_paderborn.fujaba.views.ViewDiagram;
import de.upb.tools.fca.FEmptyIterator;


/**
 * Dialog for the activity panels
 *
 * <h2>Associations</h2>
 *
 * <pre>
 *                n                panels                0..1
 * ActivityPanel --------------------------------------------- EditActivityDialog
 *                activityPanels   {sorted,}   activityDialog
 * </pre>
 *
 * @author    $Author: cschneid $
 * @version   $Revision: 1.33 $
 */
public class EditActivityDialog extends AbstractDialog implements TextEditAction.ClosableDialog
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   JPanel propertiesPanel;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   CardLayout cardLayout;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ButtonGroup buttonGroup;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private ASGElement asgElement;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   ActivityPanel currentActivityPanel;


   /**
    * Get the asgElement attribute of the EditActivityDialog object
    *
    * @return   The asgElement value
    */
   public ASGElement getAsgElement()
   {
      return asgElement;
   }


   /**
    * Constructor for class EditActivityDialog
    *
    * @param frame       No description provided
    * @param asgElement  No description provided
    */
   public EditActivityDialog (JFrame frame, ASGElement asgElement)
   {
      super (frame, true);

      this.asgElement = asgElement;
      addToActivityPanels (new StartActivityPanel (asgElement));
      addToActivityPanels (new StartActivityPanelForMethod (asgElement));
      addToActivityPanels (new StateActivityPanel (asgElement));
      addToActivityPanels (new StoryActivityPanel (asgElement));
      addToActivityPanels (new StatementActivityPanel (asgElement));
      addToActivityPanels (new NopActivityPanel (asgElement));
      addToActivityPanels (new StopActivityPanel (asgElement));

      createUserInterface();
      initialize();
   } // constructor


   /**
    * Contains the complete user interface of the dialog, except "Ok" and "Cancel" buttons.
    * Called by createUserInterface ().
    *
    * @return   JPanel that is positioned in the middle of the dialog.
    * @see      de.uni_paderborn.fujaba.gui.AbstractDialog#createUserInterface
    */
   protected JPanel createMainPanel()
   {
      JPanel panel = new JPanel();
      panel.setLayout (new BorderLayout());

      if (asgElement instanceof UMLActivityDiagram ||
          (asgElement instanceof ViewDiagram &&  ((ViewDiagram) asgElement).getOriginalDiagram() instanceof UMLActivityDiagram))
      {
         setTitle ("New Activity");

         JPanel buttonPanel = new JPanel();
         buttonPanel.setLayout (new BoxLayout (buttonPanel, BoxLayout.Y_AXIS));
         buttonPanel.setBorder (new TitledBorder ("Activity"));
         panel.add (buttonPanel, BorderLayout.WEST);

         cardLayout = new CardLayout();
         propertiesPanel = new JPanel();
         propertiesPanel.setLayout (cardLayout);
         propertiesPanel.setBorder (new TitledBorder ("Properties"));
         panel.add (propertiesPanel, BorderLayout.CENTER);

         buttonGroup = new ButtonGroup();

         ActivityPanel tmpActivityPanel;
         boolean firstSelected = false;

         Iterator iter = iteratorOfActivityPanels();
         while (iter.hasNext())
         {
            tmpActivityPanel = (ActivityPanel) iter.next();

            JRadioButton button = new JRadioButton (tmpActivityPanel.getName());
            button.addChangeListener (new RadioButtonListener());
            cardLayout.addLayoutComponent (propertiesPanel, button.getText());

            buttonGroup.add (button);
            buttonPanel.add (button);

            propertiesPanel.add (tmpActivityPanel, tmpActivityPanel.getName());

            if (!firstSelected && button.isEnabled())
            {
               button.setSelected (true);
               firstSelected = true;
               cardLayout.show (propertiesPanel, button.getText());
               currentActivityPanel = tmpActivityPanel;
            }

         }
      }
      else
      {
         ActivityPanel tmpActivityPanel = null;
         Iterator iter = iteratorOfActivityPanels();
         boolean found = false;
         while (iter.hasNext() && !found)
         {
            tmpActivityPanel = (ActivityPanel) iter.next();
            if (tmpActivityPanel.isResponsible())
            {
               found = true;
            }
         }

         if (tmpActivityPanel != null && found)
         {
            setTitle ("Edit " + tmpActivityPanel.getName() + " Activity");
            panel.setBorder (new TitledBorder ("Properties"));
            panel.add (tmpActivityPanel);
            currentActivityPanel = tmpActivityPanel;
         }
         else
         {
            setTitle ("Edit Activity");
            panel.setBorder (new TitledBorder ("Error"));
            panel.setLayout (new FlowLayout (FlowLayout.CENTER));
            panel.add (new JLabel ("Item not editable!"));
            currentActivityPanel = null;
         }
      }
      return panel;
   } // createMainPanel


   /**
    * Read the parameters which can be modified and put all information into the dialogs user
    * interface. Called by initialize ().
    *
    * @see   de.uni_paderborn.fujaba.gui.AbstractDialog#initialize
    */
   protected void unparse()
   {
      Iterator iter = iteratorOfActivityPanels();
      while (iter.hasNext())
      {
         ActivityPanel tmpActivityPanel = (ActivityPanel) iter.next();
         if (tmpActivityPanel.isResponsible())
         {
            tmpActivityPanel.unparse();
         }
      }

      if (asgElement instanceof UMLActivityDiagram ||
          (asgElement instanceof ViewDiagram &&  ((ViewDiagram) asgElement).getOriginalDiagram() instanceof UMLActivityDiagram))
      {
         getOkButton().setText ("Add");
      }
      else
      {
         getOkButton().setText ("Modify");
      }
   } // unparse


   /**
    * Get all information from the user interface and put them back to the parameters that
    * can be modified by this dialog. Called by okPressed ().
    *
    * @see   de.uni_paderborn.fujaba.gui.AbstractDialog#okPressed
    */
   protected void parse()
   {
      if (currentActivityPanel != null)
      {
         //FujabaLog is called from the current activity panel
         asgElement = currentActivityPanel.okPressed();
      }
   } // parse


   /**
    * okPressed and cancelPressed from AbstractDialog, overwritten for FujabaLog
    */
   public void okPressed()
   {
      super.okPressed();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void cancelPressed()
   {
      super.cancelPressed();
   }


   /**
    * <pre>
    *                n                panels                0..1
    * ActivityPanel --------------------------------------------- EditActivityDialog
    *                activityPanels   {sorted,}   activityDialog
    * </pre>
    */
   private LinkedList activityPanels;


   /**
    * Access method for an one to n association.
    *
    * @param elem  The object added.
    */
   public void addToActivityPanels (ActivityPanel elem)
   {
      if ( (elem != null) && !this.hasInActivityPanels (elem))
      {
         if (this.activityPanels == null)
         {
            // Create a new container for the activityPanels association.
            this.activityPanels = new LinkedList();
         }

         this.activityPanels.add (elem);
         elem.setActivityDialog (this);
      }
   } // addToActivityPanels


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    * @return      No description provided
    */
   public boolean hasInActivityPanels (ActivityPanel elem)
   {
      if (this.activityPanels == null)
      {
         return false;
      }
      else
      {
         return this.activityPanels.contains (elem);
      }
   } // hasInActivityPanels


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfActivityPanels()
   {
      return  ( (this.activityPanels == null)
         ? FEmptyIterator.get()
         : this.activityPanels.iterator());
   } // iteratorOfActivityPanels


   /**
    * Get the fromActivityPanels attribute of the EditActivityDialog object
    *
    * @param name  No description provided
    * @return      The fromActivityPanels value
    */
   public ActivityPanel getFromActivityPanels (String name)
   {
      ActivityPanel searchedPanel = null;

      Iterator iter = iteratorOfActivityPanels();
      while (iter.hasNext() &&  (searchedPanel == null))
      {
         ActivityPanel currentPanel = (ActivityPanel) iter.next();
         if (currentPanel.getName().equals (name))
         {
            searchedPanel = currentPanel;
         }
      }

      return searchedPanel;
   } // getFromActivityPanels


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param elem  No description provided
    */
   public void removeFromActivityPanels (ActivityPanel elem)
   {
      if (this.hasInActivityPanels (elem))
      {
         this.activityPanels.remove (elem);
         elem.setActivityDialog (null);
      }
   } // removeFromActivityPanels


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfActivityPanels()
   {
      if (this.activityPanels == null)
      {
         return 0;
      }
      else
      {
         return this.activityPanels.size();
      }
   } // sizeOfActivityPanels


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromActivityPanels()
   {
      ActivityPanel tmpActivityPanel;
      Iterator iter = iteratorOfActivityPanels();

      while (iter.hasNext())
      {
         tmpActivityPanel = (ActivityPanel) iter.next();
         iter.remove();
         tmpActivityPanel.setActivityDialog (null);
      }
   } // removeAllFromActivityPanels


   // ===================================== inner classes ====================================

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: cschneid $
    * @version   $Revision: 1.33 $
    */
   private class RadioButtonListener implements ChangeListener
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param e  No description provided
       */
      public void stateChanged (ChangeEvent e)
      {
         JRadioButton currentButton = (JRadioButton) e.getSource();
         if (currentButton.isSelected())
         {
            cardLayout.show (propertiesPanel, currentButton.getText());
            currentActivityPanel = getFromActivityPanels (currentButton.getText());
         }
      } // stateChanged

   } // RadioButtonListener


   /**
    * only to implement ClosableDialog interface (not used)
    */
   public void hideWhileEditing()
   {
      //don't hide
   }
}

/*
 * $Log: EditActivityDialog.java,v $
 * Revision 1.33  2005/02/08 15:46:21  cschneid
 * path search bugfix, method signature can be edited in diagram
 *
 */
