/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.update;

import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Method;
import java.util.EventListener;

import javax.swing.*;

import de.uni_paderborn.fujaba.fsa.FSADerivePropertyMethodsCache;
import de.uni_paderborn.fujaba.fsa.FSATextComponent;
import de.uni_paderborn.fujaba.fsa.swing.JTextEditor;


/**
 * Update between a logic property and the text/document of a TextComponent
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.15 $
 */
public class TextComponentUpdater
    extends LogicAndFsaUpdater implements FocusListener, PropertyChangeListener
{
   /**
    * Constructor for class TextComponentUpdater
    */
   public TextComponentUpdater()
   {
      super();
      super.setFsaAttrName ("text");
   }


   /**
    * translate whatever logic value to String
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public Object translateLogicToFsa (Object value)
   {
      if (value == null)
      {
         return "";
      }
      else
      {
         return value.toString();
      }
   }


   /**
    * Sets the fsaAttrName attribute of the TextComponentUpdater object
    *
    * @param name  The new fsaAttrName value
    * @return      No description provided
    */
   public boolean setFsaAttrName (String name)
   {
      if ("text".equals (name))
      {
         return true;
      }
      else
      {
         throw new UnsupportedOperationException ("FsaAttrName cannot be changed from " +
            "\"text\" in TextComponentUpdater");
      }
   }


   /**
    * Access method for an one to n association.
    */
   protected void addListener()
   {
      if (! (getLogicObject() == null || getLogicAttrName() == null ||
         getFsaObject() == null || getFsaAttrName() == null))
      {
         if (getLogicListener() == null)
         {
            setLogicListener (this);
         }
         ListenerHelper.get().addPropertyChangeListener (getLogicObject(), getLogicAttrName(), (PropertyChangeListener) getLogicListener());

         if (getFsaListener() == null)
         {
            setFsaListener (this);
         }
         getFsaObject().addPropertyChangeListener ("document", (PropertyChangeListener) getFsaListener());

         JTextEditor editor = (JTextEditor) getFsaObject().getJComponent();
         JEditorPane comp = editor.getEditorPane();
         comp.addFocusListener ((FocusListener) getFsaListener());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   protected void removeListener()
   {
      if (! (getLogicObject() == null || getLogicAttrName() == null ||
         getFsaObject() == null || getFsaAttrName() == null))
      {
         if (getLogicListener() != null)
         {
            ListenerHelper.get().removePropertyChangeListener (getLogicObject(), getLogicAttrName(), (PropertyChangeListener) getLogicListener());
         }

         EventListener listener = getFsaListener();
         if (listener != null)
         {
            getFsaObject().removePropertyChangeListener ("document", (PropertyChangeListener) listener);

            JTextEditor editor = (JTextEditor) getFsaObject().getJComponent();
            JEditorPane comp = editor.getEditorPane();
            comp.removeFocusListener ((FocusListener) listener);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void initialize()
   {
      super.initialize();
      focused =  (getFsaObject() != null && getFsaObject().getJComponent() != null &&
         getFsaObject().getJComponent().hasFocus());
   }


   /**
    * Sets the logicValue attribute of the TextComponentUpdater object
    *
    * @param value  The new logicValue value
    */
   public void setLogicValue (Object value)
   {
      Method[] methods = FSADerivePropertyMethodsCache.get()
         .getPropertyMethods (getLogicAttrName(), getLogicObject());

      Method setMethod = methods[1];

      try
      {
         setMethod.invoke (getLogicObject(), new Object[]
            {
            value
            }
            );
      }
      catch (Exception e)
      {
         throw new RuntimeException ("Exception in " + this + ".setLogicValue: " + e.getMessage());
      }
   }


   /**
    * Get the logicValue attribute of the TextComponentUpdater object
    *
    * @return   The logicValue value
    */
   public Object getLogicValue()
   {
      Method[] methods = FSADerivePropertyMethodsCache.get()
         .getPropertyMethods (getLogicAttrName(), getLogicObject());

      Method getMethod = methods[0];

      Object result;
      try
      {
         result = getMethod.invoke (getLogicObject(), null);
      }
      catch (Exception e)
      {
         throw new RuntimeException ("Exception in " + this + ".getLogicValue: " + e.getMessage());
      }
      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean focused = false;


   /**
    * Get the focused attribute of the TextComponentUpdater object
    *
    * @return   The focused value
    */
   public boolean isFocused()
   {
      return focused;
   }


   /**
    * Sets the focused attribute of the TextComponentUpdater object
    *
    * @param focused  The new focused value
    */
   void setFocused (boolean focused)
   {
      this.focused = focused;
   }
   // ----------------------------- Listener Implementation ------------------------

   /**
    * Listens for changes to the Document and updates the contentsChanged property
    *
    * @param e  No description provided
    */
   public void propertyChange (PropertyChangeEvent e)
   {
      if (e.getSource() == getLogicObject())
      {
         Object value = e.getNewValue();
         String text = (String) getTranslator().translateLogicToFsa (value);

          ((FSATextComponent) getFsaObject()).setText (text);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void focusGained (FocusEvent e)
   {
      focused = true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void focusLost (FocusEvent e)
   {
      focused = false;
      if (getFsaObject() != null)
      {
         setLogicValue (getTranslator().translateFsaToLogic (
             ((FSATextComponent) getFsaObject()).getText()));
      }
   }
}

/*
 * $Log: TextComponentUpdater.java,v $
 * Revision 1.15  2004/11/03 10:18:02  lowende
 * Javadoc warnings removed.
 *
 */
