/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.fsa.swing.border;

import java.awt.*;

import de.uni_paderborn.fujaba.fsa.swing.LineStyle;


/**
 * No comment provided by developer, please add a comment to improve documentation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.14 $
 */
public class OvalBorder extends AbstractColorBorder
{
   /**
    * Constructor for class OvalBorder
    */
   public OvalBorder()
   {
      super();
   }


   /**
    * Creates a new OvalBorder with the given parameters.
    *
    * @param opaque  if set to true, the inner area of the oval will be cleared while drawing
    * @param stroke  stroke to use while rendering the border. Can be used to increase the
    *      border thickness.
    */
   public OvalBorder (boolean opaque, Stroke stroke)
   {
      super();
      setBorderOpaque (opaque);
      setStroke (stroke);
   }


   /**
    * Creates a new OvalBorder with the given parameters.
    *
    * @param opaque  if set to true, the inner area of the oval will be cleared while drawing
    */
   public OvalBorder (boolean opaque)
   {
      super();
      setBorderOpaque (opaque);
   }


   /**
    * Get the borderInsets attribute of the OvalBorder object
    *
    * @param c  No description provided
    * @return   The borderInsets value
    */
   public Insets getBorderInsets (Component c)
   {
      return getBorderInsets (c, null);
   }


   /**
    * Get the borderInsets attribute of the OvalBorder object
    *
    * @param c       No description provided
    * @param insets  No description provided
    * @return        The borderInsets value
    */
   public Insets getBorderInsets (Component c, Insets insets)
   {
      Dimension size = c.getSize();

      int w =  ((int) size.getWidth() / 2) - 1;
      int h =  ((int) size.getHeight() / 2) - 1;

      // 1.4142135624 = Math.sqrt(2)
      int xradius =  (new Double (w * 1.4142135624)).intValue();
      int yradius =  (new Double (h * 1.4142135624)).intValue();

      int insetTop = yradius - h +  (getThickness() - 1);
      int insetBottom = insetTop;
      int insetLeft = xradius - w +  (getThickness() - 1);
      int insetRight = insetLeft;

      if (doubleOval)
      {
         insetTop += 10 +  (getThickness() - 1);
         insetRight += 10 +  (getThickness() - 1);
      }

      if (insets == null)
      {
         insets = new Insets (insetTop, insetLeft, insetBottom, insetRight);
      }
      else
      {
         insets.top = insetTop;
         insets.left = insetLeft;
         insets.bottom = insetBottom;
         insets.right = insetRight;
      }

      return insets;
   }



   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean opaqueFlag;


   /**
    * Sets the borderOpaque attribute of the OvalBorder object
    *
    * @param opaque  The new borderOpaque value
    */
   public void setBorderOpaque (boolean opaque)
   {
      opaqueFlag = opaque;
   }


   /**
    * Get the borderOpaque attribute of the OvalBorder object
    *
    * @return   The borderOpaque value
    */
   public boolean isBorderOpaque()
   {
      return opaqueFlag;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Stroke stroke = null;


   /**
    * Get the stroke attribute of the OvalBorder object
    *
    * @return   The stroke value
    */
   public Stroke getStroke()
   {
      return stroke;
   }


   /**
    * Sets the stroke attribute of the OvalBorder object
    *
    * @param stroke  The new stroke value
    */
   public void setStroke (Stroke stroke)
   {
      this.stroke = stroke;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean dashedFlag;


   /**
    * Sets the borderDashed attribute of the OvalBorder object
    *
    * @param dashed  The new borderDashed value
    */
   public void setBorderDashed (boolean dashed)
   {
      dashedFlag = dashed;

      if (dashed)
      {
         BasicStroke defaultStroke = LineStyle.DASHED;
         BasicStroke stroke = new BasicStroke (thickness, defaultStroke.getEndCap(), defaultStroke.getLineJoin(), defaultStroke.getMiterLimit(), defaultStroke.getDashArray(), defaultStroke.getDashPhase());
         setStroke (stroke);
      }
      else
      {
         setStroke (new BasicStroke (thickness));
      }
   }


   /**
    * Get the borderDashed attribute of the OvalBorder object
    *
    * @return   The borderDashed value
    */
   public boolean isBorderDashed()
   {
      return dashedFlag;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int thickness = 1;


   /**
    * Sets the thickness attribute of the OvalBorder object
    *
    * @param newThickness  The new thickness value
    */
   public void setThickness (int newThickness)
   {
      thickness = newThickness;

      if (thickness < 0)
      {
         thickness = 0;
      }

      BasicStroke stroke;
      if (getStroke() != null)
      {
         BasicStroke defaultStroke = (BasicStroke) getStroke();
         stroke = new BasicStroke (thickness, defaultStroke.getEndCap(), defaultStroke.getLineJoin(), defaultStroke.getMiterLimit(), defaultStroke.getDashArray(), defaultStroke.getDashPhase());
      }
      else
      {
         stroke = new BasicStroke (thickness);
      }
      setStroke (stroke);
   }


   /**
    * Get the thickness attribute of the OvalBorder object
    *
    * @return   The thickness value
    */
   public int getThickness()
   {
      return thickness;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private boolean doubleOval = false;


   /**
    * Sets the doubleOval attribute of the OvalBorder object
    *
    * @param value  The new doubleOval value
    */
   public void setDoubleOval (boolean value)
   {
      this.doubleOval = value;
   }


   /**
    * This method draws an oval border around the component, filling the oval inside depending
    * on {@link #isBorderOpaque}.
    *
    * @param c       No description provided
    * @param g       No description provided
    * @param x       No description provided
    * @param y       No description provided
    * @param width   No description provided
    * @param height  No description provided
    */
   public void paintBorder (Component c,
                            Graphics g,
                            int x,
                            int y,
                            int width,
                            int height)
   {
      Color oldColor = g.getColor();

      if (isBorderOpaque())
      {
         g.setColor (c.getBackground());
         g.fillArc (x, y, width - 1, height - 1, 0, 360);
      }

      Stroke oldStroke = null;

      if (g instanceof Graphics2D)
      {
         Graphics2D g2d = (Graphics2D) g;

         oldStroke = g2d.getStroke();

         if (getStroke() != null)
         {
            g2d.setStroke (getStroke());
         }
      }

      g.setColor (getBorderColor());

      int thicknessOffset = thickness / 2;

      if (doubleOval)
      {
         g.drawArc (x + thicknessOffset + 10, y + thicknessOffset, width - 1 - thicknessOffset - 10, height - 1 - thicknessOffset - 10, 0, 360);
         g.setColor (c.getBackground());
         g.fillArc (x + thicknessOffset, y + thicknessOffset + 10, width - 1 - thicknessOffset - 10, height - 1 - thicknessOffset - 10, 0, 360);
         g.setColor (getBorderColor());
         g.drawArc (x + thicknessOffset, y + thicknessOffset + 10, width - 1 - thicknessOffset - 10, height - 1 - thicknessOffset - 10, 0, 360);
      }
      else
      {
         g.drawArc (x + thicknessOffset, y + thicknessOffset, width - 1 - thicknessOffset, height - 1 - thicknessOffset, 0, 360);
      }

      if (g instanceof Graphics2D)
      {
          ((Graphics2D) g).setStroke (oldStroke);
      }

      g.setColor (oldColor);
   }
}

/*
 * $Log: OvalBorder.java,v $
 * Revision 1.14  2004/11/03 10:18:00  lowende
 * Javadoc warnings removed.
 *
 */
